<?php
// filepath: c:\stylescore2025\htdocs\v2\admin\heat_management_api.php
include '../includes/auth.php';
include '../includes/db.php';

header('Content-Type: application/json');

try {
    $event_id = $_POST['event_id'] ?? $_GET['event_id'] ?? '';
    $action = $_POST['action'] ?? $_GET['action'] ?? '';
    
    if (!$event_id) {
        throw new Exception('Event ID is required');
    }
    
    switch ($action) {
        case 'get_data':
            echo json_encode(getHeatManagementData($pdo, $event_id));
            break;
            
        case 'get_heats':
            // Simple endpoint to get just the heats list
            echo json_encode(getHeats($pdo, $event_id));
            break;
            
        case 'update_settings':
            echo json_encode(updateHeatSettings($pdo, $event_id, $_POST));
            break;
            
        case 'update_heat_status':
            $heat_number = $_POST['heat_number'] ?? '';
            $status = $_POST['status'] ?? '';
            echo json_encode(updateHeatStatus($pdo, $event_id, $heat_number, $status));
            break;
            
        case 'get_participants':
            $heat_number = $_GET['heat_number'] ?? '';
            echo json_encode(getHeatParticipants($pdo, $event_id, $heat_number));
            break;
            
        case 'get_queue':
            $heat_number = $_GET['heat_number'] ?? '';
            $mode = $_GET['mode'] ?? 'start_list';
            
            // Debug logging
            error_log("get_queue API called - event_id: $event_id, heat_number: $heat_number, mode: $mode");
            
            $result = getParticipantQueue($pdo, $event_id, $heat_number, $mode);
            
            // Debug logging
            error_log("get_queue result - queue count: " . count($result['queue'] ?? []));
            
            echo json_encode($result);
            break;
            
        case 'update_participant_status':
            echo json_encode(updateParticipantStatus($pdo, $event_id, $_POST));
            break;
            
        case 'deactivate_all':
            echo json_encode(deactivateAllHeats($pdo, $event_id));
            break;
            
        case 'get_judge_status':
            $heat_number = $_GET['heat_number'] ?? '';
            $run_number = $_GET['run_number'] ?? 1;
            echo json_encode(getJudgeStatus($pdo, $event_id, $heat_number, $run_number));
            break;
            
        case 'get_heat_run_scores':
            $heat_number = $_GET['heat_number'] ?? '';
            $run_number = $_GET['run_number'] ?? 1;
            echo json_encode(getHeatRunScores($pdo, $event_id, $heat_number, $run_number));
            break;
            
        case 'get_participant_runs':
            $heat_number = $_GET['heat_number'] ?? '';
            echo json_encode(getParticipantRuns($pdo, $event_id, $heat_number));
            break;
            
        case 'swap_positions':
            $participant1_id = $_POST['participant1_id'] ?? '';
            $participant2_id = $_POST['participant2_id'] ?? '';
            $participant1_order = $_POST['participant1_order'] ?? '';
            $participant2_order = $_POST['participant2_order'] ?? '';
            echo json_encode(swapParticipantPositions($pdo, $participant1_id, $participant2_id, $participant1_order, $participant2_order));
            break;
            
        case 'get_participant_cards':
            $heat_number = $_GET['heat_number'] ?? '';
            $layout = $_GET['layout'] ?? 'full'; // 'full', 'compact', 'controls', 'info'
            $mode = $_GET['mode'] ?? 'sort_order';
            echo json_encode(getParticipantCards($pdo, $event_id, $heat_number, $layout, $mode));
            break;
            
        case 'update_heat':
            $assignment_id = $_POST['assignment_id'] ?? '';
            $new_heat = $_POST['new_heat'] ?? '';
            echo json_encode(updateParticipantHeat($pdo, $assignment_id, $new_heat, $event_id));
            break;
            
        case 'update_order':
            $heat_number = $_POST['heat_number'] ?? '';
            $order = json_decode($_POST['order'] ?? '[]', true);
            echo json_encode(updateHeatOrder($pdo, $event_id, $heat_number, $order));
            break;
            
        case 'delete_assignment':
            $assignment_id = $_POST['assignment_id'] ?? '';
            echo json_encode(deleteParticipantAssignment($pdo, $assignment_id, $event_id));
            break;
            
        default:
            throw new Exception('Invalid action');
    }
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['error' => $e->getMessage()]);
}

function getHeatManagementData($pdo, $event_id) {
    // Get event info
    $event_stmt = $pdo->prepare("SELECT name, heats_total FROM events WHERE id = ?");
    $event_stmt->execute([$event_id]);
    $event = $event_stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$event) {
        throw new Exception('Event not found');
    }
    
    // Get heat settings - handle missing bib_performing column gracefully
    try {
        $heats_stmt = $pdo->prepare("
            SELECT heat_number, heat_name, runs_count, is_active, active_run, 
                   bib_on_start, bib_latest_on_run, bib_performing,
                   time_start, categories, scoring_type
            FROM event_heat_settings 
            WHERE event_id = ? 
            ORDER BY heat_number
        ");
        $heats_stmt->execute([$event_id]);
        $heats = $heats_stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Fallback if bib_performing column doesn't exist
        $heats_stmt = $pdo->prepare("
            SELECT heat_number, heat_name, runs_count, is_active, active_run, 
                   bib_on_start, bib_latest_on_run, NULL as bib_performing,
                   time_start, categories, scoring_type
            FROM event_heat_settings 
            WHERE event_id = ? 
            ORDER BY heat_number
        ");
        $heats_stmt->execute([$event_id]);
        $heats = $heats_stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Get participant counts per heat
    $participant_counts_stmt = $pdo->prepare("
        SELECT heat_number, COUNT(*) as count 
        FROM event_participants 
        WHERE event_id = ? 
        GROUP BY heat_number
    ");
    $participant_counts_stmt->execute([$event_id]);
    $participant_counts = [];
    foreach ($participant_counts_stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $participant_counts[$row['heat_number']] = $row['count'];
    }
    
    // Process heats
    foreach ($heats as &$heat) {
        $heat['categories'] = json_decode($heat['categories'] ?: '[]', true);
        $heat['participant_count'] = $participant_counts[$heat['heat_number']] ?? 0;
        $heat['is_active'] = (bool)$heat['is_active'];
    }
    
    // Find current active heat
    $active_heat = null;
    foreach ($heats as $heat) {
        if ($heat['is_active']) {
            $active_heat = $heat;
            break;
        }
    }
    
    return [
        'event' => $event,
        'heats' => $heats,
        'active_heat' => $active_heat,
        'total_heats' => $event['heats_total']
    ];
}

function getHeats($pdo, $event_id) {
    // Simple function to get just the heats list
    try {
        $heats_stmt = $pdo->prepare("
            SELECT heat_number, heat_name, runs_count, is_active, active_run, 
                   time_start, categories, scoring_type
            FROM event_heat_settings 
            WHERE event_id = ? 
            ORDER BY heat_number
        ");
        $heats_stmt->execute([$event_id]);
        $heats = $heats_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Process heats
        foreach ($heats as &$heat) {
            $heat['categories'] = json_decode($heat['categories'] ?: '[]', true);
            $heat['is_active'] = (bool)$heat['is_active'];
        }
        
        return [
            'success' => true,
            'heats' => $heats
        ];
    } catch (Exception $e) {
        return [
            'success' => false,
            'error' => $e->getMessage()
        ];
    }
}

function updateHeatSettings($pdo, $event_id, $data) {
    $active_heat = $data['active_heat'] ?? null;
    $active_run = intval($data['active_run'] ?? 1);
    $bib_on_start = $data['bib_on_start'] ?? null;
    $bib_latest_on_run = $data['bib_latest_on_run'] ?? null;
    $bib_performing = $data['bib_performing'] ?? null;
    
    // Start transaction
    $pdo->beginTransaction();
    
    try {
        // First, deactivate all heats
        $deactivate_stmt = $pdo->prepare("
            UPDATE event_heat_settings 
            SET is_active = 0 
            WHERE event_id = ?
        ");
        $deactivate_stmt->execute([$event_id]);
        
        if ($active_heat) {
            // Get current settings for the heat - handle missing bib_performing column
            try {
                $current_stmt = $pdo->prepare("
                    SELECT bib_on_start, bib_latest_on_run, bib_performing 
                    FROM event_heat_settings 
                    WHERE event_id = ? AND heat_number = ?
                ");
                $current_stmt->execute([$event_id, $active_heat]);
                $current = $current_stmt->fetch(PDO::FETCH_ASSOC);
            } catch (PDOException $e) {
                // Fallback if bib_performing column doesn't exist
                $current_stmt = $pdo->prepare("
                    SELECT bib_on_start, bib_latest_on_run, NULL as bib_performing 
                    FROM event_heat_settings 
                    WHERE event_id = ? AND heat_number = ?
                ");
                $current_stmt->execute([$event_id, $active_heat]);
                $current = $current_stmt->fetch(PDO::FETCH_ASSOC);
            }
            $current = $current_stmt->fetch(PDO::FETCH_ASSOC);
            
            // Handle BIB on start change
            $new_bib_latest = $bib_latest_on_run;
            if ($bib_on_start && $current && $current['bib_on_start'] && $current['bib_on_start'] != $bib_on_start) {
                // Previous BIB on start becomes latest on run
                $new_bib_latest = $current['bib_on_start'];
            }
            
            // Activate the selected heat and update settings
            try {
                $activate_stmt = $pdo->prepare("
                    UPDATE event_heat_settings 
                    SET is_active = 1, active_run = ?, bib_on_start = ?, bib_latest_on_run = ?, bib_performing = ?
                    WHERE event_id = ? AND heat_number = ?
                ");
                $activate_stmt->execute([
                    $active_run,
                    $bib_on_start ?: null,
                    $new_bib_latest ?: null,
                    $bib_performing ?: null,
                    $event_id,
                    $active_heat
                ]);
            } catch (PDOException $e) {
                // Fallback if bib_performing column doesn't exist
                $activate_stmt = $pdo->prepare("
                    UPDATE event_heat_settings 
                    SET is_active = 1, active_run = ?, bib_on_start = ?, bib_latest_on_run = ?
                    WHERE event_id = ? AND heat_number = ?
                ");
                $activate_stmt->execute([
                    $active_run,
                    $bib_on_start ?: null,
                    $new_bib_latest ?: null,
                    $event_id,
                    $active_heat
                ]);
            }
        }
        
        $pdo->commit();
        
        return [
            'success' => true,
            'message' => $active_heat ? "Heat $active_heat activated with Run $active_run" : 'All heats deactivated',
            'active_heat' => $active_heat,
            'active_run' => $active_run
        ];
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}

function getHeatParticipants($pdo, $event_id, $heat_number) {
    if (!$heat_number) {
        return ['participants' => []];
    }
    
    $participants_stmt = $pdo->prepare("
        SELECT 
            ep.id as event_participant_id,
            ep.bib_number,
            ep.heat_number,
            ep.updated_at as ep_updated_at,
            p.id as participant_id,
            p.name,
            p.first_name,
            p.last_name,
            p.country,
            p.date_of_birth,
            p.bib,
            p.gender,
            p.fis_code,
            p.licence_code,
            p.licence_type,
            p.club,
            p.category as participant_category,
            p.created_at,
            p.imported_at,
            p.updated_at,
            ep.category as event_category
        FROM event_participants ep
        LEFT JOIN participants p ON ep.participant_id = p.id
        WHERE ep.event_id = ? AND ep.heat_number = ?
        ORDER BY ep.sort_order
    ");
    $participants_stmt->execute([$event_id, $heat_number]);
    $participants = $participants_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get current heat settings
    $heat_stmt = $pdo->prepare("
        SELECT bib_on_start, bib_latest_on_run 
        FROM event_heat_settings 
        WHERE event_id = ? AND heat_number = ?
    ");
    $heat_stmt->execute([$event_id, $heat_number]);
    $heat_settings = $heat_stmt->fetch(PDO::FETCH_ASSOC);
    
    // Process participants data
    foreach ($participants as &$participant) {
        $bib = $participant['bib_number'];
        
        // Determine status based on heat settings
        if ($heat_settings['bib_on_start'] == $bib) {
            $participant['status'] = 'on_start';
        } elseif ($heat_settings['bib_latest_on_run'] == $bib) {
            $participant['status'] = 'latest_run';
        } else {
            $participant['status'] = 'waiting';
        }
        
        // Use the most appropriate name fields
        if (empty($participant['first_name']) && empty($participant['last_name'])) {
            // Fallback to name field if first/last names are empty
            $nameParts = explode(' ', $participant['name'] ?? '', 2);
            $participant['display_first_name'] = $nameParts[0] ?? '';
            $participant['display_last_name'] = $nameParts[1] ?? '';
        } else {
            $participant['display_first_name'] = $participant['first_name'] ?? '';
            $participant['display_last_name'] = $participant['last_name'] ?? '';
        }
        
        // Use event category if available, otherwise participant category
        $participant['display_category'] = $participant['event_category'] ?? $participant['participant_category'] ?? 'Unknown';
        
        // Format country display
        $participant['display_country'] = $participant['country'] ?? '';
        
        // Calculate age if date of birth is available
        if ($participant['date_of_birth']) {
            $dob = new DateTime($participant['date_of_birth']);
            $now = new DateTime();
            $participant['age'] = $dob->diff($now)->y;
        } else {
            $participant['age'] = null;
        }
        
        // Format license information
        $participant['license_info'] = '';
        if ($participant['licence_code']) {
            $participant['license_info'] = $participant['licence_code'];
            if ($participant['licence_type']) {
                $participant['license_info'] .= ' (' . $participant['licence_type'] . ')';
            }
        }
        
        // Clean up display fields
        $participant['display_club'] = $participant['club'] ?? '';
        $participant['display_fis_code'] = $participant['fis_code'] ?? '';
    }
    
    return [
        'participants' => $participants,
        'heat_settings' => $heat_settings
    ];
}

function deactivateAllHeats($pdo, $event_id) {
    $stmt = $pdo->prepare("
        UPDATE event_heat_settings 
        SET is_active = 0, bib_on_start = NULL 
        WHERE event_id = ?
    ");
    $stmt->execute([$event_id]);
    
    return [
        'success' => true,
        'message' => 'All heats deactivated'
    ];
}

function getParticipantQueue($pdo, $event_id, $heat_number, $mode = 'start_list') {
    if (!$heat_number) {
        error_log("getParticipantQueue: heat_number is empty");
        return ['queue' => []];
    }
    
    error_log("getParticipantQueue: event_id=$event_id, heat_number=$heat_number, mode=$mode");
    
    // Get heat settings to determine current and latest participants
    try {
        $heat_stmt = $pdo->prepare("
            SELECT 
                bib_on_start, 
                bib_latest_on_run, 
                bib_performing, 
                runs_count, 
                active_run, 
                status
            FROM event_heat_settings 
            WHERE event_id = ? AND heat_number = ?
        ");
        $heat_stmt->execute([$event_id, $heat_number]);
        $heat_settings = $heat_stmt->fetch(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        // Fallback if bib_performing column doesn't exist
        $heat_stmt = $pdo->prepare("
            SELECT 
                bib_on_start, 
                bib_latest_on_run, 
                NULL as bib_performing,
                runs_count,
                active_run,
                status
            FROM event_heat_settings 
            WHERE event_id = ? AND heat_number = ?
        ");
        $heat_stmt->execute([$event_id, $heat_number]);
        $heat_settings = $heat_stmt->fetch(PDO::FETCH_ASSOC);
    }

    if (!$heat_settings) {
        $heat_settings = [
            'bib_on_start' => null,
            'bib_latest_on_run' => null,
            'bib_performing' => null,
            'runs_count' => 1,
            'active_run' => 1,
            'status' => 'pending'
        ];
    } else {
        $heat_settings['runs_count'] = (int)($heat_settings['runs_count'] ?? 1) ?: 1;
        $heat_settings['active_run'] = (int)($heat_settings['active_run'] ?? 1) ?: 1;
        $heat_settings['status'] = $heat_settings['status'] ?? 'pending';
    }
    
    // Get participants for this heat
    $participants_stmt = $pdo->prepare("
        SELECT 
            ep.id as event_participant_id,
            ep.bib_number,
            ep.sort_order,
            ep.participant_status,
            ep.updated_at,
            p.first_name,
            p.last_name,
            p.name,
            p.country,
            p.club,
            ep.category as event_category,
            p.category as participant_category
        FROM event_participants ep
        LEFT JOIN participants p ON ep.participant_id = p.id
        WHERE ep.event_id = ? AND ep.heat_number = ?
        ORDER BY " . ($mode === 'start_list' ? 'ep.sort_order' : 'ep.bib_number') . "
    ");
    $participants_stmt->execute([$event_id, $heat_number]);
    $participants = $participants_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    error_log("getParticipantQueue: Found " . count($participants) . " participants for event_id=$event_id, heat_number=$heat_number");
    
    // Process participants and determine queue status
    $queue = [];
    $latestBib = $heat_settings['bib_latest_on_run'] ?? null;
    
    // Get all performing participants from event_participants.participant_status
    $performing_stmt = $pdo->prepare("
        SELECT bib_number 
        FROM event_participants 
        WHERE event_id = ? AND heat_number = ? AND participant_status = 'performing'
    ");
    $performing_stmt->execute([$event_id, $heat_number]);
    $performing_bibs = $performing_stmt->fetchAll(PDO::FETCH_COLUMN);
    
    foreach ($participants as $participant) {
        $bib = $participant['bib_number'];
        
        // Set display names
        if (empty($participant['first_name']) && empty($participant['last_name'])) {
            $nameParts = explode(' ', $participant['name'] ?? '', 2);
            $participant['display_first_name'] = $nameParts[0] ?? '';
            $participant['display_last_name'] = $nameParts[1] ?? '';
        } else {
            $participant['display_first_name'] = $participant['first_name'] ?? '';
            $participant['display_last_name'] = $participant['last_name'] ?? '';
        }
        
        // Set display category
        $participant['display_category'] = $participant['event_category'] ?? $participant['participant_category'] ?? 'Unknown';
        $participant['display_country'] = $participant['country'] ?? '';
        $participant['display_club'] = $participant['club'] ?? '';
        
        // Determine status in queue - check participant_status for performing
        if (in_array($bib, $performing_bibs)) {
            $participant['status'] = 'current';
            $participant['queue_position'] = 0;
        } elseif ($bib == $latestBib) {
            $participant['status'] = 'completed';
            $participant['queue_position'] = -1; // Completed participants go to end
        } else {
            $participant['status'] = 'waiting';
            $participant['queue_position'] = count($queue) + 1;
        }
        
        $queue[] = $participant;
    }
    
    // Sort queue: current first, then waiting in order, completed last
    usort($queue, function($a, $b) {
        if ($a['status'] === 'current') return -1;
        if ($b['status'] === 'current') return 1;
        if ($a['status'] === 'completed' && $b['status'] !== 'completed') return 1;
        if ($b['status'] === 'completed' && $a['status'] !== 'completed') return -1;
        return $a['sort_order'] - $b['sort_order'];
    });
    
    // Reassign queue positions after sorting
    foreach ($queue as $index => &$participant) {
        if ($participant['status'] === 'current') {
            $participant['queue_position'] = 0;
        } elseif ($participant['status'] === 'waiting') {
            $participant['queue_position'] = $index;
        }
    }
    
    // Set next participant status
    if (count($queue) > 1 && $queue[1]['status'] === 'waiting') {
        $queue[1]['status'] = 'next';
    }
    
    return [
        'queue' => $queue,
        'heat_settings' => $heat_settings,
        'mode' => $mode
    ];
}

function updateParticipantStatus($pdo, $event_id, $data) {
    $bib_number = $data['bib_number'] ?? null;
    $status = $data['status'] ?? null;
    
    if (!$bib_number || !$status) {
        throw new Exception('BIB number and status are required');
    }
    
    // Get the participant ID and heat number
    $participant_stmt = $pdo->prepare("
        SELECT id, heat_number 
        FROM event_participants 
        WHERE event_id = ? AND bib_number = ?
        LIMIT 1
    ");
    $participant_stmt->execute([$event_id, $bib_number]);
    $participant = $participant_stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$participant) {
        throw new Exception('Participant not found');
    }
    
    $heat_number = $participant['heat_number'];
    $participant_id = $participant['id'];
    
    // Determine the currently active run for this heat (defaults to 1)
    $active_run_stmt = $pdo->prepare("SELECT active_run FROM event_heat_settings WHERE event_id = ? AND heat_number = ? LIMIT 1");
    $active_run_stmt->execute([$event_id, $heat_number]);
    $active_run_row = $active_run_stmt->fetch(PDO::FETCH_ASSOC);
    $active_run = isset($active_run_row['active_run']) ? (int)$active_run_row['active_run'] : 1;
    if ($active_run < 1) {
        $active_run = 1;
    }
    
    // Update based on status
    switch ($status) {
        case 'started':
            // Update event_heat_settings.bib_performing
            $stmt = $pdo->prepare("
                UPDATE event_heat_settings 
                SET bib_performing = ? 
                WHERE event_id = ? AND heat_number = ?
            ");
            $stmt->execute([$bib_number, $event_id, $heat_number]);
            
            // Update event_participants.participant_status to 'performing'
            $participant_update = $pdo->prepare("
                UPDATE event_participants 
                SET participant_status = 'performing',
                    run_number = ? 
                WHERE id = ?
            ");
            $participant_update->execute([$active_run, $participant_id]);
            
            $message = "Participant $bib_number marked as performing";
            break;
            
        case 'completed':
            $stmt = $pdo->prepare("
                UPDATE event_heat_settings 
                SET bib_latest_on_run = ?, bib_performing = NULL 
                WHERE event_id = ? AND heat_number = ?
            ");
            $stmt->execute([$bib_number, $event_id, $heat_number]);
            
            // Clear performing status from participant
            $participant_update = $pdo->prepare("
                UPDATE event_participants 
                SET participant_status = 'approved',
                    run_number = ? 
                WHERE id = ?
            ");
            $participant_update->execute([$active_run, $participant_id]);
            
            $message = "Participant $bib_number marked as completed";
            break;
        
        case 'initial':
            // Clear performing bib and mark participant as stopped
            $stmt = $pdo->prepare("
                UPDATE event_heat_settings 
                SET bib_performing = NULL 
                WHERE event_id = ? AND heat_number = ? AND bib_performing = ?
            ");
            $stmt->execute([$event_id, $heat_number, $bib_number]);

            $participant_update = $pdo->prepare("
                UPDATE event_participants 
                SET participant_status = 'initial',
                    run_number = ? 
                WHERE id = ?
            ");
            $participant_update->execute([$active_run, $participant_id]);

            $message = "Participant $bib_number stopped";
            break;

        case 'reset':
            // Reset participant status to initial and ensure not marked as performing
            $participant_update = $pdo->prepare("
                UPDATE event_participants 
                SET participant_status = 'initial',
                    run_number = ? 
                WHERE id = ?
            ");
            $participant_update->execute([$active_run, $participant_id]);

            $stmt = $pdo->prepare("
                UPDATE event_heat_settings 
                SET bib_performing = CASE WHEN bib_performing = ? THEN NULL ELSE bib_performing END 
                WHERE event_id = ? AND heat_number = ?
            ");
            $stmt->execute([$bib_number, $event_id, $heat_number]);

            $message = "Participant $bib_number reset to initial";
            break;
            
        default:
            throw new Exception('Invalid status');
    }
    
    return [
        'success' => true,
        'message' => $message,
        'bib_number' => $bib_number,
        'status' => $status
    ];
}

function updateHeatStatus($pdo, $event_id, $heat_number, $status) {
    if (!$heat_number) {
        return [
            'success' => false,
            'message' => 'Heat number is required'
        ];
    }
    
    $valid_statuses = ['pending', 'active', 'finished', 'cancelled', 'rescheduled'];
    if (!in_array($status, $valid_statuses)) {
        return [
            'success' => false,
            'message' => 'Invalid status. Must be one of: ' . implode(', ', $valid_statuses)
        ];
    }
    
    try {
        // If setting to active, deactivate all other heats first
        if ($status === 'active') {
            $deactivate_stmt = $pdo->prepare("
                UPDATE event_heat_settings 
                SET is_active = 0, status = 'pending'
                WHERE event_id = ? AND heat_number != ?
            ");
            $deactivate_stmt->execute([$event_id, $heat_number]);
            
            // Set this heat as active
            $update_stmt = $pdo->prepare("
                UPDATE event_heat_settings 
                SET is_active = 1, status = ?
                WHERE event_id = ? AND heat_number = ?
            ");
            $update_stmt->execute([$status, $event_id, $heat_number]);
        } else {
            // For non-active statuses, just update status and set is_active to 0
            $update_stmt = $pdo->prepare("
                UPDATE event_heat_settings 
                SET status = ?, is_active = 0
                WHERE event_id = ? AND heat_number = ?
            ");
            $update_stmt->execute([$status, $event_id, $heat_number]);
        }
        
        return [
            'success' => true,
            'message' => "Heat $heat_number status updated to $status",
            'heat_number' => $heat_number,
            'status' => $status
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Database error: ' . $e->getMessage()
        ];
    }
}

function getJudgeStatus($pdo, $event_id, $heat_number, $run_number) {
    if (!$heat_number) {
        return [
            'success' => false,
            'message' => 'No active heat'
        ];
    }
    
    // Get judges assigned to this event
    $judges_stmt = $pdo->prepare("
        SELECT 
            ej.id,
            ej.judge_id,
            ej.panel_number,
            ej.judge_role,
            u.name as judge_name,
            u.email as judge_email
        FROM event_judges ej
        LEFT JOIN users u ON ej.judge_id = u.id
        WHERE ej.event_id = ?
        ORDER BY ej.panel_number, ej.judge_role
    ");
    $judges_stmt->execute([$event_id]);
    $judges = $judges_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($judges)) {
        return [
            'success' => true,
            'judges' => [],
            'heat_number' => $heat_number,
            'run_number' => $run_number,
            'total_participants' => 0,
            'average_progress' => 0
        ];
    }
    
    // Get total participants in this heat
    $participants_stmt = $pdo->prepare("
        SELECT COUNT(*) as total
        FROM event_participants
        WHERE event_id = ? AND heat_number = ?
    ");
    $participants_stmt->execute([$event_id, $heat_number]);
    $total_participants = $participants_stmt->fetchColumn();
    
    // For each judge, get their scoring status
    $judge_status = [];
    foreach ($judges as $judge) {
        // Count scores submitted by this judge for this heat/run
        $scores_stmt = $pdo->prepare("
            SELECT 
                COUNT(*) as scores_count,
                MAX(created_at) as last_score_time
            FROM scoring_results sr
            JOIN event_participants ep ON sr.participant_id = ep.id
            WHERE ep.event_id = ?
                AND ep.heat_number = ?
                AND sr.run_number = ?
                AND sr.judge_id = ?
        ");
        $scores_stmt->execute([$event_id, $heat_number, $run_number, $judge['judge_id']]);
        $score_data = $scores_stmt->fetch(PDO::FETCH_ASSOC);
        
        // Check if judge is online (has activity in last 5 minutes)
        $is_online = false;
        if ($score_data['last_score_time']) {
            $last_activity = strtotime($score_data['last_score_time']);
            $five_minutes_ago = time() - (5 * 60);
            $is_online = $last_activity > $five_minutes_ago;
        }
        
        $judge_status[] = [
            'judge_id' => $judge['judge_id'],
            'judge_name' => $judge['judge_name'] ?: 'Judge #' . $judge['judge_id'],
            'judge_email' => $judge['judge_email'],
            'panel_number' => $judge['panel_number'],
            'judge_role' => $judge['judge_role'],
            'scores_submitted' => (int)$score_data['scores_count'],
            'last_score_time' => $score_data['last_score_time'],
            'is_online' => $is_online,
            'progress_percent' => $total_participants > 0 
                ? round(($score_data['scores_count'] / $total_participants) * 100, 1) 
                : 0
        ];
    }
    
    // Calculate average progress
    $total_progress = array_sum(array_column($judge_status, 'progress_percent'));
    $average_progress = count($judge_status) > 0 ? $total_progress / count($judge_status) : 0;
    
    return [
        'success' => true,
        'judges' => $judge_status,
        'heat_number' => $heat_number,
        'run_number' => $run_number,
        'total_participants' => (int)$total_participants,
        'average_progress' => round($average_progress, 1)
    ];
}

/**
 * Get all scores for a specific heat/run combination
 */
function getHeatRunScores($pdo, $event_id, $heat_number, $run_number) {
    try {
        if (!$event_id || !$heat_number || !$run_number) {
            return [
                'success' => false,
                'message' => 'Event ID, heat number, and run number are required'
            ];
        }
        
        // Query the scores table for all scores in this heat/run
        $stmt = $pdo->prepare("
            SELECT 
                s.id,
                s.run_id,
                s.judge_id,
                s.score_value,
                s.figures_json,
                s.status,
                s.created_at,
                r.run_number,
                r.event_participant_id,
                ep.bib_number,
                ep.participant_id,
                p.first_name,
                p.last_name,
                u.username as judge_name
            FROM scores s
            JOIN runs r ON s.run_id = r.id
            JOIN event_participants ep ON r.event_participant_id = ep.id
            JOIN participants p ON ep.participant_id = p.id
            LEFT JOIN users u ON s.judge_id = u.id
            WHERE ep.event_id = ?
                AND ep.heat_number = ?
                AND r.run_number = ?
            ORDER BY ep.bib_number, s.judge_id
        ");
        
        $stmt->execute([$event_id, $heat_number, $run_number]);
        $scores = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Parse JSON fields
        foreach ($scores as &$score) {
            if (isset($score['figures_json']) && is_string($score['figures_json'])) {
                $score['figures_json'] = json_decode($score['figures_json'], true);
            }
        }
        
        return [
            'success' => true,
            'scores' => $scores,
            'heat_number' => $heat_number,
            'run_number' => $run_number,
            'count' => count($scores)
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Failed to get heat/run scores: ' . $e->getMessage()
        ];
    }
}

/**
 * Swap sort_order positions of two participants
 */
function swapParticipantPositions($pdo, $participant1_id, $participant2_id, $participant1_order, $participant2_order) {
    try {
        if (!$participant1_id || !$participant2_id) {
            return [
                'success' => false,
                'message' => 'Both participant IDs are required'
            ];
        }
        
        // Start transaction
        $pdo->beginTransaction();
        
        // Update participant 1
        $stmt1 = $pdo->prepare("UPDATE event_participants SET sort_order = ? WHERE id = ?");
        $stmt1->execute([$participant1_order, $participant1_id]);
        
        // Update participant 2
        $stmt2 = $pdo->prepare("UPDATE event_participants SET sort_order = ? WHERE id = ?");
        $stmt2->execute([$participant2_order, $participant2_id]);
        
        // Commit transaction
        $pdo->commit();
        
        return [
            'success' => true,
            'message' => 'Positions swapped successfully'
        ];
        
    } catch (Exception $e) {
        // Rollback on error
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        
        return [
            'success' => false,
            'message' => 'Failed to swap positions: ' . $e->getMessage()
        ];
    }
}

/**
 * Get participant cards with configurable layout
 * 
 * @param PDO $pdo Database connection
 * @param int $event_id Event ID
 * @param int $heat_number Heat number
 * @param string $layout Layout mode: 'full', 'compact', 'controls', 'info'
 *   - full: All details with position controls and score breakdown
 *   - compact: Minimal details with controls
 *   - controls: Standard details with position controls
 *   - info: Informative only, no controls (read-only display)
 * @param string $mode Sort mode: 'sort_order' or 'bib'
 * @return array Response with participants and layout metadata
 */
function getParticipantCards($pdo, $event_id, $heat_number, $layout = 'full', $mode = 'sort_order') {
    try {
        // Get participant queue data
        $queueData = getParticipantQueue($pdo, $event_id, $heat_number, $mode);
        
        if (!isset($queueData['queue'])) {
            return [
                'success' => false,
                'message' => 'Failed to retrieve participant queue'
            ];
        }
        
        // Determine which fields to include based on layout
        $includeControls = in_array($layout, ['full', 'compact', 'controls']);
        $includeDetails = in_array($layout, ['full', 'info']);
        $includeFigures = in_array($layout, ['full', 'info']);
        
        // Process participants based on layout
        $participants = array_map(function($participant) use ($includeControls, $includeDetails, $includeFigures) {
            $card = [
                'event_participant_id' => $participant['event_participant_id'],
                'bib_number' => $participant['bib_number'],
                'display_first_name' => $participant['display_first_name'],
                'display_last_name' => $participant['display_last_name'],
                'display_category' => $participant['display_category'],
                'sort_order' => $participant['sort_order'],
                'status' => $participant['status'],
                'has_score' => $participant['has_score'] ?? false,
                'total_score' => $participant['total_score'] ?? null,
                'judge_count' => $participant['judge_count'] ?? 0,
            ];
            
            // Add details if layout supports it
            if ($includeDetails) {
                $card['display_country'] = $participant['display_country'] ?? '';
                $card['display_club'] = $participant['display_club'] ?? '';
                $card['participant_status'] = $participant['participant_status'] ?? null;
            }
            
            // Add figures if layout supports it
            if ($includeFigures && isset($participant['figures_json'])) {
                $card['figures_json'] = $participant['figures_json'];
            }
            
            // Add control flags
            $card['show_controls'] = $includeControls;
            
            return $card;
        }, $queueData['queue']);
        
        return [
            'success' => true,
            'participants' => $participants,
            'layout' => $layout,
            'mode' => $mode,
            'heat_settings' => $queueData['heat_settings'] ?? [],
            'metadata' => [
                'include_controls' => $includeControls,
                'include_details' => $includeDetails,
                'include_figures' => $includeFigures,
                'total_participants' => count($participants)
            ]
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Failed to get participant cards: ' . $e->getMessage()
        ];
    }
}

/**
 * Get runs data for all participants in a heat from the runs table
 */
function getParticipantRuns($pdo, $event_id, $heat_number) {
    try {
        if (!$heat_number) {
            throw new Exception('Heat number is required');
        }
        
        // Get all runs for participants in this heat
        // Note: runs table only has: id, event_participant_id, run_number
        $stmt = $pdo->prepare("
            SELECT r.id, r.event_participant_id, r.run_number,
                   ep.bib_number
            FROM runs r
            INNER JOIN event_participants ep ON r.event_participant_id = ep.id
            WHERE ep.event_id = ? AND ep.heat_number = ?
            ORDER BY ep.bib_number, r.run_number
        ");
        $stmt->execute([$event_id, $heat_number]);
        $runs = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        return [
            'success' => true,
            'runs' => $runs,
            'count' => count($runs)
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Failed to get participant runs: ' . $e->getMessage(),
            'runs' => []
        ];
    }
}

/**
 * Update participant's heat assignment (for drag-drop between heats)
 */
function updateParticipantHeat($pdo, $assignment_id, $new_heat, $event_id) {
    try {
        if (!$assignment_id || !$new_heat) {
            throw new Exception('Assignment ID and new heat are required');
        }
        
        // Update the heat assignment
        $stmt = $pdo->prepare("
            UPDATE event_participants 
            SET heat_number = ?, updated_at = NOW()
            WHERE id = ? AND event_id = ?
        ");
        $stmt->execute([$new_heat, $assignment_id, $event_id]);
        
        if ($stmt->rowCount() === 0) {
            throw new Exception('Assignment not found or not updated');
        }
        
        return [
            'status' => 'success',
            'message' => 'Participant moved to heat ' . $new_heat
        ];
        
    } catch (Exception $e) {
        return [
            'status' => 'error',
            'message' => 'Failed to update heat: ' . $e->getMessage()
        ];
    }
}

/**
 * Update sort order for participants in a heat
 */
function updateHeatOrder($pdo, $event_id, $heat_number, $order) {
    try {
        if (!$heat_number || !is_array($order)) {
            throw new Exception('Heat number and order array are required');
        }
        
        // Update sort order for each participant
        $pdo->beginTransaction();
        
        $stmt = $pdo->prepare("
            UPDATE event_participants 
            SET sort_order = ?, updated_at = NOW()
            WHERE id = ? AND event_id = ? AND heat_number = ?
        ");
        
        foreach ($order as $index => $assignment_id) {
            $sort_order = $index + 1; // 1-based ordering
            $stmt->execute([$sort_order, $assignment_id, $event_id, $heat_number]);
        }
        
        $pdo->commit();
        
        return [
            'status' => 'success',
            'message' => 'Heat order updated',
            'updated_count' => count($order)
        ];
        
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        return [
            'status' => 'error',
            'message' => 'Failed to update order: ' . $e->getMessage()
        ];
    }
}

/**
 * Delete a participant assignment from a heat
 */
function deleteParticipantAssignment($pdo, $assignment_id, $event_id) {
    try {
        if (!$assignment_id) {
            throw new Exception('Assignment ID is required');
        }
        
        // Remove heat assignment (set to null instead of deleting the record)
        $stmt = $pdo->prepare("
            UPDATE event_participants 
            SET heat_number = NULL, sort_order = NULL, updated_at = NOW()
            WHERE id = ? AND event_id = ?
        ");
        $stmt->execute([$assignment_id, $event_id]);
        
        if ($stmt->rowCount() === 0) {
            throw new Exception('Assignment not found');
        }
        
        return [
            'status' => 'success',
            'message' => 'Participant removed from heat'
        ];
        
    } catch (Exception $e) {
        return [
            'status' => 'error',
            'message' => 'Failed to delete assignment: ' . $e->getMessage()
        ];
    }
}

?>