<?php
/**
 * API Endpoint: Heat Configuration Modals
 * Generates heat configuration modals dynamically for use across different panels
 */

header('Content-Type: application/json');
require_once '../includes/auth.php';
require_once '../includes/db.php';

try {
    $event_id = $_GET['event_id'] ?? '';
    $heat_numbers = $_GET['heat_numbers'] ?? ''; // Comma-separated heat numbers, or 'all'
    
    if (!$event_id) {
        echo json_encode(['error' => 'Missing event_id parameter', 'success' => false]);
        exit;
    }
    
    // Load event data
    $event_stmt = $pdo->prepare("
        SELECT e.*, sf.name as format_name, sf.sport as discipline, sf.mode as format_mode,
               GROUP_CONCAT(DISTINCT sfj.min_judges ORDER BY sfj.min_judges SEPARATOR ', ') as min_judges,
               GROUP_CONCAT(DISTINCT sfmc.config_value ORDER BY sfmc.config_key SEPARATOR ', ') as format_config
        FROM events e
        LEFT JOIN scoring_formats sf ON e.scoring_format = sf.format_id
        LEFT JOIN scoring_format_judges sfj ON sf.format_id = sfj.format_id
        LEFT JOIN scoring_format_mode_config sfmc ON sf.format_id = sfmc.format_id
        WHERE e.id = ?
        GROUP BY e.id
    ");
    $event_stmt->execute([$event_id]);
    $event_data = $event_stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$event_data) {
        echo json_encode(['error' => 'Event not found', 'success' => false]);
        exit;
    }
    
    $heats_total = intval($event_data['heats_total']);
    
    // Load event categories
    $categories_stmt = $pdo->prepare("
        SELECT id, category_name 
        FROM event_categories 
        WHERE event_id = ? AND is_main_category = 1 
        ORDER BY category_name
    ");
    $categories_stmt->execute([$event_id]);
    $event_categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get participant counts per heat
    $participant_counts_stmt = $pdo->prepare("
        SELECT heat_number, COUNT(*) as participant_count 
        FROM event_participants 
        WHERE event_id = ? 
        GROUP BY heat_number
    ");
    $participant_counts_stmt->execute([$event_id]);
    $heat_participant_counts = [];
    foreach ($participant_counts_stmt->fetchAll(PDO::FETCH_ASSOC) as $count_row) {
        $heat_participant_counts[$count_row['heat_number']] = $count_row['participant_count'];
    }
    
    // Load heat settings
    $settings_stmt = $pdo->prepare("
        SELECT heat_number, heat_name, scoring_type, runs_count, runs_scoring_method,
               time_start, estimate_time_per_participant, categories, flow_type,
               flow_source_heat, flow_participants_per_category, flow_position_range,
               is_active, active_run, status, required_figures,
               judges_required, scale_min, scale_max, precision_decimal, drop_rule,
               difficulty_level, course_length, time_limit, weather_dependent,
               format_description, scoring_method, diversity_rules_enabled,
               scoring_layout_config
        FROM event_heat_settings 
        WHERE event_id = ?
        ORDER BY heat_number
    ");
    $settings_stmt->execute([$event_id]);
    $heat_settings = [];
    foreach ($settings_stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $row['categories'] = json_decode($row['categories'] ?: '[]', true);
        $row['required_figures'] = json_decode($row['required_figures'] ?: '[]', true);
        $row['participant_count'] = $heat_participant_counts[$row['heat_number']] ?? 0;
        $heat_settings[$row['heat_number']] = $row;
    }
    
    // Determine which heats to generate
    $heats_to_generate = [];
    if ($heat_numbers === 'all') {
        $heats_to_generate = range(1, $heats_total);
    } else {
        $heats_to_generate = array_filter(
            array_map('intval', explode(',', $heat_numbers)),
            function($h) use ($heats_total) { return $h >= 1 && $h <= $heats_total; }
        );
    }
    
    if (empty($heats_to_generate)) {
        echo json_encode(['error' => 'No valid heat numbers provided', 'success' => false]);
        exit;
    }
    
    // Generate modals HTML
    ob_start();
    
    foreach ($heats_to_generate as $h) {
        $setting = $heat_settings[$h] ?? [
            'heat_name' => '', 
            'scoring_type' => 'Points',
            'runs_count' => 1,
            'runs_scoring_method' => 'best_from_all',
            'time_start' => '',
            'estimate_time_per_participant' => 0,
            'categories' => [],
            'flow_type' => 'none',
            'flow_source_heat' => null,
            'flow_participants_per_category' => 0,
            'flow_position_range' => null,
            'is_active' => 0,
            'active_run' => 1,
            'status' => 'pending',
            'format_name' => $event_data['format_name'] ?? '',
            'discipline' => $event_data['discipline'] ?? '',
            'difficulty_level' => '',
            'course_length' => 0,
            'time_limit' => 0,
            'weather_dependent' => 0,
            'format_description' => '',
            'scoring_method' => 'standard',
            'required_figures' => [],
            'judges_required' => 5,
            'scale_min' => $event_data['scale_min'] ?? 0,
            'scale_max' => $event_data['scale_max'] ?? 100,
            'precision_decimal' => $event_data['precision_decimal'] ?? 0,
            'drop_rule' => null,
            'diversity_rules_enabled' => 1,
            'scoring_layout_config' => null,
            'participant_count' => 0
        ];
        
        // Merge with actual heat settings if they exist
        if (isset($heat_settings[$h])) {
            $setting = array_merge($setting, $heat_settings[$h]);
        }
        
        // Set variables needed by the template
        $selected_event_id = $event_id;
        
        // Include the modal template (extracted from admin_heat_config.php)
        include __DIR__ . '/heat_config_modal_template.php';
    }
    
    $modals_html = ob_get_clean();
    
    echo json_encode([
        'success' => true,
        'html' => $modals_html,
        'heat_numbers' => $heats_to_generate,
        'event_id' => $event_id,
        'heat_settings' => $heat_settings,
        'heat_participant_counts' => $heat_participant_counts
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'error' => 'Server error: ' . $e->getMessage(),
        'success' => false
    ]);
}
