<?php
include '../includes/auth.php';
include '../includes/db.php';
include '../includes/event_metadata_helpers.php';
include '../includes/validate_event_access.php';

$event_id = $_GET['event_id'] ?? $_COOKIE['event_id'] ?? $_COOKIE['selectedEventId'] ?? null;
$message = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_metadata'])) {
    $event_id = $_POST['event_id'];
    
    // Build metadata array from form data
    $metadata = ['metadata_fields' => [], 'judges' => []];
    
    // Process all fields
    if (isset($_POST['field_key']) && is_array($_POST['field_key'])) {
        foreach ($_POST['field_key'] as $index => $fieldKey) {
            $fieldValue = trim($_POST['field_value'][$index] ?? '');
            
            // Only save fields with values (or system fields)
            $isSystem = isset($_POST['field_is_system'][$index]) && $_POST['field_is_system'][$index] === '1';
            
            if (!empty($fieldValue) || $isSystem) {
                $metadata['metadata_fields'][] = [
                    'group' => trim($_POST['field_group'][$index] ?? ''),
                    'field_key' => trim($fieldKey),
                    'field_label' => trim($_POST['field_label'][$index] ?? ''),
                    'field_value' => $fieldValue,
                    'field_order' => (int)($_POST['field_order'][$index] ?? 999),
                    'is_system' => $isSystem
                ];
            }
        }
    }
    
    // Handle judges array
    if (isset($_POST['judge_name']) && is_array($_POST['judge_name'])) {
        foreach ($_POST['judge_name'] as $index => $name) {
            if (!empty(trim($name))) {
                $metadata['judges'][] = [
                    'position' => $index + 1,
                    'name' => trim($name),
                    'license' => trim($_POST['judge_license'][$index] ?? '')
                ];
            }
        }
    }
    
    try {
        if (saveEventMetadata($pdo, $event_id, $metadata)) {
            $message = '<div class="alert alert-success alert-dismissible fade show">
                <i class="fas fa-check-circle me-2"></i>Event metadata saved successfully!
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
        } else {
            $message = '<div class="alert alert-danger">Failed to save metadata.</div>';
        }
    } catch (Exception $e) {
        $message = '<div class="alert alert-danger">Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
    }
}

// Load current metadata
$metadata = [];
if ($event_id) {
    $metadata = getEventMetadata($pdo, $event_id);
}

// Organize fields by group
$fieldsByGroup = getFieldsByGroup($metadata);

// Load events for dropdown
$events = $pdo->query("SELECT id, name, date FROM events ORDER BY date DESC")->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Event Metadata - StyleScore</title>
    <?php include_once '../includes/stylesheets.php'; ?>
    <link rel="stylesheet" href="https://code.jquery.com/ui/1.13.2/themes/base/jquery-ui.css">
    <style>
        .metadata-section { border-left: 4px solid #0d6efd; }
        .field-row { 
            transition: all 0.2s;
            background: white;
            border-radius: 0.375rem;
        }
        .field-row:hover { background-color: rgba(0, 123, 255, 0.05); }
        .field-row.is-system { border-left: 3px solid #6c757d; }
        .field-row.is-custom { border-left: 3px solid #198754; }
        .drag-handle { cursor: move; color: #6c757d; }
        .drag-handle:hover { color: #0d6efd; }
        .group-header { 
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 0.375rem;
        }
        .ui-sortable-helper {
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            opacity: 0.9;
        }
        .ui-sortable-placeholder {
            border: 2px dashed #0d6efd;
            background-color: rgba(13, 110, 253, 0.1);
            visibility: visible !important;
            height: 60px;
            border-radius: 0.375rem;
        }
        .sortable-container {
            min-height: 50px;
        }
    </style>
</head>
<body class="body-bg-aurora-bright">
<?php include '../menu.php'; ?>

<div class="container-fluid container-StyleScore p-4">
    <!-- Event Actions Menu -->
    <div class="row mb-3">
        <div class="col-12">
            <?php 
            $use_localStorage = true;
            $show_header = false;
            include 'event_actions_menu.php'; 
            ?>
        </div>
    </div>

    <!-- Header -->
    <div class="d-flex justify-content-between align-items-center mb-3">
        <div>
            <h1 class="h4 mb-1">
                <i class="fas fa-database text-primary me-2"></i>Event Metadata Manager
            </h1>
            <p class="text-muted mb-0 small">Flexible metadata with custom fields and ordering</p>
        </div>
        <div class="d-flex gap-2">
            <button type="button" class="btn btn-outline-info btn-sm" data-bs-toggle="modal" data-bs-target="#addFieldModal">
                <i class="fas fa-plus-circle me-1"></i>Add Custom Field
            </button>
            <button type="button" class="btn btn-outline-secondary btn-sm" data-bs-toggle="modal" data-bs-target="#helpModal">
                <i class="fas fa-question-circle me-1"></i>Help
            </button>
        </div>
    </div>

    <!-- Messages -->
    <?php if ($message): ?>
        <?php echo $message; ?>
    <?php endif; ?>

    <?php if ($event_id): ?>
        <form method="post" id="metadataForm">
            <input type="hidden" name="event_id" value="<?= $event_id ?>">
            <input type="hidden" name="save_metadata" value="1">
            
            <div class="row g-3">
                <!-- Metadata Fields by Group -->
                <div class="col-lg-8">
                    <?php foreach ($fieldsByGroup as $groupName => $fields): ?>
                        <div class="card shadow-sm mb-3">
                            <div class="card-header group-header p-3">
                                <div class="d-flex justify-content-between align-items-center">
                                    <h6 class="mb-0">
                                        <i class="fas fa-folder me-2"></i><?= htmlspecialchars($groupName) ?>
                                    </h6>
                                    <span class="badge bg-light text-dark"><?= count($fields) ?> fields</span>
                                </div>
                            </div>
                            <div class="card-body p-2 sortable-container" data-group="<?= htmlspecialchars($groupName) ?>">
                                <?php foreach ($fields as $field): ?>
                                    <div class="field-row p-3 mb-2 border <?= $field['is_system'] ? 'is-system' : 'is-custom' ?>">
                                        <div class="row g-2 align-items-center">
                                            <!-- Drag Handle -->
                                            <div class="col-auto">
                                                <i class="fas fa-grip-vertical drag-handle" title="Drag to reorder"></i>
                                            </div>
                                            
                                            <!-- Field Order -->
                                            <div class="col-auto">
                                                <input type="number" 
                                                       class="form-control form-control-sm text-center field-order-input" 
                                                       name="field_order[]" 
                                                       value="<?= (int)($field['field_order'] ?? 1) ?>" 
                                                       style="width: 60px;" 
                                                       min="1" 
                                                       title="Display order"
                                                       readonly>
                                            </div>
                                            
                                            <!-- Field Label -->
                                            <div class="col-md-3">
                                                <input type="text" 
                                                       class="form-control form-control-sm fw-medium" 
                                                       name="field_label[]" 
                                                       value="<?= htmlspecialchars($field['field_label'] ?? '') ?>"
                                                       placeholder="Field Label"
                                                       <?= $field['is_system'] ? 'readonly' : '' ?>>
                                            </div>
                                            
                                            <!-- Field Value -->
                                            <div class="col-md-5">
                                                <input type="text" 
                                                       class="form-control form-control-sm" 
                                                       name="field_value[]" 
                                                       value="<?= htmlspecialchars($field['field_value'] ?? '') ?>"
                                                       placeholder="Enter value">
                                            </div>
                                            
                                            <!-- Field Type Badge & Actions -->
                                            <div class="col-md text-end">
                                                <?php if ($field['is_system']): ?>
                                                    <span class="badge bg-secondary me-2" title="System field - cannot be renamed or deleted">
                                                        <i class="fas fa-lock me-1"></i>System
                                                    </span>
                                                <?php else: ?>
                                                    <span class="badge bg-success me-2" title="Custom field - can be edited or removed">
                                                        <i class="fas fa-edit me-1"></i>Custom
                                                    </span>
                                                    <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeField(this)" title="Remove field">
                                                        <i class="fas fa-times"></i>
                                                    </button>
                                                <?php endif; ?>
                                            </div>
                                            
                                            <!-- Hidden fields -->
                                            <input type="hidden" name="field_key[]" value="<?= htmlspecialchars($field['field_key'] ?? '') ?>">
                                            <input type="hidden" name="field_group[]" value="<?= htmlspecialchars($field['group'] ?? '') ?>">
                                            <input type="hidden" name="field_is_system[]" value="<?= $field['is_system'] ? '1' : '0' ?>">
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>

                <!-- Right Sidebar -->
                <div class="col-lg-4">
                    
                    <!-- Judges List -->
                    <div class="card shadow-sm mb-3">
                        <div class="card-header bg-warning">
                            <div class="d-flex justify-content-between align-items-center">
                                <h6 class="mb-0"><i class="fas fa-users me-2"></i>Judges</h6>
                                <button type="button" class="btn btn-sm btn-dark" onclick="addJudgeRow()">
                                    <i class="fas fa-plus me-1"></i>Add Judge
                                </button>
                            </div>
                        </div>
                        <div class="card-body p-2">
                            <div id="judgesContainer">
                                <?php if (!empty($metadata['judges'])): ?>
                                    <?php foreach ($metadata['judges'] as $index => $judge): ?>
                                        <div class="field-row p-2 mb-2 border">
                                            <div class="row g-2 align-items-center">
                                                <div class="col-1 text-center">
                                                    <span class="badge bg-primary"><?= $index + 1 ?></span>
                                                </div>
                                                <div class="col-6">
                                                    <input type="text" class="form-control form-control-sm" 
                                                           name="judge_name[]" 
                                                           value="<?= htmlspecialchars($judge['name'] ?? '') ?>"
                                                           placeholder="Judge name" required>
                                                </div>
                                                <div class="col-4">
                                                    <input type="text" class="form-control form-control-sm" 
                                                           name="judge_license[]" 
                                                           value="<?= htmlspecialchars($judge['license'] ?? '') ?>"
                                                           placeholder="License #">
                                                </div>
                                                <div class="col-1">
                                                    <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeJudgeRow(this)">
                                                        <i class="fas fa-times"></i>
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <p class="text-muted text-center py-3 mb-0">
                                        <i class="fas fa-info-circle me-1"></i>
                                        No judges added. Click "Add Judge" to add entries.
                                    </p>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>

                    <!-- Quick Actions -->
                    <div class="card shadow-sm">
                        <div class="card-header bg-info text-white">
                            <h6 class="mb-0"><i class="fas fa-bolt me-2"></i>Quick Actions</h6>
                        </div>
                        <div class="card-body">
                            <div class="d-grid gap-2">
                                <button type="button" class="btn btn-outline-primary btn-sm" onclick="previewMetadata()">
                                    <i class="fas fa-eye me-2"></i>Preview
                                </button>
                                <button type="submit" class="btn btn-success">
                                    <i class="fas fa-save me-2"></i>Save All Changes
                                </button>
                                <button type="button" class="btn btn-outline-secondary btn-sm" onclick="resetToDefaults()">
                                    <i class="fas fa-undo me-2"></i>Reset to Defaults
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </form>

    <?php else: ?>
        <div class="card shadow-sm">
            <div class="card-body text-center py-5">
                <i class="fas fa-calendar-alt fa-3x text-muted opacity-25 mb-3"></i>
                <h5 class="text-muted">No Event Selected</h5>
                <p class="text-muted">Select an event to manage its metadata</p>
            </div>
        </div>
    <?php endif; ?>
</div>

<!-- Add Custom Field Modal -->
<div class="modal fade" id="addFieldModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white">
                <h5 class="modal-title"><i class="fas fa-plus-circle me-2"></i>Add Custom Field</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label class="form-label">Group</label>
                    <select class="form-select" id="newFieldGroup">
                        <option value="">Select or type new group...</option>
                        <?php foreach (array_keys($fieldsByGroup) as $groupName): ?>
                            <option value="<?= htmlspecialchars($groupName) ?>"><?= htmlspecialchars($groupName) ?></option>
                        <?php endforeach; ?>
                    </select>
                    <input type="text" class="form-control mt-2" id="newFieldGroupCustom" placeholder="Or enter new group name">
                </div>
                <div class="mb-3">
                    <label class="form-label">Field Label</label>
                    <input type="text" class="form-control" id="newFieldLabel" placeholder="e.g., Chief Observer">
                </div>
                <div class="mb-3">
                    <label class="form-label">Field Value (optional)</label>
                    <input type="text" class="form-control" id="newFieldValue" placeholder="Enter initial value">
                </div>
                <div class="mb-3">
                    <label class="form-label">Display Order</label>
                    <input type="number" class="form-control" id="newFieldOrder" value="999" min="1">
                    <small class="text-muted">Tip: Use 2.5 to insert between order 2 and 3</small>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="addCustomField()">
                    <i class="fas fa-plus me-1"></i>Add Field
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Preview Modal -->
<div class="modal fade" id="previewModal" tabindex="-1">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="fas fa-eye me-2"></i>Metadata Preview</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="previewContent">
                <!-- Preview will be generated here -->
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<!-- Help Modal -->
<div class="modal fade" id="helpModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="fas fa-question-circle me-2"></i>Event Metadata Help</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <h6>What is Event Metadata?</h6>
                <p>Event metadata stores flexible information about your event. You can add custom fields to any group and control the display order.</p>
                
                <h6 class="mt-3">Features:</h6>
                <ul>
                    <li><strong>System Fields:</strong> Pre-defined fields (locked icon) that cannot be renamed or deleted</li>
                    <li><strong>Custom Fields:</strong> Your own fields (green badge) that can be edited, reordered, or removed</li>
                    <li><strong>Field Ordering:</strong> Control display order with numbers (use decimals like 2.5 to insert between)</li>
                    <li><strong>Groups:</strong> Organize fields into logical sections (can add custom fields to existing groups)</li>
                </ul>
                
                <h6 class="mt-3">Examples:</h6>
                <ul>
                    <li><strong>Add "Chief Observer" between "Chief of Competition" and "Head Judge":</strong><br>
                        Group: Jury Information, Label: Chief Observer, Order: 2.5</li>
                    <li><strong>Create custom group:</strong><br>
                        Group: Safety Information, Label: Medical Officer, Order: 1</li>
                </ul>
                
                <h6 class="mt-3">Tips:</h6>
                <ul>
                    <li>Drag the grip icon to reorder fields visually</li>
                    <li>Leave field values empty if not applicable</li>
                    <li>Custom fields appear with green "Custom" badge</li>
                    <li>Use consistent units for technical data</li>
                </ul>
            </div>
        </div>
    </div>
</div>

<?php include 'footer.php'; ?>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://code.jquery.com/ui/1.13.2/jquery-ui.min.js"></script>

<script>
// Auto-resolve event from localStorage
document.addEventListener('DOMContentLoaded', function() {
    <?php if (empty($event_id)): ?>
    const storedEventId = localStorage.getItem('selectedEventId') || getCookie('selectedEventId') || getCookie('event_id');
    if (storedEventId) {
        window.location.href = 'event_metadata.php?event_id=' + storedEventId;
    }
    <?php endif; ?>
    
    // Initialize sortable for all field groups
    initializeSortable();
});

function initializeSortable() {
    $('.sortable-container').sortable({
        handle: '.drag-handle',
        placeholder: 'ui-sortable-placeholder',
        axis: 'y',
        cursor: 'move',
        opacity: 0.8,
        tolerance: 'pointer',
        start: function(event, ui) {
            ui.placeholder.height(ui.item.height());
        },
        update: function(event, ui) {
            // Recalculate order values for all items in this group
            updateOrderValues($(this));
        }
    });
}

function updateOrderValues(container) {
    const fields = container.find('.field-row');
    fields.each(function(index) {
        const orderInput = $(this).find('.field-order-input');
        // Set order as index + 1 (1-based numbering)
        orderInput.val(index + 1);
    });
    
    // Visual feedback
    container.addClass('border-success');
    setTimeout(() => {
        container.removeClass('border-success');
    }, 500);
}

function getCookie(name) {
    const value = `; ${document.cookie}`;
    const parts = value.split(`; ${name}=`);
    if (parts.length === 2) return parts.pop().split(';').shift();
    return null;
}

let judgeCounter = <?= count($metadata['judges'] ?? []) ?>;

function addJudgeRow() {
    judgeCounter++;
    const container = document.getElementById('judgesContainer');
    
    // Remove empty message if exists
    const emptyMsg = container.querySelector('p.text-muted');
    if (emptyMsg) emptyMsg.remove();
    
    const row = document.createElement('div');
    row.className = 'field-row p-2 mb-2 border';
    row.innerHTML = `
        <div class="row g-2 align-items-center">
            <div class="col-1 text-center">
                <span class="badge bg-primary">${judgeCounter}</span>
            </div>
            <div class="col-6">
                <input type="text" class="form-control form-control-sm" name="judge_name[]" placeholder="Judge name" required>
            </div>
            <div class="col-4">
                <input type="text" class="form-control form-control-sm" name="judge_license[]" placeholder="License #">
            </div>
            <div class="col-1">
                <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeJudgeRow(this)">
                    <i class="fas fa-times"></i>
                </button>
            </div>
        </div>
    `;
    container.appendChild(row);
    renumberJudges();
}

function removeJudgeRow(button) {
    const container = document.getElementById('judgesContainer');
    button.closest('.field-row').remove();
    
    if (container.children.length === 0) {
        container.innerHTML = `<p class="text-muted text-center py-3 mb-0">
            <i class="fas fa-info-circle me-1"></i>
            No judges added. Click "Add Judge" to add entries.
        </p>`;
        judgeCounter = 0;
    } else {
        renumberJudges();
    }
}

function renumberJudges() {
    const rows = document.querySelectorAll('#judgesContainer .field-row');
    rows.forEach((row, index) => {
        row.querySelector('.badge').textContent = index + 1;
    });
    judgeCounter = rows.length;
}

function addCustomField() {
    const group = document.getElementById('newFieldGroupCustom').value.trim() || 
                  document.getElementById('newFieldGroup').value;
    const label = document.getElementById('newFieldLabel').value.trim();
    const value = document.getElementById('newFieldValue').value.trim();
    const order = document.getElementById('newFieldOrder').value;
    
    if (!group || !label) {
        alert('Please enter both group and label for the custom field');
        return;
    }
    
    // Generate field key from label
    const fieldKey = 'custom_' + label.toLowerCase().replace(/[^a-z0-9]+/g, '_');
    
    // Find or create group card
    let groupCard = document.querySelector(`[data-group="${group}"]`);
    if (!groupCard) {
        // Create new group
        const newGroupHtml = `
            <div class="card shadow-sm mb-3">
                <div class="card-header group-header p-3">
                    <div class="d-flex justify-content-between align-items-center">
                        <h6 class="mb-0">
                            <i class="fas fa-folder me-2"></i>${group}
                        </h6>
                        <span class="badge bg-light text-dark">1 fields</span>
                    </div>
                </div>
                <div class="card-body p-2 sortable-container" data-group="${group}"></div>
            </div>
        `;
        document.querySelector('.col-lg-8').insertAdjacentHTML('beforeend', newGroupHtml);
        groupCard = document.querySelector(`[data-group="${group}"]`);
        
        // Initialize sortable for new group
        $(groupCard).sortable({
            handle: '.drag-handle',
            placeholder: 'ui-sortable-placeholder',
            axis: 'y',
            cursor: 'move',
            opacity: 0.8,
            tolerance: 'pointer',
            start: function(event, ui) {
                ui.placeholder.height(ui.item.height());
            },
            update: function(event, ui) {
                updateOrderValues($(this));
            }
        });
    }
    
    // Add field to group
    const fieldHtml = `
        <div class="field-row p-3 mb-2 border is-custom">
            <div class="row g-2 align-items-center">
                <div class="col-auto">
                    <i class="fas fa-grip-vertical drag-handle" title="Drag to reorder"></i>
                </div>
                <div class="col-auto">
                    <input type="number" class="form-control form-control-sm text-center field-order-input" 
                           name="field_order[]" value="${order}" style="width: 60px;" min="1" title="Display order" readonly>
                </div>
                <div class="col-md-3">
                    <input type="text" class="form-control form-control-sm fw-medium" 
                           name="field_label[]" value="${label}" placeholder="Field Label">
                </div>
                <div class="col-md-5">
                    <input type="text" class="form-control form-control-sm" 
                           name="field_value[]" value="${value}" placeholder="Enter value">
                </div>
                <div class="col-md text-end">
                    <span class="badge bg-success me-2" title="Custom field - can be edited or removed">
                        <i class="fas fa-edit me-1"></i>Custom
                    </span>
                    <button type="button" class="btn btn-sm btn-outline-danger" onclick="removeField(this)" title="Remove field">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                <input type="hidden" name="field_key[]" value="${fieldKey}">
                <input type="hidden" name="field_group[]" value="${group}">
                <input type="hidden" name="field_is_system[]" value="0">
            </div>
        </div>
    `;
    groupCard.insertAdjacentHTML('beforeend', fieldHtml);
    
    // Refresh sortable after adding new field
    $(groupCard).sortable('refresh');
    updateOrderValues($(groupCard));
    
    // Close modal and reset form
    bootstrap.Modal.getInstance(document.getElementById('addFieldModal')).hide();
    document.getElementById('newFieldGroup').value = '';
    document.getElementById('newFieldGroupCustom').value = '';
    document.getElementById('newFieldLabel').value = '';
    document.getElementById('newFieldValue').value = '';
    document.getElementById('newFieldOrder').value = '999';
}

function removeField(button) {
    if (confirm('Remove this field?')) {
        button.closest('.field-row').remove();
    }
}

function previewMetadata() {
    const formData = new FormData(document.getElementById('metadataForm'));
    const groups = {};
    
    // Organize fields by group
    const fieldKeys = formData.getAll('field_key[]');
    const fieldLabels = formData.getAll('field_label[]');
    const fieldValues = formData.getAll('field_value[]');
    const fieldGroups = formData.getAll('field_group[]');
    const fieldOrders = formData.getAll('field_order[]');
    
    fieldKeys.forEach((key, index) => {
        const group = fieldGroups[index];
        const value = fieldValues[index];
        
        if (value) {
            if (!groups[group]) groups[group] = [];
            groups[group].push({
                order: parseFloat(fieldOrders[index]),
                label: fieldLabels[index],
                value: value
            });
        }
    });
    
    // Sort and generate HTML
    let html = '<div class="row">';
    let col = 0;
    
    for (const [groupName, fields] of Object.entries(groups)) {
        // Sort by order
        fields.sort((a, b) => a.order - b.order);
        
        if (col % 2 === 0 && col > 0) html += '</div><div class="row mt-3">';
        
        html += `<div class="col-md-6">
            <h6 class="text-primary border-bottom pb-2">
                <i class="fas fa-folder me-2"></i>${groupName}
            </h6>
            <dl class="row">`;
        
        fields.forEach(field => {
            html += `<dt class="col-sm-5">${field.label}:</dt><dd class="col-sm-7">${field.value}</dd>`;
        });
        
        html += `</dl></div>`;
        col++;
    }
    
    // Add judges
    const judgeNames = formData.getAll('judge_name[]');
    const judgeLicenses = formData.getAll('judge_license[]');
    
    if (judgeNames.some(name => name.trim())) {
        if (col % 2 === 0 && col > 0) html += '</div><div class="row mt-3">';
        
        html += `<div class="col-md-6">
            <h6 class="text-warning border-bottom pb-2">
                <i class="fas fa-users me-2"></i>Judges
            </h6>
            <ol>`;
        
        judgeNames.forEach((name, index) => {
            if (name.trim()) {
                const license = judgeLicenses[index] ? ` <span class="badge bg-secondary">${judgeLicenses[index]}</span>` : '';
                html += `<li>${name}${license}</li>`;
            }
        });
        
        html += `</ol></div>`;
    }
    
    html += '</div>';
    
    document.getElementById('previewContent').innerHTML = html;
    new bootstrap.Modal(document.getElementById('previewModal')).show();
}

function resetToDefaults() {
    if (confirm('Reset all metadata to default fields? This will remove all custom fields and values.')) {
        window.location.href = window.location.pathname + '?event_id=<?= $event_id ?>&reset=1';
    }
}
</script>

</body>
</html>
