<?php
session_start();
include '../includes/auth.php';
include '../includes/db.php';
include '../includes/TokenSystem.php';
//include '../includes/permissions.php';

// Check if user has permission to delete events
if (!hasPermission('delete_events')) {
    $_SESSION['error_message'] = 'You do not have permission to delete events.';
    header('Location: events.php?error_message=You do not have permission to delete events.');
    exit;
}

$id = $_GET['id'] ?? $_GET['id'] ?? $_POST['id'] ?? null;
$confirm = $_GET['confirm'] ?? $_POST['confirm'] ?? null;
$action = $_GET['action'] ?? $_POST['action'] ?? null;

// Handle export to JSON action
if ($action === 'export' && $id) {
    try {
        $export_data = [
            'exported_at' => date('Y-m-d H:i:s'),
            'event' => null,
            'participants' => [],
            'judges' => [],
            'scores' => [],
            'control_points' => [],
            'user_assignments' => []
        ];
        
        // Get event data
        $stmt = $pdo->prepare("SELECT * FROM events WHERE id = ?");
        $stmt->execute([$id]);
        $export_data['event'] = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Get participants
        try {
            $stmt = $pdo->prepare("SELECT ep.*, p.* FROM event_participants ep 
                                  LEFT JOIN participants p ON ep.participant_id = p.id 
                                  WHERE ep.event_id = ?");
            $stmt->execute([$id]);
            $export_data['participants'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {}
        
        // Get judge assignments
        try {
            $stmt = $pdo->prepare("SELECT ja.*, u.username FROM judge_assignments ja 
                                  LEFT JOIN users u ON ja.judge_id = u.id 
                                  WHERE ja.event_id = ?");
            $stmt->execute([$id]);
            $export_data['judges'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {}
        
        // Get scoring results
        try {
            $stmt = $pdo->prepare("SELECT * FROM scoring_results WHERE event_id = ?");
            $stmt->execute([$id]);
            $export_data['scores'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            // Try alternative scores table
            try {
                $stmt = $pdo->prepare("SELECT s.* FROM scores s 
                                      INNER JOIN runs r ON s.run_id = r.id 
                                      INNER JOIN event_participants ep ON r.event_participant_id = ep.id 
                                      WHERE ep.event_id = ?");
                $stmt->execute([$id]);
                $export_data['scores'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            } catch (Exception $e2) {}
        }
        
        // Get control points
        try {
            $stmt = $pdo->prepare("SELECT * FROM event_control_points WHERE event_id = ?");
            $stmt->execute([$id]);
            $export_data['control_points'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {}
        
        // Get user assignments
        try {
            $stmt = $pdo->prepare("SELECT uae.*, u.username FROM user_assigned_events uae 
                                  LEFT JOIN users u ON uae.user_id = u.id 
                                  WHERE uae.event_id = ?");
            $stmt->execute([$id]);
            $export_data['user_assignments'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {}
        
        // Set headers for download
        $filename = 'event_' . $id . '_backup_' . date('Y-m-d_His') . '.json';
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Cache-Control: no-cache, must-revalidate');
        header('Expires: 0');
        
        echo json_encode($export_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit;
        
    } catch (Exception $e) {
        $_SESSION['error_message'] = 'Error exporting event data: ' . $e->getMessage();
        header('Location: event_delete.php?id=' . $id);
        exit;
    }
}

if (!$id) {
    $_SESSION['error_message'] = 'No event ID provided.';
    header('Location: events.php?error_message=No event ID provided.');
    exit;
}

// Fetch event details
try {
    $stmt = $pdo->prepare("SELECT * FROM events WHERE id = ?");
    $stmt->execute([$id]);
    $event = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$event) {
        $_SESSION['error_message'] = 'Event not found.';
        header('Location: events.php?error_message=Event not found.');
        exit;
    }
    
    // Get related data counts - check each table individually
    $counts = [
        'participants_count' => 0,
        'judges_count' => 0,
        'scores_count' => 0,
        'assigned_users_count' => 0,
        'control_points_count' => 0
    ];
    
    // Count event participants
    try {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM event_participants WHERE event_id = ?");
        $stmt->execute([$id]);
        $counts['participants_count'] = $stmt->fetchColumn();
    } catch (Exception $e) {
        // Table might not exist
    }
    
    // Count judge assignments
    try {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM judge_assignments WHERE event_id = ?");
        $stmt->execute([$id]);
        $counts['judges_count'] = $stmt->fetchColumn();
    } catch (Exception $e) {
        // Table might not exist
    }
    
    // Count scoring results (might be in different tables)
    try {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM scoring_results WHERE event_id = ?");
        $stmt->execute([$id]);
        $counts['scores_count'] = $stmt->fetchColumn();
    } catch (Exception $e) {
        // Try alternative scores table
        try {
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM scores s 
                                  INNER JOIN runs r ON s.run_id = r.id 
                                  INNER JOIN event_participants ep ON r.event_participant_id = ep.id 
                                  WHERE ep.event_id = ?");
            $stmt->execute([$id]);
            $counts['scores_count'] = $stmt->fetchColumn();
        } catch (Exception $e2) {
            // Neither table exists
        }
    }
    
    // Count user assignments
    try {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM user_assigned_events WHERE event_id = ?");
        $stmt->execute([$id]);
        $counts['assigned_users_count'] = $stmt->fetchColumn();
    } catch (Exception $e) {
        // Table might not exist
    }
    
    // Count control points
    try {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM event_control_points WHERE event_id = ?");
        $stmt->execute([$id]);
        $counts['control_points_count'] = $stmt->fetchColumn();
    } catch (Exception $e) {
        // Table might not exist
    }
    
} catch (Exception $e) {
    $_SESSION['error_message'] = 'Error loading event: ' . $e->getMessage();
    header('Location: events.php');
    exit;
}

// Handle deletion confirmation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $confirm === 'yes') {
    try {
        $pdo->beginTransaction();
        
        // Delete related records in order (respecting foreign key constraints)
        // Wrap each deletion in try-catch to handle missing tables
        
        // 1. Delete scoring results
        try {
            $pdo->prepare("DELETE FROM scoring_results WHERE event_id = ?")->execute([$id]);
        } catch (Exception $e) {
            // Table might not exist
        }
        
        // 2. Delete judge assignments
        try {
            $pdo->prepare("DELETE FROM judge_assignments WHERE event_id = ?")->execute([$id]);
        } catch (Exception $e) {
            // Table might not exist
        }
        
        // 3. Delete control points
        try {
            $pdo->prepare("DELETE FROM event_control_points WHERE event_id = ?")->execute([$id]);
        } catch (Exception $e) {
            // Table might not exist
        }
        
        // 4. Delete heat settings
        try {
            $pdo->prepare("DELETE FROM heat_settings WHERE event_id = ?")->execute([$id]);
        } catch (Exception $e) {
            // Table might not exist
        }
        
        // 5. Delete result configurations
        try {
            $pdo->prepare("DELETE FROM result_configurations WHERE event_id = ?")->execute([$id]);
        } catch (Exception $e) {
            // Table might not exist
        }
        
        // 6. Delete user assigned events
        try {
            $pdo->prepare("DELETE FROM user_assigned_events WHERE event_id = ?")->execute([$id]);
        } catch (Exception $e) {
            // Table might not exist
        }
        
        // 7. Delete scores (older table structure)
        try {
            $pdo->prepare("DELETE s FROM scores s 
                          INNER JOIN runs r ON s.run_id = r.id 
                          INNER JOIN event_participants ep ON r.event_participant_id = ep.id 
                          WHERE ep.event_id = ?")->execute([$id]);
        } catch (Exception $e) {
            // Might not exist or already deleted
        }
        
        // 8. Delete runs
        try {
            $pdo->prepare("DELETE r FROM runs r 
                          INNER JOIN event_participants ep ON r.event_participant_id = ep.id 
                          WHERE ep.event_id = ?")->execute([$id]);
        } catch (Exception $e) {
            // Might not exist or already deleted
        }
        
        // 9. Delete event participants (this should cascade if FK constraints are set)
        try {
            $pdo->prepare("DELETE FROM event_participants WHERE event_id = ?")->execute([$id]);
        } catch (Exception $e) {
            // Table might not exist
        }
        
        // 10. Finally delete the event itself
        $stmt = $pdo->prepare("DELETE FROM events WHERE id = ?");
        $stmt->execute([$id]);
        
        $pdo->commit();
        
        $_SESSION['success_message'] = 'Event "' . htmlspecialchars($event['name']) . '" has been successfully deleted.';
        header('Location: events.php?success_message=' . urlencode($_SESSION['success_message']));
        exit;
        
    } catch (Exception $e) {
        $pdo->rollBack();
        $_SESSION['error_message'] = 'Error deleting event: ' . $e->getMessage();
        header('Location: events.php?error_message=' . urlencode($_SESSION['error_message']));
        exit;
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Delete Event - StyleScore</title>
    <?php include_once '../includes/stylesheets.php'; ?>
    <style>
        .danger-zone {
            border: 2px solid #dc3545;
            border-radius: 8px;
            background: #fff5f5;
        }
        .stat-card {
            border-left: 4px solid #dc3545;
        }
    </style>
</head>
<body class="body-bg-aurora-bright">
    <?php include '../menu.php'; ?>

    <div class="container-fluid container-StyleScore p-4">
        <div class="row justify-content-center">
            <div class="col-lg-8 col-xl-6">
                
                <!-- Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h1 class="h3 mb-1">
                            <i class="fas fa-exclamation-triangle text-danger me-2"></i>Delete Event
                        </h1>
                        <p class="text-muted mb-0">This action cannot be undone</p>
                    </div>
                    <a href="events.php" class="btn btn-outline-secondary">
                        <i class="fas fa-arrow-left me-1"></i>Back to Events
                    </a>
                </div>

                <!-- Warning Alert -->
                <div class="alert alert-danger d-flex align-items-center mb-4">
                    <i class="fas fa-exclamation-circle fa-2x me-3"></i>
                    <div>
                        <h5 class="mb-1">Warning: Permanent Deletion</h5>
                        <p class="mb-0">You are about to permanently delete this event and all associated data. This action cannot be reversed.</p>
                    </div>
                </div>

                <!-- Event Information -->
                <div class="card shadow-sm mb-4">
                    <div class="card-header bg-danger text-white">
                        <h5 class="mb-0">
                            <i class="fas fa-calendar-times me-2"></i>Event to be Deleted
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="text-muted small">Event Name</label>
                                <div class="fw-bold"><?= htmlspecialchars($event['name']) ?></div>
                            </div>
                            <div class="col-md-6">
                                <label class="text-muted small">Event ID</label>
                                <div class="fw-bold">#<?= $event['id'] ?></div>
                            </div>
                            <div class="col-md-6">
                                <label class="text-muted small">Date</label>
                                <div class="fw-bold"><?= date('F j, Y', strtotime($event['date'])) ?></div>
                            </div>
                            <div class="col-md-6">
                                <label class="text-muted small">Location</label>
                                <div class="fw-bold"><?= htmlspecialchars($event['location'] ?? 'N/A') ?></div>
                            </div>
                            <div class="col-md-6">
                                <label class="text-muted small">Sport</label>
                                <div class="fw-bold"><?= htmlspecialchars($event['sport_discipline'] ?? 'N/A') ?></div>
                            </div>
                            <div class="col-md-6">
                                <label class="text-muted small">Status</label>
                                <div>
                                    <span class="badge bg-<?= $event['status'] === 'active' ? 'success' : 'secondary' ?>">
                                        <?= ucfirst($event['status'] ?? 'unknown') ?>
                                    </span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Related Data Impact -->
                <div class="card shadow-sm mb-4">
                    <div class="card-header bg-light">
                        <h5 class="mb-0">
                            <i class="fas fa-database me-2 text-warning"></i>Data That Will Be Deleted
                        </h5>
                    </div>
                    <div class="card-body">
                        <p class="text-muted mb-3">The following associated data will be permanently removed:</p>
                        
                        <div class="row g-3">
                            <div class="col-md-6">
                                <div class="stat-card card bg-light">
                                    <div class="card-body py-2">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <i class="fas fa-users text-danger me-2"></i>
                                                <strong>Participants</strong>
                                            </div>
                                            <span class="badge bg-danger"><?= $counts['participants_count'] ?></span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="stat-card card bg-light">
                                    <div class="card-body py-2">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <i class="fas fa-star text-danger me-2"></i>
                                                <strong>Scores</strong>
                                            </div>
                                            <span class="badge bg-danger"><?= $counts['scores_count'] ?></span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="stat-card card bg-light">
                                    <div class="card-body py-2">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <i class="fas fa-gavel text-danger me-2"></i>
                                                <strong>Judge Assignments</strong>
                                            </div>
                                            <span class="badge bg-danger"><?= $counts['judges_count'] ?></span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="stat-card card bg-light">
                                    <div class="card-body py-2">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <i class="fas fa-map-marker-alt text-danger me-2"></i>
                                                <strong>Control Points</strong>
                                            </div>
                                            <span class="badge bg-danger"><?= $counts['control_points_count'] ?></span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="stat-card card bg-light">
                                    <div class="card-body py-2">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <i class="fas fa-user-check text-danger me-2"></i>
                                                <strong>User Assignments</strong>
                                            </div>
                                            <span class="badge bg-danger"><?= $counts['assigned_users_count'] ?></span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="alert alert-warning mt-3 mb-0">
                            <i class="fas fa-info-circle me-2"></i>
                            <strong>Note:</strong> All runs, heat settings, result configurations, and other related data will also be permanently deleted.
                        </div>
                        
                        <div class="alert alert-info mt-3 mb-0">
                            <i class="fas fa-download me-2"></i>
                            <strong>Backup Recommendation:</strong> Before deleting, you can download all event data as a JSON backup file.
                            <div class="mt-2">
                            <?php
                            // Check if user can afford this action
                            if (!TokenSystem::canAfford('event_delete_backup', $_SESSION['user_id'])) {
                                $balance = TokenSystem::getBalance($_SESSION['user_id']);
                                echo "<span class=\"badge bg-warning\">Insufficient tokens. Current balance: $balance</span>";
                                ?>
                                <a href="" class="btn btn-info btn-sm disabled">
                                    <i class="fas fa-file-download me-1"></i>Download Event Backup (JSON)
                                </a>
                                <?php
                            } else {
                                $cost = TokenSystem::price('event_delete_backup');
                                    if ($cost !== null) {
                                        echo "<span class=\"badge bg-info-subtle text-info\">This action costs <span class=\"fw-bold text-black bg-white p-1 rounded\">{$cost}</span> tokens</span>";
                                    }
                                ?>
                                <a href="event_delete.php?id=<?= $id ?>&action=export" class="btn btn-info btn-sm">
                                    <i class="fas fa-file-download me-1"></i>Download Event Backup (JSON)
                                </a>
                                <?php
                            }
                            ?>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Confirmation Section -->
                <div class="danger-zone p-4">
                    <h5 class="text-danger mb-3">
                        <i class="fas fa-shield-alt me-2"></i>Confirm Deletion
                    </h5>
                    
                    <p class="mb-3">To confirm deletion of <strong><?= htmlspecialchars($event['name']) ?></strong>, please type the event name below and click the delete button.</p>
                    
                    <form method="POST" id="deleteForm" onsubmit="return confirmDeletion(event)">
                        <input type="hidden" name="event_id" value="<?= $id ?>">
                        <input type="hidden" name="confirm" value="yes">
                        
                        <div class="mb-3">
                            <label for="confirmName" class="form-label fw-bold">
                                Type event name to confirm: <code><?= htmlspecialchars($event['name']) ?></code>
                            </label>
                            <input type="text" 
                                   class="form-control" 
                                   id="confirmName" 
                                   placeholder="Enter event name"
                                   autocomplete="off"
                                   required>
                        </div>
                        
                        <div class="form-check mb-3">
                            <input class="form-check-input" type="checkbox" id="confirmCheckbox" required>
                            <label class="form-check-label" for="confirmCheckbox">
                                I understand that this action is <strong>permanent and irreversible</strong>
                            </label>
                        </div>
                        
                        <div class="d-flex justify-content-between align-items-center">
                            <a href="events.php" class="btn btn-outline-secondary">
                                <i class="fas fa-times me-1"></i>Cancel
                            </a>
                            <button type="submit" class="btn btn-danger" id="deleteBtn" disabled>
                                <i class="fas fa-trash me-1"></i>Delete Event Permanently
                            </button>
                        </div>
                    </form>
                </div>

            </div>
        </div>
    </div>

    <!-- Final Confirmation Modal -->
    <div class="modal fade" id="finalConfirmModal" tabindex="-1" aria-labelledby="finalConfirmModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-danger">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title" id="finalConfirmModalLabel">
                        <i class="fas fa-exclamation-triangle me-2"></i>Final Warning
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-danger mb-3">
                        <strong>THIS CANNOT BE UNDONE!</strong>
                    </div>
                    
                    <p class="mb-2"><strong>Are you absolutely sure you want to delete this event?</strong></p>
                    
                    <div class="bg-light p-3 rounded mb-3">
                        <p class="mb-2"><strong>Event:</strong> <?= htmlspecialchars($event['name']) ?></p>
                        <p class="mb-1"><strong>This will delete:</strong></p>
                        <ul class="mb-0">
                            <li><?= $counts['participants_count'] ?> participants</li>
                            <li><?= $counts['scores_count'] ?> scores</li>
                            <li><?= $counts['judges_count'] ?> judge assignments</li>
                            <li><?= $counts['control_points_count'] ?> control points</li>
                            <li>All related data</li>
                        </ul>
                    </div>
                    
                    <p class="text-danger mb-0">
                        <i class="fas fa-exclamation-circle me-1"></i>
                        <strong>This action is permanent and cannot be reversed!</strong>
                    </p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-1"></i>Cancel
                    </button>
                    <button type="button" class="btn btn-danger" id="finalDeleteBtn">
                        <i class="fas fa-trash me-1"></i>Delete Permanently
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        const eventName = <?= json_encode($event['name']) ?>;
        const confirmInput = document.getElementById('confirmName');
        const confirmCheckbox = document.getElementById('confirmCheckbox');
        const deleteBtn = document.getElementById('deleteBtn');
        const deleteForm = document.getElementById('deleteForm');
        let finalConfirmModal = null;
        
        function checkConfirmation() {
            const nameMatches = confirmInput.value.trim() === eventName;
            const checkboxChecked = confirmCheckbox.checked;
            
            deleteBtn.disabled = !(nameMatches && checkboxChecked);
            
            if (nameMatches) {
                confirmInput.classList.remove('is-invalid');
                confirmInput.classList.add('is-valid');
            } else if (confirmInput.value.trim() !== '') {
                confirmInput.classList.remove('is-valid');
                confirmInput.classList.add('is-invalid');
            } else {
                confirmInput.classList.remove('is-valid', 'is-invalid');
            }
        }
        
        confirmInput.addEventListener('input', checkConfirmation);
        confirmCheckbox.addEventListener('change', checkConfirmation);
        
        function confirmDeletion(event) {
            event.preventDefault();
            
            if (confirmInput.value.trim() !== eventName) {
                showAlert('Event name does not match. Please type the exact event name to confirm deletion.', 'warning');
                return false;
            }
            
            if (!confirmCheckbox.checked) {
                showAlert('You must check the confirmation checkbox.', 'warning');
                return false;
            }
            
            // Show Bootstrap modal for final confirmation
            finalConfirmModal = new bootstrap.Modal(document.getElementById('finalConfirmModal'));
            finalConfirmModal.show();
            
            return false;
        }
        
        // Handle final confirmation from modal
        document.getElementById('finalDeleteBtn').addEventListener('click', function() {
            // Close modal
            finalConfirmModal.hide();
            
            // Show loading state
            deleteBtn.disabled = true;
            deleteBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i>Deleting...';
            
            // Submit form
            deleteForm.submit();
        });
        
        function showAlert(message, type = 'info') {
            const alertDiv = document.createElement('div');
            alertDiv.className = `alert alert-${type} alert-dismissible fade show position-fixed top-0 start-50 translate-middle-x mt-3`;
            alertDiv.style.zIndex = '9999';
            alertDiv.innerHTML = `
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
            `;
            document.body.appendChild(alertDiv);
            
            setTimeout(() => {
                alertDiv.remove();
            }, 5000);
        }
    </script>

    <?php include 'footer.php'; ?>
</body>
</html>
