<?php
include '../includes/auth.php';
include '../includes/db.php';
include '../includes/validate_event_access.php';

/**
 * Division Templates - Pre-configured category sets for common event types
 */
$category_templates = [
    'park_pipe' => [
        'name' => 'Park & Pipe Competitions',
        'description' => 'Standard FIS categories for Park & Pipe events (HP- Halfpipe, SS - Slopestyle, BA - Big Air, RE - Rail Events)',
        'icon' => 'fa-snowboarding',
        'categories' => [
            [
                'category_name' => 'Senior',
                'start_year' => null,
                'end_year' => 2007,
                'gender' => 'open',
                'requires_fis_number' => true,
                'requires_licence_number' => false,
                'is_main_category' => true,
                'use_for_ranking' => true,
                'country_rule' => 'all'
            ],
            [
                'category_name' => 'Junior / U18',
                'start_year' => 2008,
                'end_year' => 2010,
                'gender' => 'open',
                'requires_fis_number' => false,
                'requires_licence_number' => false,
                'is_main_category' => true,
                'use_for_ranking' => true,
                'country_rule' => 'all'
            ],
            [
                'category_name' => 'Junior młodszy / U15',
                'start_year' => 2011,
                'end_year' => 2012,
                'gender' => 'open',
                'requires_fis_number' => false,
                'requires_licence_number' => false,
                'is_main_category' => true,
                'use_for_ranking' => true,
                'country_rule' => 'all'
            ],
            [
                'category_name' => 'Młodzik / U13',
                'start_year' => 2013,
                'end_year' => 2015,
                'gender' => 'open',
                'requires_fis_number' => false,
                'requires_licence_number' => false,
                'is_main_category' => true,
                'use_for_ranking' => true,
                'country_rule' => 'all'
            ]
        ]
    ],
    'skicross' => [
        'name' => 'Skicross Competitions',
        'description' => 'Standard categories for SX - Skicross events',
        'icon' => 'fa-skiing',
        'categories' => [
            [
                'category_name' => 'Senior',
                'start_year' => null,
                'end_year' => 2004,
                'gender' => 'open',
                'requires_fis_number' => true,
                'requires_licence_number' => false,
                'is_main_category' => true,
                'use_for_ranking' => true,
                'country_rule' => 'all'
            ],
            [
                'category_name' => 'Junior / U-21',
                'start_year' => 2005,
                'end_year' => 2007,
                'gender' => 'open',
                'requires_fis_number' => false,
                'requires_licence_number' => false,
                'is_main_category' => true,
                'use_for_ranking' => true,
                'country_rule' => 'all'
            ],
            [
                'category_name' => 'Junior młodszy / U-18',
                'start_year' => 2008,
                'end_year' => 2009,
                'gender' => 'open',
                'requires_fis_number' => false,
                'requires_licence_number' => false,
                'is_main_category' => true,
                'use_for_ranking' => true,
                'country_rule' => 'all'
            ]
        ]
    ]
];

/**
 * Generate consistent colors based on category name
 * Uses HSL color space to ensure balanced, readable colors
 */
function generateCategoryColor($categoryName, $options = []) {
    if (empty($categoryName) || !is_string($categoryName)) {
        $categoryName = 'Default Division';
    }

    $defaults = [
        'saturation_min' => 0.45,
        'saturation_max' => 0.75,
        'lightness_min' => 0.35,
        'lightness_max' => 0.65,
        'alpha' => 1.0
    ];

    $options = array_merge($defaults, $options);

    $hash = hash('sha256', strtolower(trim($categoryName)));

    $hue_seed = hexdec(substr($hash, 0, 8));
    $sat_seed = hexdec(substr($hash, 8, 8));
    $light_seed = hexdec(substr($hash, 16, 8));

    $hue = ($hue_seed % 360);

    $saturation = $options['saturation_min'] + (($sat_seed % 1000) / 1000) * ($options['saturation_max'] - $options['saturation_min']);
    $lightness = $options['lightness_min'] + (($light_seed % 1000) / 1000) * ($options['lightness_max'] - $options['lightness_min']);

    $rgb = hslToRgb($hue, $saturation, $lightness);

    $hex = sprintf('#%02x%02x%02x', $rgb[0], $rgb[1], $rgb[2]);

    $light_variant = hslToRgb($hue, $saturation * 0.6, min(0.85, $lightness + 0.25));
    $hex_light = sprintf('#%02x%02x%02x', $light_variant[0], $light_variant[1], $light_variant[2]);

    $dark_variant = hslToRgb($hue, min(1.0, $saturation * 1.2), max(0.2, $lightness - 0.25));
    $hex_dark = sprintf('#%02x%02x%02x', $dark_variant[0], $dark_variant[1], $dark_variant[2]);

    $text_color = getContrastColor($hex);

    return [
        'background-color' => $hex,
        'color' => $text_color,
        'border-color' => $hex_dark,
        'background-light' => $hex_light,
        'css_class' => 'category-' . slugify($categoryName)
    ];
}

function hslToRgb($h, $s, $l) {
    $h = $h / 360;

    if ($s == 0) {
        $r = $g = $b = $l;
    } else {
        $hue2rgb = function($p, $q, $t) {
            if ($t < 0) $t += 1;
            if ($t > 1) $t -= 1;
            if ($t < 1/6) return $p + ($q - $p) * 6 * $t;
            if ($t < 1/2) return $q;
            if ($t < 2/3) return $p + ($q - $p) * (2/3 - $t) * 6;
            return $p;
        };

        $q = $l < 0.5 ? $l * (1 + $s) : $l + $s - $l * $s;
        $p = 2 * $l - $q;

        $r = $hue2rgb($p, $q, $h + 1/3);
        $g = $hue2rgb($p, $q, $h);
        $b = $hue2rgb($p, $q, $h - 1/3);
    }

    return [round($r * 255), round($g * 255), round($b * 255)];
}

function getContrastColor($hex_color) {
    $hex_color = ltrim($hex_color, '#');
    $r = hexdec(substr($hex_color, 0, 2));
    $g = hexdec(substr($hex_color, 2, 2));
    $b = hexdec(substr($hex_color, 4, 2));

    $luminance = (0.299 * $r + 0.587 * $g + 0.114 * $b) / 255;
    return $luminance > 0.5 ? '#000000' : '#ffffff';
}

function slugify($text) {
    $text = preg_replace('/[^a-zA-Z0-9]+/', '-', $text);
    return strtolower(trim($text, '-'));
}

$event_id = $_GET['event_id'] ?? null;
$edit_id = $_GET['edit'] ?? null;
$message = '';

if (!$event_id) {
    header('Location: events.php');
    exit;
}

$event_stmt = $pdo->prepare("SELECT * FROM events WHERE id = ?");
$event_stmt->execute([$event_id]);
$event = $event_stmt->fetch(PDO::FETCH_ASSOC);

if (!$event) {
    header('Location: events.php?msg=event_not_found');
    exit;
}

$edit_category = null;
if ($edit_id) {
    $edit_stmt = $pdo->prepare("SELECT * FROM event_categories WHERE id = ? AND event_id = ?");
    $edit_stmt->execute([$edit_id, $event_id]);
    $edit_category = $edit_stmt->fetch(PDO::FETCH_ASSOC);

    if (!$edit_category) {
        header('Location: event_categories.php?event_id=' . $event_id . '&msg=category_not_found');
        exit;
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && (isset($_POST['create_category']) || isset($_POST['update_category']))) {
    try {
        $category_name = trim($_POST['category_name']);
        $start_year = $_POST['start_year'] ?: null;
        $end_year = $_POST['end_year'] ?: null;
        $gender = $_POST['gender'];
        $requires_fis = isset($_POST['requires_fis']) ? 1 : 0;
        $requires_licence = isset($_POST['requires_licence']) ? 1 : 0;
        $is_main_category = isset($_POST['is_main_category']) ? 1 : 0;
        $use_for_ranking = isset($_POST['use_for_ranking']) ? 1 : 0;
        $country_rule = $_POST['country_rule'];
        $country_codes = $_POST['country_codes'] ? explode(',', str_replace(' ', '', $_POST['country_codes'])) : [];

        $colors = generateCategoryColor($category_name);
        $colors_json = json_encode($colors);

        if (isset($_POST['update_category'])) {
            $category_id = $_POST['category_id'];
            $stmt = $pdo->prepare("
                UPDATE event_categories 
                SET category_name = ?, start_year = ?, end_year = ?, gender = ?, 
                    requires_fis_number = ?, requires_licence_number = ?, 
                    is_main_category = ?, use_for_ranking = ?,
                    country_rule = ?, country_codes = ?, colors = ?, updated_at = CURRENT_TIMESTAMP
                WHERE id = ? AND event_id = ?
            ");

            $stmt->execute([
                $category_name, $start_year, $end_year, $gender,
                $requires_fis, $requires_licence, $is_main_category, $use_for_ranking,
                $country_rule, json_encode($country_codes), $colors_json, $category_id, $event_id
            ]);

            header('Location: event_categories.php?event_id=' . $event_id . '&msg=updated');
            exit;
        } else {
            $stmt = $pdo->prepare("
                INSERT INTO event_categories 
                (event_id, category_name, start_year, end_year, gender, requires_fis_number, requires_licence_number, is_main_category, use_for_ranking, country_rule, country_codes, colors) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");

            $stmt->execute([
                $event_id, $category_name, $start_year, $end_year, $gender,
                $requires_fis, $requires_licence, $is_main_category, $use_for_ranking,
                $country_rule, json_encode($country_codes), $colors_json
            ]);

            header('Location: event_categories.php?event_id=' . $event_id . '&msg=created');
            exit;
        }
    } catch (Exception $e) {
        $error_type = isset($_POST['update_category']) ? 'update_error' : 'create_error';
        header('Location: event_categories.php?event_id=' . $event_id . '&msg=' . $error_type . '&error=' . urlencode($e->getMessage()));
        exit;
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_category'])) {
    try {
        $category_id = $_POST['category_id'];
        $stmt = $pdo->prepare('DELETE FROM event_categories WHERE id = ? AND event_id = ?');
        $stmt->execute([$category_id, $event_id]);

        header('Location: event_categories.php?event_id=' . $event_id . '&msg=deleted');
        exit;
    } catch (Exception $e) {
        header('Location: event_categories.php?event_id=' . $event_id . '&msg=delete_error&error=' . urlencode($e->getMessage()));
        exit;
    }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['regenerate_color'])) {
    try {
        $category_id = $_POST['category_id'];

        $category_stmt = $pdo->prepare('SELECT category_name FROM event_categories WHERE id = ? AND event_id = ?');
        $category_stmt->execute([$category_id, $event_id]);
        $category_data = $category_stmt->fetch(PDO::FETCH_ASSOC);

        if ($category_data) {
            $colors = generateCategoryColor($category_data['category_name']);
            $colors_json = json_encode($colors);

            $update_stmt = $pdo->prepare('UPDATE event_categories SET colors = ? WHERE id = ? AND event_id = ?');
            $update_stmt->execute([$colors_json, $category_id, $event_id]);

            header('Location: event_categories.php?event_id=' . $event_id . '&msg=color_regenerated');
            exit;
        } else {
            header('Location: event_categories.php?event_id=' . $event_id . '&msg=category_not_found');
            exit;
        }
    } catch (Exception $e) {
        header('Location: event_categories.php?event_id=' . $event_id . '&msg=color_error&error=' . urlencode($e->getMessage()));
        exit;
    }
}

// Handle template application
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['apply_template'])) {
    try {
        $template_id = $_POST['template_id'] ?? null;
        $gender_mode = $_POST['gender_mode'] ?? 'open'; // 'open' or 'split'
        
        if (!$template_id || !isset($category_templates[$template_id])) {
            throw new Exception('Invalid template selected');
        }
        
        $template = $category_templates[$template_id];
        $created_count = 0;
        
        // Begin transaction
        $pdo->beginTransaction();
        
        foreach ($template['categories'] as $category_data) {
            // Determine which genders to create based on mode
            $genders_to_create = [];
            
            if ($gender_mode === 'split') {
                // Create separate Male and Female categories
                $genders_to_create = [
                    ['code' => 'M', 'suffix' => ' - Men'],
                    ['code' => 'F', 'suffix' => ' - Women']
                ];
            } else {
                // Create single Open category
                $genders_to_create = [
                    ['code' => 'open', 'suffix' => '']
                ];
            }
            
            // Create category for each gender variant
            foreach ($genders_to_create as $gender_variant) {
                $category_name = $category_data['category_name'] . $gender_variant['suffix'];
                
                // Generate colors for the category
                $colors = generateCategoryColor($category_name);
                $colors_json = json_encode($colors);
                
                // Prepare country codes JSON
                $country_codes_json = json_encode([]);
                
                // Insert category
                $stmt = $pdo->prepare("
                    INSERT INTO event_categories 
                    (event_id, category_name, start_year, end_year, gender, 
                     requires_fis_number, requires_licence_number, country_rule, 
                     country_codes, is_main_category, use_for_ranking, colors)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ");
                
                $stmt->execute([
                    $event_id,
                    $category_name,
                    $category_data['start_year'] ?? null,
                    $category_data['end_year'] ?? null,
                    $gender_variant['code'], // Use the gender variant code
                    !empty($category_data['requires_fis_number']) ? 1 : 0,
                    !empty($category_data['requires_licence_number']) ? 1 : 0,
                    $category_data['country_rule'] ?? 'all',
                    $country_codes_json,
                    !empty($category_data['is_main_category']) ? 1 : 0,
                    !empty($category_data['use_for_ranking']) ? 1 : 0,
                    $colors_json
                ]);
                
                $created_count++;
            }
        }
        
        $pdo->commit();
        
        header("Location: event_categories.php?event_id=" . $event_id . "&msg=template_applied&count=" . $created_count);
        exit;
        
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        header("Location: event_categories.php?event_id=" . $event_id . "&msg=template_error&error=" . urlencode($e->getMessage()));
        exit;
    }
}

// Handle auto-assignment
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['auto_assign'])) {
    try {
        $assignments_made = 0;
        $assignment_log = [];
        $summary_stats = [
            'total_participants' => 0,
            'assigned' => 0,
            'reassigned' => 0,
            'unmatched' => 0,
            'already_assigned' => 0,
            'category_counts' => []
        ];
        
        // Get all participants for this event
        $participants_stmt = $pdo->prepare("
            SELECT p.*, ep.id as event_participant_id, ep.category as current_category
            FROM participants p 
            INNER JOIN event_participants ep ON p.id = ep.participant_id 
            WHERE ep.event_id = ?
        ");
        $participants_stmt->execute([$event_id]);
        $participants = $participants_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get ALL categories for this event, not just main ones
        $categories_stmt = $pdo->prepare("SELECT * FROM event_categories WHERE event_id = ? ORDER BY is_main_category DESC, category_name");
        $categories_stmt->execute([$event_id]);
        $categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get only MAIN categories for matching primary category
        $main_categories = array_filter($categories, function($cat) {
            return $cat['is_main_category'] == 1;
        });
        
        if (empty($main_categories)) {
            header("Location: event_categories.php?event_id=" . $event_id . "&msg=auto_assign_no_main_categories");
            exit;
        }
        
        if (empty($participants)) {
            header("Location: event_categories.php?event_id=" . $event_id . "&msg=auto_assign_no_participants");
            exit;
        }
        
        $summary_stats['total_participants'] = count($participants);
        
        foreach ($participants as $participant) {
            $matched_main_category = null;
            $matched_main_score = -1;
            $match_reasons = [];
            $all_matching_category_ids = []; // Track ALL matching category IDs
            
            // First pass: Find the best matching MAIN category (for the 'category' column)
            foreach ($main_categories as $category) {
                $current_match_score = 0;
                $category_matches = true;
                $reasons = [];
                
                // Check birth year restrictions
                if ($category['start_year'] || $category['end_year']) {
                    $birth_year = null;
                    if ($participant['date_of_birth']) {
                        $birth_year = date('Y', strtotime($participant['date_of_birth']));
                    } elseif (isset($participant['birth_year']) && $participant['birth_year']) {
                        $birth_year = $participant['birth_year'];
                    }
                    
                    if ($birth_year) {
                        if ($category['start_year'] && $birth_year < $category['start_year']) {
                            $category_matches = false;
                            $reasons[] = "Too old (born {$birth_year}, need >= {$category['start_year']})";
                        }
                        if ($category['end_year'] && $birth_year > $category['end_year']) {
                            $category_matches = false;
                            $reasons[] = "Too young (born {$birth_year}, need <= {$category['end_year']})";
                        }
                        if ($category_matches) {
                            $current_match_score += 20;
                            $reasons[] = "Age match ({$birth_year})";
                        }
                    } else {
                        $current_match_score -= 10;
                        $reasons[] = "No birth year data";
                    }
                }
                
                // Check gender restrictions
                if ($category['gender'] !== 'open') {
                    $participant_gender = strtoupper(substr($participant['gender'] ?? '', 0, 1));
                    if ($participant_gender && $category['gender'] === $participant_gender) {
                        $current_match_score += 15;
                        $reasons[] = "Gender match ({$participant_gender})";
                    } else {
                        $category_matches = false;
                        $reasons[] = "Gender mismatch (need {$category['gender']}, got {$participant_gender})";
                    }
                } else {
                    $current_match_score += 5;
                    $reasons[] = "Open gender";
                }
                
                // Check FIS number requirement
                if ($category['requires_fis_number']) {
                    if (!empty($participant['fis_code'])) {
                        $current_match_score += 10;
                        $reasons[] = "Has FIS: {$participant['fis_code']}";
                    } else {
                        $category_matches = false;
                        $reasons[] = "Missing required FIS number";
                    }
                }
                
                // Check licence requirement
                if ($category['requires_licence_number']) {
                    if (!empty($participant['licence_number'])) {
                        $current_match_score += 10;
                        $reasons[] = "Has licence: {$participant['licence_number']}";
                    } else {
                        $category_matches = false;
                        $reasons[] = "Missing required licence number";
                    }
                }
                
                // Check country restrictions
                if ($category['country_rule'] !== 'all') {
                    $country_codes = json_decode($category['country_codes'] ?: '[]', true);
                    $participant_country = strtoupper($participant['country'] ?? '');
                    
                    if ($category['country_rule'] === 'include') {
                        if (in_array($participant_country, $country_codes)) {
                            $current_match_score += 10;
                            $reasons[] = "Country included ({$participant_country})";
                        } else {
                            $category_matches = false;
                            $reasons[] = "Country not in allowed list ({$participant_country})";
                        }
                    } elseif ($category['country_rule'] === 'exclude') {
                        if (in_array($participant_country, $country_codes)) {
                            $category_matches = false;
                            $reasons[] = "Country excluded ({$participant_country})";
                        } else {
                            $current_match_score += 5;
                            $reasons[] = "Country allowed ({$participant_country})";
                        }
                    }
                }
                
                // If this category matches and has a higher score, select it as the main category
                if ($category_matches && $current_match_score > $matched_main_score) {
                    $matched_main_category = $category;
                    $matched_main_score = $current_match_score;
                    $match_reasons = $reasons;
                }
            }
            
            // Second pass: Find ALL matching categories (for the 'category_id' JSON array)
            foreach ($categories as $category) {
                $category_matches = true;
                
                // Check birth year restrictions
                if ($category['start_year'] || $category['end_year']) {
                    $birth_year = null;
                    if ($participant['date_of_birth']) {
                        $birth_year = date('Y', strtotime($participant['date_of_birth']));
                    } elseif (isset($participant['birth_year']) && $participant['birth_year']) {
                        $birth_year = $participant['birth_year'];
                    }
                    
                    if ($birth_year) {
                        if ($category['start_year'] && $birth_year < $category['start_year']) {
                            $category_matches = false;
                        }
                        if ($category['end_year'] && $birth_year > $category['end_year']) {
                            $category_matches = false;
                        }
                    }
                }
                
                // Check gender restrictions
                if ($category['gender'] !== 'open') {
                    $participant_gender = strtoupper(substr($participant['gender'] ?? '', 0, 1));
                    if (!($participant_gender && $category['gender'] === $participant_gender)) {
                        $category_matches = false;
                    }
                }
                
                // Check FIS number requirement
                if ($category['requires_fis_number'] && empty($participant['fis_code'])) {
                    $category_matches = false;
                }
                
                // Check licence requirement
                if ($category['requires_licence_number'] && empty($participant['licence_number'])) {
                    $category_matches = false;
                }
                
                // Check country restrictions
                if ($category['country_rule'] !== 'all') {
                    $country_codes = json_decode($category['country_codes'] ?: '[]', true);
                    $participant_country = strtoupper($participant['country'] ?? '');
                    
                    if ($category['country_rule'] === 'include') {
                        if (!in_array($participant_country, $country_codes)) {
                            $category_matches = false;
                        }
                    } elseif ($category['country_rule'] === 'exclude') {
                        if (in_array($participant_country, $country_codes)) {
                            $category_matches = false;
                        }
                    }
                }
                
                // If this category matches, add its ID to the matching list
                if ($category_matches) {
                    $all_matching_category_ids[] = (int)$category['id'];
                }
            }
            
            // Process the assignment result
            $participant_name = $participant['first_name'] . ' ' . $participant['last_name'];
            $current_category = $participant['current_category'];
            
            if ($matched_main_category) {
                $new_category = $matched_main_category['category_name'];
                
                // Initialize category count if not exists
                if (!isset($summary_stats['category_counts'][$new_category])) {
                    $summary_stats['category_counts'][$new_category] = 0;
                }
                
                // Store all matching category IDs as JSON in a separate column
                $category_ids_json = json_encode($all_matching_category_ids);
                
                if ($current_category !== $new_category) {
                    // Update the participant's category AND category_id (use the main category's ID)
                    $update_stmt = $pdo->prepare("
                        UPDATE event_participants 
                        SET category = ?, category_id = ?, matching_category_ids = ?
                        WHERE id = ?
                    ");
                    $update_stmt->execute([
                        $new_category, 
                        $matched_main_category['id'],  // Use single ID for foreign key
                        $category_ids_json,            // Store full array in separate column
                        $participant['event_participant_id']
                    ]);
                    
                    $assignments_made++;
                    $summary_stats['category_counts'][$new_category]++;
                    
                    if ($current_category) {
                        $summary_stats['reassigned']++;
                        $action = 'REASSIGNED';
                    } else {
                        $summary_stats['assigned']++;
                        $action = 'ASSIGNED';
                    }
                    
                    $assignment_log[] = [
                        'name' => $participant_name,
                        'from' => $current_category ?: 'Unassigned',
                        'to' => $new_category,
                        'action' => $action,
                        'score' => $matched_main_score,
                        'reasons' => $match_reasons,
                        'country' => $participant['country'] ?? '',
                        'gender' => $participant['gender'] ?? '',
                        'birth_year' => $participant['date_of_birth'] ? date('Y', strtotime($participant['date_of_birth'])) : ($participant['birth_year'] ?? ''),
                        'matching_categories' => count($all_matching_category_ids)
                    ];
                } else {
                    // Already in correct main category, but update category_id anyway
                    $update_stmt = $pdo->prepare("
                        UPDATE event_participants 
                        SET category_id = ?, matching_category_ids = ?
                        WHERE id = ?
                    ");
                    $update_stmt->execute([
                        $matched_main_category['id'],  // Use single ID for foreign key
                        $category_ids_json,            // Store full array in separate column
                        $participant['event_participant_id']
                    ]);
                    
                    $summary_stats['already_assigned']++;
                    $summary_stats['category_counts'][$new_category]++;
                    
                    $assignment_log[] = [
                        'name' => $participant_name,
                        'from' => $current_category,
                        'to' => $new_category,
                        'action' => 'ALREADY_ASSIGNED',
                        'score' => $matched_main_score,
                        'reasons' => $match_reasons,
                        'country' => $participant['country'] ?? '',
                        'gender' => $participant['gender'] ?? '',
                        'birth_year' => $participant['date_of_birth'] ? date('Y', strtotime($participant['date_of_birth'])) : ($participant['birth_year'] ?? ''),
                        'matching_categories' => count($all_matching_category_ids)
                    ];
                }
            } else {
                // No matching main category found
                if (!empty($all_matching_category_ids)) {
                    // We have non-main categories but no main category
                    // Choose the first matching category as the foreign key reference
                    $first_category_id = $all_matching_category_ids[0];
                    $category_ids_json = json_encode($all_matching_category_ids);
                    
                    $update_stmt = $pdo->prepare("
                        UPDATE event_participants 
                        SET category_id = ?, matching_category_ids = ?, category = NULL
                        WHERE id = ?
                    ");
                    $update_stmt->execute([
                        $first_category_id,       // Use single ID for foreign key
                        $category_ids_json,       // Store full array in separate column
                        $participant['event_participant_id']
                    ]);
                    
                    $assignment_log[] = [
                        'name' => $participant_name,
                        'from' => $current_category ?: 'Unassigned',
                        'to' => 'NO MAIN MATCH (But matches other categories)',
                        'action' => 'PARTIAL_MATCH',
                        'score' => 0,
                        'reasons' => ['Matches ' . count($all_matching_category_ids) . ' non-main categories'],
                        'country' => $participant['country'] ?? '',
                        'gender' => $participant['gender'] ?? '',
                        'birth_year' => $participant['date_of_birth'] ? date('Y', strtotime($participant['date_of_birth'])) : ($participant['birth_year'] ?? ''),
                        'matching_categories' => count($all_matching_category_ids)
                    ];
                } else {
                    // No matching category found at all
                    $update_stmt = $pdo->prepare("
                        UPDATE event_participants 
                        SET category_id = NULL, category = NULL, matching_category_ids = NULL
                        WHERE id = ?
                    ");
                    $update_stmt->execute([$participant['event_participant_id']]);
                    
                    $assignment_log[] = [
                        'name' => $participant_name,
                        'from' => $current_category ?: 'Unassigned',
                        'to' => 'NO MATCH FOUND',
                        'action' => 'UNMATCHED',
                        'score' => 0,
                        'reasons' => ['No category matches participant criteria'],
                        'country' => $participant['country'] ?? '',
                        'gender' => $participant['gender'] ?? '',
                        'birth_year' => $participant['date_of_birth'] ? date('Y', strtotime($participant['date_of_birth'])) : ($participant['birth_year'] ?? ''),
                        'matching_categories' => 0
                    ];
                }
            }
        }
        
        // Store comprehensive assignment data in session
        if (session_status() === PHP_SESSION_NONE) {
            session_start();
        }
        $_SESSION['assignment_log'] = $assignment_log;
        $_SESSION['assignments_made'] = $assignments_made;
        $_SESSION['summary_stats'] = $summary_stats;
        
        // Redirect with success message
        header("Location: event_categories.php?event_id=" . $event_id . "&msg=auto_assigned");
        exit;
    } catch (Exception $e) {
        // Redirect with error message
        header("Location: event_categories.php?event_id=" . $event_id . "&msg=auto_assign_error&error=" . urlencode($e->getMessage()));
        exit;
    }
}

// Handle manual assignment
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['manual_assign'])) {
    try {
        $count = 0;
        
        $stmt = $pdo->prepare("UPDATE event_participants SET category = ?, category_id = ? WHERE id = ?");
        $cat_stmt = $pdo->prepare("SELECT category_name FROM event_categories WHERE id = ?");
        $clear_stmt = $pdo->prepare("UPDATE event_participants SET category = NULL, category_id = NULL WHERE id = ?");
        
        // Handle single assignment
        if (isset($_POST['ep_id']) && isset($_POST['category_id'])) {
            $ep_id = $_POST['ep_id'];
            $category_id = $_POST['category_id'];
            
            if ($category_id === 'none' || empty($category_id)) {
                $clear_stmt->execute([$ep_id]);
                $count++;
            } else {
                // Get category name
                $cat_stmt->execute([$category_id]);
                $cat_name = $cat_stmt->fetchColumn();
                
                if ($cat_name) {
                    $stmt->execute([$cat_name, $category_id, $ep_id]);
                    $count++;
                }
            }
        }
        // Handle bulk assignment
        elseif (isset($_POST['assignments'])) {
            $assignments = $_POST['assignments'] ?? [];
            foreach ($assignments as $ep_id => $category_id) {
                if ($category_id === 'none' || empty($category_id)) {
                    $clear_stmt->execute([$ep_id]);
                    $count++;
                } else {
                    // Get category name
                    $cat_stmt->execute([$category_id]);
                    $cat_name = $cat_stmt->fetchColumn();
                    
                    if ($cat_name) {
                        $stmt->execute([$cat_name, $category_id, $ep_id]);
                        $count++;
                    }
                }
            }
        }
        
        header("Location: event_categories.php?event_id=" . $event_id . "&msg=manual_assigned&count=" . $count);
        exit;
    } catch (Exception $e) {
        header("Location: event_categories.php?event_id=" . $event_id . "&msg=manual_assign_error&error=" . urlencode($e->getMessage()));
        exit;
    }
}

// Handle GET messages after redirect
$message = '';
if (isset($_GET['msg'])) {
    $error_detail = isset($_GET['error']) ? htmlspecialchars($_GET['error']) : '';
    
    switch ($_GET['msg']) {
        case 'created':
            $message = '<div class="alert alert-success alert alert-dismissible fade show">
                <i class="fas fa-check-circle me-2"></i>Division created successfully!
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'updated':
            $message = '<div class="alert alert-success alert alert-dismissible fade show">
                <i class="fas fa-check-circle me-2"></i>Division updated successfully!
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'deleted':
            $message = '<div class="alert alert-warning alert alert-dismissible fade show">
                <i class="fas fa-trash me-2"></i>Division deleted successfully!
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;

        case 'create_error':
            $message = '<div class="alert alert-danger alert alert-dismissible fade show">
                <i class="fas fa-exclamation-triangle me-2"></i>Error creating category' . ($error_detail ? ': ' . $error_detail : '') . '
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'update_error':
            $message = '<div class="alert alert-danger alert alert-dismissible fade show">
                <i class="fas fa-exclamation-triangle me-2"></i>Error updating category' . ($error_detail ? ': ' . $error_detail : '') . '
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'delete_error':
            $message = '<div class="alert alert-danger alert alert-dismissible fade show">
                <i class="fas fa-exclamation-triangle me-2"></i>Error deleting category' . ($error_detail ? ': ' . $error_detail : '') . '
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'auto_assign_error':
            $message = '<div class="alert alert-danger alert alert-dismissible fade show">
                <i class="fas fa-exclamation-triangle me-2"></i>Error during auto-assignment' . ($error_detail ? ': ' . $error_detail : '') . '
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'category_not_found':
            $message = '<div class="alert alert-warning alert alert-dismissible fade show">
                <i class="fas fa-exclamation-triangle me-2"></i>Division not found or does not belong to this event.
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
             case 'auto_assign_no_categories':
            $message = '<div class="alert alert-warning alert alert-dismissible fade show">
                <i class="fas fa-exclamation-triangle me-2"></i>Cannot run auto-assignment: No categories have been created yet.
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'auto_assign_no_main_categories':
            $message = '<div class="alert alert-warning alert alert-dismissible fade show">
                <i class="fas fa-exclamation-triangle me-2"></i>Cannot run auto-assignment: No main categories have been created yet. Please mark at least one category as "Main Division".
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'auto_assigned':
            //session_start();
            $assignments_made = $_SESSION['assignments_made'] ?? 0;
            $assignment_log = $_SESSION['assignment_log'] ?? [];
            $summary_stats = $_SESSION['summary_stats'] ?? [];
            
            $message = '<div class="alert alert-info alert alert-dismissible fade show">
                <div class="d-flex justify-content-between align-items-start">
                    <div>
                        <h5 class="mb-3"><i class="fas fa-robot me-2"></i>Auto-Assignment Complete!</h5>
                        
                        <!-- Summary Statistics -->
                        <div class="row mb-3">
                            <div class="col-md-12">
                                <div class="row text-center">
                                    <div class="col">
                                        <div class="bg-primary text-white rounded p-2 mb-1">
                                            <strong>' . ($summary_stats['total_participants'] ?? 0) . '</strong>
                                        </div>
                                        <small>Total Athletes</small>
                                    </div>
                                    <div class="col">
                                        <div class="bg-success text-white rounded p-2 mb-1">
                                            <strong>' . ($summary_stats['assigned'] ?? 0) . '</strong>
                                        </div>
                                        <small>Newly Assigned</small>
                                    </div>
                                    <div class="col">
                                        <div class="bg-warning rounded p-2 mb-1">
                                            <strong>' . ($summary_stats['reassigned'] ?? 0) . '</strong>
                                        </div>
                                        <small>Reassigned</small>
                                    </div>
                                    <div class="col">
                                        <div class="bg-info text-white rounded p-2 mb-1">
                                            <strong>' . ($summary_stats['already_assigned'] ?? 0) . '</strong>
                                        </div>
                                        <small>Already Correct</small>
                                    </div>
                                    <div class="col">
                                        <div class="bg-danger text-white rounded p-2 mb-1">
                                            <strong>' . ($summary_stats['unmatched'] ?? 0) . '</strong>
                                        </div>
                                        <small>No Match</small>
                                    </div>
                                </div>
                            </div>
                        </div>';
            
            // Division Distribution
            if (!empty($summary_stats['category_counts'])) {
                $message .= '<div class="mb-3">
                    <h6><i class="fas fa-chart-pie me-2"></i>Division Distribution:</h6>
                    <div class="row">';
                
                foreach ($summary_stats['category_counts'] as $category_name => $count) {
                    $message .= '<div class="col-auto mb-1">
                        <span class="badge bg-secondary">' . htmlspecialchars($category_name) . ': <strong>' . $count . '</strong></span>
                    </div>';
                }
                
                $message .= '</div></div>';
            }
            
            $message .= '</div>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>';
            
            // Detailed Assignment Log
            if (!empty($assignment_log)) {
                $message .= '<div class="mt-3">
                    <div class="d-flex justify-content-between align-items-center mb-2">
                        <h6 class="mb-0"><i class="fas fa-list me-2"></i>Assignment Details</h6>
                        <div class="" role="group">
                            <button type="button" class="btn btn-outline-secondary" onclick="filterAssignments(\'all\')">All</button>
                            <button type="button" class="btn btn-outline-success" onclick="filterAssignments(\'ASSIGNED\')">Assigned</button>
                            <button type="button" class="btn btn-outline-warning" onclick="filterAssignments(\'REASSIGNED\')">Reassigned</button>
                            <button type="button" class="btn btn-outline-danger" onclick="filterAssignments(\'UNMATCHED\')">Unmatched</button>
                        </div>
                    </div>
                    
                    <div id="assignmentDetails" style="max-height: 400px; overflow-y: auto; font-size: 0.9em;" class="border rounded p-2">';
                
                foreach ($assignment_log as $index => $log) {
                    $action_colors = [
                        'ASSIGNED' => 'success',
                        'REASSIGNED' => 'warning',
                        'ALREADY_ASSIGNED' => 'info',
                        'UNMATCHED' => 'danger'
                    ];
                    
                    $action_color = $action_colors[$log['action']] ?? 'secondary';
                    $to_color = $log['to'] === 'NO MATCH FOUND' ? 'danger' : 'success';
                    
                    $message .= '<div class="assignment-row border-bottom py-2" data-action="' . $log['action'] . '">
                        <div class="d-flex justify-content-between align-items-start">
                            <div class="flex-grow-1">
                                <div class="d-flex align-items-center mb-1">
                                    <strong>' . htmlspecialchars($log['name']) . '</strong>
                                    <span class="badge bg-' . $action_color . ' ms-2">' . $log['action'] . '</span>
                                    <span class="badge bg-secondary ms-1">Score: ' . $log['score'] . '</span>
                                </div>
                                
                                <div class="mb-1">
                                    <span class="badge bg-light text-dark me-1">' . htmlspecialchars($log['from']) . '</span>
                                    <i class="fas fa-arrow-right mx-1 text-muted"></i>
                                    <span class="badge bg-' . $to_color . '">' . htmlspecialchars($log['to']) . '</span>
                                </div>
                                
                                <div class="small text-muted mb-1">
                                    <i class="fas fa-user me-1"></i>' . htmlspecialchars($log['gender']) . ' • 
                                    <i class="fas fa-flag me-1"></i>' . htmlspecialchars($log['country']) . ' • 
                                    <i class="fas fa-birthday-cake me-1"></i>' . htmlspecialchars($log['birth_year'] ?: 'Unknown') . '
                                </div>';
                    
                    if (!empty($log['reasons'])) {
                        $message .= '<div class="small">
                            <strong>Matching details:</strong> ' . implode(', ', array_map('htmlspecialchars', $log['reasons'])) . '
                        </div>';
                    }
                    
                    $message .= '</div>
                        </div>
                    </div>';
                }
                
                $message .= '</div></div>';
            }
            
            $message .= '</div>';
            
            // Clear session data
            unset($_SESSION['assignment_log']);
            unset($_SESSION['assignments_made']);
            unset($_SESSION['summary_stats']);
            break;
            
        case 'auto_assign_no_participants':
            $message = '<div class="alert alert-warning alert alert-dismissible fade show">
                <i class="fas fa-exclamation-triangle me-2"></i>Cannot run auto-assignment: No participants found for this event.
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;    
        case 'auto_assign_no_categories':
            $message = '<div class="alert alert-warning alert alert-dismissible fade show">
                <i class="fas fa-exclamation-triangle me-2"></i>Cannot run auto-assignment: No categories have been created yet.
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'color_regenerated':
            $message = '<div class="alert alert-success alert alert-dismissible fade show">
                <i class="fas fa-palette me-2"></i>Division color regenerated successfully!
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'color_error':
            $message = '<div class="alert alert-danger alert alert-dismissible fade show">
                <i class="fas fa-exclamation-triangle me-2"></i>Error regenerating color' . ($error_detail ? ': ' . $error_detail : '') . '
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'template_applied':
            $count = $_GET['count'] ?? 0;
            $message = '<div class="alert alert-success alert alert-dismissible fade show">
                <i class="fas fa-magic me-2"></i><strong>Template Applied!</strong> Successfully created ' . $count . ' categories from template.
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;
        case 'template_error':
            $message = '<div class="alert alert-danger alert alert-dismissible fade show">
                <i class="fas fa-exclamation-triangle me-2"></i>Error applying template' . ($error_detail ? ': ' . $error_detail : '') . '
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>';
            break;    
    }
}

// Fetch existing categories
$categories_stmt = $pdo->prepare("SELECT *, colors FROM event_categories WHERE event_id = ? ORDER BY category_name");
$categories_stmt->execute([$event_id]);
$categories_raw = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);

// Remove any potential duplicates based on ID
$categories = [];
$seen_ids = [];
foreach ($categories_raw as $category) {
    if (!in_array($category['id'], $seen_ids)) {
        $categories[] = $category;
        $seen_ids[] = $category['id'];
    }
}

// Generate colors for categories that don't have them
$updated_categories = [];
foreach ($categories as $category) {
    try {
        if (empty($category['colors'])) {
            $colors = generateCategoryColor($category['category_name']);
            $colors_json = json_encode($colors);
            
            // Update the database with generated colors
            $update_stmt = $pdo->prepare("UPDATE event_categories SET colors = ? WHERE id = ?");
            $update_stmt->execute([$colors_json, $category['id']]);
            
            // Update the category with new colors
            $category['colors'] = $colors_json;
        }
        $updated_categories[] = $category;
    } catch (Exception $e) {
        // If color generation fails, use the category without colors
        error_log("Color generation failed for category " . $category['category_name'] . ": " . $e->getMessage());
        $updated_categories[] = $category;
    }
}
$categories = $updated_categories;

// Fetch participants for manual assignment
$participants_stmt = $pdo->prepare("
    SELECT p.*, ep.category_id, ep.bib_number, ep.id as ep_id
    FROM participants p 
    JOIN event_participants ep ON p.id = ep.participant_id 
    WHERE ep.event_id = ? 
    ORDER BY p.last_name, p.first_name
");
$participants_stmt->execute([$event_id]);
$participants_event = $participants_stmt->fetchAll(PDO::FETCH_ASSOC);

// Load country codes from file
$country_codes = [];
if (file_exists('../data/country_codes.json')) {
    $country_codes = json_decode(file_get_contents('../data/country_codes.json'), true);
}

// Parse country codes for edit mode
$edit_country_codes_string = '';
if ($edit_category && $edit_category['country_codes']) {
    $edit_country_codes_array = json_decode($edit_category['country_codes'], true);
    $edit_country_codes_string = implode(', ', $edit_country_codes_array);
}

// Prepare data for category table (jExcel)
$category_table_rows = [];
foreach ($categories as $category) {
    $country_codes_array = json_decode($category['country_codes'] ?? '[]', true);
    if (!is_array($country_codes_array)) {
        $country_codes_array = [];
    }
    $country_codes_display = implode(', ', array_map('trim', $country_codes_array));

    $gender_labels = [
        'M' => 'Male Only',
        'F' => 'Female Only',
        'open' => 'Open / All'
    ];
    $gender_label = $gender_labels[$category['gender']] ?? 'Open / All';

    $requirements = [];
    if (!empty($category['requires_fis_number'])) {
        $requirements[] = 'FIS';
    }
    if (!empty($category['requires_licence_number'])) {
        $requirements[] = 'Licence';
    }
    $requirements_text = empty($requirements) ? 'None' : implode(', ', $requirements);

    $colors_decoded = json_decode($category['colors'] ?? '{}', true);
    $bg_color = $colors_decoded['background-color'] ?? '#6c757d';
    $border_color = $colors_decoded['border-color'] ?? '#495057';
    $text_color = $colors_decoded['color'] ?? '#ffffff';
    $color_preview = '<span class="category-color-chip" style="display:inline-block;width:28px;height:18px;border-radius:4px;background:' . htmlspecialchars($bg_color, ENT_QUOTES) . ';border:2px solid ' . htmlspecialchars($border_color, ENT_QUOTES) . ';"></span>';
    $color_preview .= '<span class="ms-2 small text-muted">' . htmlspecialchars($bg_color, ENT_QUOTES) . '</span>';

    $year_range = ($category['start_year'] ?: 'Any') . ' - ' . ($category['end_year'] ?: 'Any');

    $actions_html = "
        <div class=\"\">
            <a href=\"event_categories.php?event_id={$event_id}&edit={$category['id']}\" class=\"btn btn-outline-warning\" title=\"Edit Division\">
                <i class=\"fas fa-edit\"></i>
            </a>
            <button type=\"button\" class=\"btn btn-outline-danger btn-sm\" title=\"Delete Division\"
                    data-bs-toggle=\"modal\" data-bs-target=\"#deleteModal{$category['id']}\">
                <i class=\"fas fa-trash\"></i>
            </button>
            <form method=\"post\" class=\"d-inline\">
                <input type=\"hidden\" name=\"regenerate_color\" value=\"1\">
                <input type=\"hidden\" name=\"category_id\" value=\"{$category['id']}\">
                <button type=\"submit\" class=\"btn btn-outline-info btn-sm\" title=\"Regenerate Color\">
                    <i class=\"fas fa-palette\"></i>
                </button>
            </form>
        </div>
    ";

    $category_table_rows[] = [
        htmlspecialchars($category['category_name']),
        $gender_label,
        $year_range,
        !empty($category['is_main_category']) ? 'Yes' : 'No',
        !empty($category['use_for_ranking']) ? 'Yes' : 'No',
        $requirements_text,
        ucfirst($category['country_rule'] ?? 'all'),
        $country_codes_display ?: '-',
        $color_preview,
        $actions_html
    ];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Event Categories - <?= htmlspecialchars($event['name']) ?></title>
    <?php include_once '../includes/stylesheets.php'; ?>
    <style>
        .card {
            border: none;
            box-shadow: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
            border-radius: 0.5rem;
        }
        
        .card-header {
            background: transparent;
            border-bottom: 1px solid rgba(0,0,0,.125);
            font-weight: 600;
        }
        
        .category-item {
            border: 1px solid #dee2e6;
            border-radius: 0.375rem;
            padding: 0.5rem;
            margin-bottom: 0.375rem;
            background: #ffffff;
            transition: all 0.15s ease-in-out;
        }
        
        .category-item:hover {
            border-color: #0d6efd;
            box-shadow: 0 0.125rem 0.25rem rgba(13, 110, 253, 0.1);
        }
        
        .category-item.editing {
            border-color: #ffc107;
            background: #fff3cd;
        }
        
        .auto-assign-card {
            background: linear-gradient(135deg, #198754 0%, #20c997 100%);
            color: white;
            border: none;
        }
        
        .form-section {
            background: #f8f9fa;
            border: 1px solid #e9ecef;
            border-radius: 0.375rem;
            padding: 0.75rem;
            margin-bottom: 0.5rem;
        }
        
        .section-title {
            color: #495057;
            font-weight: 600;
            margin-bottom: 0.5rem;
            padding-bottom: 0.25rem;
            border-bottom: 2px solid #0d6efd;
            font-size: 0.8rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .empty-state {
            text-align: center;
            padding: 3rem 1rem;
            color: #6c757d;
        }
        
        .edit-indicator {
            background: linear-gradient(135deg, #ffc107 0%, #ffca2c 100%);
            color: #212529;
            padding: 0.5rem 1rem;
            border-radius: 0.375rem 0.375rem 0 0;
            font-weight: 600;
            border-bottom: 1px solid rgba(0,0,0,.125);
        }
        
        .assignment-row {
            border-bottom: 1px solid #e9ecef;
            padding: 0.75rem 0;
        }
        
        .assignment-row:last-child {
            border-bottom: none;
        }
        
        .btn {
            border-radius: 0.375rem;
            font-weight: 500;
        }
        
        .btn-group-sm > .btn {
            padding: 0.25rem 0.5rem;
            font-size: 0.875rem;
        }
        
        .badge {
            font-weight: 500;
        }
        
        .color-preview {
            transition: all 0.2s ease-in-out;
            cursor: pointer;
        }
        
        .color-preview:hover {
            transform: scale(1.1);
            box-shadow: 0 4px 8px rgba(0,0,0,0.2);
        }
        
        .category-item:hover .color-preview {
            box-shadow: 0 2px 4px rgba(0,0,0,0.15);
        }
        
        .color-info-badge {
            font-family: 'Courier New', monospace;
            font-size: 10px;
            border: 1px solid rgba(0,0,0,0.1);
        }
        
        .form-check-input:checked {
            background-color: #0d6efd;
            border-color: #0d6efd;
        }
        
        .text-primary {
            color: #0d6efd !important;
        }
        
        .bg-primary {
            background-color: #0d6efd !important;
        }

        .category-color-chip {
            display: inline-block;
            vertical-align: middle;
        }
        
       
       
        .btn-primary {
            background-color: #0d6efd;
            border-color: #0d6efd;
        }
        
        .btn-primary:hover {
            background-color: #0b5ed7;
            border-color: #0a58ca;
        }
        
        .alert {
            border: none;
            border-radius: 0.5rem;
        }
        
        .progress {
            height: 0.25rem;
            border-radius: 0.25rem;
        }
        
        .card-body {
            padding: 0.75rem;
        }
        
        @media (max-width: 768px) {
            .btn-group-vertical {
                flex-direction: row;
            }
            
            .category-item {
                padding: 0.375rem;
            }
            
            .form-section {
                padding: 0.5rem;
                margin-bottom: 0.375rem;
            }
        }
    </style>
</head>
<body class="body-bg-aurora-bright">
    <?php include '../menu.php'; ?>
    
    <div class="container-fluid container-StyleScore p-4">
            <!-- event action menu. -->
<div class="row mb-2">
    <div class="col-12">
        <?php       
        $use_localStorage = true;
        $show_header = false;
        //$show_public_view = false;
        include 'event_actions_menu.php'; 
        ?>
    </div>
</div>

        <!-- Page Header -->
        <div class="d-flex justify-content-between align-items-center mb-3 flex-column flex-md-row">
            <div>
                <h1 class="h4 mb-1">
                    <i class="fas fa-tags text-primary me-2"></i>Divisions & Categories for "<strong><?= htmlspecialchars($event['name']) ?></strong>"
                </h1>
                
            </div>
            <div class="btn-group">
                        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#categoryFormModal">
                            <i class="fas fa-plus me-2"></i>Add Division
                        </button>
                        <button type="button" class="btn btn-outline-primary" data-bs-toggle="modal" data-bs-target="#templateModal">
                            <i class="fas fa-magic me-2"></i>Templates
                        </button>
                        <?php 
                        $main_categories = array_filter($categories, function($cat) { return $cat['is_main_category']; });
                        ?>
                        <form method="post" class="d-inline" id="autoAssignForm">
                            <input type="hidden" name="auto_assign" value="1">
                            <button type="submit" class="btn btn-outline-success" <?= empty($main_categories) ? 'disabled' : '' ?>
                                    title="Automatically assign participants to matching categories"
                                    onclick="return confirm('This will automatically assign all participants to matching MAIN categories only. Current assignments may be changed. Continue?')">
                                <i class="fas fa-robot me-2"></i>Auto-Assign
                            </button>
                        </form>
                    
            <a href="event_dashboard.php?event_id=<?= $event_id ?>" class="btn btn-outline-secondary btn-sm">
                <i class="fas fa-arrow-left me-1"></i>Back to Dashboard
            </a></div>
        </div>

        <?php if (!empty($message)) echo $message; ?>


        <!-- Categories Table -->
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body p-0">
                        <?php if (empty($categories)): ?>
                            <div class="empty-state">
                                <i class="fas fa-inbox fa-3x mb-3 text-muted"></i>
                                <h5 class="text-muted">No Categories Yet</h5>
                                <p class="text-muted mb-3">Create your first category or apply a template to get started.</p>
                                <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#categoryFormModal">
                                    <i class="fas fa-plus me-2"></i>Add Division
                                </button>
                            </div>
                        <?php else: ?>
                            <!-- Desktop Table View -->
                            <div class="table-responsive d-none d-lg-block">
                                <table class="table table-hover mb-0">
                                    <thead class="small">
                                        <tr>
                                            <th class="small bg-dark-subtle">ID</th>
                                            <th class="small bg-dark-subtle">Name</th>
                                            <th class="small bg-dark-subtle">Gender</th>
                                            <th class="small bg-dark-subtle">Years</th>
                                            <th class="small bg-dark-subtle">Main</th>
                                            <th class="small bg-dark-subtle">Ranking</th>
                                            <th class="small bg-dark-subtle">Req.</th>
                                            <th class="small bg-dark-subtle">Country Rule</th>
                                            
                                            <th class="text-end small bg-dark-subtle">Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($categories as $category): 
                                            $country_codes_array = json_decode($category['country_codes'] ?? '[]', true);
                                            if (!is_array($country_codes_array)) {
                                                $country_codes_array = [];
                                            }
                                            $country_codes_display = implode(', ', array_map('trim', $country_codes_array));

                                            $gender_labels = [
                                                'M' => '<div><i class="fas fa-mars text-primary"></i> Male</div>',
                                                'F' => '<div><i class="fas fa-venus text-danger"></i> Female</div>',
                                                'open' => 'Open / All'
                                            ];
                                            $gender_label = $gender_labels[$category['gender']] ?? 'Open / All';

                                            $requirements = [];
                                            if (!empty($category['requires_fis_number'])) {
                                                $requirements[] = 'FIS';
                                            }
                                            if (!empty($category['requires_licence_number'])) {
                                                $requirements[] = 'Licence';
                                            }
                                            $requirements_text = empty($requirements) ? 'None' : implode(', ', $requirements);

                                            $colors_decoded = json_decode($category['colors'] ?? '{}', true);
                                            $bg_color = $colors_decoded['background-color'] ?? '#6c757d';
                                            $border_color = $colors_decoded['border-color'] ?? '#495057';
                                            $text_color = $colors_decoded['color'] ?? '#ffffff';

                                            $year_range = ($category['start_year'] ?: 'Any') . ' - ' . ($category['end_year'] ?: 'Any');
                                        ?>
                                        <tr>
                                            <td><span class="badge bg-secondary"><?= $category['id'] ?></span></td>
                                            <td><strong><?= htmlspecialchars($category['category_name']) ?></strong></td>
                                            <td><?= $gender_label ?></td>
                                            <td><?= $year_range ?></td>
                                            <td><?= !empty($category['is_main_category']) ? '<span class="badge bg-primary">Yes</span>' : '<span class="badge bg-secondary">No</span>' ?></td>
                                            <td><?= !empty($category['use_for_ranking']) ? '<span class="badge bg-success">Yes</span>' : '<span class="badge bg-secondary">No</span>' ?></td>
                                            <td><?= $requirements_text ?></td>
                                            <td><?= ucfirst($category['country_rule'] ?? 'all') ?> <?= $country_codes_display ?: '-' ?></td>
                                           
                                            <td class="text-end">
                                                <div class="btn-group-sm">
                                                    <a href="event_categories.php?event_id=<?= $event_id ?>&edit=<?= $category['id'] ?>" 
                                                       class="btn btn-outline-warning" 
                                                       title="Edit Division">
                                                        <i class="fas fa-edit"></i>
                                                    </a>
                                                    <button type="button" class="btn btn-outline-danger" title="Delete Division"
                                                            data-bs-toggle="modal" data-bs-target="#deleteModal<?= $category['id'] ?>">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                    <form method="post" class="d-inline">
                                                        <input type="hidden" name="regenerate_color" value="1">
                                                        <input type="hidden" name="category_id" value="<?= $category['id'] ?>">
                                                        <button type="submit" class="btn btn-sm btn-outline-info" title="Regenerate Color">
                                                            <i class="fas fa-palette"></i>
                                                        </button>
                                                    </form>
                                                </div>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>

                            <!-- Mobile/Tablet Card View -->
                            <div class="d-lg-none">
                                <?php foreach ($categories as $category): 
                                    $country_codes_array = json_decode($category['country_codes'] ?? '[]', true);
                                    if (!is_array($country_codes_array)) {
                                        $country_codes_array = [];
                                    }
                                    $country_codes_display = implode(', ', array_map('trim', $country_codes_array));

                                    $gender_labels = [
                                        'M' => '<i class="fas fa-mars text-info"></i> Male',
                                        'F' => '<i class="fas fa-venus text-danger"></i> Female',
                                        'open' => '<i class="fas fa-venus-mars text-success"></i> Open'
                                    ];
                                    $gender_label = $gender_labels[$category['gender']] ?? '<i class="fas fa-venus-mars text-success"></i> Open';

                                    $requirements = [];
                                    if (!empty($category['requires_fis_number'])) {
                                        $requirements[] = '<span class="badge bg-primary"><i class="fas fa-id-card"></i> FIS</span>';
                                    }
                                    if (!empty($category['requires_licence_number'])) {
                                        $requirements[] = '<span class="badge bg-info"><i class="fas fa-certificate"></i> Licence</span>';
                                    }
                                    $requirements_html = empty($requirements) ? '<span class="text-muted">None</span>' : implode(' ', $requirements);

                                    $colors_decoded = json_decode($category['colors'] ?? '{}', true);
                                    $bg_color = $colors_decoded['background-color'] ?? '#6c757d';
                                    $border_color = $colors_decoded['border-color'] ?? '#495057';
                                    $text_color = $colors_decoded['color'] ?? '#ffffff';

                                    $year_range = ($category['start_year'] ?: 'Any') . ' - ' . ($category['end_year'] ?: 'Any');
                                ?>
                                <div class="card mb-3 shadow-sm">
                                    <div class="card-header bg-light d-flex justify-content-between align-items-center">
                                        <div>
                                            <h6 class="mb-0">
                                                <?= htmlspecialchars($category['category_name']) ?>
                                            </h6>
                                            <small class="text-muted">
                                                ID: <?= $category['id'] ?>
                                            </small>
                                        </div>
                                        <div class="text-end">
                                            <?php if (!empty($category['is_main_category'])): ?>
                                                <span class="badge bg-primary mb-1 d-block">
                                                    <i class="fas fa-star"></i> Main
                                                </span>
                                            <?php endif; ?>
                                            <?php if (!empty($category['use_for_ranking'])): ?>
                                                <span class="badge bg-success">
                                                    <i class="fas fa-trophy"></i> Ranking
                                                </span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div class="card-body">
                                        <div class="row g-2 mb-3">
                                            <div class="col-6">
                                                <small class="text-muted d-block">Gender</small>
                                                <div><?= $gender_label ?></div>
                                            </div>
                                            <div class="col-6">
                                                <small class="text-muted d-block">Birth Years</small>
                                                <div><i class="fas fa-birthday-cake me-1"></i><?= $year_range ?></div>
                                            </div>
                                        </div>
                                        
                                        <div class="row g-2 mb-3">
                                            <div class="col-6">
                                                <small class="text-muted d-block">Country Rule</small>
                                                <div>
                                                    <span class="badge bg-secondary">
                                                        <?= ucfirst($category['country_rule'] ?? 'all') ?>
                                                    </span>
                                                </div>
                                            </div>
                                            <div class="col-6">
                                                <small class="text-muted d-block">Requirements</small>
                                                <div><?= $requirements_html ?></div>
                                            </div>
                                        </div>

                                        <?php if ($country_codes_display): ?>
                                        <div class="mb-3">
                                            <small class="text-muted d-block">Countries</small>
                                            <div class="small">
                                                <i class="fas fa-globe me-1"></i><?= htmlspecialchars($country_codes_display) ?>
                                            </div>
                                        </div>
                                        <?php endif; ?>

                                        <div class="d-grid gap-2">
                                            <div class="btn-group" role="group">
                                                <a href="event_categories.php?event_id=<?= $event_id ?>&edit=<?= $category['id'] ?>" 
                                                   class="btn btn-outline-warning btn-sm">
                                                    <i class="fas fa-edit me-1"></i>Edit
                                                </a>
                                                <button type="button" class="btn btn-outline-danger btn-sm"
                                                        data-bs-toggle="modal" data-bs-target="#deleteModal<?= $category['id'] ?>">
                                                    <i class="fas fa-trash me-1"></i>Delete
                                                </button>
                                                <form method="post" class="d-inline">
                                                    <input type="hidden" name="regenerate_color" value="1">
                                                    <input type="hidden" name="category_id" value="<?= $category['id'] ?>">
                                                    <button type="submit" class="btn btn-outline-info btn-sm" title="Regenerate Color">
                                                        <i class="fas fa-palette me-1"></i>Color
                                                    </button>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Manual Assignment Section -->
    <div class="container-fluid container-StyleScore px-4 pb-4">
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-header bg-light d-flex justify-content-between align-items-center">
                        <h5 class="mb-0"><i class="fas fa-user-edit me-2"></i>Manual Athletes Assignment</h5>
                        <span class="badge bg-secondary"><?= count($participants_event) ?> Athletes</span>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover table-sm align-middle" id="participantsTable">
                                <thead>
                                    <tr>
                                        <th onclick="sortTable(0)" style="cursor:pointer">BIB <i class="fas fa-sort text-muted small"></i></th>
                                        <th onclick="sortTable(1)" style="cursor:pointer">Athlete <i class="fas fa-sort text-muted small"></i></th>
                                        <th onclick="sortTable(2)" style="cursor:pointer">Gender <i class="fas fa-sort text-muted small"></i></th>
                                        <th onclick="sortTable(3)" style="cursor:pointer">Year <i class="fas fa-sort text-muted small"></i></th>
                                        <th onclick="sortTable(4)" style="cursor:pointer">Current Division <i class="fas fa-sort text-muted small"></i></th>
                                        <th>Assign To</th>
                                        <th>Action</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($participants_event as $p): ?>
                                    <tr>
                                        <td><span class="badge bg-dark"><?= $p['bib_number'] ?: '-' ?></span></td>
                                        <td>
                                            <div class="fw-bold"><?= htmlspecialchars($p['last_name'] . ', ' . $p['first_name']) ?></div>
                                            <small class="text-muted"><?= htmlspecialchars($p['club'] ?? '') ?></small>
                                        </td>
                                        <td>
                                            <?php 
                                            $gender_map = [
                                                'M' => '<i class="fas fa-mars text-primary"></i> Male',
                                                'F' => '<i class="fas fa-venus text-danger"></i> Female'
                                            ];
                                            echo $gender_map[$p['gender']] ?? $p['gender'];
                                            ?>
                                        </td>
                                        <td><?= $p['birth_year'] ?? (!empty($p['date_of_birth']) ? date('Y', strtotime($p['date_of_birth'])) : '-') ?></td>
                                        <td>
                                            <?php 
                                            $current_cat = 'Unassigned';
                                            foreach ($categories as $c) {
                                                if ($c['id'] == $p['category_id']) {
                                                    $current_cat = $c['category_name'];
                                                    break;
                                                }
                                            }
                                            echo $p['category_id'] ? '<span class="badge bg-primary">' . htmlspecialchars($current_cat) . '</span>' : '<span class="badge bg-warning">Unassigned</span>';
                                            ?>
                                        </td>
                                        <form method="post">
                                            <input type="hidden" name="manual_assign" value="1">
                                            <input type="hidden" name="ep_id" value="<?= $p['ep_id'] ?>">
                                            <td>
                                                <select name="category_id" class="form-select form-select-sm">
                                                    <option value="">-- Select Division --</option>
                                                    <?php foreach ($categories as $c): ?>
                                                    <option value="<?= $c['id'] ?>" <?= $p['category_id'] == $c['id'] ? 'selected' : '' ?>>
                                                        <?= htmlspecialchars($c['category_name']) ?>
                                                    </option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </td>
                                            <td>
                                                <button type="submit" class="btn btn-primary btn-sm">
                                                    <i class="fas fa-save"></i>
                                                </button>
                                            </td>
                                        </form>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Division Form Modal -->
    <div class="modal fade" id="categoryFormModal" tabindex="-1" aria-labelledby="categoryFormModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="categoryFormModalLabel">
                        <i class="fas <?= $edit_category ? 'fa-edit' : 'fa-plus' ?> me-2"></i>
                        <?= $edit_category ? 'Edit Division' : 'Create New Division' ?>
                    </h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form method="post" id="categoryForm">
                    <div class="modal-body">
                        <?php if ($edit_category): ?>
                            <input type="hidden" name="update_category" value="1">
                            <input type="hidden" name="category_id" value="<?= $edit_category['id'] ?>">
                        <?php else: ?>
                            <input type="hidden" name="create_category" value="1">
                        <?php endif; ?>

                        <!-- Basic Information -->
                        <div class="form-section">
                            <div class="section-title">Basic Information</div>
                            <div class="mb-3">
                                <label for="modal_category_name" class="form-label">Division Name <span class="text-danger">*</span></label>
                                <input type="text" class="form-control" id="modal_category_name" name="category_name" 
                                       value="<?= $edit_category ? htmlspecialchars($edit_category['category_name']) : '' ?>" required>
                            </div>
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label for="modal_start_year" class="form-label">Start Year</label>
                                    <input type="number" class="form-control" id="modal_start_year" name="start_year" 
                                           value="<?= $edit_category ? $edit_category['start_year'] : '' ?>" placeholder="e.g., 2000">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label for="modal_end_year" class="form-label">End Year</label>
                                    <input type="number" class="form-control" id="modal_end_year" name="end_year" 
                                           value="<?= $edit_category ? $edit_category['end_year'] : '' ?>" placeholder="e.g., 2010">
                                </div>
                            </div>
                        </div>

                        <!-- Gender & Requirements -->
                        <div class="form-section">
                            <div class="section-title">Gender & Requirements</div>
                            <div class="mb-3">
                                <label for="modal_gender" class="form-label">Gender</label>
                                <select class="form-select" id="modal_gender" name="gender">
                                    <option value="open" <?= (!$edit_category || $edit_category['gender'] === 'open') ? 'selected' : '' ?>>Open / All</option>
                                    <option value="M" <?= ($edit_category && $edit_category['gender'] === 'M') ? 'selected' : '' ?>><div><i class="fas fa-mars text-info"></i> Male</div></option>
                                    <option value="F" <?= ($edit_category && $edit_category['gender'] === 'F') ? 'selected' : '' ?>><div><i class="fas fa-venus text-danger"></i> Female</div></option>
                                </select>
                            </div>
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="modal_requires_fis" name="requires_fis" 
                                               <?= ($edit_category && $edit_category['requires_fis_number']) ? 'checked' : '' ?>>
                                        <label class="form-check-label" for="modal_requires_fis">
                                            Requires FIS Number
                                        </label>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="modal_requires_licence" name="requires_licence" 
                                               <?= ($edit_category && $edit_category['requires_licence_number']) ? 'checked' : '' ?>>
                                        <label class="form-check-label" for="modal_requires_licence">
                                            Requires Licence Number
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Division Settings -->
                        <div class="form-section">
                            <div class="section-title">Division Settings</div>
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="modal_is_main_category" name="is_main_category" 
                                               <?= ($edit_category && $edit_category['is_main_category']) ? 'checked' : '' ?>>
                                        <label class="form-check-label" for="modal_is_main_category">
                                            <strong>Main Division</strong>
                                            <div class="small text-muted">Used for primary participant assignment</div>
                                        </label>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="modal_use_for_ranking" name="use_for_ranking" 
                                               <?= ($edit_category && $edit_category['use_for_ranking']) ? 'checked' : '' ?>>
                                        <label class="form-check-label" for="modal_use_for_ranking">
                                            <strong>Use for Ranking</strong>
                                            <div class="small text-muted">Include in ranking calculations</div>
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Country Rules -->
                        <div class="form-section">
                            <div class="section-title">Country Restrictions</div>
                            <div class="mb-3">
                                <label for="modal_country_rule" class="form-label">Country Rule</label>
                                <select class="form-select" id="modal_country_rule" name="country_rule">
                                    <option value="all" <?= (!$edit_category || $edit_category['country_rule'] === 'all') ? 'selected' : '' ?>>All Countries</option>
                                    <option value="include" <?= ($edit_category && $edit_category['country_rule'] === 'include') ? 'selected' : '' ?>>Only Specific Countries</option>
                                    <option value="exclude" <?= ($edit_category && $edit_category['country_rule'] === 'exclude') ? 'selected' : '' ?>>Exclude Specific Countries</option>
                                </select>
                            </div>
                            <div class="mb-3" id="modal_countryCodesGroup" style="display: <?= ($edit_category && in_array($edit_category['country_rule'], ['include', 'exclude'])) ? 'block' : 'none' ?>;">
                                <label for="modal_countryCodesInput" class="form-label">Country Codes</label>
                                <input type="text" class="form-control" id="modal_countryCodesInput" name="country_codes" 
                                       value="<?= $edit_category ? htmlspecialchars($edit_country_codes_string) : '' ?>" 
                                       placeholder="e.g., USA, CAN, GBR">
                                <div class="form-text">Enter country codes separated by commas</div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas <?= $edit_category ? 'fa-save' : 'fa-plus' ?> me-2"></i>
                            <?= $edit_category ? 'Update Division' : 'Create Division' ?>
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Confirmation Modals -->
    <?php foreach ($categories as $category): ?>
    <div class="modal fade" id="deleteModal<?= $category['id'] ?>" tabindex="-1" aria-labelledby="deleteModalLabel<?= $category['id'] ?>" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title" id="deleteModalLabel<?= $category['id'] ?>">
                        <i class="fas fa-exclamation-triangle me-2"></i>Confirm Delete
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <p class="mb-3">Are you sure you want to delete this category?</p>
                    <div class="alert alert-warning mb-3">
                        <strong><i class="fas fa-tag me-2"></i><?= htmlspecialchars($category['category_name']) ?></strong>
                    </div>
                    <p class="text-muted small mb-0">
                        <i class="fas fa-info-circle me-1"></i>
                        This action cannot be undone. Any participants assigned to this category will need to be reassigned.
                    </p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-1"></i>Cancel
                    </button>
                    <form method="post" class="d-inline">
                        <input type="hidden" name="delete_category" value="1">
                        <input type="hidden" name="category_id" value="<?= $category['id'] ?>">
                        <button type="submit" class="btn btn-danger">
                            <i class="fas fa-trash me-1"></i>Delete Division
                        </button>
                    </form>
                </div>
            </div>
        </div>
    </div>
    <?php endforeach; ?>

    <!-- Template Selection Modal -->
    <div class="modal fade" id="templateModal" tabindex="-1" aria-labelledby="templateModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-xl modal-dialog-centered modal-dialog-scrollable">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title" id="templateModalLabel">
                        <i class="fas fa-magic me-2"></i>Division Templates
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-info mb-4">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Templates create multiple categories at once.</strong> Birth years are based on standard FIS age groups. Review the categories below before applying.
                    </div>
                    
                    <!-- Gender Selection Section -->
                    <div class="card mb-4 border-primary">
                        <div class="card-header bg-primary bg-opacity-10">
                            <h6 class="mb-0 text-primary">
                                <i class="fas fa-venus-mars me-2"></i>Gender Configuration
                            </h6>
                        </div>
                        <div class="card-body">
                            <p class="text-muted mb-3">Choose how to handle gender categories:</p>
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <div class="form-check">
                                        <input class="form-check-input gender-mode-radio" type="radio" name="gender_mode_preview" 
                                               id="gender_open" value="open" checked>
                                        <label class="form-check-label fw-bold" for="gender_open">
                                            <span class="badge bg-success me-2">
                                                <i class="fas fa-venus-mars"></i>
                                            </span>
                                            Open (All Genders)
                                        </label>
                                        <div class="text-muted small mt-1 ms-4">
                                            Creates categories open to all genders (e.g., "Senior", "Junior / U18")
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-check">
                                        <input class="form-check-input gender-mode-radio" type="radio" name="gender_mode_preview" 
                                               id="gender_split" value="split">
                                        <label class="form-check-label fw-bold" for="gender_split">
                                            <span class="badge bg-info me-1">
                                                <i class="fas fa-mars"></i>
                                            </span>
                                            <span class="badge bg-danger">
                                                <i class="fas fa-venus"></i>
                                            </span>
                                            Split by Gender
                                        </label>
                                        <div class="text-muted small mt-1 ms-4">
                                            Creates separate categories for men and women (e.g., "Senior - Men", "Senior - Women")
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row g-3">
                        <?php foreach ($category_templates as $template_id => $template): ?>
                            <div class="col-md-6">
                                <div class="card h-100 template-card" data-template="<?= $template_id ?>">
                                    <div class="card-header bg-light">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <h6 class="mb-0">
                                                <i class="fas <?= $template['icon'] ?> me-2 text-primary"></i>
                                                <?= htmlspecialchars($template['name']) ?>
                                            </h6>
                                            <span class="badge bg-primary template-category-count"><?= count($template['categories']) ?> categories</span>
                                        </div>
                                    </div>
                                    <div class="card-body">
                                        <p class="text-muted small mb-3"><?= htmlspecialchars($template['description']) ?></p>
                                        
                                        <div class="template-preview" data-template-id="<?= $template_id ?>">
                                            <?php foreach ($template['categories'] as $index => $cat): ?>
                                                <!-- Open gender version (shown by default) -->
                                                <div class="border rounded p-2 mb-2 bg-light category-preview-open">
                                                    <div class="d-flex justify-content-between align-items-start">
                                                        <div class="flex-grow-1">
                                                            <strong><?= htmlspecialchars($cat['category_name']) ?></strong>
                                                            <div class="small text-muted mt-1">
                                                                <i class="fas fa-birthday-cake me-1"></i>
                                                                Born: <?= $cat['start_year'] ?: 'Any' ?> - <?= $cat['end_year'] ?: 'Any' ?>
                                                            </div>
                                                        </div>
                                                        <div class="text-end">
                                                            <span class="badge bg-success" style="font-size: 0.7rem;">
                                                                <i class="fas fa-venus-mars"></i>
                                                            </span>
                                                            <?php if ($cat['is_main_category']): ?>
                                                                <span class="badge bg-warning text-dark me-1" style="font-size: 0.7rem;">
                                                                    <i class="fas fa-star"></i>
                                                                </span>
                                                            <?php endif; ?>
                                                            <?php if ($cat['requires_fis_number']): ?>
                                                                <span class="badge bg-primary" style="font-size: 0.7rem;">
                                                                    <i class="fas fa-id-card"></i>
                                                                </span>
                                                            <?php endif; ?>
                                                        </div>
                                                    </div>
                                                </div>
                                                
                                                <!-- Split gender versions (hidden by default) -->
                                                <div class="category-preview-split" style="display: none;">
                                                    <!-- Men's category -->
                                                    <div class="border rounded p-2 mb-2 bg-light border-info">
                                                        <div class="d-flex justify-content-between align-items-start">
                                                            <div class="flex-grow-1">
                                                                <strong><?= htmlspecialchars($cat['category_name']) ?> - Men</strong>
                                                                <div class="small text-muted mt-1">
                                                                    <i class="fas fa-birthday-cake me-1"></i>
                                                                    Born: <?= $cat['start_year'] ?: 'Any' ?> - <?= $cat['end_year'] ?: 'Any' ?>
                                                                </div>
                                                            </div>
                                                            <div class="text-end">
                                                                <span class="badge bg-info" style="font-size: 0.7rem;">
                                                                    <i class="fas fa-mars"></i> Male
                                                                </span>
                                                                <?php if ($cat['is_main_category']): ?>
                                                                    <span class="badge bg-warning text-dark me-1" style="font-size: 0.7rem;">
                                                                        <i class="fas fa-star"></i>
                                                                    </span>
                                                                <?php endif; ?>
                                                                <?php if ($cat['requires_fis_number']): ?>
                                                                    <span class="badge bg-primary" style="font-size: 0.7rem;">
                                                                        <i class="fas fa-id-card"></i>
                                                                    </span>
                                                                <?php endif; ?>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    <!-- Women's category -->
                                                    <div class="border rounded p-2 mb-2 bg-light border-danger">
                                                        <div class="d-flex justify-content-between align-items-start">
                                                            <div class="flex-grow-1">
                                                                <strong><?= htmlspecialchars($cat['category_name']) ?> - Women</strong>
                                                                <div class="small text-muted mt-1">
                                                                    <i class="fas fa-birthday-cake me-1"></i>
                                                                    Born: <?= $cat['start_year'] ?: 'Any' ?> - <?= $cat['end_year'] ?: 'Any' ?>
                                                                </div>
                                                            </div>
                                                            <div class="text-end">
                                                                <span class="badge bg-danger" style="font-size: 0.7rem;">
                                                                    <i class="fas fa-venus"></i> Female
                                                                </span>
                                                                <?php if ($cat['is_main_category']): ?>
                                                                    <span class="badge bg-warning text-dark me-1" style="font-size: 0.7rem;">
                                                                        <i class="fas fa-star"></i>
                                                                    </span>
                                                                <?php endif; ?>
                                                                <?php if ($cat['requires_fis_number']): ?>
                                                                    <span class="badge bg-primary" style="font-size: 0.7rem;">
                                                                        <i class="fas fa-id-card"></i>
                                                                    </span>
                                                                <?php endif; ?>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                    </div>
                                    <div class="card-footer bg-white border-top">
                                        <form method="post" class="apply-template-form">
                                            <input type="hidden" name="apply_template" value="1">
                                            <input type="hidden" name="template_id" value="<?= $template_id ?>">
                                            <input type="hidden" name="gender_mode" class="gender-mode-input" value="open">
                                            <button type="submit" class="btn btn-primary w-100">
                                                <i class="fas fa-check me-2"></i>Apply This Template
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-1"></i>Cancel
                    </button>
                </div>
            </div>
        </div>
    </div>

    
    <script>
        // Auto-open modal when editing
        <?php if ($edit_category): ?>
        document.addEventListener('DOMContentLoaded', function() {
            const modal = new bootstrap.Modal(document.getElementById('categoryFormModal'));
            modal.show();
        });
        <?php endif; ?>

        // Country rule visibility toggle
        document.addEventListener('DOMContentLoaded', function() {
            const countryRuleSelect = document.getElementById('modal_country_rule');
            const countryCodesGroup = document.getElementById('modal_countryCodesGroup');
            
            if (countryRuleSelect) {
                countryRuleSelect.addEventListener('change', function() {
                    if (this.value === 'include' || this.value === 'exclude') {
                        countryCodesGroup.style.display = 'block';
                    } else {
                        countryCodesGroup.style.display = 'none';
                    }
                });
            }
        });

        // Reset modal form when closed without editing
        document.addEventListener('DOMContentLoaded', function() {
            const modal = document.getElementById('categoryFormModal');
            if (modal) {
                modal.addEventListener('hidden.bs.modal', function() {
                    // Only reset if not in edit mode
                    <?php if (!$edit_category): ?>
                    document.getElementById('categoryForm').reset();
                    <?php else: ?>
                    // Remove edit parameter from URL
                    const url = new URL(window.location);
                    url.searchParams.delete('edit');
                    window.history.replaceState({}, '', url);
                    <?php endif; ?>
                });
            }
        });

        // Gender mode switching for template preview
        document.querySelectorAll('.gender-mode-radio').forEach(radio => {
            radio.addEventListener('change', function() {
                const selectedMode = this.value;
                
                // Update all hidden inputs in forms
                document.querySelectorAll('.gender-mode-input').forEach(input => {
                    input.value = selectedMode;
                });
                
                // Update preview displays
                if (selectedMode === 'open') {
                    // Show open gender previews, hide split previews
                    document.querySelectorAll('.category-preview-open').forEach(el => el.style.display = 'block');
                    document.querySelectorAll('.category-preview-split').forEach(el => el.style.display = 'none');
                    
                    // Update category counts (templates have X categories in open mode)
                    document.querySelectorAll('.template-category-count').forEach((badge, index) => {
                        const templateCard = badge.closest('.template-card');
                        const openCategories = templateCard.querySelectorAll('.category-preview-open');
                        badge.textContent = openCategories.length + ' categories';
                    });
                } else {
                    // Show split gender previews, hide open previews
                    document.querySelectorAll('.category-preview-open').forEach(el => el.style.display = 'none');
                    document.querySelectorAll('.category-preview-split').forEach(el => el.style.display = 'block');
                    
                    // Update category counts (templates have 2X categories in split mode)
                    document.querySelectorAll('.template-category-count').forEach((badge, index) => {
                        const templateCard = badge.closest('.template-card');
                        const openCategories = templateCard.querySelectorAll('.category-preview-open');
                        badge.textContent = (openCategories.length * 2) + ' categories';
                    });
                }
            });
        });
        
        // Template form submission handling
        document.querySelectorAll('.apply-template-form').forEach(form => {
            form.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const templateName = this.closest('.card').querySelector('.card-header h6').textContent.trim();
                const categoryCount = this.closest('.card').querySelector('.badge').textContent;
                const genderMode = this.querySelector('.gender-mode-input').value;
                const genderModeText = genderMode === 'split' ? 'split by gender (Men/Women)' : 'open to all genders';
                const templateId = this.querySelector('input[name="template_id"]').value;
                
                // Create confirmation modal
                const confirmModal = document.createElement('div');
                confirmModal.className = 'modal fade';
                confirmModal.id = 'confirmTemplateModal';
                confirmModal.innerHTML = `
                    <div class="modal-dialog modal-dialog-centered">
                        <div class="modal-content">
                            <div class="modal-header bg-primary text-white">
                                <h5 class="modal-title">
                                    <i class="fas fa-magic me-2"></i>Apply Template
                                </h5>
                                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body">
                                <p class="mb-3">Are you sure you want to apply this template?</p>
                                <div class="alert alert-info mb-3">
                                    <strong><i class="fas fa-layer-group me-2"></i>${templateName}</strong>
                                    <div class="small mt-2">
                                        <i class="fas fa-tags me-1"></i>This will create <strong>${categoryCount}</strong> 
                                        (${genderModeText})
                                    </div>
                                </div>
                                <p class="text-muted small mb-0">
                                    <i class="fas fa-info-circle me-1"></i>
                                    Categories will be added to your current event setup.
                                </p>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                                    <i class="fas fa-times me-1"></i>Cancel
                                </button>
                                <button type="button" class="btn btn-primary" id="confirmApplyTemplate">
                                    <i class="fas fa-check me-1"></i>Apply Template
                                </button>
                            </div>
                        </div>
                    </div>
                `;
                
                document.body.appendChild(confirmModal);
                const modal = new bootstrap.Modal(confirmModal);
                
                // Handle confirm button
                const confirmBtn = confirmModal.querySelector('#confirmApplyTemplate');
                confirmBtn.addEventListener('click', () => {
                    confirmBtn.disabled = true;
                    confirmBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Creating categories...';
                    modal.hide();
                    
                    // Submit the form
                    form.submit();
                });
                
                // Clean up modal after hidden
                confirmModal.addEventListener('hidden.bs.modal', () => {
                    confirmModal.remove();
                });
                
                modal.show();
            });
        });
        
        // Template card hover effects
        document.querySelectorAll('.template-card').forEach(card => {
            card.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-4px)';
                this.style.boxShadow = '0 8px 16px rgba(0,0,0,0.15)';
                this.style.transition = 'all 0.3s ease';
            });
            card.addEventListener('mouseleave', function() {
                this.style.transform = 'translateY(0)';
                this.style.boxShadow = '';
            });
        });

        // Modern form validation and UX
        document.addEventListener('DOMContentLoaded', function() {
            // Prevent double submission
            const forms = document.querySelectorAll('form');
            
            forms.forEach(form => {
                form.addEventListener('submit', function(e) {
                    const submitButton = form.querySelector('button[type="submit"]');
                    
                    if (submitButton && !submitButton.disabled) {
                        submitButton.disabled = true;
                        const originalHtml = submitButton.innerHTML;
                        submitButton.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Processing...';
                        
                        // Re-enable after timeout as failsafe
                        setTimeout(() => {
                            submitButton.disabled = false;
                            submitButton.innerHTML = originalHtml;
                        }, 5000);
                    }
                });
            });

            // Country code input enhancement
            const countryRules = document.querySelectorAll('input[name="country_rule"]');
            const countryCodesInput = document.querySelector('input[name="country_codes"]');
            
            function toggleCountryInput() {
                const selectedRule = document.querySelector('input[name="country_rule"]:checked')?.value;
                if (countryCodesInput) {
                    countryCodesInput.disabled = selectedRule === 'all';
                    countryCodesInput.parentElement.style.opacity = selectedRule === 'all' ? '0.6' : '1';
                }
            }
            
            countryRules.forEach(radio => {
                radio.addEventListener('change', toggleCountryInput);
            });
            
            toggleCountryInput(); // Initial state

            // Enhanced visual feedback for category classification
            const isMainCheckbox = document.getElementById('is_main_category');
            const useForRankingCheckbox = document.getElementById('use_for_ranking');
            
            function updateFormSectionVisuals() {
                const formSection = document.querySelector('.form-section:has(#is_main_category)');
                if (!formSection) return;
                
                const isMain = isMainCheckbox?.checked;
                const isRanking = useForRankingCheckbox?.checked;
                
                // Reset styles
                formSection.style.borderLeft = '';
                formSection.style.borderRight = '';
                formSection.style.background = '#f8f9fa';
                
                if (isMain && isRanking) {
                    formSection.style.background = 'linear-gradient(135deg, #fff3cd 0%, #d1ecf1 100%)';
                    formSection.style.borderLeft = '4px solid #ffc107';
                    formSection.style.borderRight = '4px solid #28a745';
                } else if (isMain) {
                    formSection.style.background = 'linear-gradient(135deg, #fff3cd 0%, #fefcf3 100%)';
                    formSection.style.borderLeft = '4px solid #ffc107';
                } else if (isRanking) {
                    formSection.style.borderRight = '4px solid #28a745';
                }
            }
            
            if (isMainCheckbox) {
                isMainCheckbox.addEventListener('change', updateFormSectionVisuals);
            }
            if (useForRankingCheckbox) {
                useForRankingCheckbox.addEventListener('change', updateFormSectionVisuals);
            }
            
            updateFormSectionVisuals(); // Initial state
        });

        // Auto-assignment form handling
        document.getElementById('autoAssignForm')?.addEventListener('submit', function(e) {
            const button = this.querySelector('button[type="submit"]');
            button.disabled = true;
            button.innerHTML = '<i class="fas fa-spinner fa-spin me-2"></i>Analyzing participants...';
            
            // Show progress indicator
            const progressDiv = document.createElement('div');
            progressDiv.className = 'mt-3';
            progressDiv.innerHTML = `
                <div class="progress" style="height: 6px;">
                    <div class="progress-bar progress-bar-striped progress-bar-animated bg-light" 
                         role="progressbar" style="width: 100%"></div>
                </div>
                <small class="text-light d-block mt-2">
                    <i class="fas fa-cogs me-1"></i>Processing participant assignments...
                </small>
            `;
            this.appendChild(progressDiv);
        });

        // Assignment details filtering
        function filterAssignments(action) {
            const rows = document.querySelectorAll('.assignment-row');
            const buttons = document.querySelectorAll('[onclick^="filterAssignments"]');
            
            // Update active button state
            buttons.forEach(btn => {
                btn.classList.remove('active');
                if (btn.textContent.trim() === action || (action === 'all' && btn.textContent.trim() === 'All')) {
                    btn.classList.add('active');
                }
            });
            
            // Filter rows with smooth transition
            rows.forEach(row => {
                if (action === 'all' || row.getAttribute('data-action') === action) {
                    row.style.display = 'block';
                    row.style.opacity = '1';
                } else {
                    row.style.opacity = '0';
                    setTimeout(() => {
                        if (row.style.opacity === '0') {
                            row.style.display = 'none';
                        }
                    }, 150);
                }
            });
        }

        // Enhanced form validation
        function validateCategoryForm() {
            const categoryName = document.querySelector('input[name="category_name"]')?.value.trim();
            
            if (!categoryName) {
                // Create modern alert
                const alert = document.createElement('div');
                alert.className = 'alert alert-danger alert-dismissible fade show position-fixed';
                alert.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
                alert.innerHTML = `
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    Division name is required.
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                `;
                document.body.appendChild(alert);
                
                // Auto-remove after 5 seconds
                setTimeout(() => {
                    if (alert.parentElement) {
                        alert.remove();
                    }
                }, 5000);
                
                return false;
            }
            
            return true;
        }

        // Scroll to editing form
        <?php if ($edit_category): ?>
        document.addEventListener('DOMContentLoaded', function() {
            const editCard = document.querySelector('.card.border-warning');
            if (editCard) {
                editCard.scrollIntoView({ 
                    behavior: 'smooth', 
                    block: 'start' 
                });
            }
        });
        <?php endif; ?>

        // Clear URL parameters after showing message
        if (window.location.search.includes('msg=')) {
            setTimeout(() => {
                const url = new URL(window.location);
                url.searchParams.delete('msg');
                url.searchParams.delete('error');
                window.history.replaceState({}, document.title, url.toString());
            }, 8000);
        }

        // Enhanced tooltips for better UX
        document.addEventListener('DOMContentLoaded', function() {
            // Initialize Bootstrap tooltips if available
            if (typeof bootstrap !== 'undefined' && bootstrap.Tooltip) {
                const tooltipTriggerList = [].slice.call(document.querySelectorAll('[title]'));
                tooltipTriggerList.map(function (tooltipTriggerEl) {
                    return new bootstrap.Tooltip(tooltipTriggerEl, {
                        delay: { show: 500, hide: 100 }
                    });
                });
            }
        });

        // Auto-hide success/info alerts
        document.addEventListener('DOMContentLoaded', function() {
            const successAlerts = document.querySelectorAll('.alert-success, .alert-info');
            successAlerts.forEach(alert => {
                setTimeout(() => {
                    if (alert && alert.parentElement) {
                        const bsAlert = new bootstrap.Alert(alert);
                        bsAlert.close();
                    }
                }, 6000);
            });
        });

        function sortTable(n) {
            var table, rows, switching, i, x, y, shouldSwitch, dir, switchcount = 0;
            table = document.getElementById("participantsTable");
            switching = true;
            // Set the sorting direction to ascending:
            dir = "asc";
            /* Make a loop that will continue until
            no switching has been done: */
            while (switching) {
                // Start by saying: no switching is done:
                switching = false;
                rows = table.rows;
                /* Loop through all table rows (except the
                first, which contains table headers): */
                for (i = 1; i < (rows.length - 1); i++) {
                    // Start by saying there should be no switching:
                    shouldSwitch = false;
                    /* Get the two elements you want to compare,
                    one from current row and one from the next: */
                    x = rows[i].getElementsByTagName("TD")[n];
                    y = rows[i + 1].getElementsByTagName("TD")[n];
                    
                    // Get text content
                    let xContent = x.textContent || x.innerText;
                    let yContent = y.textContent || y.innerText;
                    
                    // Clean up content for specific columns if needed
                    xContent = xContent.trim();
                    yContent = yContent.trim();

                    // Check if the two rows should switch place,
                    // based on the direction, asc or desc:
                    if (dir == "asc") {
                        if (isNaN(xContent) || isNaN(yContent)) {
                            if (xContent.toLowerCase() > yContent.toLowerCase()) {
                                shouldSwitch = true;
                                break;
                            }
                        } else {
                            if (Number(xContent) > Number(yContent)) {
                                shouldSwitch = true;
                                break;
                            }
                        }
                    } else if (dir == "desc") {
                        if (isNaN(xContent) || isNaN(yContent)) {
                            if (xContent.toLowerCase() < yContent.toLowerCase()) {
                                shouldSwitch = true;
                                break;
                            }
                        } else {
                            if (Number(xContent) < Number(yContent)) {
                                shouldSwitch = true;
                                break;
                            }
                        }
                    }
                }
                if (shouldSwitch) {
                    /* If a switch has been marked, make the switch
                    and mark that a switch has been done: */
                    rows[i].parentNode.insertBefore(rows[i + 1], rows[i]);
                    switching = true;
                    // Each time a switch is done, increase this count by 1:
                    switchcount ++;
                } else {
                    /* If no switching has been done AND the direction is "asc",
                    set the direction to "desc" and run the while loop again. */
                    if (switchcount == 0 && dir == "asc") {
                        dir = "desc";
                        switching = true;
                    }
                }
            }
            
            // Update sort icons
            const headers = table.getElementsByTagName("TH");
            for (let j = 0; j < headers.length; j++) {
                const icon = headers[j].querySelector("i");
                if (icon) {
                    icon.className = "fas fa-sort text-muted small";
                    if (j === n) {
                        icon.className = dir === "asc" ? "fas fa-sort-up text-primary small" : "fas fa-sort-down text-primary small";
                    }
                }
            }
        }
    </script>

    <?php include 'footer.php'; ?>
</body>
</html>