<?php
include '../includes/auth.php';
include '../includes/db.php';

$message = '';

// Handle custom field operations
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_field'])) {
        try {
            $stmt = $pdo->prepare("
                INSERT INTO participant_custom_field_definitions 
                (field_key, field_label, field_type, field_options, is_searchable, is_required, display_order)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $options = null;
            if ($_POST['field_type'] === 'select' && !empty($_POST['field_options'])) {
                $options = json_encode(array_map('trim', explode(',', $_POST['field_options'])));
            }
            
            $stmt->execute([
                $_POST['field_key'],
                $_POST['field_label'],
                $_POST['field_type'],
                $options,
                isset($_POST['is_searchable']) ? 1 : 0,
                isset($_POST['is_required']) ? 1 : 0,
                $_POST['display_order'] ?? 0
            ]);
            
            $message = '<div class="alert alert-success"><i class="fas fa-check-circle me-2"></i>Custom field added successfully!</div>';
        } catch (Exception $e) {
            $message = '<div class="alert alert-danger"><i class="fas fa-exclamation-triangle me-2"></i>Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
    
    if (isset($_POST['update_field'])) {
        try {
            $stmt = $pdo->prepare("
                UPDATE participant_custom_field_definitions 
                SET field_label = ?, field_type = ?, field_options = ?, 
                    is_searchable = ?, is_required = ?, display_order = ?, is_active = ?
                WHERE id = ?
            ");
            
            $options = null;
            if ($_POST['field_type'] === 'select' && !empty($_POST['field_options'])) {
                $options = json_encode(array_map('trim', explode(',', $_POST['field_options'])));
            }
            
            $stmt->execute([
                $_POST['field_label'],
                $_POST['field_type'],
                $options,
                isset($_POST['is_searchable']) ? 1 : 0,
                isset($_POST['is_required']) ? 1 : 0,
                $_POST['display_order'] ?? 0,
                isset($_POST['is_active']) ? 1 : 0,
                $_POST['field_id']
            ]);
            
            $message = '<div class="alert alert-success"><i class="fas fa-check-circle me-2"></i>Custom field updated successfully!</div>';
        } catch (Exception $e) {
            $message = '<div class="alert alert-danger"><i class="fas fa-exclamation-triangle me-2"></i>Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
    
    if (isset($_POST['delete_field'])) {
        try {
            $stmt = $pdo->prepare("DELETE FROM participant_custom_field_definitions WHERE id = ?");
            $stmt->execute([$_POST['field_id']]);
            $message = '<div class="alert alert-success"><i class="fas fa-check-circle me-2"></i>Custom field deleted successfully!</div>';
        } catch (Exception $e) {
            $message = '<div class="alert alert-danger"><i class="fas fa-exclamation-triangle me-2"></i>Error: ' . htmlspecialchars($e->getMessage()) . '</div>';
        }
    }
}

// Fetch all custom field definitions
$fields = $pdo->query("
    SELECT * FROM participant_custom_field_definitions 
    ORDER BY display_order ASC, id ASC
")->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Custom Fields Manager</title>
    <?php include_once '../includes/stylesheets.php'; ?>
</head>
<body class="body-bg-aurora-bright">
    <?php include '../menu.php'; ?>
    <?php $use_localStorage = true; $show_header = false; include 'event_actions_menu.php'; ?>
       
    <div class="container-fluid p-4 container-StyleScore">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h1 class="h3 mb-1">
                    <i class="fas fa-sliders-h text-primary me-2"></i>Custom Fields Manager
                </h1>
                <p class="text-muted mb-0">Define custom fields for participant data</p>
            </div>
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addFieldModal">
                <i class="fas fa-plus me-2"></i>Add Custom Field
            </button>
        </div>
        
        <?= $message ?>
        
        <div class="card shadow-sm">
            <div class="card-header bg-white">
                <h5 class="mb-0"><i class="fas fa-list me-2"></i>Custom Fields</h5>
            </div>
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table table-hover mb-0">
                        <thead class="table-light">
                            <tr>
                                <th>Order</th>
                                <th>Field Key</th>
                                <th>Label</th>
                                <th>Type</th>
                                <th>Options</th>
                                <th>Searchable</th>
                                <th>Required</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($fields)): ?>
                                <tr>
                                    <td colspan="9" class="text-center text-muted py-4">
                                        <i class="fas fa-inbox fa-2x mb-2"></i>
                                        <p>No custom fields defined yet.</p>
                                    </td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($fields as $field): ?>
                                    <tr>
                                        <td><?= $field['display_order'] ?></td>
                                        <td><code><?= htmlspecialchars($field['field_key']) ?></code></td>
                                        <td><?= htmlspecialchars($field['field_label']) ?></td>
                                        <td><span class="badge bg-info"><?= htmlspecialchars($field['field_type']) ?></span></td>
                                        <td>
                                            <?php if ($field['field_options']): ?>
                                                <small class="text-muted">
                                                    <?= implode(', ', array_slice(json_decode($field['field_options'], true), 0, 3)) ?>
                                                    <?php if (count(json_decode($field['field_options'], true)) > 3): ?>...<?php endif; ?>
                                                </small>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($field['is_searchable']): ?>
                                                <i class="fas fa-check text-success"></i>
                                            <?php else: ?>
                                                <i class="fas fa-times text-muted"></i>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($field['is_required']): ?>
                                                <i class="fas fa-check text-danger"></i>
                                            <?php else: ?>
                                                <i class="fas fa-times text-muted"></i>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($field['is_active']): ?>
                                                <span class="badge bg-success">Active</span>
                                            <?php else: ?>
                                                <span class="badge bg-secondary">Inactive</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <button class="btn btn-sm btn-outline-primary" onclick='editField(<?= json_encode($field) ?>)'>
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <button class="btn btn-sm btn-outline-danger" onclick="deleteField(<?= $field['id'] ?>, '<?= htmlspecialchars($field['field_label']) ?>')">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Add Field Modal -->
    <div class="modal fade" id="addFieldModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title"><i class="fas fa-plus me-2"></i>Add Custom Field</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="post">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Field Key <span class="text-danger">*</span></label>
                            <input type="text" name="field_key" class="form-control" required pattern="[a-z0-9_]+" 
                                   placeholder="e.g., team_name" title="Lowercase letters, numbers, and underscores only">
                            <small class="text-muted">Unique identifier (lowercase, no spaces)</small>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Field Label <span class="text-danger">*</span></label>
                            <input type="text" name="field_label" class="form-control" required placeholder="e.g., Team Name">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Field Type</label>
                            <select name="field_type" class="form-select" id="addFieldType" onchange="toggleOptions('add')">
                                <option value="text">Text</option>
                                <option value="number">Number</option>
                                <option value="date">Date</option>
                                <option value="select">Dropdown Select</option>
                                <option value="textarea">Text Area</option>
                            </select>
                        </div>
                        <div class="mb-3 d-none" id="addOptionsGroup">
                            <label class="form-label">Options (comma-separated)</label>
                            <input type="text" name="field_options" class="form-control" placeholder="Option 1, Option 2, Option 3">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Display Order</label>
                            <input type="number" name="display_order" class="form-control" value="0">
                        </div>
                        <div class="form-check mb-2">
                            <input class="form-check-input" type="checkbox" name="is_searchable" id="addSearchable" checked>
                            <label class="form-check-label" for="addSearchable">Searchable</label>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="is_required" id="addRequired">
                            <label class="form-check-label" for="addRequired">Required</label>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="add_field" class="btn btn-primary">Add Field</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Edit Field Modal -->
    <div class="modal fade" id="editFieldModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title"><i class="fas fa-edit me-2"></i>Edit Custom Field</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="post">
                    <input type="hidden" name="field_id" id="editFieldId">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">Field Key</label>
                            <input type="text" id="editFieldKey" class="form-control" disabled>
                            <small class="text-muted">Key cannot be changed</small>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Field Label <span class="text-danger">*</span></label>
                            <input type="text" name="field_label" id="editFieldLabel" class="form-control" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Field Type</label>
                            <select name="field_type" class="form-select" id="editFieldType" onchange="toggleOptions('edit')">
                                <option value="text">Text</option>
                                <option value="number">Number</option>
                                <option value="date">Date</option>
                                <option value="select">Dropdown Select</option>
                                <option value="textarea">Text Area</option>
                            </select>
                        </div>
                        <div class="mb-3 d-none" id="editOptionsGroup">
                            <label class="form-label">Options (comma-separated)</label>
                            <input type="text" name="field_options" id="editFieldOptions" class="form-control">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Display Order</label>
                            <input type="number" name="display_order" id="editDisplayOrder" class="form-control">
                        </div>
                        <div class="form-check mb-2">
                            <input class="form-check-input" type="checkbox" name="is_searchable" id="editSearchable">
                            <label class="form-check-label" for="editSearchable">Searchable</label>
                        </div>
                        <div class="form-check mb-2">
                            <input class="form-check-input" type="checkbox" name="is_required" id="editRequired">
                            <label class="form-check-label" for="editRequired">Required</label>
                        </div>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="is_active" id="editActive">
                            <label class="form-check-label" for="editActive">Active</label>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="update_field" class="btn btn-primary">Update Field</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Delete Confirmation Modal -->
    <div class="modal fade" id="deleteFieldModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title"><i class="fas fa-exclamation-triangle me-2"></i>Confirm Delete</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <form method="post">
                    <input type="hidden" name="field_id" id="deleteFieldId">
                    <div class="modal-body">
                        <p>Are you sure you want to delete the custom field "<strong id="deleteFieldName"></strong>"?</p>
                        <div class="alert alert-warning">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            This will remove the field definition, but existing data in participants will be preserved.
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="delete_field" class="btn btn-danger">Delete Field</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <script>
        function toggleOptions(mode) {
            const typeSelect = document.getElementById(mode + 'FieldType');
            const optionsGroup = document.getElementById(mode + 'OptionsGroup');
            
            if (typeSelect.value === 'select') {
                optionsGroup.classList.remove('d-none');
            } else {
                optionsGroup.classList.add('d-none');
            }
        }
        
        function editField(field) {
            document.getElementById('editFieldId').value = field.id;
            document.getElementById('editFieldKey').value = field.field_key;
            document.getElementById('editFieldLabel').value = field.field_label;
            document.getElementById('editFieldType').value = field.field_type;
            document.getElementById('editDisplayOrder').value = field.display_order;
            document.getElementById('editSearchable').checked = field.is_searchable == 1;
            document.getElementById('editRequired').checked = field.is_required == 1;
            document.getElementById('editActive').checked = field.is_active == 1;
            
            if (field.field_options) {
                const options = JSON.parse(field.field_options);
                document.getElementById('editFieldOptions').value = options.join(', ');
            }
            
            toggleOptions('edit');
            new bootstrap.Modal(document.getElementById('editFieldModal')).show();
        }
        
        function deleteField(id, label) {
            document.getElementById('deleteFieldId').value = id;
            document.getElementById('deleteFieldName').textContent = label;
            new bootstrap.Modal(document.getElementById('deleteFieldModal')).show();
        }
    </script>
    
<?php include 'footer.php'; ?>
</body>
</html>
