<?php
/**
 * SSH2 Extension Checker for StyleScore
 * 
 * This script checks if the PHP SSH2 extension is properly installed
 * and provides troubleshooting information.
 */

require_once '../includes/auth.php'; // Require authentication

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>SSH2 Extension Check - StyleScore</title>
    <?php include_once '../includes/stylesheets.php'; ?>
    <style>
        .status-box {
            padding: 20px;
            border-radius: 10px;
            margin: 20px 0;
            font-size: 1.2em;
        }
        .status-ok {
            background: #d4edda;
            border: 2px solid #28a745;
            color: #155724;
        }
        .status-error {
            background: #f8d7da;
            border: 2px solid #dc3545;
            color: #721c24;
        }
        .info-table {
            background: white;
            border-radius: 8px;
        }
        .info-table th {
            background: #f8f9fa;
            font-weight: 600;
        }
        code {
            background: #f4f4f4;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
        }
        .command-box {
            background: #2d2d2d;
            color: #f8f8f2;
            padding: 15px;
            border-radius: 8px;
            font-family: 'Courier New', monospace;
            overflow-x: auto;
        }
    </style>
</head>
<body class="body-bg-aurora-bright">
<?php include '../menu.php'; ?>

<div class="container-fluid container-StyleScore p-4">
    <div class="row">
        <div class="col-lg-8 offset-lg-2">
            <div class="card border-0 shadow-sm">
                <div class="card-header bg-dark text-white">
                    <h1 class="h4 mb-0">
                        <i class="fas fa-server me-2"></i>SSH2 Extension Status Check
                    </h1>
                </div>
                <div class="card-body">
                    
                    <?php
                    $ssh2_installed = function_exists('ssh2_connect');
                    $ssh2_version = $ssh2_installed ? phpversion('ssh2') : 'Not installed';
                    $php_version = phpversion();
                    ?>
                    
                    <!-- Status Box -->
                    <div class="status-box <?= $ssh2_installed ? 'status-ok' : 'status-error' ?>">
                        <?php if ($ssh2_installed): ?>
                            <i class="fas fa-check-circle fa-2x mb-2"></i>
                            <h3>✅ SSH2 Extension is Installed!</h3>
                            <p class="mb-0">SFTP publishing is ready to use.</p>
                        <?php else: ?>
                            <i class="fas fa-exclamation-triangle fa-2x mb-2"></i>
                            <h3>❌ SSH2 Extension Not Found</h3>
                            <p class="mb-0">You need to install the PHP SSH2 extension to use SFTP publishing.</p>
                        <?php endif; ?>
                    </div>

                    <!-- System Information -->
                    <h5 class="mt-4 mb-3"><i class="fas fa-info-circle me-2"></i>System Information</h5>
                    <table class="table table-bordered info-table">
                        <tr>
                            <th width="30%">PHP Version</th>
                            <td><code><?= $php_version ?></code></td>
                        </tr>
                        <tr>
                            <th>SSH2 Extension Status</th>
                            <td>
                                <?php if ($ssh2_installed): ?>
                                    <span class="badge bg-success">Installed</span>
                                <?php else: ?>
                                    <span class="badge bg-danger">Not Installed</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php if ($ssh2_installed): ?>
                        <tr>
                            <th>SSH2 Extension Version</th>
                            <td><code><?= $ssh2_version ?></code></td>
                        </tr>
                        <?php endif; ?>
                        <tr>
                            <th>Operating System</th>
                            <td><code><?= PHP_OS ?></code></td>
                        </tr>
                        <tr>
                            <th>PHP Configuration File</th>
                            <td><code><?= php_ini_loaded_file() ?></code></td>
                        </tr>
                        <tr>
                            <th>Extensions Directory</th>
                            <td><code><?= ini_get('extension_dir') ?></code></td>
                        </tr>
                    </table>

                    <!-- Available SSH2 Functions -->
                    <?php if ($ssh2_installed): ?>
                    <h5 class="mt-4 mb-3"><i class="fas fa-code me-2"></i>Available SSH2 Functions</h5>
                    <div class="alert alert-success">
                        <ul class="mb-0">
                            <li><code>ssh2_connect()</code> - Connect to SSH server ✓</li>
                            <li><code>ssh2_auth_password()</code> - Authenticate with password ✓</li>
                            <li><code>ssh2_sftp()</code> - Initialize SFTP subsystem ✓</li>
                            <li><code>ssh2_scp_send()</code> - Send file via SCP ✓</li>
                            <li><code>ssh2_sftp_mkdir()</code> - Create remote directory ✓</li>
                        </ul>
                    </div>
                    <?php endif; ?>

                    <!-- Installation Instructions -->
                    <?php if (!$ssh2_installed): ?>
                    <h5 class="mt-4 mb-3"><i class="fas fa-wrench me-2"></i>Installation Instructions</h5>
                    
                    <div class="accordion" id="installAccordion">
                        <!-- Windows -->
                        <div class="accordion-item">
                            <h2 class="accordion-header">
                                <button class="accordion-button" type="button" data-bs-toggle="collapse" data-bs-target="#collapseWindows">
                                    <i class="fab fa-windows me-2"></i>Windows (XAMPP/WAMP)
                                </button>
                            </h2>
                            <div id="collapseWindows" class="accordion-collapse collapse show" data-bs-parent="#installAccordion">
                                <div class="accordion-body">
                                    <ol>
                                        <li>Download SSH2 DLL from <a href="https://pecl.php.net/package/ssh2" target="_blank">PECL</a></li>
                                        <li>Extract and copy <code>php_ssh2.dll</code> to: <code>C:\xampp\php\ext\</code></li>
                                        <li>Edit <code>php.ini</code> and add: <code>extension=ssh2</code></li>
                                        <li>Restart Apache</li>
                                        <li>Refresh this page to verify</li>
                                    </ol>
                                    <div class="alert alert-info">
                                        <strong>Note:</strong> Make sure to download the correct version matching your PHP version (<?= $php_version ?>)
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Linux -->
                        <div class="accordion-item">
                            <h2 class="accordion-header">
                                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseLinux">
                                    <i class="fab fa-linux me-2"></i>Linux (Ubuntu/Debian)
                                </button>
                            </h2>
                            <div id="collapseLinux" class="accordion-collapse collapse" data-bs-parent="#installAccordion">
                                <div class="accordion-body">
                                    <p>Run these commands in your terminal:</p>
                                    <div class="command-box">
sudo apt-get update<br>
sudo apt-get install php-ssh2<br>
sudo systemctl restart apache2
                                    </div>
                                    <p class="mt-3">Or compile from source:</p>
                                    <div class="command-box">
sudo apt-get install libssh2-1-dev php-dev php-pear<br>
sudo pecl install ssh2<br>
echo "extension=ssh2.so" | sudo tee /etc/php/<?= substr($php_version, 0, 3) ?>/mods-available/ssh2.ini<br>
sudo phpenmod ssh2<br>
sudo systemctl restart apache2
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- macOS -->
                        <div class="accordion-item">
                            <h2 class="accordion-header">
                                <button class="accordion-button collapsed" type="button" data-bs-toggle="collapse" data-bs-target="#collapseMac">
                                    <i class="fab fa-apple me-2"></i>macOS
                                </button>
                            </h2>
                            <div id="collapseMac" class="accordion-collapse collapse" data-bs-parent="#installAccordion">
                                <div class="accordion-body">
                                    <p>Using Homebrew:</p>
                                    <div class="command-box">
brew install libssh2<br>
pecl install ssh2<br>
brew services restart httpd
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="alert alert-warning mt-4">
                        <i class="fas fa-book me-2"></i>
                        For detailed instructions, see <a href="../SFTP_SETUP_GUIDE.md" target="_blank"><strong>SFTP_SETUP_GUIDE.md</strong></a>
                    </div>
                    <?php endif; ?>

                    <!-- Quick Test -->
                    <?php if ($ssh2_installed): ?>
                    <h5 class="mt-4 mb-3"><i class="fas fa-vial me-2"></i>Quick Connection Test</h5>
                    <div class="alert alert-info">
                        <p>You can now test SFTP connections in the <a href="publish_servers.php" class="alert-link"><strong>Publish Servers</strong></a> panel.</p>
                        <p class="mb-0">Click "Test Connection" when adding or editing a server to verify your SFTP credentials.</p>
                    </div>
                    <?php endif; ?>

                    <!-- Action Buttons -->
                    <div class="mt-4 d-flex gap-2">
                        <a href="publish_servers.php" class="btn btn-primary">
                            <i class="fas fa-server me-2"></i>Go to Publish Servers
                        </a>
                        <button onclick="location.reload()" class="btn btn-secondary">
                            <i class="fas fa-sync-alt me-2"></i>Refresh Check
                        </button>
                        <a href="../SFTP_SETUP_GUIDE.md" target="_blank" class="btn btn-outline-info">
                            <i class="fas fa-book me-2"></i>Setup Guide
                        </a>
                    </div>

                </div>
            </div>
        </div>
    </div>
</div>

</body>
</html>
