<?php
include '../includes/auth.php';
include '../includes/db.php';
include '../includes/bib_helpers.php';

// Get event_id from GET, POST, Cookie, or fallback to null
$event_id = $_GET['event_id'] ?? $_POST['event_id'] ?? $_COOKIE['event_id'] ?? $_COOKIE['selectedEventId'] ?? null;

// Handle new series submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_series'])) {
    if (!empty($_POST['name']) && !empty($_POST['event_id'])) {
        $event_id = $_POST['event_id'];
        $name = $_POST['name'];
        $color = $_POST['color'];
        $start = (int)$_POST['start_number'];
        $end = (int)$_POST['end_number'];
        $missing = trim($_POST['missing_numbers']);
        $additional = trim($_POST['additional_numbers']);

        // Prepare full number list for new series
        $new_range = range($start, $end);
        $new_missing = array_filter(array_map('trim', explode(',', $missing)), 'is_numeric');
        $new_additional = array_filter(array_map('trim', explode(',', $additional)), 'is_numeric');
        $all_new = array_unique(array_merge($new_range, $new_additional));
        $all_new = array_diff($all_new, $new_missing);

        // Check for conflicts with existing series in this event
        $conflict_series = $pdo->prepare("SELECT * FROM bib_series WHERE event_id = ?");
        $conflict_series->execute([$event_id]);

        $has_conflict = false;
        foreach ($conflict_series as $existing) {
            $existing_range = range($existing['start_number'], $existing['end_number']);
            $existing_missing = array_filter(array_map('trim', explode(',', $existing['missing_numbers'])), 'is_numeric');
            $existing_additional = array_filter(array_map('trim', explode(',', $existing['additional_numbers'])), 'is_numeric');
            $existing_all = array_unique(array_merge($existing_range, $existing_additional));
            $existing_all = array_diff($existing_all, $existing_missing);

            $overlap = array_intersect($all_new, $existing_all);
            if (count($overlap) > 0) {
                $has_conflict = true;
                $conflict_details = [
                    'series_name' => $existing['name'],
                    'conflicts' => implode(', ', $overlap)
                ];
                break;
            }
        }

        if ($has_conflict && !isset($_POST['force_create'])) {
            $_SESSION['conflict_message'] = [
                'series_name' => $conflict_details['series_name'],
                'conflicts' => $conflict_details['conflicts'],
                'post_data' => $_POST
            ];
            header("Location: bib_manager.php?event_id=$event_id&show_conflict=1");
            exit;
        }

        $check = $pdo->prepare("SELECT COUNT(*) FROM bib_series WHERE event_id = ? AND name = ?");
        $check->execute([$event_id, $name]);
        if ($check->fetchColumn() == 0) {
            $stmt = $pdo->prepare("INSERT INTO bib_series (event_id, name, color, start_number, end_number, missing_numbers, additional_numbers) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$event_id, $name, $color, $start, $end, $missing, $additional]);
            $_SESSION['success_message'] = "Series '{$name}' created successfully!";
            header("Location: bib_manager.php?event_id=$event_id");
            exit;
        }
    }
}

// Handle update series submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_series'])) {
    if (!empty($_POST['series_id']) && !empty($_POST['name']) && !empty($_POST['event_id'])) {
        $series_id = $_POST['series_id'];
        $event_id = $_POST['event_id'];
        $name = $_POST['name'];
        $color = $_POST['color'];
        $start = (int)$_POST['start_number'];
        $end = (int)$_POST['end_number'];
        $missing = trim($_POST['missing_numbers']);
        $additional = trim($_POST['additional_numbers']);

        // Prepare full number list for updated series
        $new_range = range($start, $end);
        $new_missing = array_filter(array_map('trim', explode(',', $missing)), 'is_numeric');
        $new_additional = array_filter(array_map('trim', explode(',', $additional)), 'is_numeric');
        $all_new = array_unique(array_merge($new_range, $new_additional));
        $all_new = array_diff($all_new, $new_missing);

        // Check for conflicts with OTHER existing series in this event
        $conflict_series = $pdo->prepare("SELECT * FROM bib_series WHERE event_id = ? AND id != ?");
        $conflict_series->execute([$event_id, $series_id]);

        $has_conflict = false;
        foreach ($conflict_series as $existing) {
            $existing_range = range($existing['start_number'], $existing['end_number']);
            $existing_missing = array_filter(array_map('trim', explode(',', $existing['missing_numbers'])), 'is_numeric');
            $existing_additional = array_filter(array_map('trim', explode(',', $existing['additional_numbers'])), 'is_numeric');
            $existing_all = array_unique(array_merge($existing_range, $existing_additional));
            $existing_all = array_diff($existing_all, $existing_missing);

            $overlap = array_intersect($all_new, $existing_all);
            if (count($overlap) > 0) {
                $has_conflict = true;
                $conflict_details = [
                    'series_name' => $existing['name'],
                    'conflicts' => implode(', ', $overlap)
                ];
                break;
            }
        }

        if ($has_conflict && !isset($_POST['force_update'])) {
            $_SESSION['conflict_message'] = [
                'series_name' => $conflict_details['series_name'],
                'conflicts' => $conflict_details['conflicts'],
                'post_data' => $_POST,
                'is_update' => true
            ];
            header("Location: bib_manager.php?event_id=$event_id&show_conflict=1");
            exit;
        }

        $stmt = $pdo->prepare("UPDATE bib_series SET name = ?, color = ?, start_number = ?, end_number = ?, missing_numbers = ?, additional_numbers = ? WHERE id = ?");
        $stmt->execute([$name, $color, $start, $end, $missing, $additional, $series_id]);
        $_SESSION['success_message'] = "Series '{$name}' updated successfully!";
        header("Location: bib_manager.php?event_id=$event_id");
        exit;
    }
}

// Handle delete
if (isset($_GET['delete_series'])) {
    $stmt = $pdo->prepare("DELETE FROM bib_series WHERE id = ?");
    $stmt->execute([$_GET['delete_series']]);
    $_SESSION['success_message'] = "Series deleted successfully!";
    header("Location: bib_manager.php?event_id=" . (int)$_GET['event_id']);
    exit;
}

// Fetch events
$events = $pdo->query("SELECT id, name, date FROM events ORDER BY id DESC")->fetchAll(PDO::FETCH_ASSOC);

// Fetch BIB series and participants
$bib_series = [];
$participants = [];
$series_stats = [];

if ($event_id) {
    $stmt = $pdo->prepare("SELECT * FROM bib_series WHERE event_id = ? ORDER BY start_number ASC");
    $stmt->execute([$event_id]);
    $bib_series = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $participants = get_event_participants_for_bib_distribution($pdo, $event_id);
    
    // Calculate series stats
    foreach ($bib_series as $series) {
        $all_numbers = range($series['start_number'], $series['end_number']);
        $missing = array_filter(array_map('trim', explode(',', $series['missing_numbers'] ?? '')), 'is_numeric');
        $additional = array_filter(array_map('trim', explode(',', $series['additional_numbers'] ?? '')), 'is_numeric');
        $available = array_unique(array_merge($all_numbers, $additional));
        $available = array_values(array_diff($available, $missing));
        sort($available);
        
        $used = [];
        foreach ($participants as $p) {
            $bib_val = $p['bib_number'] ?? $p['bib'] ?? null;
            if ($bib_val && in_array((int)$bib_val, $available)) {
                $used[] = (int)$bib_val;
            }
        }
        
        $series_stats[] = [
            'series' => $series,
            'available_numbers' => array_values(array_diff($available, $used)),
            'used_numbers' => $used,
            'total' => count($available)
        ];
    }
    
    // Get unique categories from participants
    $unique_categories = [];
    foreach ($participants as $p) {
        $cat = $p['categories'] ?? $p['category'] ?? '';
        if ($cat) {
            // Handle comma-separated categories (from GROUP_CONCAT)
            $cats = array_map('trim', explode(',', $cat));
            foreach ($cats as $c) {
                if ($c && !in_array($c, $unique_categories)) {
                    $unique_categories[] = $c;
                }
            }
        }
    }
    sort($unique_categories);
}

$countries = $pdo->query("SELECT DISTINCT country FROM participants WHERE country IS NOT NULL AND country != '' ORDER BY country")->fetchAll(PDO::FETCH_COLUMN);

// Get custom field definitions
$custom_field_defs = [];
try {
    $custom_field_defs = $pdo->query("
        SELECT * FROM participant_custom_field_definitions 
        WHERE is_active = 1 
        ORDER BY display_order ASC, id ASC
    ")->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    // Table might not exist yet, silently fail
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>BIB Manager</title>
    <?php include_once '../includes/stylesheets.php'; ?>
    
    <style>
        .bib-series-card {
            transition: all 0.3s ease;
            border-left: 4px solid;
        }
        .bib-series-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }
        .color-pick {
            cursor: pointer;
            transition: transform 0.2s;
        }
        .color-pick:hover {
            transform: scale(1.1);
        }
        .participant-row {
            transition: background-color 0.2s;
        }
        .participant-row:hover {
            background-color: rgba(0,123,255,0.05);
        }
        .bib-input {
            width: 80px;
        }
        .series-badge {
            font-size: 0.75rem;
            padding: 0.25rem 0.5rem;
        }
    </style>
</head>
<body class="body-bg-aurora-bright">
<?php include '../menu.php'; ?>

<div class="container-fluid container-StyleScore p-4">
    <!-- Event Actions Menu -->
    <div class="row mb-3">
        <div class="col-12">
            <?php $use_localStorage = true; $show_header = false; include 'event_actions_menu.php'; ?>
        </div>
    </div>

    <!-- Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3 mb-1">
                <i class="fas fa-tags text-primary me-2"></i>BIB Manager
            </h1>
            <p class="text-muted mb-0">Create series and distribute BIB numbers</p>
        </div>
        <button type="button" class="btn btn-outline-secondary btn-sm" data-bs-toggle="modal" data-bs-target="#helpModal">
            <i class="fas fa-circle-question me-1"></i>Help
        </button>
    </div>

    <!-- Success/Error Messages -->
    <?php if (isset($_SESSION['success_message'])): ?>
    <div class="alert alert-success alert-dismissible fade show">
        <i class="fas fa-check-circle me-2"></i><?= $_SESSION['success_message'] ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    <?php unset($_SESSION['success_message']); endif; ?>

    <!-- Event Selection -->
    <?php if (empty($event_id)): ?>
    <div class="card bg-ss-sidebar text-white mb-4">
        <div class="card-header">
            <h5 class="mb-0"><i class="fas fa-calendar-alt me-2"></i>Event Selection</h5>
        </div>
        <div class="card-body">
            <form method="get">
                <div class="row g-3 align-items-end">
                    <div class="col-md-8">
                        <label class="form-label text-white">Select Event:</label>
                        <select name="event_id" class="form-select" onchange="this.form.submit()">
                            <option value="">-- Choose Event --</option>
                            <?php foreach ($events as $e): ?>
                                <option value="<?= $e['id'] ?>"><?= htmlspecialchars($e['name']) ?> (<?= date('M d, Y', strtotime($e['date'])) ?>)</option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-4">
                        <button type="submit" class="btn btn-light">
                            <i class="fas fa-search me-2"></i>Load Event
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>
    <?php endif; ?>

    <?php if ($event_id): ?>
    
    <!-- Two-Pane Layout -->
    <div class="row g-4">
        
        <!-- LEFT PANE: Series Management & Distribution -->
        <div class="col-lg-4 col-xxl-3">
            
            <!-- Create New Series -->
            <div class="card shadow-sm mb-3">
                <div class="card-header bg-white">
                    <h6 class="mb-0"><i class="fas fa-plus-circle text-success me-2"></i>Create Series</h6>
                </div>
                <div class="card-body p-3">
                    <form method="post" id="createSeriesForm">
                        <input type="hidden" name="event_id" value="<?= $event_id ?>">
                        <input type="hidden" name="create_series" value="1">
                        
                        <div class="mb-3">
                            <label class="form-label small fw-medium">Series Name</label>
                            <input type="text" name="name" class="form-control form-control-sm" placeholder="Elite, Juniors, etc." required>
                        </div>
                        
                        <div class="row g-2 mb-3">
                            <div class="col-6">
                                <label class="form-label small fw-medium">Start #</label>
                                <input type="number" name="start_number" class="form-control form-control-sm" placeholder="1" required>
                            </div>
                            <div class="col-6">
                                <label class="form-label small fw-medium">End #</label>
                                <input type="number" name="end_number" class="form-control form-control-sm" placeholder="100" required>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label small fw-medium">Missing Numbers</label>
                            <input type="text" name="missing_numbers" class="form-control form-control-sm" placeholder="13, 27, 42">
                            <small class="text-muted">Comma separated</small>
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label small fw-medium">Additional Numbers</label>
                            <input type="text" name="additional_numbers" class="form-control form-control-sm" placeholder="201, 202">
                        </div>
                        
                        <div class="mb-3">
                            <label class="form-label small fw-medium">Color</label>
                            <div class="d-flex gap-2 flex-wrap mb-2">
                                <button type="button" class="btn p-0 color-pick border border-2" data-color="#000000" style="width: 35px; height: 35px; background-color: #000000; border-radius: 6px;"></button>
                                <button type="button" class="btn p-0 color-pick border border-2" data-color="#007BFF" style="width: 35px; height: 35px; background-color: #007BFF; border-radius: 6px;"></button>
                                <button type="button" class="btn p-0 color-pick border border-2" data-color="#DC3545" style="width: 35px; height: 35px; background-color: #DC3545; border-radius: 6px;"></button>
                                <button type="button" class="btn p-0 color-pick border border-2" data-color="#28A745" style="width: 35px; height: 35px; background-color: #28A745; border-radius: 6px;"></button>
                                <button type="button" class="btn p-0 color-pick border border-2" data-color="#FD7E14" style="width: 35px; height: 35px; background-color: #FD7E14; border-radius: 6px;"></button>
                                <button type="button" class="btn p-0 color-pick border border-2" data-color="#6F42C1" style="width: 35px; height: 35px; background-color: #6F42C1; border-radius: 6px;"></button>
                            </div>
                            <input type="text" name="color" id="colorInput" class="form-control form-control-sm" placeholder="#000000" value="#000000" pattern="^#[0-9A-Fa-f]{6}$">
                        </div>
                        
                        <button type="submit" class="btn btn-success btn-sm w-100">
                            <i class="fas fa-plus me-2"></i>Create Series
                        </button>
                    </form>
                </div>
            </div>
            
            <!-- Series Overview -->
            <div class="card shadow-sm mb-3">
                <div class="card-header bg-white d-flex justify-content-between align-items-center">
                    <h6 class="mb-0"><i class="fas fa-list text-info me-2"></i>Series</h6>
                    <span class="badge bg-secondary"><?= count($bib_series) ?></span>
                </div>
                <div class="card-body p-2">
                    <?php if (empty($bib_series)): ?>
                        <div class="text-center py-4">
                            <i class="fas fa-tags fa-2x text-muted mb-2"></i>
                            <p class="text-muted small mb-0">No series created</p>
                        </div>
                    <?php else: ?>
                        <div class="d-grid gap-2">
                            <?php foreach ($series_stats as $stat): 
                                $series = $stat['series'];
                                $available_count = count($stat['available_numbers']);
                                $used_count = count($stat['used_numbers']);
                                $percentage = $stat['total'] > 0 ? round(($used_count / $stat['total']) * 100) : 0;
                            ?>
                            <div class="card bib-series-card" style="border-left-color: <?= $series['color'] ?>;">
                                <div class="card-body p-2">
                                    <div class="d-flex justify-content-between align-items-start mb-2">
                                        <div class="flex-grow-1">
                                            <div class="fw-bold small"><?= htmlspecialchars($series['name']) ?></div>
                                            <div class="text-muted" style="font-size: 0.75rem;"><?= $series['start_number'] ?>-<?= $series['end_number'] ?></div>
                                        </div>
                                        <div class="btn-group btn-group-sm">
                                            <button class="btn btn-outline-info btn-sm" onclick="showSeriesDetails(<?= $series['id'] ?>)" title="Details">
                                                <i class="fas fa-info-circle"></i>
                                            </button>
                                            <button class="btn btn-outline-secondary btn-sm" onclick="editSeries(<?= $series['id'] ?>)" title="Edit">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <button class="btn btn-outline-success btn-sm" onclick="showAvailableNumbers(<?= $series['id'] ?>)" title="Available Numbers">
                                                <i class="fas fa-list-ol"></i>
                                            </button>
                                            <button class="btn btn-outline-primary btn-sm" onclick="assignNextAvailable(<?= $series['id'] ?>)" title="Assign Next">
                                                <i class="fas fa-plus"></i>
                                            </button>
                                            <button class="btn btn-outline-danger btn-sm" onclick="confirmDelete('<?= htmlspecialchars($series['name']) ?>', <?= $series['id'] ?>, <?= $event_id ?>); return false;" title="Delete">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </div>
                                    </div>
                                    <div class="progress" style="height: 6px;">
                                        <div class="progress-bar" style="width: <?= $percentage ?>%; background-color: <?= $series['color'] ?>;"></div>
                                    </div>
                                    <div class="d-flex justify-content-between mt-1" style="font-size: 0.7rem;">
                                        <span class="text-muted"><?= $used_count ?> used</span>
                                        <span class="text-success"><?= $available_count ?> free</span>
                                    </div>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Quick Actions -->
            <div class="card shadow-sm">
                <div class="card-header bg-white">
                    <h6 class="mb-0"><i class="fas fa-bolt text-warning me-2"></i>Actions</h6>
                </div>
                <div class="card-body p-3">
                    <div class="d-grid gap-2">
                        <button class="btn btn-primary btn-sm" onclick="autoAssignBIBs()">
                            <i class="fas fa-magic me-2"></i>Auto Assign
                        </button>
                        <button class="btn btn-secondary btn-sm" onclick="validateBIBs()">
                            <i class="fas fa-check-double me-2"></i>Validate
                        </button>
                        <button class="btn btn-info btn-sm" onclick="showDetailedReport()">
                            <i class="fas fa-file-alt me-2"></i>Report
                        </button>
                        <button class="btn btn-outline-danger btn-sm" onclick="confirmClearAll()">
                            <i class="fas fa-eraser me-2"></i>Clear All
                        </button>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- RIGHT PANE: Participant Distribution Table -->
        <div class="col-lg-8 col-xxl-9">
            <div class="card shadow-sm">
                <div class="card-header bg-white">
                    <div class="d-flex justify-content-between align-items-center">
                        <h6 class="mb-0"><i class="fas fa-users text-primary me-2"></i>Athlete Distribution</h6>
                        <div class="text-end">
                            <span class="badge bg-primary me-2" id="assignedCount">0</span> assigned
                            <span class="badge bg-secondary me-2" id="unassignedCount">0</span> unassigned
                            <span class="badge bg-danger" id="conflictCount">0</span> conflicts
                        </div>
                    </div>
                </div>
                
                <!-- Filters -->
                <div class="card-body bg-light border-bottom p-3">
                    <div class="row g-2 align-items-end">
                        <div class="col-md-2">
                            <label class="form-label small mb-1">Filter by Status</label>
                            <select id="filterStatus" class="form-select form-select-sm" onchange="applyFilters()">
                                <option value="all">All Athletes</option>
                                <option value="assigned">✓ Has BIB</option>
                                <option value="unassigned">✗ No BIB</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label small mb-1">Filter by Series</label>
                            <select id="filterSeries" class="form-select form-select-sm" onchange="applyFilters()">
                                <option value="all">All Series</option>
                                <?php foreach ($series_stats as $stat): ?>
                                    <option value="<?= $stat['series']['id'] ?>">
                                        <?= htmlspecialchars($stat['series']['name']) ?> (<?= $stat['series']['start_number'] ?>-<?= $stat['series']['end_number'] ?>)
                                    </option>
                                <?php endforeach; ?>
                                <option value="unknown">⚠️ Unknown Series</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <label class="form-label small mb-1">Filter by Division</label>
                            <select id="filterCategory" class="form-select form-select-sm" onchange="applyFilters()">
                                <option value="all">All Divisions</option>
                                <?php foreach ($unique_categories as $cat): ?>
                                    <option value="<?= htmlspecialchars($cat) ?>"><?= htmlspecialchars($cat) ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label small mb-1">Search</label>
                            <input type="text" id="searchParticipant" class="form-control form-control-sm" placeholder="Name, club..." onkeyup="applyFilters()">
                        </div>
                        <div class="col-md-1">
                            <button class="btn btn-outline-info btn-sm w-100" onclick="showColumnVisibilityModal()" title="Column Visibility">
                                <i class="fas fa-columns"></i>
                            </button>
                        </div>
                        <div class="col-md-1">
                            <button class="btn btn-outline-secondary btn-sm w-100" onclick="clearFilters()">
                                <i class="fas fa-times me-1"></i>Clear
                            </button>
                        </div>
                    </div>
                    <div class="mt-2">
                        <small class="text-muted">
                            <i class="fas fa-info-circle me-1"></i>
                            <span id="visibleCount"><?= count($participants) ?></span> of <?= count($participants) ?> participants visible
                        </small>
                    </div>
                </div>
                
                <div class="card-body p-0">
                    <?php if (empty($participants)): ?>
                        <div class="text-center py-5">
                            <i class="fas fa-user-slash fa-3x text-muted mb-3"></i>
                            <h5 class="text-muted">No Athletes</h5>
                            <p class="text-muted">No athletes assigned to this event</p>
                        </div>
                    <?php else: ?>
                        <form method="post" action="bib_distribute_save.php" id="bibAssignForm">
                            <input type="hidden" name="event_id" value="<?= $event_id ?>">
                            
                            <div class="table-responsive">
                                <table class="table table-sm table-hover mb-0" id="participantTable">
                                    <thead class="table-light sticky-top">
                                        <tr>
                                            <th style="width: 60px;" class="sortable" data-sort="current-bib">
                                                Current BIB <i class="fas fa-sort ms-1 text-muted sort-icon"></i>
                                            </th>
                                            <th class="sortable" data-sort="name">
                                                Name <i class="fas fa-sort ms-1 text-muted sort-icon"></i>
                                            </th>
                                            <th style="width: 80px;" class="sortable" data-sort="gender">
                                                Gender <i class="fas fa-sort ms-1 text-muted sort-icon"></i>
                                            </th>
                                            <th style="width: 80px;" class="sortable" data-sort="year">
                                                Year <i class="fas fa-sort ms-1 text-muted sort-icon"></i>
                                            </th>
                                            <th style="width: 120px;" class="sortable" data-sort="country">
                                                Country <i class="fas fa-sort ms-1 text-muted sort-icon"></i>
                                            </th>
                                            <th style="width: 150px;" class="sortable" data-sort="club">
                                                Club <i class="fas fa-sort ms-1 text-muted sort-icon"></i>
                                            </th>
                                            <th style="width: 120px;" class="sortable" data-sort="category" data-col-index="6">
                                                Division <i class="fas fa-sort ms-1 text-muted sort-icon"></i>
                                            </th>
                                            <?php 
                                            $custom_col_index = 7;
                                            foreach ($custom_field_defs as $field): 
                                            ?>
                                            <th style="width: 120px;" class="sortable" data-sort="custom_<?= htmlspecialchars($field['field_key']) ?>" data-col-index="<?= $custom_col_index++ ?>">
                                                <?= htmlspecialchars($field['field_label']) ?> <i class="fas fa-sort ms-1 text-muted sort-icon"></i>
                                            </th>
                                            <?php endforeach; ?>
                                            <th style="width: 100px;" class="text-center" data-col-index="<?= $custom_col_index++ ?>">New BIB</th>
                                            <th style="width: 150px;" data-col-index="<?= $custom_col_index ?>">Series</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($participants as $p): 
                                            $current_bib = $p['bib_number'] ?? $p['bib'] ?? null;
                                            $country = $p['country'] ?? '';
                                            $category = $p['categories'] ?? $p['category'] ?? '';
                                            
                                            // Find series color for current BIB
                                            $bib_color = '#6c757d'; // default gray
                                            $bib_series_name = '';
                                            $bib_series_id = '';
                                            if ($current_bib) {
                                                foreach ($series_stats as $stat) {
                                                    $all_nums = array_merge($stat['available_numbers'], $stat['used_numbers']);
                                                    if (in_array((int)$current_bib, $all_nums)) {
                                                        $bib_color = $stat['series']['color'];
                                                        $bib_series_name = $stat['series']['name'];
                                                        $bib_series_id = $stat['series']['id'];
                                                        break;
                                                    }
                                                }
                                            }
                                            
                                            // Calculate birth year
                                            $dob = $p['date_of_birth'] ?? '';
                                            $birth_year = '';
                                            if ($dob && preg_match('/^(\d{4})/', $dob, $matches)) {
                                                $birth_year = $matches[1];
                                            }
                                            
                                            // Parse custom fields for data attributes
                                            $custom_fields_data = [];
                                            if (!empty($p['custom_fields'])) {
                                                $custom_fields_data = is_string($p['custom_fields']) ? json_decode($p['custom_fields'], true) : $p['custom_fields'];
                                            }
                                            
                                            // Build custom field attributes string
                                            $custom_attrs = '';
                                            foreach ($custom_field_defs as $field) {
                                                $value = $custom_fields_data[$field['field_key']] ?? '';
                                                $custom_attrs .= ' data-custom-' . htmlspecialchars($field['field_key']) . '="' . htmlspecialchars(strtolower($value)) . '"';
                                            }
                                        ?>
                                        <tr class="participant-row" 
                                            data-participant-id="<?= $p['id'] ?>"
                                            data-bib="<?= $current_bib ?: '' ?>"
                                            data-series-id="<?= $bib_series_id ?>"
                                            data-series-name="<?= htmlspecialchars($bib_series_name) ?>"
                                            data-name="<?= htmlspecialchars(strtolower($p['first_name'] . ' ' . $p['last_name'])) ?>"
                                            data-club="<?= htmlspecialchars(strtolower($p['club'] ?? '')) ?>"
                                            data-category="<?= htmlspecialchars(strtolower($category)) ?>"
                                            data-gender="<?= htmlspecialchars($p['gender'] ?? '') ?>"
                                            data-country="<?= htmlspecialchars($country) ?>"
                                            data-birth-year="<?= htmlspecialchars($birth_year) ?>"<?= $custom_attrs ?>>
                                            <td class="text-center">
                                                <?php if ($current_bib): ?>
                                                    <span class="badge" style="background-color: <?= $bib_color ?>; color: white;" title="<?= htmlspecialchars($bib_series_name) ?>"><?= $current_bib ?></span>
                                                <?php else: ?>
                                                    <span class="text-muted">-</span>
                                                <?php endif; ?>
                                            </td>
                                            <td class="fw-medium"><?= htmlspecialchars($p['first_name'] . ' ' . $p['last_name']) ?></td>
                                            <td>
                                                <?php if (($p['gender'] ?? '') === 'M'): ?>
                                                    <span class="badge bg-primary"><i class="fas fa-male me-1"></i>M</span>
                                                <?php elseif (($p['gender'] ?? '') === 'F'): ?>
                                                    <span class="badge bg-danger"><i class="fas fa-female me-1"></i>F</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?= htmlspecialchars($birth_year) ?>
                                            </td>
                                            <td class="country-flag">
                                                <?php 
                                                $flags = ['CZ'=>'🇨🇿','PL'=>'🇵🇱','SK'=>'🇸🇰','DE'=>'🇩🇪','AT'=>'🇦🇹','FR'=>'🇫🇷','IT'=>'🇮🇹','CH'=>'🇨🇭','US'=>'🇺🇸','CA'=>'🇨🇦','GB'=>'🇬🇧','NL'=>'🇳🇱','BE'=>'🇧🇪','ES'=>'🇪🇸','SE'=>'🇸🇪','NO'=>'🇳🇴','FI'=>'🇫🇮','RU'=>'🇷🇺'];
                                                echo isset($flags[$country]) && $country ? $flags[$country] . ' ' : '';
                                                echo htmlspecialchars($country);
                                                ?>
                                            </td>
                                            <td><?= htmlspecialchars($p['club'] ?? '') ?></td>
                                            <td data-col-index="6"><?= htmlspecialchars($category) ?></td>
                                            <?php 
                                            // Render custom field cells (data already parsed above)
                                            $custom_col_idx = 7;
                                            foreach ($custom_field_defs as $field): 
                                                $value = $custom_fields_data[$field['field_key']] ?? '';
                                            ?>
                                            <td data-col-index="<?= $custom_col_idx++ ?>"><?= htmlspecialchars($value) ?></td>
                                            <?php endforeach; ?>
                                            <td class="text-center" data-col-index="<?= $custom_col_idx++ ?>">
                                                <input type="number" 
                                                       name="bibs[<?= $p['id'] ?>]" 
                                                       class="form-control form-control-sm bib-input text-center" 
                                                       value="<?= $current_bib ?? '' ?>"
                                                       min="1"
                                                       data-participant-id="<?= $p['id'] ?>"
                                                       onchange="updateBIBPreview(this)">
                                            </td>
                                            <td data-col-index="<?= $custom_col_idx ?>">
                                                <span id="series-indicator-<?= $p['id'] ?>"></span>
                                            </td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            
                            <div class="card-footer bg-white">
                                <div class="d-flex justify-content-between align-items-center">
                                    <div class="text-muted small">
                                        <i class="fas fa-users me-1"></i><?= count($participants) ?> athletes total
                                    </div>
                                    <button type="submit" class="btn btn-success">
                                        <i class="fas fa-save me-2"></i>Save All BIB Assignments
                                    </button>
                                </div>
                            </div>
                        </form>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <?php endif; ?>
</div>

<!-- Edit Series Modal -->
<div class="modal fade" id="editSeriesModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="fas fa-edit me-2"></i>Edit Series</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form method="post" id="editSeriesForm">
                    <input type="hidden" name="event_id" value="<?= $event_id ?>">
                    <input type="hidden" name="update_series" value="1">
                    <input type="hidden" name="series_id" id="edit_series_id">
                    
                    <div class="mb-3">
                        <label class="form-label small fw-medium">Series Name</label>
                        <input type="text" name="name" id="edit_name" class="form-control" required>
                    </div>
                    
                    <div class="row g-2 mb-3">
                        <div class="col-6">
                            <label class="form-label small fw-medium">Start #</label>
                            <input type="number" name="start_number" id="edit_start_number" class="form-control" required>
                        </div>
                        <div class="col-6">
                            <label class="form-label small fw-medium">End #</label>
                            <input type="number" name="end_number" id="edit_end_number" class="form-control" required>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label small fw-medium">Missing Numbers</label>
                        <input type="text" name="missing_numbers" id="edit_missing_numbers" class="form-control">
                        <small class="text-muted">Comma separated</small>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label small fw-medium">Additional Numbers</label>
                        <input type="text" name="additional_numbers" id="edit_additional_numbers" class="form-control">
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label small fw-medium">Color</label>
                        <div class="d-flex gap-2 flex-wrap mb-2">
                             <button type="button" class="btn p-0 color-pick-edit border border-2" data-color="#000000" style="width: 35px; height: 35px; background-color: #000000; border-radius: 6px;"></button>
                             <button type="button" class="btn p-0 color-pick-edit border border-2" data-color="#007BFF" style="width: 35px; height: 35px; background-color: #007BFF; border-radius: 6px;"></button>
                             <button type="button" class="btn p-0 color-pick-edit border border-2" data-color="#DC3545" style="width: 35px; height: 35px; background-color: #DC3545; border-radius: 6px;"></button>
                             <button type="button" class="btn p-0 color-pick-edit border border-2" data-color="#28A745" style="width: 35px; height: 35px; background-color: #28A745; border-radius: 6px;"></button>
                             <button type="button" class="btn p-0 color-pick-edit border border-2" data-color="#FD7E14" style="width: 35px; height: 35px; background-color: #FD7E14; border-radius: 6px;"></button>
                             <button type="button" class="btn p-0 color-pick-edit border border-2" data-color="#6F42C1" style="width: 35px; height: 35px; background-color: #6F42C1; border-radius: 6px;"></button>
                        </div>
                        <input type="text" name="color" id="edit_color" class="form-control" pattern="^#[0-9A-Fa-f]{6}$">
                    </div>
                    
                    <div class="d-grid">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>Update Series
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<!-- Conflict Warning Modal -->
<div class="modal fade" id="conflictModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title"><i class="fas fa-exclamation-triangle me-2"></i>Conflict Detected</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p class="mb-2">BIB numbers overlap with series <strong id="conflictSeriesName"></strong>.</p>
                <p class="mb-0">Conflicting numbers: <code id="conflictNumbers"></code></p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" onclick="forceCreateSeries()">
                    <i class="fas fa-check me-2"></i>Ignore and Create
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Generic Alert Modal -->
<div class="modal fade" id="alertModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="alertTitle">Notice</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="alertMessage"></div>
            <div class="modal-footer">
                <button type="button" class="btn btn-primary" data-bs-dismiss="modal">OK</button>
            </div>
        </div>
    </div>
</div>

<!-- Generic Confirm Modal -->
<div class="modal fade" id="confirmModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="confirmTitle">Confirm</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="confirmMessage"></div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="confirmButton">Confirm</button>
            </div>
        </div>
    </div>
</div>

<!-- Delete Series Confirmation Modal -->
<div class="modal fade" id="deleteSeriesModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title"><i class="fas fa-trash-alt me-2"></i>Delete Series</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p class="mb-2">Delete series <strong id="deleteSeriesName"></strong>?</p>
                <p class="mb-0 text-danger"><i class="fas fa-exclamation-triangle me-1"></i>This cannot be undone.</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-danger" id="deleteSeriesButton">
                    <i class="fas fa-trash-alt me-1"></i>Delete
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Series Details Modal -->
<div class="modal fade" id="seriesDetailsModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="seriesDetailsModalLabel">Series Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="seriesDetailsContent"></div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<!-- Available Numbers Modal -->
<div class="modal fade" id="availableNumbersModal" tabindex="-1">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="availableNumbersModalLabel">Available Numbers</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="availableNumbersContent"></div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" onclick="printAvailableNumbers()">
                    <i class="fas fa-print me-2"></i>Print
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Detailed Report Modal -->
<div class="modal fade" id="detailedReportModal" tabindex="-1">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">BIB Distribution Report</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <ul class="nav nav-tabs mb-3" id="reportTabs" role="tablist">
                    <li class="nav-item">
                        <button class="nav-link active" data-bs-toggle="tab" data-bs-target="#assignmentsTab">
                            <i class="fas fa-list-check me-1"></i>Assignments
                        </button>
                    </li>
                    <li class="nav-item">
                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#freeNumbersTab">
                            <i class="fas fa-list me-1"></i>Free Numbers
                        </button>
                    </li>
                    <li class="nav-item">
                        <button class="nav-link" data-bs-toggle="tab" data-bs-target="#combinedTab">
                            <i class="fas fa-table me-1"></i>Combined View
                        </button>
                    </li>
                </ul>
                <div class="tab-content">
                    <div class="tab-pane fade show active" id="assignmentsTab">
                        <div id="assignmentsContent"></div>
                    </div>
                    <div class="tab-pane fade" id="freeNumbersTab">
                        <div id="freeNumbersContent"></div>
                    </div>
                    <div class="tab-pane fade" id="combinedTab">
                        <div id="combinedContent"></div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-primary" onclick="printModalContent()">
                    <i class="fas fa-print me-2"></i>Print
                </button>
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<!-- Help Modal -->
<div class="modal fade" id="helpModal" tabindex="-1">
    <div class="modal-dialog modal-xl modal-dialog-scrollable">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title"><i class="fas fa-circle-question me-2 text-info"></i>BIB Manager Help</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body p-0" style="height:80vh;">
                <iframe src="../help/bib_assign_help.html" style="border:0;width:100%;height:100%;"></iframe>
            </div>
        </div>
    </div>
</div>

<!-- Column Visibility Modal -->
<div class="modal fade" id="columnVisibilityModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-info text-white">
                <h5 class="modal-title"><i class="fas fa-columns me-2"></i>Column Visibility</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <p class="text-muted small mb-3">Select which columns to display in the athlete table</p>
                <div id="columnVisibilityOptions" class="d-flex flex-column gap-2">
                    <!-- Column checkboxes will be dynamically generated here -->
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="resetColumnVisibility()">
                    <i class="fas fa-undo me-1"></i>Reset to Default
                </button>
                <button type="button" class="btn btn-primary" onclick="applyColumnVisibility()">
                    <i class="fas fa-check me-1"></i>Apply Changes
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Advanced Auto Assign Modal -->
<div class="modal fade" id="advancedAssignModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Advanced BIB Assignment</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="row mb-3">
                    <div class="col-md-6">
                        <h6>Filters</h6>
                        <div id="assignFilters"></div>
                        <button class="btn btn-sm btn-outline-secondary mt-2" onclick="addAssignFilter()">
                            <i class="fas fa-plus me-1"></i> Add Filter
                        </button>
                    </div>
                    <div class="col-md-6">
                        <h6>Sorting</h6>
                        <div id="assignSorts"></div>
                        <button class="btn btn-sm btn-outline-secondary mt-2" onclick="addAssignSort()">
                            <i class="fas fa-plus me-1"></i> Add Sort
                        </button>
                    </div>
                </div>
                
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label class="form-label">Randomize Top N</label>
                        <input type="number" id="assignRandomizeN" class="form-control" placeholder="0 to disable">
                        <small class="text-muted">Shuffle the first N participants after sorting</small>
                    </div>
                    <div class="col-md-6 d-flex align-items-end">
                        <button class="btn btn-primary w-100" onclick="updateAssignPreview()">
                            <i class="fas fa-sync me-1"></i> Update Preview
                        </button>
                    </div>
                </div>

                <div class="row mb-3 bg-light p-2 rounded">
                    <div class="col-md-6">
                        <label class="form-label">Target Series</label>
                        <select id="assignTargetSeries" class="form-select" onchange="updateAssignStartNumber()">
                            <!-- Populated by JS -->
                        </select>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">Start Number</label>
                        <input type="number" id="assignStartNumber" class="form-control" onchange="updateAssignPreview()">
                    </div>
                </div>

                <h6>Preview (<span id="assignPreviewCount">0</span> participants)</h6>
                <div class="table-responsive" style="max-height: 300px; overflow-y: auto;">
                    <table class="table table-sm table-striped">
                        <thead>
                            <tr>
                                <th>#</th>
                                <th>Name</th>
                                <th>Division</th>
                                <th>Club</th>
                                <th>Current BIB</th>
                                <th>New BIB</th>
                            </tr>
                        </thead>
                        <tbody id="assignPreviewBody"></tbody>
                    </table>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-success" onclick="applyAdvancedAssign()">Apply Assignment</button>
            </div>
        </div>
    </div>
</div>

<script>
    // Data from PHP
    const bibSeries = <?= json_encode($bib_series) ?>;
    const seriesStats = <?= json_encode($series_stats) ?>;
    const customFieldDefs = <?= json_encode($custom_field_defs) ?>;
    const participantsData = <?= json_encode($participants) ?>;

    // Column Visibility Management
    let columnVisibilityModal = null;
    let hiddenColumns = new Set(JSON.parse(localStorage.getItem('bib_hiddenColumns') || '[]'));
    
    // Build column definitions dynamically
    function buildColumnDefinitions() {
        const baseColumns = [
            { index: 0, name: 'Current BIB', mandatory: true },
            { index: 1, name: 'Name', mandatory: true },
            { index: 2, name: 'Gender', mandatory: false },
            { index: 3, name: 'Year', mandatory: false },
            { index: 4, name: 'Country', mandatory: false },
            { index: 5, name: 'Club', mandatory: false },
            { index: 6, name: 'Division', mandatory: false }
        ];
        
        let currentIndex = 7;
        
        // Add custom field columns
        if (customFieldDefs && Array.isArray(customFieldDefs)) {
            customFieldDefs.forEach(field => {
                if (field.is_active == 1 || field.is_active === true) {
                    baseColumns.push({
                        index: currentIndex,
                        name: field.field_label + ' (Custom)',
                        mandatory: false,
                        isCustomField: true
                    });
                    currentIndex++;
                }
            });
        }
        
        // New BIB and Series columns (always last, mandatory)
        baseColumns.push({ index: currentIndex, name: 'New BIB', mandatory: true });
        baseColumns.push({ index: currentIndex + 1, name: 'Series', mandatory: true });
        
        return baseColumns;
    }
    
    let columnDefinitions = buildColumnDefinitions();
    
    function showColumnVisibilityModal() {
        if (!columnVisibilityModal) {
            columnVisibilityModal = new bootstrap.Modal(document.getElementById('columnVisibilityModal'));
        }
        
        const container = document.getElementById('columnVisibilityOptions');
        container.innerHTML = '';
        
        columnDefinitions.forEach(col => {
            const isHidden = hiddenColumns.has(col.index);
            const isVisible = !isHidden;
            
            const div = document.createElement('div');
            div.className = 'form-check';
            div.innerHTML = `
                <input class="form-check-input" type="checkbox" 
                       id="col_${col.index}" 
                       value="${col.index}" 
                       ${isVisible ? 'checked' : ''}
                       ${col.mandatory ? 'disabled' : ''}>
                <label class="form-check-label" for="col_${col.index}">
                    ${col.name}
                    ${col.mandatory ? '<span class="badge bg-secondary ms-2">Required</span>' : ''}
                </label>
            `;
            container.appendChild(div);
        });
        
        columnVisibilityModal.show();
    }
    
    function applyColumnVisibility() {
        hiddenColumns.clear();
        
        const table = document.getElementById('participantTable');
        if (!table) return;
        
        columnDefinitions.forEach(col => {
            if (!col.mandatory) {
                const checkbox = document.getElementById('col_' + col.index);
                const isVisible = checkbox && checkbox.checked;
                
                // Hide/show header th
                const headerCells = table.querySelectorAll('thead th[data-col-index="' + col.index + '"]');
                headerCells.forEach(cell => {
                    cell.style.display = isVisible ? '' : 'none';
                });
                
                // Hide/show body td
                const bodyCells = table.querySelectorAll('tbody td[data-col-index="' + col.index + '"]');
                bodyCells.forEach(cell => {
                    cell.style.display = isVisible ? '' : 'none';
                });
                
                if (!isVisible) {
                    hiddenColumns.add(col.index);
                }
            }
        });
        
        localStorage.setItem('bib_hiddenColumns', JSON.stringify(Array.from(hiddenColumns)));
        
        if (columnVisibilityModal) {
            columnVisibilityModal.hide();
        }
        
        showAlert('Column visibility updated successfully', 'Success');
    }
    
    function resetColumnVisibility() {
        hiddenColumns.clear();
        localStorage.removeItem('bib_hiddenColumns');
        
        const table = document.getElementById('participantTable');
        if (table) {
            const allCells = table.querySelectorAll('th[data-col-index], td[data-col-index]');
            allCells.forEach(cell => {
                cell.style.display = '';
            });
        }
        
        columnDefinitions.forEach(col => {
            const checkbox = document.getElementById('col_' + col.index);
            if (checkbox && !col.mandatory) {
                checkbox.checked = true;
            }
        });
        
        showAlert('Column visibility reset to default', 'Success');
    }
    
    function applyInitialColumnVisibility() {
        const table = document.getElementById('participantTable');
        if (!table) return;
        
        hiddenColumns.forEach(colIndex => {
            const headerCells = table.querySelectorAll('thead th[data-col-index="' + colIndex + '"]');
            headerCells.forEach(cell => {
                cell.style.display = 'none';
            });
            
            const bodyCells = table.querySelectorAll('tbody td[data-col-index="' + colIndex + '"]');
            bodyCells.forEach(cell => {
                cell.style.display = 'none';
            });
        });
    }
    
    // Apply saved column visibility on page load
    document.addEventListener('DOMContentLoaded', function() {
        setTimeout(applyInitialColumnVisibility, 100);
        initializeSorting();
        sortTable('current-bib');
        updateCounts();
    });
    
    // Table Sorting Functionality
    let currentSort = { column: null, direction: 'asc' };
    
    function initializeSorting() {
        const sortableHeaders = document.querySelectorAll('.sortable');
        sortableHeaders.forEach(header => {
            header.style.cursor = 'pointer';
            header.addEventListener('click', function() {
                const sortKey = this.getAttribute('data-sort');
                sortTable(sortKey);
            });
        });
    }
    
    function sortTable(sortKey) {
        const tbody = document.querySelector('#participantTable tbody');
        if (!tbody) return;
        
        const rows = Array.from(tbody.querySelectorAll('tr'));
        
        // Determine sort direction
        if (currentSort.column === sortKey) {
            currentSort.direction = currentSort.direction === 'asc' ? 'desc' : 'asc';
        } else {
            currentSort.column = sortKey;
            currentSort.direction = 'asc';
        }
        
        // Update sort icons
        document.querySelectorAll('.sortable i').forEach(icon => {
            icon.className = 'fas fa-sort ms-1 text-muted sort-icon';
        });
        const activeHeader = document.querySelector('[data-sort="' + sortKey + '"]');
        if (activeHeader) {
            const icon = activeHeader.querySelector('i');
            if (icon) {
                icon.className = currentSort.direction === 'asc' 
                    ? 'fas fa-sort-up ms-1 text-primary sort-icon'
                    : 'fas fa-sort-down ms-1 text-primary sort-icon';
            }
        }
        
        // Sort rows
        rows.sort((a, b) => {
            let aVal, bVal;
            
            // Handle custom fields - get data directly from row attributes
            if (sortKey.startsWith('custom_')) {
                // Convert custom_field_key to data-custom-field_key
                const fieldKey = sortKey.substring(7);
                const attrName = 'data-custom-' + fieldKey;
                aVal = a.getAttribute(attrName);
                bVal = b.getAttribute(attrName);
                
                // Fallback to empty string if null
                aVal = aVal || '';
                bVal = bVal || '';
            } else {
                // Handle standard columns using data attributes
                switch(sortKey) {
                    case 'current-bib':
                        aVal = parseInt(a.getAttribute('data-bib')) || 0;
                        bVal = parseInt(b.getAttribute('data-bib')) || 0;
                        break;
                    case 'name':
                        aVal = a.getAttribute('data-name') || '';
                        bVal = b.getAttribute('data-name') || '';
                        break;
                    case 'gender':
                        aVal = a.getAttribute('data-gender') || '';
                        bVal = b.getAttribute('data-gender') || '';
                        break;
                    case 'year':
                        aVal = a.getAttribute('data-birth-year') || '';
                        bVal = b.getAttribute('data-birth-year') || '';
                        break;
                    case 'country':
                        aVal = a.getAttribute('data-country') || '';
                        bVal = b.getAttribute('data-country') || '';
                        break;
                    case 'club':
                        aVal = a.getAttribute('data-club') || '';
                        bVal = b.getAttribute('data-club') || '';
                        break;
                    case 'category':
                        aVal = a.getAttribute('data-category') || '';
                        bVal = b.getAttribute('data-category') || '';
                        break;
                    default:
                        return 0;
                }
            }
            
            // Handle numeric vs string comparison
            if (typeof aVal === 'number' && typeof bVal === 'number') {
                return currentSort.direction === 'asc' ? aVal - bVal : bVal - aVal;
            }
            
            // String comparison (case-insensitive)
            aVal = String(aVal).toLowerCase();
            bVal = String(bVal).toLowerCase();
            
            if (currentSort.direction === 'asc') {
                return aVal.localeCompare(bVal, undefined, { numeric: true });
            } else {
                return bVal.localeCompare(aVal, undefined, { numeric: true });
            }
        });
        
        // Re-append rows in sorted order
        rows.forEach(row => tbody.appendChild(row));
    }
    
    function applyFilters() {
        const status = document.getElementById('filterStatus').value;
        const series = document.getElementById('filterSeries').value;
        const category = document.getElementById('filterCategory').value;
        const search = document.getElementById('searchParticipant').value.toLowerCase();
        
        const rows = document.querySelectorAll('#participantTable tbody tr');
        
        rows.forEach(row => {
            const hasBib = row.getAttribute('data-bib') !== '';
            const rowSeriesId = row.getAttribute('data-series-id');
            const rowCategory = row.getAttribute('data-category');
            const name = row.getAttribute('data-name');
            const club = row.getAttribute('data-club');
            
            let show = true;
            
            // Status Filter
            if (status === 'assigned' && !hasBib) show = false;
            if (status === 'unassigned' && hasBib) show = false;
            
            // Series Filter
            if (series !== 'all' && rowSeriesId != series) show = false;
            
            // Category Filter
            if (category !== 'all' && rowCategory !== category.toLowerCase()) show = false;
            
            // Search Filter
            if (search && !name.includes(search) && !club.includes(search)) show = false;
            
            row.style.display = show ? '' : 'none';
        });
        
        updateCounts();
    }
    
    function updateCounts() {
        const rows = document.querySelectorAll('#participantTable tbody tr');
        let assigned = 0;
        let unassigned = 0;
        
        rows.forEach(row => {
            // Only count visible rows? Or all rows? Usually all rows for the top badges as they represent the event status.
            // But wait, if I filter, I might want to know how many in the filtered set.
            // The badges are in the header "Assigned" "Unassigned", usually implies total.
            // Let's stick to total for now, or maybe visible. 
            // Given the UI "Assigned" "Unassigned", it's likely total.
            
            const hasBib = row.querySelector('.bib-input').value !== '';
            if (hasBib) assigned++; else unassigned++;
        });
        
        const assignedEl = document.getElementById('assignedCount');
        const unassignedEl = document.getElementById('unassignedCount');
        
        if (assignedEl) assignedEl.textContent = assigned;
        if (unassignedEl) unassignedEl.textContent = unassigned;
    }

    // --- Rich Modal Functions & Exports (Ported from bib_distribute.php) ---

    function showSeriesDetails(id) {
        const stat = seriesStats.find(s => s.series.id == id);
        if (!stat) {
            showAlert('Series not found', 'error');
            return;
        }
        
        const series = stat.series;
        const totalRange = series.end_number - series.start_number + 1;
        const assignedCount = stat.used_numbers ? stat.used_numbers.length : 0;
        const availableCount = stat.available_numbers ? stat.available_numbers.length : 0;
        const usagePercentage = totalRange > 0 ? Math.round((assignedCount / totalRange) * 100) : 0;
        
        // Get missing and additional numbers
        const missingNumbers = series.missing_numbers ? 
            series.missing_numbers.split(',').map(n => parseInt(n.trim())).filter(n => !isNaN(n)) : [];
        const additionalNumbers = series.additional_numbers ? 
            series.additional_numbers.split(',').map(n => parseInt(n.trim())).filter(n => !isNaN(n)) : [];
        
        const content = `
            <div class="row">
                <div class="col-md-6">
                    <div class="border-0 shadow-none card border-primary">
                        <div class="border-0 shadow-none card-header" style="background-color: ${series.color}; color: white;">
                            <h6 class="mb-0">
                                <i class="fas fa-tag me-2"></i>${series.name}
                            </h6>
                        </div>
                        <div class="border-0 shadow-none card-body">
                            <table class="table table-sm">
                                <tr>
                                    <td><strong>Range:</strong></td>
                                    <td>${series.start_number} - ${series.end_number}</td>
                                </tr>
                                <tr>
                                    <td><strong>Total in Range:</strong></td>
                                    <td>${totalRange}</td>
                                </tr>
                                <tr>
                                    <td><strong>Available Numbers:</strong></td>
                                    <td><span class="badge bg-success">${availableCount}</span></td>
                                </tr>
                                <tr>
                                    <td><strong>Assigned Numbers:</strong></td>
                                    <td><span class="badge bg-primary">${assignedCount}</span></td>
                                </tr>
                                <tr>
                                    <td><strong>Usage:</strong></td>
                                    <td>
                                        <div class="progress" style="height: 20px;">
                                            <div class="progress-bar" role="progressbar" 
                                                 style="width: ${usagePercentage}%; background-color: ${series.color};" 
                                                 aria-valuenow="${usagePercentage}" aria-valuemin="0" aria-valuemax="100">
                                                ${usagePercentage}%
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                            </table>
                        </div>
                    </div>
                </div>
                <div class="col-md-6">
                    <div class="border-0 shadow-none card">
                        <div class="border-0 shadow-none card-header">
                            <h6 class="mb-0">Series Configuration</h6>
                        </div>
                        <div class="border-0 shadow-none card-body">
                            ${missingNumbers.length > 0 ? `
                                <div class="mb-3">
                                    <h6 class="text-warning">
                                        <i class="fas fa-exclamation-triangle me-1"></i>Missing from Range:
                                    </h6>
                                    <div class="d-flex flex-wrap gap-1">
                                        ${missingNumbers.map(num => 
                                            `<span class="badge bg-warning text-dark">${num}</span>`
                                        ).join('')}
                                    </div>
                                </div>
                            ` : ''}
                            
                            ${additionalNumbers.length > 0 ? `
                                <div class="mb-3">
                                    <h6 class="text-info">
                                        <i class="fas fa-plus-circle me-1"></i>Additional Numbers:
                                    </h6>
                                    <div class="d-flex flex-wrap gap-1">
                                        ${additionalNumbers.map(num => 
                                            `<span class="badge bg-info">${num}</span>`
                                        ).join('')}
                                    </div>
                                </div>
                            ` : ''}
                            
                            <div class="d-grid gap-2">
                                <button class="btn btn-outline-primary btn-sm" onclick="showAvailableNumbers('${series.id}')">
                                    <i class="fas fa-list me-1"></i>View Available Numbers
                                </button>
                                <button class="btn btn-outline-success btn-sm" onclick="exportSeriesInfo('${series.id}')">
                                    <i class="fas fa-download me-1"></i>Export Series Info
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        document.getElementById('seriesDetailsContent').innerHTML = content;
        document.getElementById('seriesDetailsModalLabel').textContent = `Series Details - ${series.name}`;
        new bootstrap.Modal(document.getElementById('seriesDetailsModal')).show();
    }

    function showAvailableNumbers(id) {
        const stat = seriesStats.find(s => s.series.id == id);
        if (!stat) {
            showAlert('Series not found', 'error');
            return;
        }
        
        const series = stat.series;
        const availableNumbers = stat.available_numbers.sort((a, b) => a - b);
        const usedNumbers = stat.used_numbers ? stat.used_numbers.sort((a, b) => a - b) : [];
        
        // Create columns for better display
        const itemsPerColumn = 15;
        const columns = [];
        for (let i = 0; i < availableNumbers.length; i += itemsPerColumn) {
            columns.push(availableNumbers.slice(i, i + itemsPerColumn));
        }
        
        const content = `
            <div class="row">
                <div class="col-md-8">
                    <div class="border-0 shadow-none card">
                        <div class="border-0 shadow-none card-header" style="background-color: ${series.color}; color: white;">
                            <h6 class="mb-0">
                                <i class="fas fa-list me-2"></i>Available Numbers - ${series.name}
                                <span class="badge bg-light text-dark ms-2">${availableNumbers.length} available</span>
                            </h6>
                            <small>Range: ${series.start_number} - ${series.end_number}</small>
                        </div>
                        <div class="border-0 shadow-none card-body" style="max-height: 400px; overflow-y: auto;">
                            <div class="row">
                                ${columns.map((column, index) => `
                                    <div class="col-md-${Math.min(12, Math.ceil(12 / columns.length))} mb-3">
                                        <div class="d-flex flex-wrap gap-1">
                                            ${column.map(num => `
                                                <span class="badge text-white" style="background-color: ${series.color}; font-size: 12px; margin: 1px;">
                                                    ${num}
                                                </span>
                                            `).join('')}
                                        </div>
                                    </div>
                                `).join('')}
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-md-4">
                    <div class="border-0 shadow-none card border-info">
                        <div class="border-0 shadow-none card-header bg-light">
                            <h6 class="mb-0">Statistics</h6>
                        </div>
                        <div class="border-0 shadow-none card-body">
                            <table class="table table-sm">
                                <tr>
                                    <td><strong>Total Range:</strong></td>
                                    <td>${series.end_number - series.start_number + 1}</td>
                                </tr>
                                <tr>
                                    <td><strong>Available:</strong></td>
                                    <td><span class="badge bg-success">${availableNumbers.length}</span></td>
                                </tr>
                                <tr>
                                    <td><strong>Assigned:</strong></td>
                                    <td><span class="badge bg-primary">${usedNumbers.length}</span></td>
                                </tr>
                            </table>
                            
                            ${usedNumbers.length > 0 ? `
                                <div class="mt-3">
                                    <h6 class="text-primary">Assigned Numbers:</h6>
                                    <div style="max-height: 150px; overflow-y: auto;">
                                        <div class="d-flex flex-wrap gap-1">
                                            ${usedNumbers.map(num => `
                                                <span class="badge bg-primary" style="font-size: 11px; margin: 1px;">
                                                    ${num}
                                                </span>
                                            `).join('')}
                                        </div>
                                    </div>
                                </div>
 ` : ''}
                            
                            <div class="mt-3 d-grid gap-2">
                                <button class="btn btn-outline-primary btn-sm" onclick="copyAvailableNumbers('${series.id}')">
                                    <i class="fas fa-copy me-1"></i>Copy Numbers
                                </button>
                                <button class="btn btn-outline-success btn-sm" onclick="exportAvailableCSV('${series.id}')">
                                    <i class="fas fa-file-csv me-1"></i>Export CSV
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        document.getElementById('availableNumbersContent').innerHTML = content;
        document.getElementById('availableNumbersModalLabel').textContent = `Available Numbers - ${series.name}`;
        new bootstrap.Modal(document.getElementById('availableNumbersModal')).show();
    }

    function showDetailedReport() {
        // Collect current assignments from the table
        const assignments = [];
        const rows = document.querySelectorAll('#participantTable tbody tr');
        
        rows.forEach(row => {
            const input = row.querySelector('.bib-input');
            const bibNumber = parseInt(input.value);
            
            if (bibNumber && !isNaN(bibNumber)) {
                const participantName = row.getAttribute('data-name');
                
                // Find series for this BIB
                let seriesInfo = null;
                for (const stat of seriesStats) {
                    const allNums = [...stat.available_numbers, ...stat.used_numbers];
                    if (allNums.includes(bibNumber)) {
                        seriesInfo = stat.series;
                        break;
                    }
                }
                
                assignments.push({
                    bibNumber: bibNumber,
                    participantName: participantName,
                    series: seriesInfo
                });
            }
        });
        
        // Sort assignments by BIB number
        assignments.sort((a, b) => a.bibNumber - b.bibNumber);
        
        // Generate content for each tab
        generateAssignmentsContent(assignments);
        generateFreeNumbersContent(assignments);
        generateCombinedContent(assignments);
        
        new bootstrap.Modal(document.getElementById('detailedReportModal')).show();
    }

    function generateAssignmentsContent(assignments) {
        const content = document.getElementById('assignmentsContent');
        
        // Group assignments by series
        const assignmentsBySeries = {};
        assignments.forEach(assignment => {
            const seriesId = assignment.series ? assignment.series.id : 'unknown';
            const seriesName = assignment.series ? assignment.series.name : 'Unknown Series';
            const seriesColor = assignment.series ? assignment.series.color : '#6c757d';
            
            if (!assignmentsBySeries[seriesId]) {
                assignmentsBySeries[seriesId] = {
                    name: seriesName,
                    color: seriesColor,
                    assignments: []
                };
            }
            assignmentsBySeries[seriesId].assignments.push(assignment);
        });
        
        let html = '<h5 class="mb-3">BIB Assignments by Series</h5>';
        
        Object.values(assignmentsBySeries).forEach(seriesGroup => {
            html += `
                <div class="border-0 shadow-none card mb-3">
                    <div class="border-0 shadow-none card-header" style="background-color: ${seriesGroup.color}; color: white;">
                        <h6 class="mb-0">
                            <i class="fas fa-tag me-2"></i>${seriesGroup.name}
                            <span class="badge bg-light text-dark ms-2">${seriesGroup.assignments.length} assigned</span>
                        </h6>
                    </div>
                    <div class="border-0 shadow-none card-body p-0">
                        <div class="table-responsive">
                            <table class="table table-sm mb-0">
                                <thead class="table-light">
                                    <tr>
                                        <th>BIB #</th>
                                        <th>Participant</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    ${seriesGroup.assignments.map(assignment => `
                                        <tr>
                                            <td>
                                                <span class="badge text-white" style="background-color: ${seriesGroup.color};">
                                                    ${assignment.bibNumber}
                                                </span>
                                            </td>
                                            <td>${assignment.participantName}</td>
                                        </tr>
                                    `).join('')}
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            `;
        });
        
        content.innerHTML = html;
    }

    function generateFreeNumbersContent(assignments) {
        const content = document.getElementById('freeNumbersContent');
        const assignedNumbers = new Set(assignments.map(a => a.bibNumber));
        
        let html = '<h5 class="mb-3">Free BIB Numbers by Series</h5>';
        
        seriesStats.forEach(stat => {
            const series = stat.series;
            // Recalculate free numbers based on current assignments in the UI, not just DB state
            // Actually, stat.available_numbers might be stale if user changed inputs.
            // Let's reconstruct available numbers from series definition minus assignedNumbers
            
            // Reconstruct all valid numbers for this series
            const range = [];
            for(let i=parseInt(series.start_number); i<=parseInt(series.end_number); i++) range.push(i);
            
            const missing = (series.missing_numbers || '').split(',').map(n => parseInt(n.trim())).filter(n => !isNaN(n));
            const additional = (series.additional_numbers || '').split(',').map(n => parseInt(n.trim())).filter(n => !isNaN(n));
            let allValid = [...range, ...additional].filter(n => !missing.includes(n));
            allValid = [...new Set(allValid)];
            
            const freeNumbers = allValid.filter(num => !assignedNumbers.has(num));
            freeNumbers.sort((a, b) => a - b);
            
            html += `
                <div class="border-0 shadow-none card mb-3">
                    <div class="border-0 shadow-none card-header" style="background-color: ${series.color}; color: white;">
                        <div class="d-flex justify-content-between align-items-center">
                            <h6 class="mb-0">
                                <i class="fas fa-tag me-2"></i>${series.name}
                                <span class="badge bg-light text-dark ms-2">${freeNumbers.length} free</span>
                            </h6>
                            <button class="btn btn-sm btn-light" onclick="copyTableToClipboard('free-table-${series.id}')">
                                <i class="fas fa-copy me-1"></i>Copy Table
                            </button>
                        </div>
                        <small>Range: ${series.start_number} - ${series.end_number}</small>
                    </div>
                    <div class="border-0 shadow-none card-body">
                        <div class="row">
                            <div class="col-md-8">
                                <h6 class="text-success">Available BIB Numbers - Manual Assignment Table:</h6>
                                <div class="table-responsive" style="max-height: 400px;">
                                    <table class="table table-striped table-sm" id="free-table-${series.id}">
                                        <thead class="table-success sticky-top">
                                            <tr>
                                                <th style="width: 100px;">BIB Number</th>
                                                <th>Participant Name</th>
                                                <th>Notes</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            ${freeNumbers.map(num => `
                                                <tr>
                                                    <td>
                                                        <span class="badge text-white fw-bold" style="background-color: ${series.color}; font-size: 14px; margin: 1px;">
                                                            ${num}
                                                        </span>
                                                    </td>
                                                    <td style="border-bottom: 1px dotted #ccc; height: 40px;">&nbsp;</td>
                                                    <td style="border-bottom: 1px dotted #ccc; height: 40px;">&nbsp;</td>
                                                </tr>
                                            `).join('')}
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <h6 class="text-info">Series Information:</h6>
                                <div class="border-0 shadow-none card border-info">
                                    <div class="border-0 shadow-none card-body">
                                        <p><strong>Total in Range:</strong> ${series.end_number - series.start_number + 1}</p>
                                        <p><strong>Free Numbers:</strong> ${freeNumbers.length}</p>
                                        
                                        <div class="mt-3">
                                            <button class="btn btn-sm btn-outline-primary w-100" onclick="exportSeriesCSV('${series.id}', '${series.name}', ${JSON.stringify(freeNumbers).replace(/"/g, '&quot;')})">
                                                <i class="fas fa-file-csv me-1"></i>Export to CSV
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            `;
        });
        
        content.innerHTML = html;
    }

    function generateCombinedContent(assignments) {
        const content = document.getElementById('combinedContent');
        const assignedNumbers = new Set(assignments.map(a => a.bibNumber));
        
        let html = '<h5 class="mb-3">Complete BIB Overview by Series</h5>';
        
        seriesStats.forEach(stat => {
            const series = stat.series;
            const assignedInSeries = assignments.filter(a => a.series && a.series.id === series.id);
            
            // Reconstruct all valid numbers
            const range = [];
            for(let i=parseInt(series.start_number); i<=parseInt(series.end_number); i++) range.push(i);
            const missing = (series.missing_numbers || '').split(',').map(n => parseInt(n.trim())).filter(n => !isNaN(n));
            const additional = (series.additional_numbers || '').split(',').map(n => parseInt(n.trim())).filter(n => !isNaN(n));
            let allValid = [...range, ...additional].filter(n => !missing.includes(n));
            allValid = [...new Set(allValid)];
            
            const freeNumbers = allValid.filter(num => !assignedNumbers.has(num));
            
            html += `
                <div class="border-0 shadow-none card mb-4">
                    <div class="border-0 shadow-none card-header" style="background-color: ${series.color}; color: white;">
                        <div class="d-flex justify-content-between align-items-center">
                            <h6 class="mb-0">
                                <i class="fas fa-tag me-2"></i>${series.name}
                            </h6>
                            <div>
                                <span class="badge bg-success me-1">${assignedInSeries.length} Assigned</span>
                                <span class="badge bg-secondary">${freeNumbers.length} Free</span>
                            </div>
                        </div>
                        <small>Range: ${series.start_number} - ${series.end_number}</small>
                    </div>
                    <div class="border-0 shadow-none card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <h6 class="text-success">Assigned Numbers:</h6>
                                <div class="table-responsive" style="max-height: 300px;">
                                    <table class="table table-sm">
                                        <thead class="table-success">
                                            <tr>
                                                <th>BIB #</th>
                                                <th>Participant</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            ${assignedInSeries.map(assignment => `
                                                <tr>
                                                    <td>
                                                        <span class="badge text-white" style="background-color: ${series.color};">
                                                            ${assignment.bibNumber}
                                                        </span>
                                                    </td>
                                                    <td>${assignment.participantName}</td>
                                                </tr>
                                            `).join('')}
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <h6 class="text-secondary">Free Numbers:</h6>
                                <div class="d-flex flex-wrap gap-1 mb-3" style="max-height: 200px; overflow-y: auto;">
                                    ${freeNumbers.map(num => 
                                        `<span class="badge text-white" style="background-color: ${series.color};">${num}</span>`
                                    ).join('')}
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            `;
        });
        
        content.innerHTML = html;
    }

    // Export series to CSV
    function exportSeriesCSV(seriesId, seriesName, freeNumbers) {
        let csvContent = "BIB Number,Participant Name,Notes\n";
        
        freeNumbers.forEach(num => {
            csvContent += `${num},"",""\n`;
        });
        
        // Create download link
        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement("a");
        const url = URL.createObjectURL(blob);
        link.setAttribute("href", url);
        link.setAttribute("download", `free_bibs_${seriesName.replace(/[^a-zA-Z0-9]/g, '_')}.csv`);
        link.style.visibility = 'hidden';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }

    // Export series information
    function exportSeriesInfo(seriesId) {
        const stat = seriesStats.find(s => s.series.id == seriesId);
        if (!stat) return;
        
        const series = stat.series;
        const totalRange = series.end_number - series.start_number + 1;
        const assignedCount = stat.used_numbers ? stat.used_numbers.length : 0;
        const availableCount = stat.available_numbers ? stat.available_numbers.length : 0;
        
        let csvContent = "Series Information\n";
        csvContent += `Series Name,${series.name}\n`;
        csvContent += `Range,${series.start_number} - ${series.end_number}\n`;
        csvContent += `Total Numbers in Range,${totalRange}\n`;
        csvContent += `Available Numbers,${availableCount}\n`;
        csvContent += `Assigned Numbers,${assignedCount}\n`;
        csvContent += `Usage Percentage,${Math.round((assignedCount / totalRange) * 100)}%\n`;
        csvContent += `Color,${series.color}\n\n`;
        
        if (series.missing_numbers) {
            csvContent += "Missing Numbers\n";
            csvContent += `${series.missing_numbers}\n\n`;
        }
        
        if (series.additional_numbers) {
            csvContent += "Additional Numbers\n";
            csvContent += `${series.additional_numbers}\n\n`;
        }
        
        if (stat.available_numbers && stat.available_numbers.length > 0) {
            csvContent += "Available Numbers\n";
            csvContent += `${stat.available_numbers.join(',')}\n\n`;
        }
        
        if (stat.used_numbers && stat.used_numbers.length > 0) {
            csvContent += "Assigned Numbers\n";
            csvContent += `${stat.used_numbers.join(',')}\n`;
        }
        
        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement("a");
        const url = URL.createObjectURL(blob);
        link.setAttribute("href", url);
        link.setAttribute("download", `series_info_${series.name.replace(/[^a-zA-Z0-9]/g, '_')}.csv`);
        link.style.visibility = 'hidden';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }

    // Copy available numbers to clipboard
    function copyAvailableNumbers(seriesId) {
        const stat = seriesStats.find(s => s.series.id == seriesId);
        if (!stat || !stat.available_numbers) return;
        
        const numbersText = stat.available_numbers.sort((a, b) => a - b).join(', ');
        
        navigator.clipboard.writeText(numbersText).then(() => {
            showAlert('Available numbers copied to clipboard!', 'success');
        }).catch(err => {
            showAlert('Failed to copy numbers. Please select and copy manually.', 'error');
        });
    }

    // Export available numbers as CSV
    function exportAvailableCSV(seriesId) {
        const stat = seriesStats.find(s => s.series.id == seriesId);
        if (!stat || !stat.available_numbers) return;
        
        const series = stat.series;
        const availableNumbers = stat.available_numbers.sort((a, b) => a - b);
        
        let csvContent = "Available BIB Numbers\n";
        csvContent += `Series: ${series.name}\n`;
        csvContent += `Range: ${series.start_number} - ${series.end_number}\n`;
        csvContent += `Total Available: ${availableNumbers.length}\n\n`;
        csvContent += "BIB Number,Participant Name,Notes\n";
        
        availableNumbers.forEach(num => {
            csvContent += `${num},"",""\n`;
        });
        
        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement("a");
        const url = URL.createObjectURL(blob);
        link.setAttribute("href", url);
        link.setAttribute("download", `available_bibs_${series.name.replace(/[^a-zA-Z0-9]/g, '_')}.csv`);
        link.style.visibility = 'hidden';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }

    // Copy table to clipboard
    function copyTableToClipboard(tableId) {
        const table = document.getElementById(tableId);
        if (!table) return;
        
        let csvContent = '';
        const rows = table.querySelectorAll('tr');
        
        rows.forEach(row => {
            const cells = row.querySelectorAll('th, td');
            const rowData = Array.from(cells).map(cell => {
                // Clean up the cell content for CSV
                let text = cell.textContent.trim();
                if (text === '\u00A0') text = ''; // Replace non-breaking space
                return `"${text}"`;
            });
            csvContent += rowData.join('\t') + '\n';
        });
        
        navigator.clipboard.writeText(csvContent).then(() => {
            showAlert('Table copied to clipboard!', 'success');
        }).catch(err => {
            showAlert('Failed to copy table.', 'error');
        });
    }

    function editSeries(id) {
        const series = bibSeries.find(s => s.id == id);
        if (!series) return;
        
        document.getElementById('edit_series_id').value = series.id;
        document.getElementById('edit_name').value = series.name;
        document.getElementById('edit_start_number').value = series.start_number;
        document.getElementById('edit_end_number').value = series.end_number;
        document.getElementById('edit_missing_numbers').value = series.missing_numbers;
        document.getElementById('edit_additional_numbers').value = series.additional_numbers;
        document.getElementById('edit_color').value = series.color;
        
        new bootstrap.Modal(document.getElementById('editSeriesModal')).show();
    }

    // Print available numbers
    function printAvailableNumbers() {
        const modalContent = document.getElementById('availableNumbersContent');
        if (!modalContent) return;
        
        const seriesName = document.getElementById('availableNumbersModalLabel').textContent;
        
        const printWindow = window.open('', '_blank');
        printWindow.document.write(`
            <!DOCTYPE html>
            <html>
            <head>
                <title>${seriesName}</title>
                <style>
                    body { font-family: Arial, sans-serif; padding: 20px; }
                    .card { border: 1px solid #ddd; margin-bottom: 15px; }
                    .card-header { background-color: #f8f9fa; padding: 10px; font-weight: bold; }
                    .card-body { padding: 10px; }
                    .badge { padding: 2px 8px; border-radius: 3px; margin: 1px; }
                    .text-white { color: white !important; }
                    .table { width: 100%; border-collapse: collapse; }
                    .table th, .table td { border: 1px solid #ddd; padding: 8px; text-align: left; }
                    @media print {
                        .btn, button { display: none !important; }
                    }
                </style>
            </head>
            <body>
                <h2>${seriesName}</h2>
                <p>Generated on: ${new Date().toLocaleString()}</p>
                ${modalContent.innerHTML}
            </body>
            </html>
        `);
        printWindow.document.close();
        printWindow.print();
    }

    // Print modal content
    function printModalContent() {
        const eventName = document.querySelector('#event_id option:checked')?.textContent || 'Event Report';
        const currentDate = new Date().toLocaleDateString();
        
        // Get active tab content
        const activeTabEl = document.querySelector('#reportTabs .nav-link.active');
        const activeTab = activeTabEl ? activeTabEl.textContent.trim() : 'Report';
        
        const activeContentEl = document.querySelector('.tab-content .tab-pane.active');
        const activeContent = activeContentEl ? activeContentEl.innerHTML : '';
        
        const printHTML = `
    <!DOCTYPE html>
    <html>
    <head>
        <meta charset="UTF-8">
        <title>BIB Report - ${eventName} - ${activeTab}</title>
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
        <style>
            @media print {
                body { margin: 0; font-size: 12px; }
                .card-header { -webkit-print-color-adjust: exact; color-adjust: exact; }
                .badge { -webkit-print-color-adjust: exact; color-adjust: exact; }
            }
            body { font-family: Arial, sans-serif; margin: 20px; }
            .badge { display: inline-block; padding: 0.25em 0.5em; }
        </style>
    </head>
    <body>
        <div class="container-fluid">
            <h2>BIB Report - ${activeTab}</h2>
            <p><strong>Event:</strong> ${eventName}</p>
            <p><strong>Generated:</strong> ${currentDate}</p>
            
            ${activeContent}
        </div>
    </body>
    </html>`;
        
        const printWindow = window.open('', '_blank', 'width=1200,height=800');
        printWindow.document.write(printHTML);
        printWindow.document.close();
        printWindow.print();
    }

    // Advanced Auto Assign Logic
    let assignPreviewData = [];
    let assignModalInstance = null;

    function autoAssignBIBs() {
        if (!assignModalInstance) {
            assignModalInstance = new bootstrap.Modal(document.getElementById('advancedAssignModal'));
        }
        
        // Reset modal state
        document.getElementById('assignFilters').innerHTML = '';
        document.getElementById('assignSorts').innerHTML = '';
        document.getElementById('assignRandomizeN').value = '';
        document.getElementById('assignPreviewBody').innerHTML = '';
        document.getElementById('assignPreviewCount').textContent = '0';
        
        // Populate Series
        populateAssignSeries();
        
        // Add default sort (Name ASC)
        addAssignSort();
        
        assignModalInstance.show();
        updateAssignPreview();
    }

    function getAvailableFields() {
        const fields = [
            { value: 'first_name', label: 'First Name' },
            { value: 'last_name', label: 'Last Name' },
            { value: 'gender', label: 'Gender' },
            { value: 'birth_year', label: 'Year' },
            { value: 'country', label: 'Country' },
            { value: 'club', label: 'Club' },
            { value: 'category_name', label: 'Division' },
            { value: 'has_bib', label: 'Has BIB / Current BIB' }
        ];
        
        if (customFieldDefs) {
            customFieldDefs.forEach(field => {
                fields.push({
                    value: 'custom_' + field.field_key,
                    label: field.field_label
                });
            });
        }
        return fields;
    }

    function addAssignFilter() {
        const container = document.getElementById('assignFilters');
        const fields = getAvailableFields();
        
        const div = document.createElement('div');
        div.className = 'input-group mb-2';
        
        // Create Field Select
        const fieldSelect = document.createElement('select');
        fieldSelect.className = 'form-select form-select-sm assign-filter-field';
        fields.forEach(f => {
            const opt = document.createElement('option');
            opt.value = f.value;
            opt.textContent = f.label;
            fieldSelect.appendChild(opt);
        });

        // Create Value Select
        const valueSelect = document.createElement('select');
        valueSelect.className = 'form-select form-select-sm assign-filter-value';
        
        // Create Remove Button
        const removeBtn = document.createElement('button');
        removeBtn.className = 'btn btn-outline-danger btn-sm';
        removeBtn.innerHTML = '<i class="fas fa-times"></i>';
        removeBtn.onclick = function() { div.remove(); };

        // Append elements
        div.appendChild(fieldSelect);
        div.appendChild(valueSelect);
        div.appendChild(removeBtn);
        container.appendChild(div);

        // Add change listener
        fieldSelect.addEventListener('change', () => populateFilterValues(fieldSelect, valueSelect));
        
        // Initial population
        populateFilterValues(fieldSelect, valueSelect);
    }

    function populateFilterValues(fieldSelect, valueSelect) {
        const field = fieldSelect.value;
        
        if (field === 'has_bib') {
            valueSelect.innerHTML = '<option value="">-- Select Value --</option>';
            valueSelect.innerHTML += '<option value="YES">YES</option>';
            valueSelect.innerHTML += '<option value="NO">NO</option>';
            return;
        }

        const values = new Set();
        
        participantsData.forEach(p => {
            let val = '';
            if (field.startsWith('custom_')) {
                const fieldName = field.replace('custom_', '');
                let cf = p.custom_fields;
                if (typeof cf === 'string') {
                    try { cf = JSON.parse(cf); } catch(e) { cf = {}; }
                }
                val = cf ? (cf[fieldName] || '') : '';
            } else {
                val = p[field] || '';
                // Handle category special case if needed
                if (field === 'category_name') {
                    val = p.categories || p.category || '';
                }
                // Handle birth_year
                if (field === 'birth_year') {
                     const dob = p.date_of_birth || '';
                     const match = dob.match(/^(\d{4})/);
                     val = match ? match[1] : '';
                }
            }
            if (val !== null && val !== undefined && String(val).trim() !== '') {
                values.add(String(val).trim());
            }
        });

        // Sort values
        const sortedValues = Array.from(values).sort((a, b) => a.localeCompare(b, undefined, { numeric: true }));
        
        // Clear and repopulate
        valueSelect.innerHTML = '<option value="">-- Select Value --</option>';
        sortedValues.forEach(v => {
            const opt = document.createElement('option');
            opt.value = v;
            opt.textContent = v;
            valueSelect.appendChild(opt);
        });
    }

    function addAssignSort() {
        const container = document.getElementById('assignSorts');
        const fields = getAvailableFields();
        
        const div = document.createElement('div');
        div.className = 'input-group mb-2';
        div.innerHTML = `
            <select class="form-select form-select-sm assign-sort-field">
                ${fields.map(f => `<option value="${f.value}">${f.label}</option>`).join('')}
            </select>
            <select class="form-select form-select-sm assign-sort-dir" style="max-width: 100px;">
                <option value="asc">ASC</option>
                <option value="desc">DESC</option>
            </select>
            <button class="btn btn-outline-danger btn-sm" onclick="this.parentElement.remove()">
                <i class="fas fa-times"></i>
            </button>
        `;
        container.appendChild(div);
    }

    function updateAssignPreview() {
        // 1. Filter
        let filtered = [...participantsData];
        const filterRows = document.querySelectorAll('#assignFilters .input-group');
        
        // Collect active fields for dynamic columns
        const activeFields = new Set();
        const defaultFields = ['first_name', 'last_name', 'category_name', 'club', 'bib_number', 'has_bib'];

        filterRows.forEach(row => {
            const field = row.querySelector('.assign-filter-field').value;
            const value = row.querySelector('.assign-filter-value').value.toLowerCase();
            
            if (value) {
                if (!defaultFields.includes(field)) activeFields.add(field);

                filtered = filtered.filter(p => {
                    if (field === 'has_bib') {
                        const hasBib = (p.bib_number || p.bib) ? 'yes' : 'no';
                        return hasBib === value;
                    }

                    let pVal = '';
                    if (field.startsWith('custom_')) {
                        const fieldName = field.replace('custom_', '');
                        // Handle custom fields structure (could be JSON string or object)
                        let cf = p.custom_fields;
                        if (typeof cf === 'string') {
                            try { cf = JSON.parse(cf); } catch(e) { cf = {}; }
                        }
                        pVal = cf ? (cf[fieldName] || '') : '';
                    } else {
                        pVal = p[field] || '';
                        // Handle category special case if needed
                        if (field === 'category_name') {
                            pVal = p.categories || p.category || '';
                        }
                        // Handle birth_year
                        if (field === 'birth_year') {
                             const dob = p.date_of_birth || '';
                             const match = dob.match(/^(\d{4})/);
                             pVal = match ? match[1] : '';
                        }
                    }
                    return String(pVal).trim().toLowerCase() === value;
                });
            }
        });

        // 2. Sort
        const sortRows = document.querySelectorAll('#assignSorts .input-group');
        if (sortRows.length > 0) {
            filtered.sort((a, b) => {
                for (const row of sortRows) {
                    const field = row.querySelector('.assign-sort-field').value;
                    const dir = row.querySelector('.assign-sort-dir').value;
                    
                    if (!defaultFields.includes(field)) activeFields.add(field);

                    let aVal, bVal;
                    if (field.startsWith('custom_')) {
                        const fieldName = field.replace('custom_', '');
                        let cfA = a.custom_fields;
                        let cfB = b.custom_fields;
                        if (typeof cfA === 'string') { try { cfA = JSON.parse(cfA); } catch(e) { cfA = {}; } }
                        if (typeof cfB === 'string') { try { cfB = JSON.parse(cfB); } catch(e) { cfB = {}; } }
                        
                        aVal = cfA ? (cfA[fieldName] || '') : '';
                        bVal = cfB ? (cfB[fieldName] || '') : '';
                    } else {
                        aVal = a[field] || '';
                        bVal = b[field] || '';
                        if (field === 'category_name') {
                            aVal = a.categories || a.category || '';
                            bVal = b.categories || b.category || '';
                        }
                        if (field === 'birth_year') {
                             const dobA = a.date_of_birth || '';
                             const matchA = dobA.match(/^(\d{4})/);
                             aVal = matchA ? matchA[1] : '';

                             const dobB = b.date_of_birth || '';
                             const matchB = dobB.match(/^(\d{4})/);
                             bVal = matchB ? matchB[1] : '';
                        }
                    }
                    
                    if (aVal !== bVal) {
                        const comparison = String(aVal).localeCompare(String(bVal), undefined, { numeric: true });
                        return dir === 'asc' ? comparison : -comparison;
                    }
                }
                return 0;
            });
        }

        // 3. Randomize Top N
        const randomizeN = parseInt(document.getElementById('assignRandomizeN').value) || 0;
        if (randomizeN > 0 && filtered.length > 0) {
            const topN = filtered.slice(0, randomizeN);
            const rest = filtered.slice(randomizeN);
            
            // Fisher-Yates Shuffle
            for (let i = topN.length - 1; i > 0; i--) {
                const j = Math.floor(Math.random() * (i + 1));
                [topN[i], topN[j]] = [topN[j], topN[i]];
            }
            
            filtered = [...topN, ...rest];
        }

        assignPreviewData = filtered;
        
        // Calculate New BIBs
        const seriesId = document.getElementById('assignTargetSeries').value;
        const startNumVal = document.getElementById('assignStartNumber').value;
        let currentNum = parseInt(startNumVal);
        let available = new Set();
        let stat = null;

        if (seriesId && !isNaN(currentNum)) {
            stat = seriesStats.find(s => s.series.id == seriesId);
            if (stat) {
                available = new Set(stat.available_numbers);
            }
        }

        // Update Table Header
        const theadRow = document.querySelector('#advancedAssignModal table thead tr');
        let headerHTML = `
            <th>#</th>
            <th>Name</th>
            <th>Division</th>
            <th>Club</th>
            <th>Current BIB</th>
        `;
        
        const availableFields = getAvailableFields();
        activeFields.forEach(field => {
            const fieldObj = availableFields.find(f => f.value === field);
            const label = fieldObj ? fieldObj.label : field;
            headerHTML += `<th>${label}</th>`;
        });

        headerHTML += `<th>New BIB</th>`;
        theadRow.innerHTML = headerHTML;

        // Render Body
        const tbody = document.getElementById('assignPreviewBody');
        tbody.innerHTML = '';
        document.getElementById('assignPreviewCount').textContent = filtered.length;
        
        filtered.forEach((p, index) => {
            const tr = document.createElement('tr');
            const cat = p.categories || p.category || '-';
            
            let newBib = '-';
            if (stat && available.size > 0) {
                // Find next available number >= currentNum
                while (!available.has(currentNum) && currentNum <= stat.series.end_number) {
                    currentNum++;
                }
                
                if (available.has(currentNum)) {
                    newBib = `<span class="badge" style="background-color: ${stat.series.color}">${currentNum}</span>`;
                    available.delete(currentNum); // Mark as used for this preview
                    currentNum++;
                } else {
                    newBib = '<span class="text-danger">No numbers</span>';
                }
            }

            let rowHTML = `
                <td>${index + 1}</td>
                <td>${p.first_name} ${p.last_name}</td>
                <td>${cat}</td>
                <td>${p.club || '-'}</td>
                <td>${p.bib_number || '-'}</td>
            `;

            activeFields.forEach(field => {
                let val = '';
                if (field.startsWith('custom_')) {
                    const fieldName = field.replace('custom_', '');
                    let cf = p.custom_fields;
                    if (typeof cf === 'string') { try { cf = JSON.parse(cf); } catch(e) { cf = {}; } }
                    val = cf ? (cf[fieldName] || '-') : '-';
                } else if (field === 'birth_year') {
                     const dob = p.date_of_birth || '';
                     const match = dob.match(/^(\d{4})/);
                     val = match ? match[1] : '-';
                } else {
                    val = p[field] || '-';
                }
                rowHTML += `<td>${val}</td>`;
            });

            rowHTML += `<td>${newBib}</td>`;
            tr.innerHTML = rowHTML;
            tbody.appendChild(tr);
        });
    }

    function populateAssignSeries() {
        const select = document.getElementById('assignTargetSeries');
        select.innerHTML = '<option value="">-- Select Series --</option>';
        
        if (bibSeries) {
            bibSeries.forEach(s => {
                select.innerHTML += `<option value="${s.id}">${s.name} (${s.start_number}-${s.end_number})</option>`;
            });
        }
    }

    function updateAssignStartNumber() {
        const seriesId = document.getElementById('assignTargetSeries').value;
        const input = document.getElementById('assignStartNumber');
        
        if (!seriesId) {
            input.value = '';
            updateAssignPreview(); // Update preview when series is cleared
            return;
        }
        
        const stat = seriesStats.find(s => s.series.id == seriesId);
        if (stat && stat.available_numbers && stat.available_numbers.length > 0) {
            // Find first available number
            const sorted = [...stat.available_numbers].sort((a, b) => a - b);
            input.value = sorted[0];
        } else {
            input.value = '';
        }
        updateAssignPreview(); // Update preview when series/start number changes
    }

    function confirmClearAll() {
        const modal = new bootstrap.Modal(document.getElementById('confirmModal'));
        document.getElementById('confirmTitle').textContent = 'Clear All Assignments';
        document.getElementById('confirmMessage').innerHTML = `
            <p class="text-danger"><i class="fas fa-exclamation-triangle me-2"></i>Warning</p>
            <p>Are you sure you want to clear <strong>ALL</strong> assigned BIB numbers currently in the form?</p>
            <p class="small text-muted">This action only clears the input fields. Changes will not be permanent until you click "Save All BIB Assignments".</p>
        `;
        
        const confirmBtn = document.getElementById('confirmButton');
        confirmBtn.className = 'btn btn-danger';
        confirmBtn.textContent = 'Clear All';
        confirmBtn.onclick = function() {
            clearAllBIBs();
            modal.hide();
        };
        
        modal.show();
    }

    function clearAllBIBs() {
        const inputs = document.querySelectorAll('.bib-input');
        let count = 0;
        
        inputs.forEach(input => {
            if (input.value) {
                input.value = '';
                updateBIBPreview(input);
                count++;
            }
        });
        
        if (typeof showAlert === 'function') {
            showAlert(`Cleared ${count} BIB assignments.`, 'success');
        }
    }

    function applyAdvancedAssign() {
        if (assignPreviewData.length === 0) return;
        
        const seriesId = document.getElementById('assignTargetSeries').value;
        const startNumVal = document.getElementById('assignStartNumber').value;
        
        if (!seriesId || !startNumVal) {
            alert('Please select a target series and start number.');
            return;
        }
        
        let currentNum = parseInt(startNumVal);
        const stat = seriesStats.find(s => s.series.id == seriesId);
        
        if (!stat) return;
        
        const available = new Set(stat.available_numbers);
        
        assignPreviewData.forEach(p => {
            // Find row
            const row = document.querySelector(`tr[data-participant-id="${p.id}"]`);
            if (row) {
                const input = row.querySelector('.bib-input');
                if (input) {
                    // Find next available number >= currentNum
                    while (!available.has(currentNum) && currentNum <= stat.series.end_number) {
                        currentNum++;
                    }
                    
                    if (available.has(currentNum)) {
                        input.value = currentNum;
                        updateBIBPreview(input);
                        available.delete(currentNum); // Mark as used for this batch
                        currentNum++;
                    }
                }
            }
        });
        
        assignModalInstance.hide();
        if (typeof showAlert === 'function') {
            showAlert('BIB numbers assigned based on preview order.', 'success');
        } else {
            alert('BIB numbers assigned based on preview order.');
        }
    }

    function updateBIBPreview(input) {
        const row = input.closest('tr');
        const participantId = row.dataset.participantId;
        const bibNumber = parseInt(input.value);
        const indicator = document.getElementById(`series-indicator-${participantId}`);
        
        if (!bibNumber || isNaN(bibNumber)) {
            indicator.innerHTML = '<span class="text-muted small">No series</span>';
            return;
        }
        
        // Find which series this BIB belongs to
        let foundSeries = null;
        if (seriesStats) {
            for (const stat of seriesStats) {
                const allNums = [...stat.available_numbers, ...stat.used_numbers];
                if (allNums.includes(bibNumber)) {
                    foundSeries = stat.series;
                    break;
                }
            }
        }
        
        if (foundSeries) {
            indicator.innerHTML = `<span class="badge" style="background-color: ${foundSeries.color}; color: white;">${foundSeries.name}</span>`;
        } else {
            indicator.innerHTML = '<span class="badge bg-danger">Invalid BIB</span>';
        }
    }

    function assignNextAvailable(seriesId) {
        if (!seriesStats || seriesStats.length === 0) {
            if (typeof showAlert === 'function') showAlert('No BIB series available', 'warning');
            return;
        }
        
        const inputs = document.querySelectorAll('.bib-input');
        const usedNumbers = new Set();
        
        // Get currently used numbers in the form
        inputs.forEach(input => {
            if (input.value && !isNaN(parseInt(input.value))) {
                usedNumbers.add(parseInt(input.value));
            }
        });
        
        // Find series data
        const stat = seriesStats.find(s => s.series.id == seriesId);
        if (!stat || !stat.available_numbers) {
            if (typeof showAlert === 'function') showAlert('Series not found or no available numbers', 'warning');
            return;
        }
        
        // Find available numbers from this series that are NOT used in the form
        const availableInSeries = stat.available_numbers.filter(num => !usedNumbers.has(num));
        availableInSeries.sort((a, b) => a - b);
        
        if (availableInSeries.length === 0) {
            if (typeof showAlert === 'function') showAlert(`No available numbers in ${stat.series.name} series`, 'warning');
            return;
        }
        
        // Find first unassigned participant (visible rows only)
        const unassignedInput = Array.from(inputs).find(input => {
             const row = input.closest('tr');
             return !input.value && row && row.style.display !== 'none';
        });

        if (!unassignedInput) {
            if (typeof showAlert === 'function') showAlert('All visible participants already have BIB numbers', 'info');
            return;
        }
        
        // Assign the first available number
        unassignedInput.value = availableInSeries[0];
        updateBIBPreview(unassignedInput);
        
        // Scroll to the row
        unassignedInput.scrollIntoView({ behavior: 'smooth', block: 'center' });
        
        // Flash the input
        unassignedInput.classList.add('bg-success', 'text-white');
        setTimeout(() => {
            unassignedInput.classList.remove('bg-success', 'text-white');
        }, 1000);
    }
</script>

<?php include_once 'footer.php'; ?>
</body>
</html>