<?php
// API endpoint for user management operations
error_reporting(E_ALL); // Enable error reporting for debugging
ini_set('display_errors', 1);

include '../../includes/auth.php';
include '../../includes/db.php';


header('Content-Type: application/json');

// Check permissions - users need at least view permission
$user_role = $_SESSION['role'] ?? $_SESSION['user_type'] ?? '';
$current_user_id = $_SESSION['user_id'] ?? 0;

// Make current_user_id available globally for all functions
$GLOBALS['current_user_id'] = $current_user_id;

if (!hasAnyPermission(['view_all_users', 'view_created_users'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized: Missing required permissions']);
    exit;
}

$action = $_GET['action'] ?? $_POST['action'] ?? '';

try {
    switch ($action) {
        case 'get_user':
            if (!isset($_GET['id'])) {
                echo json_encode(['success' => false, 'message' => 'Missing user ID']);
                exit;
            }
            $result = getUser($pdo, $_GET['id']);
            echo json_encode($result);
            break;
            
        case 'get_group_members':
            echo json_encode(getGroupMembers($pdo, $_GET['group_id']));
            break;
            
        case 'get_group':
            echo json_encode(getGroup($pdo, $_GET['id']));
            break;
            
        case 'search_users':
            echo json_encode(searchUsers($pdo, $_GET['q']));
            break;
            
        case 'update_user_status':
            echo json_encode(updateUserStatus($pdo, $_POST, $user_role));
            break;
            
        case 'bulk_action':
            echo json_encode(bulkAction($pdo, $_POST, $user_role));
            break;
            
        case 'get_events':
            echo json_encode(getEvents($pdo));
            break;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Invalid action']);
    }
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
}

function getUser($pdo, $user_id) {
    try {
        $stmt = $pdo->prepare("
            SELECT u.id, u.username, u.password, u.role, u.created_at, u.last_login, u.judge_id,
                   p.full_name, p.email, p.phone, p.address, p.emergency_contact, 
                   p.emergency_phone, p.notes, p.avatar, p.is_active,
                   GROUP_CONCAT(DISTINCT g.id) as group_ids,
                   GROUP_CONCAT(DISTINCT g.name) as group_names,
                   GROUP_CONCAT(DISTINCT uae.event_id) as assigned_event_ids
            FROM users u
            LEFT JOIN user_profiles p ON u.id = p.user_id
            LEFT JOIN user_group_memberships ugm ON u.id = ugm.user_id
            LEFT JOIN user_groups g ON ugm.group_id = g.id
            LEFT JOIN user_assigned_events uae ON u.id = uae.user_id
            WHERE u.id = ?
            GROUP BY u.id, u.username, u.password, u.role, u.created_at, u.last_login, u.judge_id,
                     p.full_name, p.email, p.phone, p.address, p.emergency_contact, 
                     p.emergency_phone, p.notes, p.avatar, p.is_active
        ");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user) {
            // Get event details if assigned
            if ($user['assigned_event_ids']) {
                $event_ids = explode(',', $user['assigned_event_ids']);
                $placeholders = str_repeat('?,', count($event_ids) - 1) . '?';
                
                // Dynamic column detection for events table
                try {
                    $columns = $pdo->query("SHOW COLUMNS FROM events")->fetchAll(PDO::FETCH_COLUMN);
                    $dateColumn = in_array('event_date', $columns) ? 'event_date' : 'date';
                    $hasStatus = in_array('status', $columns);
                    
                    $statusSelect = $hasStatus ? ', status' : ', "active" as status';
                    
                    $event_stmt = $pdo->prepare("
                        SELECT id, name, $dateColumn as event_date $statusSelect
                        FROM events 
                        WHERE id IN ($placeholders)
                        ORDER BY $dateColumn DESC
                    ");
                    $event_stmt->execute($event_ids);
                    $user['assigned_events'] = $event_stmt->fetchAll(PDO::FETCH_ASSOC);
                } catch (Exception $e) {
                    error_log("Error fetching event details: " . $e->getMessage());
                    $user['assigned_events'] = [];
                }
            } else {
                $user['assigned_events'] = [];
            }
            
            return ['success' => true, 'user' => $user];
        } else {
            return ['success' => false, 'message' => 'User not found'];
        }
    } catch (Exception $e) {
        // Fallback to basic user info if extended tables don't exist
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user) {
            // Add default values for missing profile fields
            $user['full_name'] = $user['full_name'] ?? ucwords(str_replace('_', ' ', $user['username']));
            $user['email'] = $user['email'] ?? '';
            $user['phone'] = $user['phone'] ?? '';
            $user['is_active'] = $user['is_active'] ?? 1;
            $user['notes'] = $user['notes'] ?? '';
            return ['success' => true, 'user' => $user];
        } else {
            return ['success' => false, 'message' => 'User not found'];
        }
    }
}

function getGroupMembers($pdo, $group_id) {
    try {
        // Check if tables exist first
        $pdo->query("SELECT 1 FROM user_group_memberships LIMIT 1");
        $pdo->query("SELECT 1 FROM user_profiles LIMIT 1");
        
        // First, try to get the actual column structure
        $columns = $pdo->query("SHOW COLUMNS FROM user_group_memberships")->fetchAll(PDO::FETCH_COLUMN);
        
        // Determine which date column to use
        $dateColumn = 'NOW() as joined_at'; // Default fallback
        if (in_array('created_at', $columns)) {
            $dateColumn = 'ugm.created_at as joined_at';
        } elseif (in_array('joined_at', $columns)) {
            $dateColumn = 'ugm.joined_at';
        } elseif (in_array('added_at', $columns)) {
            $dateColumn = 'ugm.added_at as joined_at';
        }
        
        $stmt = $pdo->prepare("
            SELECT u.id, u.username, p.full_name, p.email, u.role,
                   ugm.role_in_group, $dateColumn
            FROM user_group_memberships ugm
            JOIN users u ON ugm.user_id = u.id
            LEFT JOIN user_profiles p ON u.id = p.user_id
            WHERE ugm.group_id = ?
            ORDER BY u.username
        ");
        $stmt->execute([$group_id]);
        $members = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        return ['success' => true, 'members' => $members];
    } catch (Exception $e) {
        // If tables don't exist, return empty result
        if (strpos($e->getMessage(), "doesn't exist") !== false) {
            return ['success' => true, 'members' => [], 'message' => 'Group membership tables not yet created'];
        }
        
        // Try a simpler query without date columns
        try {
            $stmt = $pdo->prepare("
                SELECT u.id, u.username, p.full_name, p.email, u.role,
                       ugm.role_in_group, 'Member since setup' as joined_at
                FROM user_group_memberships ugm
                JOIN users u ON ugm.user_id = u.id
                LEFT JOIN user_profiles p ON u.id = p.user_id
                WHERE ugm.group_id = ?
                ORDER BY u.username
            ");
            $stmt->execute([$group_id]);
            $members = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return ['success' => true, 'members' => $members];
        } catch (Exception $e2) {
            return ['success' => false, 'message' => $e2->getMessage()];
        }
    }
}

function getGroup($pdo, $group_id) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM user_groups WHERE id = ?");
        $stmt->execute([$group_id]);
        $group = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($group) {
            return ['success' => true, 'group' => $group];
        } else {
            return ['success' => false, 'message' => 'Group not found'];
        }
    } catch (Exception $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}

function searchUsers($pdo, $query) {
    $search = "%$query%";
    $stmt = $pdo->prepare("
        SELECT u.id, u.username, u.role, p.full_name, p.email
        FROM users u
        LEFT JOIN user_profiles p ON u.id = p.user_id
        WHERE u.username LIKE ? OR p.full_name LIKE ? OR p.email LIKE ?
        ORDER BY p.full_name, u.username
        LIMIT 20
    ");
    $stmt->execute([$search, $search, $search]);
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    return ['success' => true, 'users' => $users];
}

function updateUserStatus($pdo, $data, $user_role) {
    $user_id = intval($data['user_id']);
    $status = $data['status'] === 'active' ? 1 : 0;
    
    $stmt = $pdo->prepare("UPDATE user_profiles SET is_active = ? WHERE user_id = ?");
    $stmt->execute([$status, $user_id]);
    
    return ['success' => true, 'message' => 'User status updated'];
}

function bulkAction($pdo, $data, $user_role) {
    $user_ids = $data['user_ids'] ?? [];
    $action = $data['bulk_action'];
    
    if (empty($user_ids)) {
        return ['success' => false, 'message' => 'No users selected'];
    }
    
    switch ($action) {
        case 'activate':
            $placeholders = str_repeat('?,', count($user_ids) - 1) . '?';
            $stmt = $pdo->prepare("UPDATE user_profiles SET is_active = 1 WHERE user_id IN ($placeholders)");
            $stmt->execute($user_ids);
            return ['success' => true, 'message' => count($user_ids) . ' users activated'];
            
        case 'deactivate':
            $placeholders = str_repeat('?,', count($user_ids) - 1) . '?';
            $stmt = $pdo->prepare("UPDATE user_profiles SET is_active = 0 WHERE user_id IN ($placeholders)");
            $stmt->execute($user_ids);
            return ['success' => true, 'message' => count($user_ids) . ' users deactivated'];
            
        case 'delete':
            if ($user_role !== 'super_admin') {
                return ['success' => false, 'message' => 'Only super admin can delete users'];
            }
            $placeholders = str_repeat('?,', count($user_ids) - 1) . '?';
            $stmt = $pdo->prepare("DELETE FROM users WHERE id IN ($placeholders)");
            $stmt->execute($user_ids);
            return ['success' => true, 'message' => count($user_ids) . ' users deleted'];
            
        default:
            return ['success' => false, 'message' => 'Invalid bulk action'];
    }
}

function getEvents($pdo) {
    try {
        $current_user_id = $GLOBALS['current_user_id'] ?? 0;
        
        // Check which columns exist
        $columns = $pdo->query("SHOW COLUMNS FROM events")->fetchAll(PDO::FETCH_COLUMN);
        
        // Build query based on available columns
        $selectColumns = "e.id, e.name";
        
        if (in_array('event_date', $columns)) {
            $selectColumns .= ", e.event_date";
        } elseif (in_array('date', $columns)) {
            $selectColumns .= ", e.date as event_date";
        }
        
        if (in_array('status', $columns)) {
            $selectColumns .= ", e.status";
        }
        
        // Filter events based on permissions
        $where_clause = "1=1";
        $params = [];
        
        if (hasPermission('view_all_users')) {
            // Can see all events
            $where_clause = "1=1";
        } else {
            // Can only see events they're assigned to
            $where_clause = "EXISTS (
                SELECT 1 FROM user_assigned_events uae 
                WHERE uae.event_id = e.id 
                AND uae.user_id = ?
            )";
            $params[] = $current_user_id;
        }
        
        $order_by = in_array('event_date', $columns) ? 'e.event_date' : (in_array('date', $columns) ? 'e.date' : 'e.id');
        
        $stmt = $pdo->prepare("
            SELECT $selectColumns 
            FROM events e
            WHERE $where_clause
            ORDER BY $order_by DESC, e.name ASC
        ");
        
        $stmt->execute($params);
        $events = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        return ['success' => true, 'events' => $events];
    } catch (Exception $e) {
        return ['success' => false, 'message' => $e->getMessage()];
    }
}
?>
