<?php
/**
 * AJAX Unassign Handler
 * Removes participants from all heats in an event
 */

session_start();
require_once '../includes/db.php';
require_once '../includes/auth.php';

header('Content-Type: application/json');

// Check authentication
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['status' => 'error', 'message' => 'Authentication required']);
    exit;
}

// Validate required parameters
if (!isset($_POST['event_id']) || !isset($_POST['participant_ids'])) {
    echo json_encode(['status' => 'error', 'message' => 'Missing required parameters']);
    exit;
}

$event_id = (int)$_POST['event_id'];
$participant_ids = $_POST['participant_ids'];

// Validate participant_ids is an array
if (!is_array($participant_ids) || empty($participant_ids)) {
    echo json_encode(['status' => 'error', 'message' => 'No participants selected']);
    exit;
}

// Sanitize participant IDs
$participant_ids = array_map('intval', $participant_ids);

try {
    // Begin transaction
    $pdo->beginTransaction();
    
    // Set heat_number to NULL for these participants (unassign from heats but keep in event)
    $placeholders = str_repeat('?,', count($participant_ids) - 1) . '?';
    $params = array_merge([$event_id], $participant_ids);
    
    $stmt = $pdo->prepare("
        UPDATE event_participants 
        SET heat_number = NULL, updated_at = NOW()
        WHERE event_id = ? 
        AND participant_id IN ($placeholders)
    ");
    
    $stmt->execute($params);
    $updatedCount = $stmt->rowCount();
    
    // Commit transaction
    $pdo->commit();
    
    // Return success response
    echo json_encode([
        'status' => 'success',
        'message' => "Unassigned {$updatedCount} participant(s) from all heats",
        'updated_count' => $updatedCount,
        'participant_ids' => $participant_ids
    ]);
    
} catch (PDOException $e) {
    // Rollback on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    error_log("Unassignment error: " . $e->getMessage());
    echo json_encode([
        'status' => 'error',
        'message' => 'Database error: ' . $e->getMessage()
    ]);
}
