<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

include '../includes/auth.php';
include '../includes/db.php';
include '../includes/validate_event_access.php';
include '../includes/TokenSystem.php';

$message = '';

// Handle event selection - check cookie for persistence
$selected_event_id = $_GET['event_id'] ?? $_COOKIE['event_id'] ?? $_COOKIE['selectedEventId'] ?? '';
$heats_total = 0;
$heat_settings = [];
$event_categories = [];

// Handle add heat action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_heat']) && isset($_POST['event_id'])) {
    $event_id = intval($_POST['event_id']);
    $user_id = $_SESSION['user_id'];
    
    try {
        // Charge tokens for adding a heat
        $result = TokenSystem::charge('add_heat', $user_id, [
            'event_id' => $event_id,
            'reference_type' => 'event'
        ]);
        
        if (!$result['success']) {
            echo json_encode([
                'success' => false, 
                'message' => $result['message'],
                'balance' => TokenSystem::getBalance($user_id)
            ]);
            exit;
        }
        
        // Get current heats_total
        $stmt = $pdo->prepare("SELECT heats_total FROM events WHERE id = ?");
        $stmt->execute([$event_id]);
        $current_heats = $stmt->fetchColumn();
        
        if ($current_heats === false) {
            echo json_encode(['success' => false, 'message' => 'Event not found']);
            exit;
        }
        
        // Increment heats_total
        $new_heats_total = $current_heats + 1;
        $update_stmt = $pdo->prepare("UPDATE events SET heats_total = ? WHERE id = ?");
        $update_stmt->execute([$new_heats_total, $event_id]);
        
        // Create default heat settings for the new heat
        $new_heat_number = $new_heats_total;
        $insert_stmt = $pdo->prepare("
            INSERT INTO event_heat_settings 
            (event_id, heat_number, heat_name, scoring_type, runs_count, runs_scoring_method, 
             time_start, estimate_time_per_participant, categories, status, is_active, active_run)
            VALUES (?, ?, ?, 'Points', 1, 'best_from_all', NULL, 0, '[]', 'pending', 0, 0)
        ");
        $insert_stmt->execute([$event_id, $new_heat_number, ""]);
        
        echo json_encode([
            'success' => true, 
            'message' => "Heat $new_heat_number added successfully! ({$result['charged_amount']} tokens charged)",
            'new_heats_total' => $new_heats_total,
            'new_balance' => $result['new_balance']
        ]);
        exit;
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        exit;
    }
}

// Handle flow-only save (from node editor)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_flow_only']) && isset($_POST['event_id'])) {
    $event_id = intval($_POST['event_id']);
    
    // Process flow settings for each heat
    for ($h = 1; $h <= 100; $h++) { // Max 100 heats
        // Check if this heat exists
        $stmt = $pdo->prepare("SELECT heat_number FROM event_heat_settings WHERE event_id = ? AND heat_number = ?");
        $stmt->execute([$event_id, $h]);
        if (!$stmt->fetch()) {
            continue; // Skip non-existent heats
        }
        
        // Get flow settings for this heat
        $flow_type = $_POST['flow_type'][$h] ?? 'none';
        $flow_source_heat = null;
        $flow_participants_per_category = intval($_POST['flow_participants_per_category'][$h] ?? 0);
        $flow_position_range = trim($_POST['flow_position_range'][$h] ?? '');
        $layout_x = isset($_POST['layout_x'][$h]) ? intval($_POST['layout_x'][$h]) : null;
        $layout_y = isset($_POST['layout_y'][$h]) ? intval($_POST['layout_y'][$h]) : null;
        $hue_offsets_json = $_POST['hue_offsets'][$h] ?? '{}';
        
        // Handle multiple source heats (array)
        if (isset($_POST['flow_source_heat'][$h]) && is_array($_POST['flow_source_heat'][$h])) {
            $source_heats = array_map('intval', array_filter($_POST['flow_source_heat'][$h]));
            if (!empty($source_heats)) {
                $flow_source_heat = json_encode($source_heats);
            }
        }
        
        // Validate flow settings
        if ($flow_type === 'none' || empty($flow_source_heat)) {
            $flow_source_heat = null;
            $flow_participants_per_category = 0;
            $flow_position_range = null;
            $flow_type = 'none';
        }
        
        // Update flow-related fields AND layout positions
        $update = $pdo->prepare("
            UPDATE event_heat_settings 
            SET flow_type = ?, 
                flow_source_heat = ?, 
                flow_participants_per_category = ?,
                flow_position_range = ?,
                layout_x = ?,
                layout_y = ?,
                hue_offsets = ?
            WHERE event_id = ? AND heat_number = ?
        ");
        $update->execute([
            $flow_type,
            $flow_source_heat,
            $flow_participants_per_category,
            $flow_position_range ?: null,
            $layout_x,
            $layout_y,
            $hue_offsets_json,
            $event_id,
            $h
        ]);
    }
    
    echo json_encode(['success' => true, 'message' => 'Flow configuration saved successfully']);
    exit;
}

// Handle saving settings (only for selected event)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_GET['event_id']) && isset($_POST['save_settings'])) {
    $event_id = intval($_POST['event_id']);
    
    // First, deactivate all heats and runs for this event (set to pending if not finished/cancelled)
    $pdo->prepare("UPDATE event_heat_settings SET is_active = 0, active_run = 0, status = 'pending' WHERE event_id = ? AND status NOT IN ('finished', 'cancelled')")->execute([$event_id]);
    
    if (isset($_POST['heat_name'])) {
        foreach ($_POST['heat_name'] as $heat_number => $name) {
            $heat_name = trim($name);
            $scoring_type = $_POST['scoring_type'][$heat_number] ?? 'Points';
            $runs_count = intval($_POST['runs_count'][$heat_number] ?? 1);
            $runs_scoring_method = $_POST['runs_scoring_method'][$heat_number] ?? 'best_from_all';
            $time_start = $_POST['time_start'][$heat_number] ?? null;
            $estimate_time_per_participant = intval($_POST['estimate_time_per_participant'][$heat_number] ?? 0);
            
            // Handle multiple categories
            $categories = isset($_POST['categories'][$heat_number]) ? $_POST['categories'][$heat_number] : [];
            $categories_json = json_encode($categories);
            
            // Handle format-related settings
            $format_name = $_POST['format_name'][$heat_number] ?? ($event_data['format_name'] ?? '');
            $discipline = $_POST['discipline'][$heat_number] ?? ($event_data['discipline'] ?? '');
            $difficulty_level = $_POST['difficulty_level'][$heat_number] ?? '';
            $course_length = intval($_POST['course_length'][$heat_number] ?? 0);
            $time_limit = intval($_POST['time_limit'][$heat_number] ?? 0);
            $weather_dependent = isset($_POST['weather_dependent'][$heat_number]) ? 1 : 0;
            $format_description = $_POST['format_description'][$heat_number] ?? '';
            $scoring_method = $_POST['scoring_method'][$heat_number] ?? 'standard';
            $required_figures = isset($_POST['required_figures'][$heat_number]) ? 
                                json_encode($_POST['required_figures'][$heat_number]) : '[]';
            $judges_required = intval($_POST['judges_required'][$heat_number] ?? 5);
            $scale_min = floatval($_POST['scale_min'][$heat_number] ?? ($event_data['scale_min'] ?? 0));
            $scale_max = floatval($_POST['scale_max'][$heat_number] ?? ($event_data['scale_max'] ?? 100));
            $precision_decimal = intval($_POST['precision_decimal'][$heat_number] ?? ($event_data['precision_decimal'] ?? 0));
            
            // Handle drop_rule setting (NULL means use event default, empty string converted to NULL)
            $drop_rule = isset($_POST['drop_rule'][$heat_number]) ? trim($_POST['drop_rule'][$heat_number]) : null;
            $drop_rule = ($drop_rule === '') ? null : $drop_rule;
            
            // Handle flow settings - only process if not disabled
            $flow_type = $_POST['flow_type'][$heat_number] ?? 'none';
            $flow_source_heat = null;
            $flow_participants_per_category = 0;
            $flow_position_range = null;
            
            if ($flow_type !== 'none') {
                // Support multiple source heats (array or single value)
                $source_heat_input = $_POST['flow_source_heat'][$heat_number] ?? null;
                
                if ($source_heat_input) {
                    // If it's already an array (from multiselect)
                    if (is_array($source_heat_input)) {
                        $flow_source_heat = json_encode(array_map('intval', array_filter($source_heat_input)));
                    } else {
                        // Single value - convert to JSON array for consistency
                        $flow_source_heat = json_encode([intval($source_heat_input)]);
                    }
                }
                
                $flow_participants_per_category = intval($_POST['flow_participants_per_category'][$heat_number] ?? 0);
                $flow_position_range = trim($_POST['flow_position_range'][$heat_number] ?? '');
                
                // Validate position range format (e.g., "1-3", "4-8")
                if (!empty($flow_position_range) && !preg_match('/^\d+-\d+$/', $flow_position_range)) {
                    $flow_position_range = null; // Invalid format
                }
                
                // Validate flow settings
                if ($flow_participants_per_category < 1) {
                    $flow_participants_per_category = 0;
                    $flow_type = 'none'; // Reset to none if invalid
                }
                
                if ($flow_type === 'promotion' && !$flow_source_heat) {
                    $flow_type = 'none'; // Reset to none if no source heat selected
                }
            }
            
            // Handle diversity rules setting
            $diversity_rules_enabled = isset($_POST['diversity_rules_enabled'][$heat_number]) ? 1 : 0;
            
            $is_active = isset($_POST['is_active']) && $_POST['is_active'] == $heat_number ? 1 : 0;
            $active_run = 0;
            
            // Determine heat status
            $status = 'pending'; // Default status
            if (isset($_POST['heat_status'][$heat_number])) {
                $status = $_POST['heat_status'][$heat_number];
            } elseif ($is_active) {
                $status = 'active';
            }
            
            // If heat is active, set the active run
            if ($status === 'active' && isset($_POST['active_run'][$heat_number])) {
                $active_run = intval($_POST['active_run'][$heat_number]);
            }
            
            // Process heat-specific scoring layout configuration
            $heat_scoring_layout_config = null;
            if (isset($_POST['heat_scoring_layout'][$heat_number])) {
                $heatLayoutData = $_POST['heat_scoring_layout'][$heat_number];
                
                // Handle checkboxes - unchecked checkboxes don't appear in POST data
                $availableComponents = [
                    'criteriaInputGrid', 'finalCalculatedScore', 'figuresCard', 
                    'latestScores', 'otherHeatScores', 'diversityValidation', 
                    'figureHistory', 'formatInfo', 'drawingBoardButton'
                ];
                
                // Ensure all components have an enabled status (default false if not present)
                if (isset($heatLayoutData['cards'])) {
                    foreach ($availableComponents as $component) {
                        if (!isset($heatLayoutData['cards'][$component]['enabled'])) {
                            $heatLayoutData['cards'][$component]['enabled'] = false;
                        } else {
                            // Convert string "1" to boolean true
                            $heatLayoutData['cards'][$component]['enabled'] = 
                                ($heatLayoutData['cards'][$component]['enabled'] == '1');
                        }
                    }
                }
                
                // Convert layout options to proper types
                if (isset($heatLayoutData['layout']['compactMode'])) {
                    $heatLayoutData['layout']['compactMode'] = 
                        ($heatLayoutData['layout']['compactMode'] == '1');
                }
                if (isset($heatLayoutData['layout']['hideEmptyCards'])) {
                    $heatLayoutData['layout']['hideEmptyCards'] = 
                        ($heatLayoutData['layout']['hideEmptyCards'] == '1');
                }
                
                $heat_scoring_layout_config = json_encode($heatLayoutData);
            }
            
            $stmt = $pdo->prepare("
                INSERT INTO event_heat_settings (
                    event_id, heat_number, heat_name, scoring_type, runs_count, runs_scoring_method,
                    time_start, estimate_time_per_participant, categories, 
                    flow_type, flow_source_heat, flow_participants_per_category, flow_position_range,
                    diversity_rules_enabled,
                    is_active, active_run, status,
                    format_name, discipline, difficulty_level, course_length, time_limit,
                    weather_dependent, format_description, scoring_method, required_figures,
                    judges_required, scale_min, scale_max, precision_decimal, drop_rule, scoring_layout_config
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                ON DUPLICATE KEY UPDATE 
                    heat_name = VALUES(heat_name),
                    scoring_type = VALUES(scoring_type),
                    runs_count = VALUES(runs_count),
                    runs_scoring_method = VALUES(runs_scoring_method),
                    time_start = VALUES(time_start),
                    estimate_time_per_participant = VALUES(estimate_time_per_participant),
                    categories = VALUES(categories),
                    flow_type = VALUES(flow_type),
                    flow_source_heat = VALUES(flow_source_heat),
                    flow_participants_per_category = VALUES(flow_participants_per_category),
                    flow_position_range = VALUES(flow_position_range),
                    diversity_rules_enabled = VALUES(diversity_rules_enabled),
                    is_active = VALUES(is_active),
                    active_run = VALUES(active_run),
                    status = VALUES(status),
                    format_name = VALUES(format_name),
                    discipline = VALUES(discipline),
                    difficulty_level = VALUES(difficulty_level),
                    course_length = VALUES(course_length),
                    time_limit = VALUES(time_limit),
                    weather_dependent = VALUES(weather_dependent),
                    format_description = VALUES(format_description),
                    scoring_method = VALUES(scoring_method),
                    required_figures = VALUES(required_figures),
                    judges_required = VALUES(judges_required),
                    scale_min = VALUES(scale_min),
                    scale_max = VALUES(scale_max),
                    precision_decimal = VALUES(precision_decimal),
                    drop_rule = VALUES(drop_rule),
                    scoring_layout_config = VALUES(scoring_layout_config),
                    updated_at = CURRENT_TIMESTAMP
            ");
            
            $stmt->execute([
                $event_id, $heat_number, $heat_name, $scoring_type, $runs_count, $runs_scoring_method,
                $time_start, $estimate_time_per_participant, $categories_json,
                $flow_type, $flow_source_heat, $flow_participants_per_category, $flow_position_range,
                $diversity_rules_enabled,
                $is_active, $active_run, $status,
                $format_name, $discipline, $difficulty_level, $course_length, $time_limit,
                $weather_dependent, $format_description, $scoring_method, $required_figures,
                $judges_required, $scale_min, $scale_max, $precision_decimal, $drop_rule, $heat_scoring_layout_config
            ]);
        }
        $message = '<div class="alert alert-success"><i class="fas fa-check-circle me-2"></i>Heat settings saved successfully!</div>';
    }
}

// Handle deactivate request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['deactivate_heat'])) {
    $event_id = intval($_POST['event_id']);
    $pdo->prepare("UPDATE event_heat_settings SET is_active = 0, active_run = 0, status = 'pending' WHERE event_id = ?")->execute([$event_id]);
    $message = '<div class="alert alert-info"><i class="fas fa-info-circle me-2"></i>All heats have been deactivated.</div>';
}

// Handle active run change request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['change_active_run'])) {
    $event_id = intval($_POST['event_id']);
    $heat_number = intval($_POST['heat_number']);
    $direction = intval($_POST['direction']);
    
    // Get current active run
    $stmt = $pdo->prepare("SELECT active_run, runs_count FROM event_heat_settings WHERE event_id = ? AND heat_number = ? AND status = 'active'");
    $stmt->execute([$event_id, $heat_number]);
    $heat = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($heat) {
        $current_run = intval($heat['active_run']);
        $runs_count = intval($heat['runs_count']);
        $new_run = $current_run + $direction;
        
        // Validate new run number
        if ($new_run >= 1 && $new_run <= $runs_count) {
            $update = $pdo->prepare("UPDATE event_heat_settings SET active_run = ? WHERE event_id = ? AND heat_number = ?");
            $update->execute([$new_run, $event_id, $heat_number]);
            $message = '<div class="alert alert-success"><i class="fas fa-check-circle me-2"></i>Active run changed to Run ' . $new_run . ' for Heat ' . $heat_number . '</div>';
        } else {
            $message = '<div class="alert alert-warning"><i class="fas fa-exclamation-triangle me-2"></i>Invalid run number. Must be between 1 and ' . $runs_count . '</div>';
        }
    } else {
        $message = '<div class="alert alert-danger"><i class="fas fa-times-circle me-2"></i>Heat ' . $heat_number . ' is not active.</div>';
    }
}

// Handle close heat request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['close_heat'])) {
    $event_id = intval($_POST['event_id']);
    $heat_number = intval($_POST['heat_number']);
    
    // Set heat to finished status
    $update = $pdo->prepare("UPDATE event_heat_settings SET is_active = 0, active_run = 0, status = 'finished' WHERE event_id = ? AND heat_number = ?");
    $update->execute([$event_id, $heat_number]);
    
    $message = '<div class="alert alert-success"><i class="fas fa-check-circle me-2"></i>Heat ' . $heat_number . ' has been marked as finished.</div>';
}

// Handle cancel heat request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cancel_heat'])) {
    $event_id = intval($_POST['event_id']);
    $heat_number = intval($_POST['heat_number']);
    
    // Set heat to cancelled status
    $update = $pdo->prepare("UPDATE event_heat_settings SET is_active = 0, active_run = 0, status = 'cancelled' WHERE event_id = ? AND heat_number = ?");
    $update->execute([$event_id, $heat_number]);
    
    $message = '<div class="alert alert-warning"><i class="fas fa-ban me-2"></i>Heat ' . $heat_number . ' has been cancelled.</div>';
}

// Handle reschedule heat request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['reschedule_heat'])) {
    $event_id = intval($_POST['event_id']);
    $heat_number = intval($_POST['heat_number']);
    
    // Set heat to rescheduled status
    $update = $pdo->prepare("UPDATE event_heat_settings SET is_active = 0, active_run = 0, status = 'rescheduled' WHERE event_id = ? AND heat_number = ?");
    $update->execute([$event_id, $heat_number]);
    
    $message = '<div class="alert alert-info"><i class="fas fa-clock me-2"></i>Heat ' . $heat_number . ' has been marked for rescheduling.</div>';
}

// Handle set pending heat request
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['set_pending_heat'])) {
    $event_id = intval($_POST['event_id']);
    $heat_number = intval($_POST['heat_number']);
    
    // Set heat to pending status
    $update = $pdo->prepare("UPDATE event_heat_settings SET is_active = 0, active_run = 0, status = 'pending' WHERE event_id = ? AND heat_number = ?");
    $update->execute([$event_id, $heat_number]);
    
    $message = '<div class="alert alert-secondary"><i class="fas fa-pause me-2"></i>Heat ' . $heat_number . ' has been set to pending status.</div>';
}

// Load events
$events = $pdo->query("SELECT id, name, heats_total FROM events ORDER BY date DESC")->fetchAll(PDO::FETCH_ASSOC);

// Load selected event settings
if ($selected_event_id) {
    $event = $pdo->prepare("
        SELECT e.*, sf.name as format_name, sf.sport as discipline, sf.mode as format_mode,
               sfj.scale_min, sfj.scale_max, sfj.scale_type, sfj.precision_decimal,
               GROUP_CONCAT(DISTINCT sfmc.config_value ORDER BY sfmc.config_key SEPARATOR ', ') as format_config
        FROM events e
        LEFT JOIN scoring_formats sf ON e.scoring_format = sf.format_id
        LEFT JOIN scoring_format_judges sfj ON sf.format_id = sfj.format_id
        LEFT JOIN scoring_format_mode_config sfmc ON sf.format_id = sfmc.format_id
        WHERE e.id = ?
        GROUP BY e.id
    ");
    $event->execute([$selected_event_id]);
    $event_data = $event->fetch(PDO::FETCH_ASSOC);
    $heats_total = intval($event_data['heats_total']);

    // Load event categories (main categories only)
    $categories_stmt = $pdo->prepare("
        SELECT id, category_name 
        FROM event_categories 
        WHERE event_id = ? AND is_main_category = 1 
        ORDER BY category_name
    ");
    $categories_stmt->execute([$selected_event_id]);
    $event_categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get participant counts per heat
    $participant_counts_stmt = $pdo->prepare("
        SELECT heat_number, COUNT(*) as participant_count 
        FROM event_participants 
        WHERE event_id = ? 
        GROUP BY heat_number
    ");
    $participant_counts_stmt->execute([$selected_event_id]);
    $heat_participant_counts = [];
    foreach ($participant_counts_stmt->fetchAll(PDO::FETCH_ASSOC) as $count_row) {
        $heat_participant_counts[$count_row['heat_number']] = $count_row['participant_count'];
    }
    
    // Load participants for bracket display
    $participants_stmt = $pdo->prepare("
        SELECT ep.heat_number, ep.bib_number, p.first_name, p.last_name, p.country
        FROM event_participants ep
        JOIN participants p ON ep.participant_id = p.id
        WHERE ep.event_id = ?
        ORDER BY ep.heat_number, ep.bib_number
    ");
    $participants_stmt->execute([$selected_event_id]);
    $heat_participants = [];
    foreach ($participants_stmt->fetchAll(PDO::FETCH_ASSOC) as $participant) {
        $heat_num = $participant['heat_number'];
        if (!isset($heat_participants[$heat_num])) {
            $heat_participants[$heat_num] = [];
        }
        $heat_participants[$heat_num][] = $participant;
    }

    // Load heat settings with new fields
    $settings = $pdo->prepare("
        SELECT heat_number, heat_name, scoring_type, runs_count, runs_scoring_method,
               time_start, estimate_time_per_participant, categories, flow_type, flow_source_heat,
               flow_participants_per_category, flow_position_range, diversity_rules_enabled, is_active, active_run, status,
               format_name, discipline, difficulty_level, course_length, time_limit,
               weather_dependent, format_description, scoring_method, required_figures,
               judges_required, scale_min, scale_max, precision_decimal, drop_rule,
               layout_x, layout_y, hue_offsets
        FROM event_heat_settings 
        WHERE event_id = ?
        ORDER BY heat_number
    ");
    $settings->execute([$selected_event_id]);
    foreach ($settings->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $row['categories'] = json_decode($row['categories'] ?: '[]', true);
        $row['required_figures'] = json_decode($row['required_figures'] ?: '[]', true);
        $row['participant_count'] = $heat_participant_counts[$row['heat_number']] ?? 0;
        $heat_settings[$row['heat_number']] = $row;
    }
}
?>
<!DOCTYPE html>
<html>
<head>
        <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Configure Heats</title>
   
</head>
<body class="body-bg-aurora-bright">
<?php include '../menu.php'; ?>
     <!-- style skin holder -->
    <?php include_once '../includes/stylesheets.php'; ?>
    
    <style>
        /* Compact Sticky Top Bar Styles */
        #heatConfigTopBar {
            position: sticky;
            top: 0;
            z-index: 1020;
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 50%, #0f3460 100%);
            box-shadow: 0 2px 10px rgba(0,0,0,0.3);
            margin: -1rem -1rem 1.5rem -1rem;
            padding: 0.75rem 1.5rem;
        }
        
        #heatConfigTopBar .control-row {
            display: flex;
            align-items: center;
            gap: 0.75rem;
            flex-wrap: wrap;
        }
        
        #heatConfigTopBar .form-select-sm {
            min-width: 180px;
            max-width: 250px;
        }
        
      
        
        #heatConfigTopBar h6 {
            color: white;
            margin: 0;
            white-space: nowrap;
        }
        
        #heatConfigTopBar .badge {
            font-size: 0.7rem;
            padding: 0.25em 0.5em;
        }
        
        .quick-actions {
            display: flex;
            gap: 0.5rem;
            margin-left: auto;
        }
        
        .deactivate-badge {
            background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
            color: white;
            padding: 0.5rem 1rem;
            border-radius: 0.375rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.875rem;
            cursor: pointer;
            transition: all 0.2s;
        }
        
        .deactivate-badge:hover {
            background: linear-gradient(135deg, #c82333 0%, #bd2130 100%);
            transform: translateY(-1px);
            box-shadow: 0 2px 8px rgba(220, 53, 69, 0.3);
        }
        
        .metrics-pills {
            display: flex;
            gap: 0.75rem;
            align-items: center;
        }
        
        .metric-pill {
            background: rgba(255, 255, 255, 0.1);
            padding: 0.375rem 0.75rem;
            border-radius: 1rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.875rem;
            color: white;
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
        
      
        
       
    </style>
        
<div class="container-fluid container-StyleScore p-4">
        
        <!-- Compact Sticky Top Bar -->
        <div id="heatConfigTopBar">
            <div class="control-row">
                <!-- Title & Badge -->
                <div class="d-flex align-items-center gap-2">
                    <h6 class="mb-0 fw-bold">
                        <i class="fas fa-fire me-1"></i>
                        Heat Configuration
                    </h6>
                    <span class="badge bg-warning text-dark">Admin</span>
                </div>
                
                <!-- Event Selector -->
                <form method="get" class="d-flex align-items-center gap-2 mb-0 d-none">
                    <select class="form-select form-select-sm" name="event_id" id="event_id" onchange="this.form.submit()">
                        <option value="">-- Select Event --</option>
                        <?php foreach ($events as $ev): ?>
                            <option value="<?= $ev['id'] ?>" <?= $ev['id'] == $selected_event_id ? 'selected' : '' ?>>
                                <?= htmlspecialchars($ev['name']) ?> (<?= $ev['heats_total'] ?> heats)
                            </option>
                        <?php endforeach; ?>
                    </select>
                </form>
                
                
                
                <!-- Quick Actions -->
                <div class="quick-actions">
                    <?php if ($selected_event_id): ?>
                        <a href="active_heat_panel.php?event_id=<?= $selected_event_id ?>" class="btn btn-success btn-sm">
                           <i data-lucide="activity" class="me-2"></i>Active Heat Panel
                        </a>
                        <a href="heats_configure.php?event_id=<?= $selected_event_id ?>" class="btn btn-primary btn-sm">
                            <i data-lucide="users" class="me-2"></i>Athletes
                        </a>
                        <a href="<?= $baseUrl ?>/v2/admin/event_start_list.php?view_type=summary_table" class="btn btn-info btn-sm">
                            <i data-lucide="trophy" class="me-2"></i>Results
                        </a>
                        
                        <?php if (!empty($active_heat)): ?>
                        <form method="post" class="mb-0" onsubmit="return confirm('Deactivate current heat?');">
                            <input type="hidden" name="event_id" value="<?= $selected_event_id ?>">
                            <input type="hidden" name="deactivate_heat" value="1">
                            <button type="submit" class="btn btn-danger btn-sm">
                                <i class="fas fa-power-off me-1"></i>Deactivate
                            </button>
                        </form>
                        <?php endif; ?>
                    <?php endif; ?>
                    
                    <button type="button" class="btn btn-outline-info btn-sm bg-white" data-bs-toggle="modal" data-bs-target="#helpModal">
                        <i class="fas fa-circle-question me-1"></i>Help
                    </button>
                </div>
            </div>
        </div>

        <!-- Event action menu (hidden) -->
        <div class=" ">
            <?php       
            $use_localStorage = true;
            $show_header = false;
            include 'event_actions_menu.php'; 
            ?>
        </div>
        
        <!-- Help Modal -->
        <div class="modal fade" id="helpModal" tabindex="-1" aria-labelledby="helpModalLabel" aria-hidden="true">
            <div class="modal-dialog modal-xl modal-dialog-scrollable">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title" id="helpModalLabel"><i class="fas fa-circle-question me-2 text-info"></i>Configure Heats — Help</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body p-0" style="height:80vh;">
                        <iframe src="../help/heats_configure_help.html" style="border:0;width:100%;height:100%;" title="Configure Heats Help"></iframe>
                    </div>
                </div>
            </div>
        </div>

        <!-- Activate Heat Modal -->
        <div class="modal fade" id="activateHeatModal" tabindex="-1" aria-labelledby="activateHeatModalLabel" aria-hidden="true">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header bg-success text-white">
                        <h5 class="modal-title" id="activateHeatModalLabel">
                            <i class="fas fa-fire me-2"></i>Activate Heat <span id="activateHeatNumber"></span>
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                        <!-- Active Heat Warning -->
                        <div id="activeHeatWarning" class="alert alert-warning d-none">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <strong>Warning:</strong> Heat <span id="currentActiveHeatNumber"></span> (<span id="currentActiveHeatName"></span>) is currently active.
                            <br>Activating this heat will deactivate the current heat.
                        </div>

                        <!-- Run Selection -->
                        <div class="mb-3">
                            <label class="form-label fw-bold">
                                <i class="fas fa-play-circle me-1"></i>Select Run to Activate:
                            </label>
                            <div id="runSelectionOptions" class="d-grid gap-2">
                                <!-- Run buttons will be inserted here -->
                            </div>
                        </div>

                        <div class="alert alert-info mb-0">
                            <i class="fas fa-info-circle me-2"></i>
                            This will make <strong>Heat <span id="confirmHeatNumber"></span></strong> the active heat for scoring.
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                            <i class="fas fa-times me-1"></i>Cancel
                        </button>
                        <button type="button" class="btn btn-success" id="confirmActivateBtn" onclick="confirmActivateHeat()">
                            <i class="fas fa-fire me-1"></i>Activate Heat
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Heat Results Modal -->
        <div class="modal fade" id="heatResultsModal" tabindex="-1" aria-labelledby="heatResultsModalLabel" aria-hidden="true">
            <div class="modal-dialog modal-fullscreen p-5">
                <div class="modal-content">
                    <div class="modal-header bg-info text-white">
                        <h5 class="modal-title" id="heatResultsModalLabel">
                            <i class="fas fa-trophy me-2"></i>Heat <span id="resultsHeatNumber"></span> - <span id="resultsHeatName"></span> Results
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body" style="max-height: 70vh; overflow-y: auto;">
                        <style>
                            #savedConfigsList .config-button {
                                font-size: 0.875rem;
                                padding: 0.5rem 0.75rem;
                                border-left: 3px solid transparent;
                                transition: all 0.2s;
                            }
                            #savedConfigsList .config-button:hover {
                                background-color: #f8f9fa;
                                border-left-color: #0dcaf0;
                            }
                            #savedConfigsList .config-button.active {
                                background-color: #e7f5ff;
                                border-left-color: #0dcaf0;
                                font-weight: 600;
                            }
                            #resultsContentDisplay table {
                                font-size: 0.875rem;
                            }
                            #resultsContentDisplay .table-hover tbody tr:hover {
                                background-color: rgba(13, 202, 240, 0.1);
                            }
                        </style>
                        <div class="row">
                            <!-- Sidebar with Configurations -->
                            <div class="col-md-3">
                                <div class="card shadow-sm mb-3">
                                    <div class="card-body p-3">
                                        <h6 class="mb-2 fw-bold text-success">
                                            <i class="fas fa-bookmark me-2"></i>Saved Reports
                                        </h6>
                                        <div id="savedConfigsList" class="list-group list-group-flush">
                                            <div class="text-center py-3 text-muted">
                                                <div class="spinner-border spinner-border-sm" role="status">
                                                    <span class="visually-hidden">Loading...</span>
                                                </div>
                                                <p class="mt-2 small">Loading configurations...</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Main Content Area -->
                            <div class="col-md-9">
                                <div id="resultsContentDisplay">
                                    <div class="alert alert-info mb-0">
                                        <i class="fas fa-info-circle me-2"></i>
                                        Select a report from the sidebar to view results.
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                            <i class="fas fa-times me-1"></i>Close
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Generic Confirmation Modal -->
        <div class="modal fade" id="actionConfirmModal" tabindex="-1" aria-labelledby="actionConfirmModalLabel" aria-hidden="true">
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title action-confirm-title" id="actionConfirmModalLabel">Confirm Action</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                        <div class="d-flex align-items-start gap-3">
                            <div class="fs-2 text-primary">
                                <i class="action-confirm-icon fas fa-question-circle"></i>
                            </div>
                            <p class="mb-0 action-confirm-message">Are you sure?</p>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-outline-secondary action-confirm-cancel" data-bs-dismiss="modal">Cancel</button>
                        <button type="button" class="btn btn-primary action-confirm-approve">Confirm</button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Flow Unlink Confirmation Modal -->
        <div class="modal fade" id="flowUnlinkConfirmModal" tabindex="-1" aria-hidden="true" style="z-index: 9999;">
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content">
                    <div class="modal-header bg-warning text-dark">
                        <h5 class="modal-title">
                            <i class="fas fa-exclamation-triangle me-2"></i>
                            <span id="flowUnlinkModalTitle">Confirm Unlink</span>
                        </h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                        <div class="d-flex align-items-start gap-3">
                            <div class="fs-2 text-warning">
                                <i class="fas fa-unlink"></i>
                            </div>
                            <p class="mb-0" id="flowUnlinkModalMessage">Are you sure?</p>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="button" class="btn btn-danger" id="flowUnlinkConfirmBtn">
                            <i class="fas fa-unlink me-1"></i>Unlink
                        </button>
                    </div>
                </div>
            </div>
        </div>
    
    <?= $message ?>

    <!-- Main Content Area -->
    <div class="row justify-content-center">
        <div class="col-12">
    
    <!-- Heat Settings -->
    <?php if ($selected_event_id && $heats_total): ?>

      <!-- Configure Heats Section -->
    <div class="shadow-none card p-4 shadow-sm mb-4">
        <div class="d-flex flex-wrap justify-content-between align-items-center gap-3 mb-3">
            <div class="d-flex flex-align-items-center gap-3 flex-row">
                <h5 class="mb-0">
                    <i class="fas fa-cogs me-2 text-success"></i>
                    Configure <?= $heats_total ?> Heat<?= $heats_total > 1 ? 's' : '' ?>
                </h5>
                <button type="button" class="btn btn-outline-success" onclick="addNewHeat()" id="addHeatBtn">
                    <i class="fas fa-plus me-1"></i>Add Heat
                </button>
            </div>
                <div class="d-flex flex-wrap align-items-center gap-3">
                    <div class="d-flex align-items-center gap-2">
                        <small class="text-muted">Auto-refresh:</small>
                        <select class="form-select form-select-sm" id="refreshInterval" style="width: auto;" onchange="setRefreshInterval()">
                            <option value="0">Manual</option>
                            <option value="5">5 seconds</option>
                            <option value="10">10 seconds</option>
                            <option value="30" selected>30 seconds</option>
                        </select>
                        <button type="button" class="btn btn-sm btn-outline-primary" onclick="loadHeatCards()" title="Refresh now">
                            <i class="fas fa-sync-alt me-1"></i>Refresh
                        </button>
                        <small class="text-muted" id="lastRefreshTime"></small>
                    </div>
                    <div class="d-flex align-items-center gap-2">
                        <small class="text-muted">Heat cards:</small>
                        <!-- Heat Card Layout Selector -->
                        <div class="btn-group btn-group-sm flex-wrap heat-layout-selector" role="group" aria-label="Heat card layout selector">
                            <button type="button" class="btn btn-outline-secondary" data-heat-layout="full" onclick="setHeatCardLayoutPreference('full')">Full</button>
                            <button type="button" class="btn btn-outline-secondary" data-heat-layout="compact" onclick="setHeatCardLayoutPreference('compact')">Compact</button>
                            <button type="button" class="btn btn-outline-secondary" data-heat-layout="monitor" onclick="setHeatCardLayoutPreference('monitor')">Monitor</button>
                            <button type="button" class="btn btn-outline-secondary" data-heat-layout="monitor-row" onclick="setHeatCardLayoutPreference('monitor-row')">Monitor Row</button>
                            <button type="button" class="btn btn-outline-secondary" data-heat-layout="minimal" onclick="setHeatCardLayoutPreference('minimal')">Minimal</button>
                            <button type="button" class="btn btn-outline-secondary" data-heat-layout="judges" onclick="setHeatCardLayoutPreference('judges')">Judges</button>
                        </div>
                    </div>
                </div>
        </div>
        
        <?php 
        // Create an array to sort heats by flow order
        $heat_order = [];
        $processed_for_order = [];
        
        // First, add all starting heats (no incoming flow)
        for ($h = 1; $h <= $heats_total; $h++) {
            $setting = $heat_settings[$h] ?? [];
            $has_incoming = false;
            
            // Check if any other heat flows to this one
            foreach ($heat_settings as $other_setting) {
                if (($other_setting['flow_source_heat'] ?? null) == $h) {
                    $has_incoming = true;
                    break;
                }
            }
            
            // If no incoming flow and not a flow destination, it's a starting heat
            if (($setting['flow_type'] ?? 'none') === 'none' && !$has_incoming) {
                $heat_order[] = $h;
                $processed_for_order[] = $h;
            }
        }
        
        // Then add heats with flow connections in order
        $remaining_iterations = $heats_total;
        while (count($processed_for_order) < $heats_total && $remaining_iterations > 0) {
            for ($h = 1; $h <= $heats_total; $h++) {
                if (in_array($h, $processed_for_order)) continue;
                
                $setting = $heat_settings[$h] ?? [];
                $source_heat = $setting['flow_source_heat'] ?? null;
                
                // If this heat flows from a heat that's already processed, add it
                if ($source_heat && in_array($source_heat, $processed_for_order)) {
                    $heat_order[] = $h;
                    $processed_for_order[] = $h;
                }
            }
            $remaining_iterations--;
        }
        
        // Add any remaining heats that couldn't be sorted
        for ($h = 1; $h <= $heats_total; $h++) {
            if (!in_array($h, $processed_for_order)) {
                $heat_order[] = $h;
            }
        }
        ?>
        
        <div class="d-flex gap-4 overflow-scroll p-3" id="heat-buttons-container" style="cursor: grab; user-select: none;">
            <!-- Heat cards will be loaded via AJAX -->
            <div class="col-12 text-center py-5">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">Loading heats...</span>
                </div>
                <p class="mt-2 text-muted">Loading heat cards...</p>
            </div>
        </div>
     
    </div>

        <!-- Heat Flow Node Editor -->
        <div class="card mb-4 shadow-sm">
            <div class="card-header text-white d-flex justify-content-between align-items-center p-2 bg-dark">
                <h5 class="mb-0">
                    <i class="fas fa-project-diagram me-2"></i>Heat Flow Builder
                </h5>
                <div class="btn-group btn-group-sm" role="group">
                    <button type="button" class="btn btn-success" onclick="saveFlowConfiguration()" id="saveFlowBtn">
                        <i class="fas fa-save me-1"></i>Save Flow
                    </button>
                    <button type="button" class="btn btn-warning" onclick="unlinkAllNodes()" title="Remove all connections">
                        <i class="fas fa-unlink me-1"></i>Unlink All
                    </button>
                    <button type="button" class="btn btn-light" onclick="resetNodeLayout()">
                        <i class="fas fa-undo me-1"></i>Reset Layout
                    </button>
                    <button type="button" class="btn btn-light" onclick="autoArrangeNodes()">
                        <i class="fas fa-magic me-1"></i>Auto Arrange
                    </button>
                </div>
            </div>
            <!-- Debug Connection Summary -->
            <div class="card-body p-3 bg-dark text-white" style="max-height: 150px; overflow-y: auto; display: none;">
                <h6 class="mb-2">
                    Connection Summary
                    <span class="badge bg-info ms-2" id="connectionCount">0</span>
                </h6>
                <div id="connectionSummary" class="small">
                    <em>No connections</em>
                </div>
            </div>
            <div class="card-body p-0 body-bg-aurora" style="min-height: 600px;">
                <style>
                    .flow-editor-container {
                        position: relative;
                        width: 100%;
                        height: 90vh;
                        overflow: hidden;
                        cursor: grab;
                    }
                    .flow-editor-canvas {
                        position: relative;
                        width: 100%;
                        height: 100%;
                        user-select: none;
                        -webkit-user-select: none;
                        -moz-user-select: none;
                        -ms-user-select: none;
                    }
                    .flow-connections-svg {
                        position: absolute;
                        top: 0;
                        left: 0;
                        width: 100%;
                        height: 100%;
                        pointer-events: none;
                        z-index: 1;
                    }
                    .flow-node {
                        opacity: .90;
                        position: absolute;
                        background: white;                       
                        border-radius: 8px;
                        padding: 0;
                        min-width: 280px;
                        max-width: 320px;
                        cursor: move;
                        z-index: 10;                        
                        transition: box-shadow 0.2s;
                        
                    }
                    .flow-node:hover {
                        box-shadow: 0 4px 16px rgba(0,0,0,0.3);
                    }
                    .flow-node.active-heat {
                        background: #d4edda;
                        border-color: #28a745;
                    }
                    .flow-node.dragging {
                        opacity: 0.8;
                        box-shadow: 0 8px 24px rgba(0,0,0,0.4);
                        cursor: grabbing;
                    }
                    .flow-node-header {
                        display: flex;
                        align-items: center;
                        justify-content: space-between;
                        margin-bottom: 8px;
                        padding-bottom: 8px;
                        border-bottom: 1px solid #dee2e6;
                    }
                    .flow-node .card-header {
                        padding: 0.75rem;
                        background-color: #f8f9fa !important;
                    }
                    .flow-node .border-top {
                        padding: 0.75rem;
                    }
                    .flow-node .flow-ports {
                        position: absolute;
                        width: 100%;
                        transform: translate(0, -50%);
                        top: 50%;
                    }
                    .flow-node-title {
                        display: flex;
                        align-items: center;
                        gap: 6px;
                        font-weight: bold;
                        font-size: 0.9rem;
                    }
                    .flow-node-badge {
                        background: #0d6efd;
                        color: white;
                        padding: 2px 8px;
                        border-radius: 4px;
                        font-size: 0.75rem;
                    }
                    .flow-node.active-heat .flow-node-badge {
                        background: #28a745;
                    }
                    .flow-node-body {
                        font-size: 0.8rem;
                        color: #6c757d;
                        margin-bottom: 12px;
                    }
                    .flow-ports {
                        display: flex;
                        justify-content: space-between;
                        align-items: center;
                        gap: 12px;
                    }
                    .flow-port {
                        position: relative;
                        width: 20px;
                        height: 20px;
                        border-radius: 50%;
                        background: #6c757d;
                        border: 2px solid white;
                        cursor: pointer;
                        transition: all 0.2s;
                    }
                    .flow-port:hover {
                        background: #0d6efd;
                        transform: scale(1.3);
                    }
                    .flow-port.active {
                        background: #28a745;
                        box-shadow: 0 0 0 4px rgba(40, 167, 69, 0.3);
                    }
                    .flow-port-in {
                        margin-left: -8px;
                        background-color: var(--ss-success);
                    }
                    .flow-port-out {
                        margin-right: -8px;
                        background-color: var(--ss-primary);
                    }
                    .flow-port-label {
                        font-size: 0.7rem;
                        color: #6c757d;
                        text-transform: uppercase;
                        font-weight: 600;
                    }
                    .connection-line {
                        fill: none;
                        stroke-width: 4;
                        stroke-linecap: round;
                    }
                    .connection-arrow {
                        fill: #40c69a;
                    }
                    .connection-line.selected {
                        stroke: url(#connectionGradientSelected);
                        stroke-width: 6;
                    }
                    .flow-connection-label {
                        font-size: 10px;
                        fill: white;
                        font-weight: bold;
                        pointer-events: none;
                    }
                    .flow-config-icon {
                        cursor: pointer;
                        transition: all 0.2s;
                    }
                    .flow-config-icon:hover {
                        transform: scale(1.2);
                    }
                    .flow-config-icon circle {
                        fill: #0d6efd;
                        stroke: white;
                        stroke-width: 2;
                    }
                    .flow-config-icon:hover circle {
                        fill: #0b5ed7;
                    }
                    .flow-config-icon text {
                        fill: white;
                        font-size: 14px;
                        font-weight: bold;
                        pointer-events: none;
                    }
                    .flow-config-modal {
                        position: fixed;
                        top: 50%;
                        left: 50%;
                        transform: translate(-50%, -50%);
                        background: white;
                        border-radius: 8px;
                        box-shadow: 0 4px 20px rgba(0,0,0,0.3);
                        z-index: 9999;
                        min-width: 500px;
                        max-width: 700px;
                        display: none;
                    }
                    .flow-config-modal.active {
                        display: block;
                    }
                    .flow-config-modal-header {
                        background: #0d6efd;
                        color: white;
                        padding: 1rem;
                        border-radius: 8px 8px 0 0;
                        cursor: move;
                        display: flex;
                        justify-content: space-between;
                        align-items: center;
                    }
                    .flow-config-modal-body {
                        padding: 1.5rem;
                        max-height: 500px;
                        overflow-y: auto;
                    }
                    .flow-config-modal-backdrop {
                        position: fixed;
                        top: 0;
                        left: 0;
                        width: 100%;
                        height: 100%;
                        background: rgba(0,0,0,0.5);
                        z-index: 9998;
                        display: none;
                    }
                    .flow-config-modal-backdrop.active {
                        display: block;
                    }
                    .flow-gear-icon, .connection-label{z-index: 5;}
                </style>
                
                <!-- Floating Zoom Control -->
                <div class="position-fixed" id="zoomControlPanel" style="bottom: 20px; right: 20px; z-index: 1050;">
                    <div class="btn-group bg-white shadow-lg rounded" role="group" aria-label="Zoom controls">
                        <button type="button" class="btn btn-outline-secondary" onclick="adjustFlowZoom(-0.1)" title="Zoom Out">
                            <i class="fas fa-search-minus"></i>
                        </button>
                        <button type="button" class="btn btn-outline-secondary" id="zoomPercentageBtn" onclick="resetFlowZoom()" title="Reset Zoom">
                            100%
                        </button>
                        <button type="button" class="btn btn-outline-secondary" onclick="adjustFlowZoom(0.1)" title="Zoom In">
                            <i class="fas fa-search-plus"></i>
                        </button>
                    </div>
                </div>
                
                <div class="flow-editor-container overflow-auto" id="flowEditorContainer">
                    <?php
                    // Build connections data for JavaScript
                    $connections = [];
                    $heat_nodes = [];
                    
                    foreach ($heat_settings as $heat) {
                        // Build node data
                        $heat_nodes[] = [
                            'heat_number' => (int)$heat['heat_number'],
                            'heat_name' => $heat['heat_name'] ?? '',
                            'is_active' => $heat['is_active'] ?? false,
                            'participant_count' => $heat_participant_counts[$heat['heat_number']] ?? 0,
                            'flow_type' => $heat['flow_type'] ?? 'none',
                            'status' => $heat['status'] ?? 'pending',
                            'runs_count' => $heat['runs_count'] ?? 1,
                            'active_run' => $heat['active_run'] ?? 0,
                            'time_start' => $heat['time_start'] ?? '',
                            'estimate_time_per_participant' => $heat['estimate_time_per_participant'] ?? 0,
                            'layout_x' => $heat['layout_x'] ?? null,
                            'layout_y' => $heat['layout_y'] ?? null,
                        ];
                        
                        // Build connections
                        if (!empty($heat['flow_source_heat'])) {
                            $decoded = json_decode($heat['flow_source_heat'], true);
                            $source_heats = is_array($decoded) ? $decoded : [$heat['flow_source_heat']];
                            $hue_offsets_map = json_decode($heat['hue_offsets'] ?? '{}', true) ?: [];
                            foreach ($source_heats as $source) {
                                $connections[] = [
                                    'from' => (int)$source,
                                    'to' => (int)$heat['heat_number'],
                                    'type' => $heat['flow_type'] ?? 'none',
                                    'count' => $heat['flow_participants_per_category'] ?? 0,
                                    'range' => $heat['flow_position_range'] ?? null,
                                    'hue_offset' => $hue_offsets_map[$source] ?? 0
                                ];
                            }
                        }
                    }
                    ?>
                    
                    <div class="flow-editor-canvas" id="flowEditorCanvas">
                        <svg class="flow-connections-svg" id="flowConnectionsSVG">
                            <defs>
                                <marker id="flowArrowhead" markerWidth="10" markerHeight="10" refX="9" refY="3" orient="auto" markerUnits="strokeWidth">
                                    <!--<polygon points="0 0, 10 3, 0 6" class="connection-arrow" />-->
                                </marker>
                                <!-- Gradient for connection lines -->
                                <linearGradient id="connectionGradient" x1="0%" y1="0%" x2="100%" y2="0%">
                                    <stop offset="0%" style="stop-color:#19a2ff;stop-opacity:1" />
                                    <stop offset="50%" style="stop-color:#B0F761;stop-opacity:1" />
                                    <stop offset="100%" style="stop-color:#39FF14;stop-opacity:1" />
                                </linearGradient>
                                <linearGradient id="connectionGradientSelected" x1="0%" y1="0%" x2="100%" y2="0%">
                                    <stop offset="0%" style="stop-color:#19a2ff;stop-opacity:1" />
                                    <stop offset="50%" style="stop-color:#B0F761;stop-opacity:1" />
                                    <stop offset="100%" style="stop-color:#39FF14;stop-opacity:1" />
                                </linearGradient>
                            </defs>
                        </svg>
                        
                        <!-- Gear icons overlay container (DOM-based for clickability) -->
                        <div id="gearIconsContainer" style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; pointer-events: none;"></div>
                        
                        <!-- Nodes will be dynamically created here -->
                    </div>
                    
                    <!-- Flow Configuration Modal -->
                    <div class="flow-config-modal-backdrop" id="flowConfigModalBackdrop"></div>
                    <div class="flow-config-modal" id="flowConfigModal">
                        <div class="flow-config-modal-header" id="flowConfigModalHeader">
                            <h6 class="mb-0">
                                <i class="fas fa-sitemap me-2"></i>Flow Configuration
                            </h6>
                            <button type="button" class="btn-close btn-close-white" onclick="closeFlowConfigModal()"></button>
                        </div>
                        <div class="flow-config-modal-body" id="flowConfigModalBody">
                            <!-- Content will be dynamically loaded -->
                        </div>
                    </div>
                    
                    <script>
                    document.addEventListener('DOMContentLoaded', function() {
                        const heatNodesData = <?= json_encode($heat_nodes) ?>;
                        const connectionsData = <?= json_encode($connections) ?>;
                        
                        // Zoom Control Functions
                        let currentZoom = parseFloat(localStorage.getItem('heatFlowZoom_<?= $selected_event_id ?>')) || 1.0;
                        
                        function applyZoom(zoom) {
                            currentZoom = Math.max(0.3, Math.min(2.0, zoom)); // Limit between 30% and 200%
                            
                            const flowContainer = document.getElementById('flowEditorContainer');
                            const heatButtons = document.getElementById('heat-buttons-container');
                            const zoomBtn = document.getElementById('zoomPercentageBtn');
                            
                            if (flowContainer) {
                                flowContainer.style.zoom = currentZoom;
                            }
                            if (heatButtons) {
                                heatButtons.style.zoom = currentZoom;
                            }
                            if (zoomBtn) {
                                zoomBtn.textContent = Math.round(currentZoom * 100) + '%';
                            }
                            
                            // Save to localStorage
                            localStorage.setItem('heatFlowZoom_<?= $selected_event_id ?>', currentZoom);
                        }
                        
                        window.adjustFlowZoom = function(delta) {
                            applyZoom(currentZoom + delta);
                        };
                        
                        window.resetFlowZoom = function() {
                            applyZoom(1.0);
                        };
                        
                        // Apply saved zoom on load
                        applyZoom(currentZoom);
                        
                        // Flow Editor State
                        const flowEditor = {
                            nodes: new Map(),
                            connections: [],
                            selectedPort: null,
                            draggedNode: null,
                            dragOffset: {x: 0, y: 0},
                            canvas: document.getElementById('flowEditorCanvas'),
                            svg: document.getElementById('flowConnectionsSVG'),
                            gearContainer: document.getElementById('gearIconsContainer'),
                            container: document.getElementById('flowEditorContainer'),
                            configModal: document.getElementById('flowConfigModal'),
                            configModalBackdrop: document.getElementById('flowConfigModalBackdrop'),
                            configModalHeader: document.getElementById('flowConfigModalHeader'),
                            isDraggingModal: false,
                            modalDragOffset: {x: 0, y: 0},
                            handlePortClick: null // Will be set below
                        };
                        
                        // Expose flowEditor globally for refresh function
                        window.flowEditor = flowEditor;
                        
                        // Initialize node positions from database first, then localStorage, or auto-arrange
                        function loadNodePositions() {
                            // First priority: Check if heatNodesData has database positions
                            const dbPositions = {};
                            let hasDbPositions = false;
                            heatNodesData.forEach(node => {
                                if (node.layout_x !== null && node.layout_y !== null) {
                                    dbPositions[node.heat_number] = {x: node.layout_x, y: node.layout_y};
                                    hasDbPositions = true;
                                }
                            });
                            if (hasDbPositions) {
                                return dbPositions;
                            }
                            
                            // Second priority: localStorage
                            const saved = localStorage.getItem('heatFlowNodePositions_<?= $selected_event_id ?>');
                            if (saved) {
                                return JSON.parse(saved);
                            }
                            return null;
                        }
                        
                        function saveNodePositions() {
                            const positions = {};
                            flowEditor.nodes.forEach((data, heatNum) => {
                                positions[heatNum] = {x: data.x, y: data.y};
                            });
                            localStorage.setItem('heatFlowNodePositions_<?= $selected_event_id ?>', JSON.stringify(positions));
                        }
                        
                        // Create node element
                        function createNodeElement(nodeData) {
                            const node = document.createElement('div');
                            node.className = 'flow-node border shadow-lg' + (nodeData.is_active ? ' active-heat' : '');
                            node.dataset.heatNumber = nodeData.heat_number;
                            
                            node.innerHTML = buildNodeHTML(nodeData);
                            
                            return node;
                        }
                        
                        // Build node HTML from data (shared template for create and update)
                        function buildNodeHTML(nodeData) {
                            // Calculate estimated end time and duration
                            let timeHtml = '';
                            if (nodeData.time_start) {
                                const startTime = nodeData.time_start;
                                const estimateMinutes = nodeData.participant_count * (nodeData.estimate_time_per_participant || 0);
                                const estimateSeconds = estimateMinutes * 60;
                                
                                // Calculate end time
                                const startParts = startTime.split(':');
                                const startDate = new Date();
                                startDate.setHours(parseInt(startParts[0]), parseInt(startParts[1]), parseInt(startParts[2] || 0));
                                const endDate = new Date(startDate.getTime() + estimateSeconds * 1000);
                                
                                const endTime = endDate.getHours().toString().padStart(2, '0') + ':' + 
                                               endDate.getMinutes().toString().padStart(2, '0');
                                const duration = Math.round(estimateMinutes) + 'min';
                                
                                timeHtml = ''
                            }
                            
                            // Status badge
                            let statusBadge = '';
                            let statusIcon = '';
                            let statusClass = 'bg-secondary';
                            
                            if (nodeData.is_active) {
                                statusIcon = 'fa-play';
                                statusClass = 'bg-success';
                                statusBadge = 'ACTIVE';
                            } else if (nodeData.status === 'completed' || nodeData.status === 'finished') {
                                statusIcon = 'fa-check';
                                statusClass = 'bg-info';
                                statusBadge = 'COMPLETED';
                            } else if (nodeData.status === 'in_progress') {
                                statusIcon = 'fa-hourglass-half';
                                statusClass = 'bg-warning';
                                statusBadge = 'IN PROGRESS';
                            } else {
                                statusIcon = 'fa-pause';
                                statusClass = 'bg-secondary';
                                statusBadge = 'PENDING';
                            }
                            
                            return '<div class="card-header bg-light border-0 p-3">' +
                                '<div class="d-flex justify-content-between align-items-start">' +
                                    '<div class="d-flex flex-column text-primary">' +
                                        '<h5 class="mb-1 fw-bold">' + nodeData.heat_name +  '</h5>' +
                                        (nodeData.heat_name ? '<small class="mb-0 fw-bold text-muted small"><i class="fas fa-fire me-2 text-muted"></i>Heat ' + nodeData.heat_number + '</small>' : '') +
                                    '</div>' +
                                    '<div class="d-flex flex-column">' +
                                        '<div class="row g-2">' +
                                            '<div class="d-flex flex-column gap-1 align-items-end">' +
                                                '<small class="badge small ' + statusClass + '">' +
                                                    '<i class="fas ' + statusIcon + ' me-1"></i>' + statusBadge +
                                                '</small>' +
                                            '</div>' +
                                        '</div>' +
                                        '<div class="row g-1">' +
                                            '<div class="col-6">' +
                                                '<div class="text-center p-1 bg-light bg-opacity-25 rounded d-flex flex-row align-items-center justify-content-center gap-1 small" title="Athletes">' +
                                                    '<i class="fas fa-users text-primary mb-1"></i>' +
                                                    '<div class="fw-bold">' + nodeData.participant_count + '</div>' +
                                                '</div>' +
                                            '</div>' +
                                            '<div class="col-6">' +
                                                '<div class="text-center p-1 bg-light bg-opacity-25 rounded d-flex flex-row align-items-center justify-content-center gap-1 small" title="Runs">' +
                                                    '<i class="fas fa-redo text-info mb-1"></i>' +
                                                    '<div class="fw-bold">' + nodeData.runs_count + '</div>' +
                                                '</div>' +
                                            '</div>' +
                                        '</div>' +
                                    '</div>' +
                                '</div>' +
                                '<small class="fw-bold text-primary-emphasis"><i class="fas fa-play-circle me-1 small"></i>Current Run</small> ' +
                                '<span class="badge small ' + (nodeData.is_active ? 'bg-success' : 'bg-secondary') + '">' + 
                                    nodeData.active_run + '/' + nodeData.runs_count + 
                                '</span>' +
                            '</div>' +
                            timeHtml +
                            '<div class="flow-ports">' +
                                '<div class="text-center">' +
                                    '<div class="flow-port flow-port-in" data-port="in" data-heat="' + nodeData.heat_number + '"></div>' +
                                    '' +
                                '</div>' +
                                '<div class="text-center">' +
                                    '<div class="flow-port flow-port-out" data-port="out" data-heat="' + nodeData.heat_number + '"></div>' +
                                    '' +
                                '</div>' +
                            '</div>' +
                            '<div class="position-absolute bottom-0 end-0 m-2" style="z-index: 10;">' +
                                '<button class="btn btn-sm btn-light border-0 shadow-sm rounded-circle p-1 " ' +
                                    'onclick="event.stopPropagation(); openHeatModal(' + nodeData.heat_number + ')" ' +
                                    'title="Configure heat settings" ' +
                                    'style="width: 32px; height: 32px; display: flex; align-items: center; justify-content: center;">' +
                                    '<i class="fas fa-cog text-primary"></i>' +
                                '</button>' +
                            '</div>';
                        }
                        
                        // Initialize nodes
                        function initializeNodes() {
                            const savedPositions = loadNodePositions();
                            
                            heatNodesData.forEach((nodeData, index) => {
                                const nodeElement = createNodeElement(nodeData);
                                
                                // Set position
                                if (savedPositions && savedPositions[nodeData.heat_number]) {
                                    nodeData.x = savedPositions[nodeData.heat_number].x;
                                    nodeData.y = savedPositions[nodeData.heat_number].y;
                                } else {
                                    // Auto-arrange in grid
                                    const col = index % 3;
                                    const row = Math.floor(index / 3);
                                    nodeData.x = 50 + col * 360;
                                    nodeData.y = 50 + row * 280;
                                }
                                
                                nodeElement.style.left = nodeData.x + 'px';
                                nodeElement.style.top = nodeData.y + 'px';
                                
                                flowEditor.canvas.appendChild(nodeElement);
                                flowEditor.nodes.set(nodeData.heat_number, {
                                    element: nodeElement,
                                    data: nodeData,
                                    x: nodeData.x,
                                    y: nodeData.y
                                });
                                
                                // Add drag handlers (mouse and touch)
                                nodeElement.addEventListener('mousedown', startDragNode);
                                nodeElement.addEventListener('touchstart', startTouchDragNode);
                                
                                // Add port click handlers
                                const ports = nodeElement.querySelectorAll('.flow-port');
                                ports.forEach(port => {
                                    port.addEventListener('click', handlePortClick);
                                });
                            });
                            
                            // Load existing connections
                            flowEditor.connections = connectionsData.map(conn => ({
                                from: conn.from,
                                to: conn.to,
                                type: conn.type,
                                count: conn.count,
                                range: conn.range,
                                hue_offset: conn.hue_offset || 0
                            }));
                            
                            //console.log('Loaded connections with hue_offset:', flowEditor.connections);
                            renderConnections();
                        }
                        
                        // Node dragging
                        function startDragNode(e) {
                            if (e.target.classList.contains('flow-port')) return;
                            
                            const node = e.currentTarget;
                            const heatNum = parseInt(node.dataset.heatNumber);
                            const nodeData = flowEditor.nodes.get(heatNum);
                            
                            flowEditor.draggedNode = nodeData;
                            const rect = node.getBoundingClientRect();
                            const containerRect = flowEditor.container.getBoundingClientRect();
                            
                            flowEditor.dragOffset = {
                                x: e.clientX - rect.left,
                                y: e.clientY - rect.top
                            };
                            
                            node.classList.add('dragging');
                            
                            document.addEventListener('mousemove', dragNode);
                            document.addEventListener('mouseup', stopDragNode);
                        }
                        
                        function dragNode(e) {
                            if (!flowEditor.draggedNode) return;
                            
                            const containerRect = flowEditor.container.getBoundingClientRect();
                            const x = e.clientX - containerRect.left - flowEditor.dragOffset.x;
                            const y = e.clientY - containerRect.top - flowEditor.dragOffset.y;
                            
                            flowEditor.draggedNode.x = Math.max(0, x);
                            flowEditor.draggedNode.y = Math.max(0, y);
                            
                            flowEditor.draggedNode.element.style.left = flowEditor.draggedNode.x + 'px';
                            flowEditor.draggedNode.element.style.top = flowEditor.draggedNode.y + 'px';
                            
                            renderConnections();
                        }
                        
                        function stopDragNode(e) {
                            if (flowEditor.draggedNode) {
                                flowEditor.draggedNode.element.classList.remove('dragging');
                                saveNodePositions();
                                flowEditor.draggedNode = null;
                            }
                            
                            document.removeEventListener('mousemove', dragNode);
                            document.removeEventListener('mouseup', stopDragNode);
                        }
                        
                        // Touch drag support (2 second hold to activate)
                        let touchHoldTimer = null;
                        let touchStartPos = null;
                        
                        function startTouchDragNode(e) {
                            if (e.target.classList.contains('flow-port')) return;
                            
                            const node = e.currentTarget;
                            const touch = e.touches[0];
                            
                            touchStartPos = { x: touch.clientX, y: touch.clientY };
                            
                            // Visual feedback - add pulsing effect
                            node.style.transition = 'transform 0.5s ease-in-out';
                            let pulseInterval = setInterval(() => {
                                if (touchHoldTimer) {
                                    node.style.transform = node.style.transform === 'scale(1.05)' ? 'scale(1)' : 'scale(1.05)';
                                }
                            }, 250);
                            
                            // Wait 2 seconds before allowing drag
                            touchHoldTimer = setTimeout(() => {
                                clearInterval(pulseInterval);
                                node.style.transition = '';
                                node.style.transform = 'scale(1)';
                                
                                const heatNum = parseInt(node.dataset.heatNumber);
                                const nodeData = flowEditor.nodes.get(heatNum);
                                
                                flowEditor.draggedNode = nodeData;
                                const rect = node.getBoundingClientRect();
                                const containerRect = flowEditor.container.getBoundingClientRect();
                                
                                flowEditor.dragOffset = {
                                    x: touch.clientX - rect.left,
                                    y: touch.clientY - rect.top
                                };
                                
                                node.classList.add('dragging');
                                
                                document.addEventListener('touchmove', dragNodeTouch, { passive: false });
                                document.addEventListener('touchend', stopDragNodeTouch);
                            }, 2000);
                            
                            // Cancel hold timer if touch moves or ends early
                            const cancelHold = () => {
                                if (touchHoldTimer) {
                                    clearTimeout(touchHoldTimer);
                                    clearInterval(pulseInterval);
                                    touchHoldTimer = null;
                                    node.style.transition = '';
                                    node.style.transform = 'scale(1)';
                                }
                            };
                            
                            const checkTouchMove = (moveEvent) => {
                                const touch = moveEvent.touches[0];
                                const dx = Math.abs(touch.clientX - touchStartPos.x);
                                const dy = Math.abs(touch.clientY - touchStartPos.y);
                                if (dx > 10 || dy > 10) {
                                    cancelHold();
                                    node.removeEventListener('touchmove', checkTouchMove);
                                }
                            };
                            
                            node.addEventListener('touchmove', checkTouchMove, { once: false });
                            node.addEventListener('touchend', () => {
                                cancelHold();
                                node.removeEventListener('touchmove', checkTouchMove);
                            }, { once: true });
                        }
                        
                        function dragNodeTouch(e) {
                            if (!flowEditor.draggedNode) return;
                            e.preventDefault(); // Prevent scrolling while dragging
                            
                            const touch = e.touches[0];
                            const containerRect = flowEditor.container.getBoundingClientRect();
                            const x = touch.clientX - containerRect.left - flowEditor.dragOffset.x;
                            const y = touch.clientY - containerRect.top - flowEditor.dragOffset.y;
                            
                            flowEditor.draggedNode.x = Math.max(0, x);
                            flowEditor.draggedNode.y = Math.max(0, y);
                            
                            flowEditor.draggedNode.element.style.left = flowEditor.draggedNode.x + 'px';
                            flowEditor.draggedNode.element.style.top = flowEditor.draggedNode.y + 'px';
                            
                            renderConnections();
                        }
                        
                        function stopDragNodeTouch(e) {
                            if (touchHoldTimer) {
                                clearTimeout(touchHoldTimer);
                                touchHoldTimer = null;
                            }
                            
                            if (flowEditor.draggedNode) {
                                flowEditor.draggedNode.element.classList.remove('dragging');
                                saveNodePositions();
                                flowEditor.draggedNode = null;
                            }
                            
                            document.removeEventListener('touchmove', dragNodeTouch);
                            document.removeEventListener('touchend', stopDragNodeTouch);
                        }
                        
                        // Port connection handling
                        function handlePortClick(e) {
                            e.stopPropagation();
                            const port = e.currentTarget;
                            const portType = port.dataset.port;
                            const heatNum = parseInt(port.dataset.heat);
                            
                            if (!flowEditor.selectedPort) {
                                // First port selected
                                flowEditor.selectedPort = {type: portType, heat: heatNum, element: port};
                                port.classList.add('active');
                            } else {
                                // Second port selected - create connection
                                const firstPort = flowEditor.selectedPort;
                                
                                // Validate connection (out -> in only)
                                if (firstPort.type === 'out' && portType === 'in' && firstPort.heat !== heatNum) {
                                    // Check if connection already exists
                                    const existingIdx = flowEditor.connections.findIndex(c => 
                                        c.from === firstPort.heat && c.to === heatNum
                                    );
                                    
                                    if (existingIdx >= 0) {
                                        // Remove existing connection
                                        flowEditor.connections.splice(existingIdx, 1);
                                        // Auto-save when disconnecting
                                        syncConnectionsToForms();
                                        showSaveNotification('Connection removed. Click "Save Flow" to persist changes.');
                                    } else {
                                        // Add new connection - open flow config modal
                                        openFlowConfigModal(firstPort.heat, heatNum);
                                    }
                                } else if (firstPort.type === 'in' && portType === 'out' && firstPort.heat !== heatNum) {
                                    // Reverse direction
                                    const existingIdx = flowEditor.connections.findIndex(c => 
                                        c.from === heatNum && c.to === firstPort.heat
                                    );
                                    
                                    if (existingIdx >= 0) {
                                        flowEditor.connections.splice(existingIdx, 1);
                                        // Auto-sync when disconnecting
                                        syncConnectionsToForms();
                                        showSaveNotification('Connection removed. Click "Save Flow" to persist changes.');
                                    } else {
                                        openFlowConfigModal(heatNum, firstPort.heat);
                                    }
                                }
                                
                                // Clear selection
                                firstPort.element.classList.remove('active');
                                flowEditor.selectedPort = null;
                                
                                renderConnections();
                                syncConnectionsToForms();
                            }
                        }
                        
                        // Render SVG connections
                        function renderConnections() {
                            // Clear existing paths and gear icons
                            const existingPaths = flowEditor.svg.querySelectorAll('path, text, g');
                            existingPaths.forEach(p => p.remove());
                            
                            // Clear DOM-based gear icons
                            flowEditor.gearContainer.innerHTML = '';
                            
                            flowEditor.connections.forEach(conn => {
                                const fromNode = flowEditor.nodes.get(conn.from);
                                const toNode = flowEditor.nodes.get(conn.to);
                                
                                if (!fromNode || !toNode) return;
                                
                                // Calculate port positions
                                const fromX = fromNode.x + fromNode.element.offsetWidth;
                                const fromY = fromNode.y + fromNode.element.offsetHeight / 2;
                                const toX = toNode.x;
                                const toY = toNode.y + toNode.element.offsetHeight / 2;
                                
                                // Calculate control points for smooth curve
                                // Start control point extends to the right of source node
                                // End control point extends to the left of target node
                                const horizontalDistance = Math.abs(toX - fromX);
                                const curveOffset = Math.min(horizontalDistance / 2, 150);
                                
                                const controlPoint1X = fromX + curveOffset;
                                const controlPoint1Y = fromY;
                                const controlPoint2X = toX - curveOffset;
                                const controlPoint2Y = toY;
                                
                                // Draw curved path with proper direction
                                const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
                                const d = 'M ' + fromX + ' ' + fromY + ' C ' + controlPoint1X + ' ' + controlPoint1Y + ', ' + controlPoint2X + ' ' + controlPoint2Y + ', ' + toX + ' ' + toY;
                                
                                path.setAttribute('class', 'connection-line');
                                path.setAttribute('d', d);
                                path.setAttribute('marker-end', 'url(#flowArrowhead)');
                                
                                // Apply hue offset to connection color (always create gradient)
                                const hueOffset = conn.hue_offset || 0;
                                ////console.log('Rendering connection', conn.from, '->', conn.to, 'with hue_offset:', hueOffset);
                                const color1 = 'hsl(' + ((180 + hueOffset) % 360) + ', 100%, 50%)';
                                const color2 = 'hsl(' + ((120 + hueOffset) % 360) + ', 90%, 70%)';
                                const color3 = 'hsl(' + ((90 + hueOffset) % 360) + ', 100%, 50%)';
                                const gradientId = 'connectionGradient_' + conn.from + '_' + conn.to;
                                
                                // Create unique gradient for this connection
                                const gradient = document.createElementNS('http://www.w3.org/2000/svg', 'linearGradient');
                                gradient.setAttribute('id', gradientId);
                                gradient.setAttribute('x1', '0%');
                                gradient.setAttribute('y1', '0%');
                                gradient.setAttribute('x2', '100%');
                                gradient.setAttribute('y2', '0%');
                                
                                const stop1 = document.createElementNS('http://www.w3.org/2000/svg', 'stop');
                                stop1.setAttribute('offset', '0%');
                                stop1.setAttribute('style', 'stop-color:' + color1 + ';stop-opacity:1');
                                
                                const stop2 = document.createElementNS('http://www.w3.org/2000/svg', 'stop');
                                stop2.setAttribute('offset', '50%');
                                stop2.setAttribute('style', 'stop-color:' + color2 + ';stop-opacity:1');
                                
                                const stop3 = document.createElementNS('http://www.w3.org/2000/svg', 'stop');
                                stop3.setAttribute('offset', '100%');
                                stop3.setAttribute('style', 'stop-color:' + color3 + ';stop-opacity:1');
                                
                                gradient.appendChild(stop1);
                                gradient.appendChild(stop2);
                                gradient.appendChild(stop3);
                                
                                flowEditor.svg.querySelector('defs').appendChild(gradient);
                                path.setAttribute('stroke', 'url(#' + gradientId + ')');
                                
                                flowEditor.svg.appendChild(path);
                                
                                // Calculate midpoint for icons
                                const midX = (fromX + toX ) / 2;
                                const midY = (fromY + toY ) / 2;
                                
                                // Calculate color based on hue offset
                                const badgeHue = (210 + hueOffset) % 360;
                                const badgeColor = 'hsl(' + badgeHue + ', 70%, 50%)';
                                const gearHue = (200 + hueOffset) % 360;
                                const gearColor = 'hsl(' + gearHue + ', 40%, 45%)';
                                const gearHoverColor = 'hsl(' + gearHue + ', 40%, 35%)';
                                const promoteHue = (140 + hueOffset) % 360;
                                const promoteColor = 'hsl(' + promoteHue + ', 60%, 45%)';
                                const promoteHoverColor = 'hsl(' + promoteHue + ', 60%, 35%)';
                                
                                // Add DOM-based label/badge for flow type and count
                                if (conn.type && conn.type !== 'none') {
                                    const badge = document.createElement('div');
                                    badge.className = 'badge connection-label';
                                    badge.style.position = 'absolute';
                                    badge.style.left = (midX - 40) + 'px';
                                    badge.style.top = (midY - 40) + 'px';
                                    badge.style.pointerEvents = 'none';
                                    badge.style.fontSize = '11px';
                                    badge.style.whiteSpace = 'nowrap';
                                    badge.style.backgroundColor = badgeColor;
                                    badge.style.color = 'white';
                                    badge.style.padding = '4px 8px';
                                    badge.style.opacity = '0.8';
                                    badge.style.borderRadius = '4px';
                                    badge.textContent = conn.type + (conn.count ? ' (' + conn.count + ')' : '');
                                    
                                    flowEditor.gearContainer.appendChild(badge);
                                }
                                
                                // Add DOM-based gear icon (clickable settings button)
                                const gearIcon = document.createElement('div');
                                gearIcon.className = 'flow-gear-icon flow-configuration-icon';
                                gearIcon.style.position = 'absolute';
                                gearIcon.style.left = (midX - 15) + 'px';
                                gearIcon.style.top = (midY - 15) + 'px';
                                gearIcon.style.width = '30px';
                                gearIcon.style.height = '30px';
                                gearIcon.style.borderRadius = '50%';
                                gearIcon.style.backgroundColor = gearColor;
                                gearIcon.style.color = 'white';
                                gearIcon.style.display = 'flex';
                                gearIcon.style.alignItems = 'center';
                                gearIcon.style.justifyContent = 'center';
                                gearIcon.style.cursor = 'pointer';
                                gearIcon.style.fontSize = '16px';
                                gearIcon.style.pointerEvents = 'auto';
                                gearIcon.style.opacity = '0.7';
                                gearIcon.style.transition = 'all 0.2s';
                                gearIcon.style.boxShadow = '0 2px 4px rgba(0,0,0,0.2)';
                                gearIcon.innerHTML = '<i class=\"fas fa-cog\"></i>';
                                gearIcon.title = 'Configure flow settings';
                                
                                gearIcon.addEventListener('mouseenter', function() {
                                    this.style.backgroundColor = gearHoverColor;
                                    this.style.transform = 'scale(1.1)';
                                });
                                
                                gearIcon.addEventListener('mouseleave', function() {
                                    this.style.backgroundColor = gearColor;
                                    this.style.transform = 'scale(1)';
                                });
                                
                                gearIcon.addEventListener('click', function(e) {
                                    //console.log('Gear icon clicked for connection:', conn.from, '->', conn.to);
                                    e.stopPropagation();
                                    e.preventDefault();
                                    window.openFlowConfigModalForConnection(conn.from, conn.to);
                                });
                                
                                flowEditor.gearContainer.appendChild(gearIcon);
                                
                                // Add promote icon next to gear icon
                                const promoteIcon = document.createElement('div');
                                promoteIcon.className = 'flow-gear-icon flow-promote-icon';
                                promoteIcon.style.position = 'absolute';
                                promoteIcon.style.left = (midX + 20) + 'px';
                                promoteIcon.style.top = ((fromY + toY) / 2 - 15) + 'px';
                                promoteIcon.style.width = '20px';
                                promoteIcon.style.height = '20px';
                                promoteIcon.style.borderRadius = '50%';
                                promoteIcon.style.backgroundColor = promoteColor;
                                promoteIcon.style.color = 'white';
                                promoteIcon.style.display = 'flex';
                                promoteIcon.style.alignItems = 'center';
                                promoteIcon.style.justifyContent = 'center';
                                promoteIcon.style.cursor = 'pointer';
                                promoteIcon.style.fontSize = '12px';
                                promoteIcon.style.pointerEvents = 'auto';
                                promoteIcon.style.transition = 'all 0.2s';
                                promoteIcon.style.boxShadow = '0 2px 4px rgba(0,0,0,0.2)';
                                promoteIcon.innerHTML = '<i class=\"fas fa-arrow-up\"></i>';
                                promoteIcon.title = 'Promote participants to this heat';
                                
                                promoteIcon.addEventListener('mouseenter', function() {
                                    this.style.backgroundColor = promoteHoverColor;
                                    this.style.transform = 'scale(1.1)';
                                });
                                
                                promoteIcon.addEventListener('mouseleave', function() {
                                    this.style.backgroundColor = promoteColor;
                                    this.style.transform = 'scale(1)';
                                });
                                
                                promoteIcon.addEventListener('click', function(e) {
                                    //console.log('Promote icon clicked for connection:', conn.from, '->', conn.to);
                                    e.stopPropagation();
                                    e.preventDefault();
                                    window.confirmAndPromoteParticipants(conn.from, conn.to, conn);
                                });
                                
                                flowEditor.gearContainer.appendChild(promoteIcon);
                            });
                            
                            // Update debug summary
                            updateConnectionSummary();
                        }
                        
                        // Update connection summary debug box
                        function updateConnectionSummary() {
                            const summaryEl = document.getElementById('connectionSummary');
                            const countEl = document.getElementById('connectionCount');
                            
                            if (!summaryEl || !countEl) return;
                            
                            countEl.textContent = flowEditor.connections.length;
                            
                            if (flowEditor.connections.length === 0) {
                                summaryEl.innerHTML = '<em>No connections</em>';
                                return;
                            }
                            
                            const html = flowEditor.connections.map((conn, idx) => {
                                const fromNode = heatNodesData.find(n => n.heat_number === conn.from);
                                const toNode = heatNodesData.find(n => n.heat_number === conn.to);
                                const fromName = fromNode ? fromNode.heat_name : '';
                                const toName = toNode ? toNode.heat_name : '';
                                
                                return '<div class="mb-1 p-1 bg-secondary bg-opacity-25 rounded">' +
                                    '<strong>#' + (idx + 1) + ':</strong> ' +
                                    'Heat ' + conn.from + (fromName ? ' (' + fromName + ')' : '') + 
                                    ' <i class="fas fa-arrow-right mx-1"></i> ' +
                                    'Heat ' + conn.to + (toName ? ' (' + toName + ')' : '') +
                                    ' | Type: <span class="badge bg-info">' + (conn.type || 'none') + '</span>' +
                                    (conn.count ? ' | Count: ' + conn.count : '') +
                                    (conn.range ? ' | Range: ' + conn.range : '') +
                                '</div>';
                            }).join('');
                            
                            summaryEl.innerHTML = html;
                        }
                        
                        // Sync connections to form inputs
                        function syncConnectionsToForms() {
                            //console.log('Syncing connections to forms. Total connections:', flowEditor.connections.length);
                            
                            // Group connections by destination heat
                            const connectionsByDest = {};
                            flowEditor.connections.forEach(conn => {
                                if (!connectionsByDest[conn.to]) {
                                    connectionsByDest[conn.to] = [];
                                }
                                connectionsByDest[conn.to].push(conn);
                            });
                            
                            //console.log('Connections by destination:', connectionsByDest);
                            
                            // Update form fields for each heat
                            Object.keys(connectionsByDest).forEach(toHeat => {
                                const conns = connectionsByDest[toHeat];
                                const sourceHeats = conns.map(c => c.from);
                                
                                // Update the multi-select field in the modal
                                const sourceSelect = document.querySelector('select[name="flow_source_heat[' + toHeat + '][]"]');
                                if (sourceSelect) {
                                    // Clear all selections first
                                    Array.from(sourceSelect.options).forEach(opt => opt.selected = false);
                                    
                                    // Select the connected source heats
                                    sourceHeats.forEach(sourceHeat => {
                                        Array.from(sourceSelect.options).forEach(opt => {
                                            if (parseInt(opt.value) === sourceHeat) {
                                                opt.selected = true;
                                            }
                                        });
                                    });
                                }
                            });
                            
                            // Clear selections for heats with no connections
                            for (let h = 1; h <= <?= $heats_total ?>; h++) {
                                if (!connectionsByDest[h]) {
                                    const sourceSelect = document.querySelector('select[name="flow_source_heat[' + h + '][]"]');
                                    if (sourceSelect) {
                                        Array.from(sourceSelect.options).forEach(opt => opt.selected = false);
                                    }
                                }
                            }
                        }
                        
                        // Open flow configuration modal for connection
                        function openFlowConfigModal(fromHeat, toHeat) {
                            // Add connection temporarily
                            flowEditor.connections.push({
                                from: fromHeat,
                                to: toHeat,
                                type: 'top',
                                count: 1,
                                range: null
                            });
                            renderConnections();
                            syncConnectionsToForms();
                            
                            // Set flow type to promotion for this heat
                            const flowTypeSelect = document.querySelector('select[name="flow_type[' + toHeat + ']"]');
                            if (flowTypeSelect && flowTypeSelect.value === 'none') {
                                flowTypeSelect.value = 'top';
                                
                                // Show the flow config section
                                const flowConfigDiv = document.getElementById('flow_config_modal_' + toHeat);
                                if (flowConfigDiv) {
                                    flowConfigDiv.classList.remove('d-none');
                                }
                            }
                            
                            // Open the heat configuration modal for toHeat
                            const modalId = 'heatModal_' + toHeat;
                            const modalEl = document.getElementById(modalId);
                            if (modalEl) {
                                const modal = new bootstrap.Modal(modalEl);
                                modal.show();
                            }
                        }
                        
                        // Open floating flow config modal for editing existing connection
                        window.openFlowConfigModalForConnection = function(fromHeat, toHeat) {
                            //console.log('Opening flow config modal for:', fromHeat, '->', toHeat);
                            
                            const modal = flowEditor.configModal;
                            const backdrop = flowEditor.configModalBackdrop;
                            const body = document.getElementById('flowConfigModalBody');
                            
                            if (!modal || !backdrop || !body) {
                                console.error('Modal elements not found:', {modal, backdrop, body});
                                return;
                            }
                            
                            // Get current connection data
                            const conn = flowEditor.connections.find(c => c.from === fromHeat && c.to === toHeat);
                            const currentFlowType = conn ? conn.type : 'none';
                            const currentCount = conn ? conn.count : 0;
                            const currentRange = conn ? conn.range : '';
                            const currentHueOffset = conn ? (conn.hue_offset || 0) : 0;
                            
                            // Get available source heats (all heats except the target)
                            const sourceHeatOptions = Array.from({length: <?= $heats_total ?>}, (_, i) => i + 1)
                                .filter(h => h !== toHeat)
                                .map(h => {
                                    const nodeData = heatNodesData.find(n => n.heat_number === h);
                                    const heatName = nodeData ? nodeData.heat_name : '';
                                    const isSelected = fromHeat === h ? ' selected' : '';
                                    return '<option value="' + h + '"' + isSelected + '>Heat ' + h + 
                                           (heatName ? ' (' + heatName + ')' : '') + '</option>';
                                }).join('');
                            
                            // Build modal content
                            body.innerHTML = '<div class="col-12">' +
                                '<div class="border-0 shadow-none card">' +
                                    '<div class="border-0 shadow-none card-body">' +
                                        '<div class="mb-3">' +
                                            '<label class="form-label fw-semibold">Flow Type</label>' +
                                            '<select class="form-select" id="modalFlowType" onchange="updatePromotionPreview(' + fromHeat + ', ' + toHeat + ')">' +
                                                '<option value="none"' + (currentFlowType === 'none' ? ' selected' : '') + '>No Flow (Manual Entry)</option>' +
                                                '<option value="promotion"' + (currentFlowType === 'promotion' ? ' selected' : '') + '>Promotion from Previous Heat</option>' +
                                                '<option value="top"' + (currentFlowType === 'top' ? ' selected' : '') + '>Top Performers</option>' +
                                                '<option value="qualifying"' + (currentFlowType === 'qualifying' ? ' selected' : '') + '>Qualifying Heat (Best from Categories)</option>' +
                                            '</select>' +
                                        '</div>' +
                                        '<div class="mb-3">' +
                                            '<label class="form-label fw-semibold">Source Heat</label>' +
                                            '<select class="form-select" id="modalSourceHeat" onchange="updatePromotionPreview(' + fromHeat + ', ' + toHeat + ')">' +
                                                sourceHeatOptions +
                                            '</select>' +
                                            '<div class="form-text">' +
                                                '<i class="fas fa-info-circle me-1"></i>Heat ' + fromHeat + ' → Heat ' + toHeat +
                                            '</div>' +
                                        '</div>' +
                                        '<div class="mb-3">' +
                                            '<label class="form-label fw-semibold">Athletes to Promote per Division</label>' +
                                            '<input type="number" class="form-control" id="modalFlowCount" value="' + currentCount + '" min="0" onchange="updatePromotionPreview(' + fromHeat + ', ' + toHeat + ')">' +
                                            '<div class="form-text">' +
                                                'Number of top athletes from each division to advance to this heat (0 = disabled)' +
                                            '</div>' +
                                        '</div>' +
                                        '<div class="mb-3">' +
                                            '<label class="form-label fw-semibold">' +
                                                'Position Range <span class="badge bg-secondary ms-1">Optional</span>' +
                                            '</label>' +
                                            '<input type="text" class="form-control" id="modalFlowRange" value="' + (currentRange || '') + '" placeholder="e.g., 1-3, 4-8, 9-12" pattern="\\d+-\\d+" onchange="updatePromotionPreview(' + fromHeat + ', ' + toHeat + ')">' +
                                            '<div class="form-text">' +
                                                '<i class="fas fa-info-circle me-1"></i>' +
                                                'Specify position range from source heat (e.g., "3-4" for 3rd and 4th place, "4-15" for positions 4-15). ' +
                                                'Leave empty to use top N athletes per division.' +
                                            '</div>' +
                                        '</div>' +
                                        '<div class="mb-3">' +
                                            '<label class="form-label fw-semibold">' +
                                                'Link Color <span class="badge bg-secondary ms-1">Visual Aid</span>' +
                                            '</label>' +
                                            '<input type="range" class="form-range" id="modalHueOffset" min="0" max="300" step="50" value="' + currentHueOffset + '" oninput="this.nextElementSibling.querySelector(\'#linkColorValue\').textContent = this.value + \'°\'; var preview = this.nextElementSibling.querySelector(\'#linkColorPreview\'); preview.style.background = \'linear-gradient(90deg, hsl(\' + ((180 + parseInt(this.value)) % 360) + \', 100%, 50%), hsl(\' + ((120 + parseInt(this.value)) % 360) + \', 90%, 70%), hsl(\' + ((90 + parseInt(this.value)) % 360) + \', 100%, 50%))\';">' +
                                            '<div class="d-flex justify-content-between align-items-center mt-2">' +
                                                '<div class="form-text">' +
                                                    '<i class="fas fa-palette me-1"></i>Adjust link color to distinguish connections from same source' +
                                                '</div>' +
                                                '<div class="d-flex align-items-center gap-2">' +
                                                    '<div id="linkColorPreview" style="width: 30px; height: 30px; border-radius: 4px; border: 2px solid #ddd; background: linear-gradient(90deg, hsl(' + ((180 + currentHueOffset) % 360) + ', 100%, 50%), hsl(' + ((120 + currentHueOffset) % 360) + ', 90%, 70%), hsl(' + ((90 + currentHueOffset) % 360) + ', 100%, 50%));"></div>' +
                                                    '<span class="badge bg-secondary" id="linkColorValue">' + currentHueOffset + '°</span>' +
                                                '</div>' +
                                            '</div>' +
                                        '</div>' +
                                        
                                        '<div class="alert alert-info">' +
                                            '<i class="fas fa-info-circle me-2"></i>' +
                                            'Flow configuration will automatically populate participants based on results from the source heat.' +
                                        '</div>' +
                                        '<div class="d-flex gap-2 justify-content-between">' +
                                            '<button type="button" class="btn btn-danger" onclick="deleteFlowConnection(' + fromHeat + ', ' + toHeat + ')" title="Remove this connection">' +
                                                '<i class="fas fa-unlink me-1"></i>Unlink' +
                                            '</button>' +
                                            '<div class="d-flex gap-2">' +
                                                '<button type="button" class="btn btn-secondary" onclick="closeFlowConfigModal()">Cancel</button>' +
                                                '<button type="button" class="btn btn-primary" onclick="saveFlowConfigFromModal(' + fromHeat + ', ' + toHeat + ')">' +
                                                    '<i class="fas fa-save me-1"></i>Save Configuration' +
                                                '</button>' +
                                            '</div>' +
                                        '</div>' +
                                    '</div>' +
                                '</div>' +
                            '</div>';
                            
                            // Show modal
                            modal.classList.add('active');
                            backdrop.classList.add('active');
                            
                            // Automatically open promotion preview modal alongside
                            setTimeout(() => openPromotionPreviewModal(fromHeat, toHeat), 100);
                        }
                        
                        // Close floating modal
                        window.closeFlowConfigModal = function() {
                            flowEditor.configModal.classList.remove('active');
                            flowEditor.configModalBackdrop.classList.remove('active');
                            
                            // Also close promotion preview modal
                            closePromotionPreviewModal();
                        };
                        
                        // Confirm and promote participants function
                        window.confirmAndPromoteParticipants = async function(fromHeat, toHeat, connection) {
                            //console.log('Confirm and promote:', fromHeat, '->', toHeat);
                            
                            // Get flow configuration
                            const conn = flowEditor.connections.find(c => c.from === fromHeat && c.to === toHeat);
                            if (!conn || !conn.count || conn.count <= 0) {
                                alert('No promotion configuration found for this flow. Please configure the flow first using the gear icon.');
                                return;
                            }
                            
                            // Get promotion preview data to show count (reuse same logic as updatePromotionPreviewModal)
                            try {
                                // Get heat settings
                                const heatSettings = <?= json_encode($heat_settings) ?>;
                                const sourceHeatConfig = heatSettings[fromHeat];
                                
                                if (!sourceHeatConfig || !sourceHeatConfig.categories) {
                                    alert('No categories configured for Heat ' + fromHeat);
                                    return;
                                }
                                
                                let categoryIds = [];
                                try {
                                    if (Array.isArray(sourceHeatConfig.categories)) {
                                        categoryIds = sourceHeatConfig.categories;
                                    } else if (typeof sourceHeatConfig.categories === 'string') {
                                        categoryIds = JSON.parse(sourceHeatConfig.categories);
                                    }
                                } catch (e) {
                                    console.error('Error parsing divisions:', e);
                                    alert('Error parsing heat divisions');
                                    return;
                                }
                                
                                if (categoryIds.length === 0) {
                                    alert('No divisions found for Heat ' + fromHeat);
                                    return;
                                }
                                
                                // Fetch data for each category
                                const heatFilter = encodeURIComponent(JSON.stringify({[fromHeat]: [1,2,3,4,5]}));
                                const allPromotedParticipants = [];
                                
                                for (const categoryId of categoryIds) {
                                    const apiUrl = '../api/summary_table_api.php?event_id=<?= $selected_event_id ?>' +
                                        '&format=json&styling=none' +
                                        '&heat_run_filter=' + heatFilter +
                                        '&category=' + categoryId +
                                        '&show_rank=true&show_bib=true&show_participant=true&show_category=true' +
                                        '&show_club=false&show_gender=true' +
                                        '&show_runs=true&show_heat_best=true' +
                                        '&heat_direction=row&sort=OverallBest&sort_direction=desc&gender=all';
                                    
                                    const response = await fetch(apiUrl);
                                    if (!response.ok) continue;
                                    
                                    const responseText = await response.text();
                                    let data;
                                    try {
                                        data = JSON.parse(responseText);
                                    } catch (e) {
                                        console.error('JSON parse error for category ' + categoryId + ':', e);
                                        continue;
                                    }
                                    
                                    if (data.success && data.participants && data.participants.length > 0) {
                                        // Determine which participants to promote
                                        let promotedParticipants = [];
                                        
                                        if (conn.range) {
                                            const rangeParts = conn.range.split('-').map(s => parseInt(s.trim()));
                                            if (rangeParts.length === 2) {
                                                promotedParticipants = data.participants.filter(p => 
                                                    p.rank >= rangeParts[0] && p.rank <= rangeParts[1]
                                                );
                                            }
                                        } else {
                                            promotedParticipants = data.participants.slice(0, conn.count);
                                        }
                                        
                                        allPromotedParticipants.push(...promotedParticipants);
                                    }
                                }
                                
                                if (allPromotedParticipants.length === 0) {
                                    alert('No participants to promote. Make sure participants have been scored in Heat ' + fromHeat);
                                    return;
                                }
                                
                                // Store promotion data for modal execution
                                window.pendingPromotionData = {
                                    fromHeat: fromHeat,
                                    toHeat: toHeat,
                                    connection: conn,
                                    participants: allPromotedParticipants
                                };
                                
                                // Get heat names from heat settings (heatSettings already declared above)
                                const fromHeatName = heatSettings[fromHeat]?.heat_name || 'Unnamed Heat';
                                const toHeatName = heatSettings[toHeat]?.heat_name || 'Unnamed Heat';
                                
                                // Populate modal
                                document.getElementById('promotionParticipantCount').textContent = 
                                    allPromotedParticipants.length + ' participant' + 
                                    (allPromotedParticipants.length !== 1 ? 's' : '');
                                document.getElementById('promotionFromHeat').value = fromHeat;
                                document.getElementById('promotionFromHeatName').value = fromHeatName;
                                document.getElementById('promotionToHeat').value = toHeat;
                                document.getElementById('promotionToHeatName').value = toHeatName;
                                
                                // Show/hide position range
                                const rangeInfo = document.getElementById('promotionRangeInfo');
                                const rangeTextEl = document.getElementById('promotionRangeText');
                                if (conn.range) {
                                    rangeTextEl.textContent = conn.range;
                                    rangeInfo.style.display = 'block';
                                } else {
                                    rangeInfo.style.display = 'none';
                                }
                                
                                // Show modal
                                const modal = new bootstrap.Modal(document.getElementById('promotionConfirmModal'));
                                modal.show();
                                
                            } catch (error) {
                                console.error('Error in confirmAndPromoteParticipants:', error);
                                alert('Error: ' + error.message);
                            }
                        };
                        
                        // Execute promotion from modal (called by modal button)
                        window.executePromotionFromModal = async function() {
                            const data = window.pendingPromotionData;
                            if (!data) {
                                alert('No promotion data found');
                                return;
                            }
                            
                            // Get selected promotion order
                            const promotionOrder = document.getElementById('promotionOrder').value;
                            
                            // Close modal
                            const modal = bootstrap.Modal.getInstance(document.getElementById('promotionConfirmModal'));
                            if (modal) modal.hide();
                            
                            // Execute promotion with selected order
                            await executePromotion(data.fromHeat, data.toHeat, data.connection, data.participants, promotionOrder);
                        };
                        
                        // Execute promotion by copying participants to target heat
                        async function executePromotion(fromHeat, toHeat, connection, promotedParticipants, promotionOrder = 'normal') {
                            try {
                                // Apply order transformation if needed
                                let orderedParticipants = [...promotedParticipants];
                                if (promotionOrder === 'reverse') {
                                    orderedParticipants.reverse();
                                }
                                
                                // Collect all event_participant IDs to promote (in correct order)
                                const eventParticipantIds = [];
                                orderedParticipants.forEach(p => {
                                    // Use event_participant_id (from event_participants table)
                                    if (p.event_participant_id) {
                                        eventParticipantIds.push(p.event_participant_id);
                                    }
                                });
                                
                                if (eventParticipantIds.length === 0) {
                                    alert('No participants to promote.');
                                    return;
                                }
                                
                                // Use existing ajax_move_to_heat.php endpoint to copy participants
                                const formData = new FormData();
                                formData.append('event_id', '<?= htmlspecialchars($selected_event_id ?? '', ENT_QUOTES) ?>');
                                formData.append('target_heat', toHeat);
                                formData.append('action_type', 'copy'); // Copy participants to maintain them in source heat
                                
                                // Add each ID as array element
                                eventParticipantIds.forEach(id => {
                                    formData.append('event_participant_ids[]', id);
                                });
                                
                                const response = await fetch('ajax_move_to_heat.php', {
                                    method: 'POST',
                                    body: formData
                                });
                                
                                const result = await response.json();
                                
                                if (result.status === 'success') {
                                    // Show success modal with promotion details
                                    showPromotionResultModal(result.message, 'success');
                                } else {
                                    // Show error modal
                                    showPromotionResultModal('Error promoting participants: ' + (result.message || 'Unknown error'), 'error');
                                }
                                
                            } catch (error) {
                                console.error('Error in executePromotion:', error);
                                showPromotionResultModal('Error: ' + error.message, 'error');
                            }
                        }

                        // Show promotion result modal
                        function showPromotionResultModal(message, type) {
                            const modalId = 'promotionResultModal';
                            let modal = document.getElementById(modalId);
                            
                            // Create modal if it doesn't exist
                            if (!modal) {
                                const modalHTML = `
                                    <div class="modal fade" id="${modalId}" tabindex="-1" aria-hidden="true">
                                        <div class="modal-dialog modal-dialog-centered">
                                            <div class="modal-content">
                                                <div class="modal-header bg-${type === 'success' ? 'success' : 'danger'} text-white">
                                                    <h5 class="modal-title">
                                                        <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-circle'} me-2"></i>
                                                        ${type === 'success' ? 'Promotion Complete' : 'Promotion Error'}
                                                    </h5>
                                                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                                                </div>
                                                <div class="modal-body">
                                                    <p class="mb-0" id="${modalId}Message"></p>
                                                </div>
                                                <div class="modal-footer">
                                                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                                    ${type === 'success' ? '<button type="button" class="btn btn-success" onclick="location.reload()"><i class="fas fa-sync me-1"></i>Reload Page</button>' : ''}
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                `;
                                document.body.insertAdjacentHTML('beforeend', modalHTML);
                                modal = document.getElementById(modalId);
                            } else {
                                // Update existing modal
                                const header = modal.querySelector('.modal-header');
                                const title = modal.querySelector('.modal-title');
                                const footer = modal.querySelector('.modal-footer');
                                
                                header.className = 'modal-header bg-' + (type === 'success' ? 'success' : 'danger') + ' text-white';
                                title.innerHTML = '<i class="fas fa-' + (type === 'success' ? 'check-circle' : 'exclamation-circle') + ' me-2"></i>' +
                                                 (type === 'success' ? 'Promotion Complete' : 'Promotion Error');
                                
                                // Update footer buttons
                                if (type === 'success') {
                                    footer.innerHTML = '<button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>' +
                                                      '<button type="button" class="btn btn-success" onclick="location.reload()"><i class="fas fa-sync me-1"></i>Reload Page</button>';
                                } else {
                                    footer.innerHTML = '<button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>';
                                }
                            }
                            
                            // Set message
                            document.getElementById(modalId + 'Message').textContent = message;
                            
                            // Show modal
                            const bsModal = new bootstrap.Modal(modal);
                            bsModal.show();
                        }
                        
                        // Promotion Preview Modal Controls
                        window.openPromotionPreviewModal = function(fromHeat, toHeat) {
                            const modal = document.getElementById('promotionPreviewModal');
                            const backdrop = document.getElementById('promotionPreviewModalBackdrop');
                            const flowConfigModal = flowEditor.configModal;
                            
                            if (!modal || !backdrop) return;
                            
                            // Position modal to the right of flow config modal
                            if (flowConfigModal) {
                                const flowRect = flowConfigModal.getBoundingClientRect();
                                const spacing = 20;
                                modal.style.left = (flowRect.right + spacing) + 'px';
                                modal.style.top = flowRect.top + 'px';
                                modal.style.transform = 'none';
                            }
                            
                            // Show modal with animation
                            modal.style.display = 'block';
                            backdrop.style.display = 'block';
                            
                            setTimeout(() => {
                                modal.style.opacity = '1';
                                backdrop.style.opacity = '0.5';
                            }, 10);
                            
                            // Load preview
                            updatePromotionPreviewModal(fromHeat, toHeat);
                        };
                        
                        window.closePromotionPreviewModal = function() {
                            const modal = document.getElementById('promotionPreviewModal');
                            const backdrop = document.getElementById('promotionPreviewModalBackdrop');
                            
                            if (!modal || !backdrop) return;
                            
                            modal.style.opacity = '0';
                            backdrop.style.opacity = '0';
                            
                            setTimeout(() => {
                                modal.style.display = 'none';
                                backdrop.style.display = 'none';
                            }, 300);
                        };
                        
                        // Make Promotion Preview modal draggable
                        const previewModal = document.getElementById('promotionPreviewModal');
                        const previewModalHeader = document.getElementById('promotionPreviewModalHeader');
                        let isDraggingPreviewModal = false;
                        let previewModalDragOffset = {x: 0, y: 0};
                        
                        previewModalHeader.addEventListener('mousedown', function(e) {
                            isDraggingPreviewModal = true;
                            const rect = previewModal.getBoundingClientRect();
                            previewModalDragOffset = {
                                x: e.clientX - rect.left,
                                y: e.clientY - rect.top
                            };
                            e.preventDefault();
                        });
                        
                        document.addEventListener('mousemove', function(e) {
                            if (!isDraggingPreviewModal) return;
                            
                            const x = e.clientX - previewModalDragOffset.x;
                            const y = e.clientY - previewModalDragOffset.y;
                            
                            previewModal.style.left = x + 'px';
                            previewModal.style.top = y + 'px';
                            previewModal.style.transform = 'none';
                        });
                        
                        document.addEventListener('mouseup', function() {
                            isDraggingPreviewModal = false;
                        });
                        
                        // Save flow config from modal
                        window.saveFlowConfigFromModal = function(fromHeat, toHeat) {
                            const flowType = document.getElementById('modalFlowType').value;
                            const flowCount = parseInt(document.getElementById('modalFlowCount').value) || 0;
                            const flowRange = document.getElementById('modalFlowRange').value.trim();
                            const hueOffset = parseInt(document.getElementById('modalHueOffset').value) || 0;
                            const newSourceHeat = parseInt(document.getElementById('modalSourceHeat').value);
                            
                            //console.log('Saving flow config:', {fromHeat, toHeat, newSourceHeat, flowType, flowCount, flowRange, hueOffset});
                            
                            // Find and update the specific connection
                            const connIdx = flowEditor.connections.findIndex(c => c.from === fromHeat && c.to === toHeat);
                            if (connIdx >= 0) {
                                // Update existing connection
                                flowEditor.connections[connIdx] = {
                                    from: newSourceHeat,
                                    to: toHeat,
                                    type: flowType,
                                    count: flowCount,
                                    range: flowRange || null,
                                    hue_offset: hueOffset
                                };
                                //console.log('Updated connection at index', connIdx, flowEditor.connections[connIdx]);
                            } else {
                                // Add new connection if not found
                                flowEditor.connections.push({
                                    from: newSourceHeat,
                                    to: toHeat,
                                    type: flowType,
                                    count: flowCount,
                                    range: flowRange || null,
                                    hue_offset: hueOffset
                                });
                                //console.log('Added new connection');
                            }
                            
                            //console.log('Total connections after update:', flowEditor.connections.length);
                            
                            // Update form fields (optional, for compatibility)
                            const flowTypeSelect = document.querySelector('select[name="flow_type[' + toHeat + ']"]');
                            if (flowTypeSelect) {
                                flowTypeSelect.value = flowType;
                            }
                            
                            const flowCountInput = document.querySelector('input[name="flow_participants_per_category[' + toHeat + ']"]');
                            if (flowCountInput) {
                                flowCountInput.value = flowCount;
                            }
                            
                            const flowRangeInput = document.querySelector('input[name="flow_position_range[' + toHeat + ']"]');
                            if (flowRangeInput) {
                                flowRangeInput.value = flowRange;
                            }
                            
                            // Sync to forms and re-render
                            syncConnectionsToForms();
                            renderConnections();
                            showSaveNotification('Flow configuration updated. Click "Save Flow" to persist.');
                            closeFlowConfigModal();
                        };
                        
                        // Make modal draggable
                        flowEditor.configModalHeader.addEventListener('mousedown', function(e) {
                            flowEditor.isDraggingModal = true;
                            const rect = flowEditor.configModal.getBoundingClientRect();
                            flowEditor.modalDragOffset = {
                                x: e.clientX - rect.left,
                                y: e.clientY - rect.top
                            };
                        });
                        
                        document.addEventListener('mousemove', function(e) {
                            if (!flowEditor.isDraggingModal) return;
                            
                            const x = e.clientX - flowEditor.modalDragOffset.x;
                            const y = e.clientY - flowEditor.modalDragOffset.y;
                            
                            flowEditor.configModal.style.left = x + 'px';
                            flowEditor.configModal.style.top = y + 'px';
                            flowEditor.configModal.style.transform = 'none';
                        });
                        
                        document.addEventListener('mouseup', function() {
                            flowEditor.isDraggingModal = false;
                        });
                        
                        // Close modal on backdrop click
                        flowEditor.configModalBackdrop.addEventListener('click', closeFlowConfigModal);
                        
                        // Update promotion preview in separate modal
                        window.updatePromotionPreviewModal = async function(fromHeat, toHeat) {
                            //console.log('updatePromotionPreviewModal called:', {fromHeat, toHeat});
                            
                            const previewContent = document.getElementById('promotionPreviewModalContent');
                            if (!previewContent) {
                                console.error('promotionPreviewModalContent element not found!');
                                return;
                            }
                            
                            // Get current flow settings from connection or modal
                            const conn = flowEditor.connections.find(c => c.from === fromHeat && c.to === toHeat);
                            
                            let sourceHeat = fromHeat;
                            let flowCount = conn ? conn.count : 0;
                            let flowRange = conn ? conn.range : '';
                            let flowType = conn ? conn.type : 'top';
                            
                            // Try to get from modal if it's open
                            const sourceHeatSelect = document.getElementById('modalSourceHeat');
                            const flowCountInput = document.getElementById('modalFlowCount');
                            const flowRangeInput = document.getElementById('modalFlowRange');
                            const flowTypeSelect = document.getElementById('modalFlowType');
                            
                            if (sourceHeatSelect && sourceHeatSelect.value) {
                                sourceHeat = parseInt(sourceHeatSelect.value);
                            }
                            if (flowCountInput && flowCountInput.value) {
                                flowCount = parseInt(flowCountInput.value) || 0;
                            }
                            if (flowRangeInput) {
                                flowRange = flowRangeInput.value.trim();
                            }
                            if (flowTypeSelect && flowTypeSelect.value) {
                                flowType = flowTypeSelect.value;
                            }
                            
                            //console.log('Flow settings:', {sourceHeat, flowCount, flowRange, flowType});
                            
                            // Show loading state
                            previewContent.innerHTML = '<div class="text-center text-muted py-3">' +
                                '<i class="fas fa-spinner fa-spin me-2"></i>Loading preview...' +
                            '</div>';
                            
                            // Don't fetch if flow type is none or count is 0
                            if (flowType === 'none' || flowCount === 0) {
                                previewContent.innerHTML = '<div class="text-center text-muted py-3">' +
                                    '<i class="fas fa-info-circle me-2"></i>Configure flow settings to see promotion preview' +
                                '</div>';
                                return;
                            }
                            
                            try {
                                // STEP 1: Get categories configured for this heat from PHP heat settings
                                const heatSettings = <?= json_encode($heat_settings) ?>;
                                //console.log('Heat settings object:', heatSettings);
                                
                                // heatSettings is an object with heat_number as keys, not an array
                                const sourceHeatConfig = heatSettings[sourceHeat];
                                
                                if (!sourceHeatConfig || !sourceHeatConfig.categories) {
                                    previewContent.innerHTML = '<div class="alert alert-warning mb-0">' +
                                        '<i class="fas fa-exclamation-triangle me-2"></i>No categories configured for Heat ' + sourceHeat +
                                    '</div>';
                                    return;
                                }
                                
                                let categoryIds = [];
                                try {
                                    // Categories might already be an array if json_decoded in PHP
                                    if (Array.isArray(sourceHeatConfig.categories)) {
                                        categoryIds = sourceHeatConfig.categories;
                                    } else if (typeof sourceHeatConfig.categories === 'string') {
                                        categoryIds = JSON.parse(sourceHeatConfig.categories);
                                    }
                                } catch (e) {
                                    console.error('Error parsing divisions:', e);
                                    categoryIds = [];
                                }
                                
                                //console.log('Heat ' + sourceHeat + ' has categories:', categoryIds);
                                
                                if (categoryIds.length === 0) {
                                    previewContent.innerHTML = '<div class="alert alert-warning mb-0">' +
                                        '<i class="fas fa-exclamation-triangle me-2"></i>No divisions found for Heat ' + sourceHeat +
                                    '</div>';
                                    return;
                                }
                                
                                // STEP 2: Fetch data for EACH category separately using category_id parameter
                                const heatFilter = encodeURIComponent(JSON.stringify({[sourceHeat]: [1,2,3,4,5]}));
                                const categoryResults = [];
                                
                                for (const categoryId of categoryIds) {
                                    //console.log('Fetching data for category ID:', categoryId);
                                    
                                    const apiUrl = '../api/summary_table_api.php?event_id=<?= $selected_event_id ?>' +
                                        '&format=json&styling=none' +
                                        '&heat_run_filter=' + heatFilter +
                                        '&category=' + categoryId +  // Use category_id parameter
                                        '&show_rank=true&show_bib=true&show_participant=true&show_category=true' +
                                        '&show_club=false&show_gender=true' +
                                        '&show_runs=true&show_heat_best=true' +
                                        '&heat_direction=row&sort=OverallBest&sort_direction=desc&gender=all';
                                    
                                    //console.log('API URL for category ID ' + categoryId + ':', apiUrl);
                                    
                                    const response = await fetch(apiUrl);
                                    
                                    // Check if response is ok
                                    if (!response.ok) {
                                        console.error('HTTP error for category ' + categoryId + ':', response.status, response.statusText);
                                        continue;
                                    }
                                    
                                    // Get response text first to check for errors
                                    const responseText = await response.text();
                                    //console.log('Response text length for category ' + categoryId + ':', responseText.length);
                                    
                                    // Try to parse JSON
                                    let data;
                                    try {
                                        data = JSON.parse(responseText);
                                    } catch (e) {
                                        console.error('JSON parse error for category ' + categoryId + ':', e);
                                        console.error('Response text:', responseText.substring(0, 500));
                                        continue;
                                    }
                                    
                                    //console.log('Category ' + categoryId + ' response:', data);
                                    
                                    if (data.success && data.participants && data.participants.length > 0) {
                                        categoryResults.push({
                                            categoryId: categoryId,
                                            categoryName: data.participants[0].category || 'Category ' + categoryId,
                                            participants: data.participants
                                        });
                                    }
                                }
                                
                                //console.log('All category results:', categoryResults);
                                
                                if (categoryResults.length === 0) {
                                    previewContent.innerHTML = '<div class="alert alert-warning mb-0">' +
                                        '<i class="fas fa-exclamation-triangle me-2"></i>No scored participants found for Heat ' + sourceHeat +
                                        '<br><small>Make sure participants have been scored in this heat.</small>' +
                                    '</div>';
                                    return;
                                }
                                
                                // STEP 3: Build preview HTML for each category
                                let html = '';
                                let totalPromoted = 0;
                                
                                categoryResults.forEach(categoryData => {
                                    const participants = categoryData.participants;
                                    const categoryName = categoryData.categoryName;
                                    
                                    //console.log('Processing category:', categoryName, 'with', participants.length, 'participants');
                                    
                                    // Determine which participants to promote
                                    let promotedParticipants = [];
                                    
                                    if (flowRange) {
                                        // Use position range
                                        const rangeParts = flowRange.split('-').map(s => parseInt(s.trim()));
                                        if (rangeParts.length === 2) {
                                            const rangeStart = rangeParts[0];
                                            const rangeEnd = rangeParts[1];
                                            promotedParticipants = participants.filter(p => 
                                                p.rank >= rangeStart && p.rank <= rangeEnd
                                            );
                                            //console.log('Using position range', rangeStart, '-', rangeEnd, ':', promotedParticipants.length, 'participants');
                                        }
                                    } else {
                                        // Use top N
                                        promotedParticipants = participants.slice(0, flowCount);
                                        //console.log('Using top', flowCount, ':', promotedParticipants.length, 'participants');
                                    }
                                    
                                    totalPromoted += promotedParticipants.length;
                                    
                                    if (promotedParticipants.length > 0) {
                                        //console.log('Adding HTML for category:', categoryName, 'with', promotedParticipants.length, 'promoted');
                                        
                                        html += '<div class="mb-3">' +
                                            '<div class="fw-bold text-primary mb-2">' +
                                                '<i class="fas fa-layer-group me-1"></i>' + categoryName +
                                                ' <span class="badge bg-primary ms-1">' + promotedParticipants.length + ' promoted</span>' +
                                            '</div>' +
                                            '<div class="list-group list-group-flush">';
                                        
                                        promotedParticipants.forEach(p => {
                                            const heatBest = (p.heat_scores && p.heat_scores[sourceHeat] && p.heat_scores[sourceHeat].best != null)
                                                ? Number(p.heat_scores[sourceHeat].best) 
                                                : 0;
                                            
                                            html += '<div class="list-group-item d-flex justify-content-between align-items-center py-2 px-3">' +
                                                '<div class="d-flex align-items-center gap-2">' +
                                                    '<span class="badge bg-secondary">#' + p.rank + '</span>' +
                                                    '<span class="badge bg-info">BIB ' + p.bib_number + '</span>' +
                                                    '<span class="fw-semibold">' + p.first_name + ' ' + p.last_name + '</span>' +
                                                    (p.gender ? '<span class="badge ' + (p.gender === 'Male' ? 'bg-primary' : 'bg-danger') + ' ms-1">' + p.gender + '</span>' : '') +
                                                '</div>' +
                                                '<div>' +
                                                    '<span class="badge bg-success">' + heatBest.toFixed(2) + '</span>' +
                                                '</div>' +
                                            '</div>';
                                        });
                                        
                                        html += '</div></div>';
                                    }
                                });
                                
                                //console.log('Total HTML length:', html.length, 'Total promoted:', totalPromoted);
                                
                                if (html === '') {
                                    previewContent.innerHTML = '<div class="alert alert-info mb-0">' +
                                        '<i class="fas fa-info-circle me-2"></i>No participants match the current flow settings' +
                                    '</div>';
                                } else {
                                    previewContent.innerHTML = 
                                        '<div class="alert alert-success mb-3">' +
                                            '<i class="fas fa-check-circle me-2"></i>' +
                                            '<strong>' + totalPromoted + ' participants</strong> will be promoted from Heat ' + sourceHeat + ' to Heat ' + toHeat +
                                        '</div>' +
                                        html;
                                }
                                
                            } catch (error) {
                                console.error('Error loading promotion preview:', error);
                                previewContent.innerHTML = '<div class="alert alert-danger mb-0">' +
                                    '<i class="fas fa-exclamation-circle me-2"></i>Error loading preview: ' + error.message +
                                '</div>';
                            }
                        };
                        
                        // Alias for HTML calls
                        window.updatePromotionPreview = window.updatePromotionPreviewModal;
                        
                        // Show save notification
                        function showSaveNotification(message) {
                            const btn = document.getElementById('saveFlowBtn');
                            if (btn) {
                                btn.classList.add('btn-warning');
                                btn.classList.remove('btn-success');
                                const originalHtml = btn.innerHTML;
                                btn.innerHTML = '<i class="fas fa-exclamation-triangle me-1"></i>Unsaved Changes';
                                
                                setTimeout(() => {
                                    btn.innerHTML = originalHtml;
                                    btn.classList.remove('btn-warning');
                                    btn.classList.add('btn-success');
                                }, 3000);
                            }
                        }
                        
                        // Save flow configuration
                        window.saveFlowConfiguration = function() {
                            const btn = document.getElementById('saveFlowBtn');
                            const originalHtml = btn.innerHTML;
                            btn.disabled = true;
                            btn.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i>Saving...';
                            
                            // Create form data with all heat flow configurations
                            const formData = new FormData();
                            formData.append('event_id', '<?= $selected_event_id ?>');
                            formData.append('save_flow_only', '1');
                            
                            // Group connections by destination heat
                            const connectionsByDest = {};
                            flowEditor.connections.forEach(conn => {
                                if (!connectionsByDest[conn.to]) {
                                    connectionsByDest[conn.to] = [];
                                }
                                connectionsByDest[conn.to].push(conn.from);
                            });
                            
                            // Add flow source data AND node positions for each heat
                            for (let h = 1; h <= <?= $heats_total ?>; h++) {
                                // Get connections for this heat
                                const heatsConnections = connectionsByDest[h] || [];
                                
                                if (heatsConnections.length > 0) {
                                    // Add source heats
                                    heatsConnections.forEach(sourceHeat => {
                                        formData.append('flow_source_heat[' + h + '][]', sourceHeat);
                                    });
                                    
                                    // Get flow settings from the first connection to this heat
                                    const firstConn = flowEditor.connections.find(c => c.to === h);
                                    if (firstConn) {
                                        formData.append('flow_type[' + h + ']', firstConn.type || 'top');
                                        formData.append('flow_participants_per_category[' + h + ']', firstConn.count || 0);
                                        formData.append('flow_position_range[' + h + ']', firstConn.range || '');
                                    }
                                    
                                    // Build hue_offsets JSON for this heat (map of source_heat => hue_offset)
                                    const hueOffsetsForHeat = {};
                                    flowEditor.connections.filter(c => c.to === h).forEach(conn => {
                                        hueOffsetsForHeat[conn.from] = conn.hue_offset || 0;
                                    });
                                    formData.append('hue_offsets[' + h + ']', JSON.stringify(hueOffsetsForHeat));
                                } else {
                                    // No connections - set to none
                                    formData.append('flow_type[' + h + ']', 'none');
                                    formData.append('flow_participants_per_category[' + h + ']', 0);
                                    formData.append('flow_position_range[' + h + ']', '');
                                    formData.append('hue_offsets[' + h + ']', '{}');
                                }
                                
                                // Add node position for this heat
                                const nodeData = flowEditor.nodes.get(h);
                                if (nodeData) {
                                    formData.append('layout_x[' + h + ']', Math.round(nodeData.x));
                                    formData.append('layout_y[' + h + ']', Math.round(nodeData.y));
                                }
                            }
                            
                            // Debug log
                            //console.log('Saving connections:', flowEditor.connections);
                            //console.log('Connections by destination:', connectionsByDest);
                            
                            // Submit via AJAX
                            fetch(window.location.href, {
                                method: 'POST',
                                body: formData
                            })
                            .then(response => response.text())
                            .then(data => {
                                // Try to parse as JSON
                                let result;
                                try {
                                    result = JSON.parse(data);
                                } catch (e) {
                                    // Not JSON, might be HTML response
                                    //console.log('Response:', data);
                                    result = {success: true}; // Assume success if no JSON
                                }
                                
                                if (result.success) {
                                    btn.disabled = false;
                                    btn.innerHTML = '<i class="fas fa-check me-1"></i>Saved!';
                                    btn.classList.remove('btn-warning', 'btn-danger');
                                    btn.classList.add('btn-success');
                                    
                                    setTimeout(() => {
                                        btn.innerHTML = originalHtml;
                                    }, 2000);
                                } else {
                                    throw new Error(result.message || 'Save failed');
                                }
                            })
                            .catch(error => {
                                btn.disabled = false;
                                btn.innerHTML = '<i class="fas fa-times me-1"></i>Error';
                                btn.classList.remove('btn-warning', 'btn-success');
                                btn.classList.add('btn-danger');
                                
                                setTimeout(() => {
                                    btn.innerHTML = originalHtml;
                                    btn.classList.remove('btn-danger');
                                    btn.classList.add('btn-success');
                                }, 3000);
                                
                                console.error('Save error:', error);
                                alert('Error saving flow configuration: ' + error.message);
                            });
                        };
                        
                        // Global functions for buttons
                        window.resetNodeLayout = function() {
                            // Show Bootstrap confirmation modal
                            const modal = document.getElementById('flowUnlinkConfirmModal');
                            const title = document.getElementById('flowUnlinkModalTitle');
                            const message = document.getElementById('flowUnlinkModalMessage');
                            const confirmBtn = document.getElementById('flowUnlinkConfirmBtn');
                            
                            title.textContent = 'Reset Node Layout';
                            message.innerHTML = 'Are you sure you want to <strong>reset the node layout</strong>? This will clear all saved positions and reload the page with default positions.';
                            
                            // Remove old event listeners and add new one
                            const newConfirmBtn = confirmBtn.cloneNode(true);
                            confirmBtn.parentNode.replaceChild(newConfirmBtn, confirmBtn);
                            
                            newConfirmBtn.addEventListener('click', function() {
                                localStorage.removeItem('heatFlowNodePositions_<?= $selected_event_id ?>');
                                location.reload();
                            });
                            
                            // Show modal
                            const bsModal = new bootstrap.Modal(modal);
                            bsModal.show();
                        };
                        
                        window.deleteFlowConnection = function(fromHeat, toHeat) {
                            // Show Bootstrap confirmation modal
                            const modal = document.getElementById('flowUnlinkConfirmModal');
                            const title = document.getElementById('flowUnlinkModalTitle');
                            const message = document.getElementById('flowUnlinkModalMessage');
                            const confirmBtn = document.getElementById('flowUnlinkConfirmBtn');
                            
                            title.textContent = 'Remove Flow Connection';
                            message.innerHTML = 'Are you sure you want to remove the flow connection from <strong>Heat ' + fromHeat + '</strong> to <strong>Heat ' + toHeat + '</strong>?';
                            
                            // Remove old event listeners and add new one
                            const newConfirmBtn = confirmBtn.cloneNode(true);
                            confirmBtn.parentNode.replaceChild(newConfirmBtn, confirmBtn);
                            
                            newConfirmBtn.addEventListener('click', function() {
                                // Remove the connection
                                flowEditor.connections = flowEditor.connections.filter(
                                    c => !(c.from === fromHeat && c.to === toHeat)
                                );
                                
                                // Clear any selected ports
                                document.querySelectorAll('.flow-port.active').forEach(port => {
                                    port.classList.remove('active');
                                });
                                flowEditor.selectedPort = null;
                                
                                // Update UI
                                renderConnections();
                                syncConnectionsToForms();
                                showSaveNotification('Connection removed. Click "Save Flow" to persist.');
                                
                                // Close modals
                                bootstrap.Modal.getInstance(modal).hide();
                                closeFlowConfigModal();
                            });
                            
                            // Show modal
                            const bsModal = new bootstrap.Modal(modal);
                            bsModal.show();
                        };
                        
                        window.autoArrangeNodes = function() {
                            // Show Bootstrap confirmation modal
                            const modal = document.getElementById('flowUnlinkConfirmModal');
                            const title = document.getElementById('flowUnlinkModalTitle');
                            const message = document.getElementById('flowUnlinkModalMessage');
                            const confirmBtn = document.getElementById('flowUnlinkConfirmBtn');
                            
                            title.textContent = 'Auto-Arrange Nodes';
                            message.innerHTML = 'Are you sure you want to <strong>auto-arrange all nodes</strong>? This will rearrange all heat nodes into a grid layout.';
                            
                            // Remove old event listeners and add new one
                            const newConfirmBtn = confirmBtn.cloneNode(true);
                            confirmBtn.parentNode.replaceChild(newConfirmBtn, confirmBtn);
                            
                            newConfirmBtn.addEventListener('click', function() {
                                // Simple grid layout
                                const nodes = Array.from(flowEditor.nodes.values());
                                const cols = Math.ceil(Math.sqrt(nodes.length));
                                
                                nodes.forEach((nodeData, index) => {
                                    const col = index % cols;
                                    const row = Math.floor(index / cols);
                                    nodeData.x = 50 + col * 360;
                                    nodeData.y = 50 + row * 280;
                                    nodeData.element.style.left = nodeData.x + 'px';
                                    nodeData.element.style.top = nodeData.y + 'px';
                                });
                                
                                saveNodePositions();
                                renderConnections();
                                showSaveNotification('Nodes auto-arranged. Click "Save Flow" to persist positions.');
                                
                                // Close modal
                                bootstrap.Modal.getInstance(modal).hide();
                            });
                            
                            // Show modal
                            const bsModal = new bootstrap.Modal(modal);
                            bsModal.show();
                        };
                        
                        window.unlinkAllNodes = function() {
                            // Show Bootstrap confirmation modal
                            const modal = document.getElementById('flowUnlinkConfirmModal');
                            const title = document.getElementById('flowUnlinkModalTitle');
                            const message = document.getElementById('flowUnlinkModalMessage');
                            const confirmBtn = document.getElementById('flowUnlinkConfirmBtn');
                            
                            title.textContent = 'Remove All Connections';
                            message.innerHTML = 'Are you sure you want to remove <strong>all flow connections</strong>? This will unlink all heats.';
                            
                            // Remove old event listeners and add new one
                            const newConfirmBtn = confirmBtn.cloneNode(true);
                            confirmBtn.parentNode.replaceChild(newConfirmBtn, confirmBtn);
                            
                            newConfirmBtn.addEventListener('click', function() {
                                // Clear all connections
                                flowEditor.connections = [];
                                
                                // Clear all selected ports
                                document.querySelectorAll('.flow-port.active').forEach(port => {
                                    port.classList.remove('active');
                                });
                                flowEditor.selectedPort = null;
                                
                                // Update UI
                                renderConnections();
                                syncConnectionsToForms();
                                showSaveNotification('All connections removed. Click "Save Flow" to persist.');
                                
                                // Close modal
                                bootstrap.Modal.getInstance(modal).hide();
                            });
                            
                            // Show modal
                            const bsModal = new bootstrap.Modal(modal);
                            bsModal.show();
                        };
                        
                        // Initialize
                        initializeNodes();
                        
                        // Listen for changes in flow source selects to update node editor
                        document.querySelectorAll('select[name^="flow_source_heat"]').forEach(select => {
                            select.addEventListener('change', function() {
                                // Extract heat number from select name
                                const match = this.name.match(/flow_source_heat\[(\d+)\]/);
                                if (!match) return;
                                
                                const toHeat = parseInt(match[1]);
                                const selectedOptions = Array.from(this.selectedOptions);
                                const selectedHeats = selectedOptions.map(opt => parseInt(opt.value));
                                
                                // Preserve existing connection data (hue_offset, etc.)
                                const existingConnections = flowEditor.connections.filter(conn => conn.to === toHeat);
                                const existingConnectionsMap = {};
                                existingConnections.forEach(conn => {
                                    existingConnectionsMap[conn.from] = {
                                        type: conn.type,
                                        count: conn.count,
                                        range: conn.range,
                                        hue_offset: conn.hue_offset || 0
                                    };
                                });
                                
                                // Remove existing connections to this heat
                                flowEditor.connections = flowEditor.connections.filter(conn => conn.to !== toHeat);
                                
                                // Add new connections, preserving existing settings
                                selectedHeats.forEach(fromHeat => {
                                    const existing = existingConnectionsMap[fromHeat];
                                    flowEditor.connections.push({
                                        from: fromHeat,
                                        to: toHeat,
                                        type: existing ? existing.type : 'top',
                                        count: existing ? existing.count : 1,
                                        range: existing ? existing.range : null,
                                        hue_offset: existing ? existing.hue_offset : 0
                                    });
                                });
                                
                                renderConnections();
                            });
                        });
                    });
                    </script>
                    
                    <?php if (empty($heat_nodes)): ?>
                        <div class="text-center text-white py-5" style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%);">
                            <i class="fas fa-info-circle fa-3x mb-3"></i>
                            <p class="lead">No heats configured</p>
                            <p>Configure heats below to see the flow builder</p>
                        </div>
                    <?php endif; ?>

        <!-- Heat Flow Visualization -->
        <?php if (!empty($heat_settings) && $heats_total > 0): ?>
            <div class="border-0 card card-body d-flex flex-row flex-wrap mb-4 overflow-auto shadow-none">
                
                
                <?php
                // Build flow chains based on actual connections
                $processed_heats = [];
                $flow_chains = [];
                
                for ($h = 1; $h <= $heats_total; $h++) {
                    if (in_array($h, $processed_heats)) continue;
                    
                    $current_chain = [];
                    $heat_to_process = $h;
                    
                    // Start a new chain with current heat
                    while ($heat_to_process && !in_array($heat_to_process, $processed_heats)) {
                        $setting = $heat_settings[$heat_to_process] ?? [];
                        
                        // Get category names for display
                        $heat_categories = $setting['categories'] ?? [];
                        $category_names = [];
                        if (empty($heat_categories)) {
                            $category_names = ['All Categories'];
                        } else {
                            foreach ($event_categories as $cat) {
                                if (in_array($cat['id'], $heat_categories)) {
                                    $category_names[] = $cat['category_name'];
                                }
                            }
                        }
                        
                        $current_chain[] = [
                            'heat_number' => $heat_to_process,
                            'heat_name' => $setting['heat_name'] ?? '',
                            'flow_type' => $setting['flow_type'] ?? 'none',
                            'flow_participants_per_category' => $setting['flow_participants_per_category'] ?? 0,
                            'flow_position_range' => $setting['flow_position_range'] ?? null,
                            'flow_source_heat' => $setting['flow_source_heat'] ?? null,
                            'is_active' => $setting['is_active'] ?? false,
                            'categories' => $category_names,
                            'participant_count' => $heat_participant_counts[$heat_to_process] ?? 0,
                            'time_start' => $setting['time_start'] ?? '',
                            'estimate_time_per_participant' => $setting['estimate_time_per_participant'] ?? 0
                        ];
                        
                        $processed_heats[] = $heat_to_process;
                        
                        // Find next heat that flows from this one
                        $next_heat = null;
                        for ($next = $heat_to_process + 1; $next <= $heats_total; $next++) {
                            $next_setting = $heat_settings[$next] ?? [];
                            if (($next_setting['flow_source_heat'] ?? null) == $heat_to_process) {
                                $next_heat = $next;
                                break;
                            }
                        }
                        
                        $heat_to_process = $next_heat;
                    }
                    
                    if (!empty($current_chain)) {
                        $flow_chains[] = $current_chain;
                    }
                }
                ?>
                
                <!-- Flow Statistics -->
                <div class="mt-3 pt-3 border-top" style="display: none;">
                    <div class="row">
                        <div class="col-md-3">
                            <div class="summary-stat">
                                <i class="fas fa-fire text-warning me-2"></i>
                                <strong><?= $heats_total ?></strong> Total Heats
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="summary-stat">
                                <i class="fas fa-stream text-info me-2"></i>
                                <strong><?= count($flow_chains) ?></strong> Flow Chain<?= count($flow_chains) > 1 ? 's' : '' ?>
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="summary-stat">
                                <?php 
                                $auto_flow_count = 0;
                                if (!empty($heat_settings)) {
                                    foreach ($heat_settings as $setting) {
                                        if (($setting['flow_type'] ?? 'none') !== 'none') {
                                            $auto_flow_count++;
                                        }
                                    }
                                }
                                ?>
                                <i class="fas fa-route text-success me-2"></i>
                                <strong><?= $auto_flow_count ?></strong> Auto Flow
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="summary-stat">
                                <?php 
                                $starting_count = 0;
                                if (!empty($heat_settings)) {
                                    foreach ($heat_settings as $h_num => $setting) {
                                        if (($setting['flow_type'] ?? 'none') === 'none') {
                                            // Check if any other heat flows from this one
                                            $has_outgoing = false;
                                            foreach ($heat_settings as $other_setting) {
                                                if (($other_setting['flow_source_heat'] ?? null) == $h_num) {
                                                    $has_outgoing = true;
                                                    break;
                                                }
                                            }
                                            if (!$has_outgoing) {
                                                $starting_count++;
                                            }
                                        }
                                    }
                                }
                                ?>
                                <i class="fas fa-play text-primary me-2"></i>
                                <strong><?= $starting_count ?></strong> Starting
                            </div>
                        </div>
                    </div>
                    
                    <!-- Flow Examples -->
                    <?php if (count($flow_chains) > 1): ?>
                        <div class="mt-3" style="display: none;">
                            <h6 class="text-muted mb-2">
                                <i class="fas fa-sitemap me-1"></i>Flow Structure:
                            </h6>
                            <?php foreach ($flow_chains as $chain_idx => $chain): ?>
                                <div class="flow-example mb-1">
                                    <small class="text-info">
                                        <strong>Chain <?= $chain_idx + 1 ?>:</strong>
                                         <?php foreach ($chain as $idx => $heat): ?>
                                            <?= !empty($heat['heat_name']) ? $heat['heat_name'] : "Heat {$heat['heat_number']}" ?>
                                            <span class="text-muted">(<?= implode(', ', array_slice($heat['categories'], 0, 2)) ?><?= count($heat['categories']) > 2 ? '...' : '' ?>)</span>
                                            <?php if ($idx < count($chain) - 1): ?>
                                                <?php 
                                                $next_heat = $chain[$idx + 1];
                                                $has_flow = ($next_heat['flow_source_heat'] ?? null) == $heat['heat_number'];
                                                ?>
                                                <?= $has_flow ? ' → ' : ' | ' ?>
                                            <?php endif; ?>
                                        <?php endforeach; ?>
                                    </small>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        <?php endif; ?>

  
    <style>
        .heat-config-modal .modal-dialog {
            width: 80vw;
            max-width: 80vw;
        }
        @media (max-width: 992px) {
            .heat-config-modal .modal-dialog {
                width: 95vw;
                max-width: 95vw;
            }
        }
        .heat-config-modal .modal-content {
            height: 90vh;
            max-height: 90vh;
            display: flex;
            flex-direction: column;
        }
        .heat-config-modal .modal-body {
            flex: 1 1 auto;
            overflow-y: auto;
            padding: 1.5rem;
            max-height: calc(90vh - 140px);
        }
        .heat-config-modal .modal-footer {
            position: sticky;
            bottom: 0;
            background: #fff;
            border-top: 1px solid #dee2e6;
            z-index: 10;
        }
    </style>

    <!-- Promotion Preview Modal -->
    <div id="promotionPreviewModalBackdrop" class="modal-backdrop" style="display: none; opacity: 0; transition: opacity 0.3s;"></div>
    <div id="promotionPreviewModal" class="card shadow-lg" style="
        position: fixed;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        width: 600px;
        max-width: 90vw;
        height: 600px;
        max-height: 80vh;
        z-index: 20000;
        display: none;
        opacity: 0;
        transition: opacity 0.3s;
        resize: vertical;
        overflow: hidden;
        min-height: 300px;
    ">
        <div id="promotionPreviewModalHeader" class="card-header bg-primary text-white d-flex justify-content-between align-items-center" style="cursor: move; user-select: none;">
            <h6 class="mb-0"><i class="fas fa-users me-2 p-2"></i>Promotion Preview</h6>
            <button type="button" class="btn btn-sm btn-light" onclick="closePromotionPreviewModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="card-body" style="overflow-y: auto; height: calc(100% - 60px);">
            <div id="promotionPreviewModalContent">
                <div class="text-center text-muted py-5">
                    <i class="fas fa-info-circle fa-3x mb-3"></i>
                    <p>Click "Open Promotion Preview" button to load preview</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Heat Confirmation Modal -->
    <div class="modal fade" id="addHeatConfirmModal" tabindex="-1" aria-labelledby="addHeatConfirmModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-success">
                <div class="modal-header bg-success text-white">
                    <h5 class="modal-title" id="addHeatConfirmModalLabel">
                        <i class="fas fa-plus-circle me-2"></i>Add New Heat
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-info mb-3">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Token Cost:</strong> This action will charge <strong><span id="addHeatTokenCost">0</span> tokens</strong> from your balance.
                    </div>
                    
                    <p class="mb-2"><strong>Are you sure you want to add a new heat to this event?</strong></p>
                    
                    <div class="bg-light p-3 rounded mb-3">
                        <p class="mb-1"><strong>What will happen:</strong></p>
                        <ul class="mb-0">
                            <li>A new heat will be created with default settings</li>
                            <li>You can configure it after creation</li>
                            <li>Athletes can be assigned to it</li>
                            <li>The heat counter will increment by 1</li>
                        </ul>
                    </div>
                    
                    <p class="text-muted mb-0">
                        <i class="fas fa-lightbulb me-1"></i>
                        <small>You can configure heat details (name, runs, categories, etc.) after creation.</small>
                    </p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-1"></i>Cancel
                    </button>
                    <button type="button" class="btn btn-success" onclick="confirmAddHeat()">
                        <i class="fas fa-plus me-1"></i>Add Heat
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Heat Confirmation Modal -->
    <div class="modal fade" id="addHeatConfirmModal" tabindex="-1" aria-labelledby="addHeatConfirmModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content border-success">
                <div class="modal-header bg-success text-white">
                    <h5 class="modal-title" id="addHeatConfirmModalLabel">
                        <i class="fas fa-plus-circle me-2"></i>Add New Heat
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-info mb-3">
                        <i class="fas fa-coins me-2"></i>
                        <strong>Token Cost:</strong> This action will charge <strong><span id="addHeatTokenCost">0</span> tokens</strong> from your balance.
                    </div>
                    
                    <p class="mb-2"><strong>Are you sure you want to add a new heat to this event?</strong></p>
                    
                    <div class="bg-light p-3 rounded mb-3">
                        <p class="mb-1"><strong>What will happen:</strong></p>
                        <ul class="mb-0">
                            <li>A new heat will be created with default settings</li>
                            <li>You can configure it after creation</li>
                            <li>Athletes can be assigned to it</li>
                            <li>The heat counter will increment by 1</li>
                        </ul>
                    </div>
                    
                    <p class="text-muted mb-0">
                        <i class="fas fa-lightbulb me-1"></i>
                        <small>You can configure heat details (name, runs, categories, etc.) after creation.</small>
                    </p>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-1"></i>Cancel
                    </button>
                    <button type="button" class="btn btn-success" onclick="confirmAddHeat()">
                        <i class="fas fa-plus me-1"></i>Add Heat
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Promotion Confirmation Modal -->
    <div class="modal fade" id="promotionConfirmModal" tabindex="-1" aria-labelledby="promotionConfirmModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered">
            <div class="modal-content">
                <div class="modal-header bg-success text-white">
                    <h5 class="modal-title" id="promotionConfirmModalLabel">
                        <i class="fas fa-arrow-up me-2"></i>Confirm Participant Promotion
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-info mb-3">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong id="promotionParticipantCount">0 participants</strong> will be promoted
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label fw-semibold">
                            <i class="fas fa-level-down-alt me-1"></i>From Heat
                        </label>
                        <div class="input-group">
                            <span class="input-group-text">Heat</span>
                            <input type="text" class="form-control" id="promotionFromHeat" readonly>
                            <input type="text" class="form-control" id="promotionFromHeatName" readonly>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label fw-semibold">
                            <i class="fas fa-level-up-alt me-1"></i>To Heat
                        </label>
                        <div class="input-group">
                            <span class="input-group-text">Heat</span>
                            <input type="text" class="form-control" id="promotionToHeat" readonly>
                            <input type="text" class="form-control" id="promotionToHeatName" readonly>
                        </div>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label fw-semibold">
                            <i class="fas fa-sort me-1"></i>Promotion Order
                        </label>
                        <select class="form-select" id="promotionOrder">
                            <option value="normal" selected>Normal Order (Rank 1 → Order 1, Rank 10 → Order 10)</option>
                            <option value="reverse">Reverse Order (Rank 1 → Order 10, Rank 10 → Order 1)</option>
                        </select>
                        <div class="form-text">
                            <i class="fas fa-info-circle me-1"></i>
                            Choose how participants should be ordered in the target heat
                        </div>
                    </div>
                    
                    <div id="promotionRangeInfo" class="alert alert-secondary mb-0" style="display: none;">
                        <small><i class="fas fa-filter me-1"></i>Position Range: <strong id="promotionRangeText"></strong></small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                        <i class="fas fa-times me-1"></i>Cancel
                    </button>
                    <button type="button" class="btn btn-success" id="confirmPromotionBtn" onclick="executePromotionFromModal()">
                        <i class="fas fa-arrow-up me-1"></i>Promote Athletes
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Heat Configuration Modals Container -->
    <div id="heatConfigModalsContainer">
        <!-- Modals will be loaded dynamically via API -->
        <div class="text-center py-5" id="modalsLoading">
            <div class="spinner-border text-primary" role="status">
                <span class="visually-hidden">Loading heat modals...</span>
            </div>
            <div class="mt-2 text-muted">Loading configuration panels...</div>
        </div>
    </div>
    <?php foreach ($heat_order as $h): ?>
        <?php 
        $setting = $heat_settings[$h] ?? [
            'heat_name' => '', 
            'scoring_type' => 'Points',
            'runs_count' => 1,
            'runs_scoring_method' => 'best_from_all',
            'time_start' => '',
            'estimate_time_per_participant' => 0,
            'categories' => [],
            'flow_type' => 'none',
            'flow_source_heat' => null,
            'flow_participants_per_category' => 0,
            'flow_position_range' => null,
            'is_active' => 0,
            'active_run' => 1,
            'status' => 'pending',
            'format_name' => $event_data['format_name'] ?? '',
            'discipline' => $event_data['discipline'] ?? '',
            'difficulty_level' => '',
            'course_length' => 0,
            'time_limit' => 0,
            'weather_dependent' => 0,
            'format_description' => '',
            'scoring_method' => 'standard',
            'required_figures' => [],
            'judges_required' => 5,
            'scale_min' => $event_data['scale_min'] ?? 0,
            'scale_max' => $event_data['scale_max'] ?? 100,
            'precision_decimal' => $event_data['precision_decimal'] ?? 0,
            'drop_rule' => null // NULL means use event default
        ];
        ?>
        
        <div class="modal fade heat-config-modal" id="heatModal<?= $h ?>" tabindex="-1" aria-labelledby="heatModalLabel<?= $h ?>" aria-hidden="true">
            <div class="modal-dialog modal-dialog-centered modal-dialog-scrollable">
                <div class="modal-content">
                    <div class="modal-header p-2">
                        <h6 class="modal-title" id="heatModalLabel<?= $h ?>">
                            <i class="fas fa-fire me-2 text-warning"></i>
                            Configure Heat <?= $h ?>
                            <?php if ($setting['is_active']): ?>
                                <span class="badge text-bg-success ms-2">ACTIVE</span>
                            <?php endif; ?>
                        </h6>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    
                    <form method="post" onsubmit="return validateHeatForm(<?= $h ?>)">
                        <input type="hidden" name="event_id" value="<?= $selected_event_id ?>">
                        <input type="hidden" name="save_settings" value="1">
                        <input type="hidden" name="single_heat" value="<?= $h ?>">
                        
                        <div class="modal-body">
                            <!-- Heat Activation Status -->
                            <?php if ($setting['is_active']): ?>
                                <div class="alert alert-success border-0 mb-4">
                                    <div class="d-flex align-items-center">
                                        <i class="fas fa-fire text-success fs-4 me-3"></i>
                                        <div>
                                            <h6 class="mb-1">This Heat is Currently Active</h6>
                                            <small class="text-muted">All scoring activity is directed to this heat</small>
                                        </div>
                                    </div>
                                </div>
                            <?php endif; ?>
                            
                            <!-- Tabbed Interface -->
                            <ul class="nav nav-tabs mb-4" id="heatConfigTabs<?= $h ?>" role="tablist">
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link active" id="basic-tab-<?= $h ?>" data-bs-toggle="tab" 
                                            data-bs-target="#basic-<?= $h ?>" type="button" role="tab">
                                        <i class="fas fa-info-circle me-2"></i>Basic Settings
                                    </button>
                                </li>
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link" id="timing-tab-<?= $h ?>" data-bs-toggle="tab" 
                                            data-bs-target="#timing-<?= $h ?>" type="button" role="tab">
                                        <i class="fas fa-clock me-2"></i>Timing & Runs
                                    </button>
                                </li>
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link" id="format-tab-<?= $h ?>" data-bs-toggle="tab" 
                                            data-bs-target="#format-<?= $h ?>" type="button" role="tab">
                                        <i class="fas fa-clipboard-list me-2"></i>Format
                                    </button>
                                </li>
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link" id="flow-tab-<?= $h ?>" data-bs-toggle="tab" 
                                            data-bs-target="#flow-<?= $h ?>" type="button" role="tab">
                                        <i class="fas fa-sitemap me-2"></i>Flow
                                    </button>
                                </li>
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link" id="activation-tab-<?= $h ?>" data-bs-toggle="tab" 
                                            data-bs-target="#activation-<?= $h ?>" type="button" role="tab">
                                        <i class="fas fa-power-off me-2"></i>Activation
                                    </button>
                                </li>
                                <li class="nav-item" role="presentation">
                                    <button class="nav-link" id="scoring-layout-tab-<?= $h ?>" data-bs-toggle="tab" 
                                            data-bs-target="#scoring-layout-<?= $h ?>" type="button" role="tab">
                                        <i class="fas fa-layout me-2"></i>Score Input Layout
                                    </button>
                                </li>
                            </ul>
                            
                            <!-- Tab Content -->
                            <div class="tab-content" id="heatConfigTabContent<?= $h ?>">
                                
                                <!-- Basic Settings Tab -->
                                <div class="tab-pane fade show active" id="basic-<?= $h ?>" role="tabpanel">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label fw-semibold">Heat Name</label>
                                                <input type="text" class="form-control" 
                                                       name="heat_name[<?= $h ?>]" 
                                                       value="<?= htmlspecialchars($setting['heat_name'] ?? '') ?>"
                                                       placeholder="Enter descriptive heat name">
                                                <div class="form-text">Optional: Give this heat a descriptive name</div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label fw-semibold">Scoring Type</label>
                                                <select class="form-select" name="scoring_type[<?= $h ?>]">
                                                    <option value="Points" <?= $setting['scoring_type'] == 'Points' ? 'selected' : '' ?>>Points Based</option>
                                                    <option value="Time" <?= $setting['scoring_type'] == 'Time' ? 'selected' : '' ?>>Time Based</option>
                                                </select>
                                                <div class="form-text">How this heat will be scored</div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="mb-4">
                                        <label class="form-label fw-semibold">Divisions</label>
                                        <?php if (empty($event_categories)): ?>
                                            <div class="alert alert-warning">
                                                <i class="fas fa-exclamation-triangle me-2"></i>
                                                No divisions defined for this event. 
                                                <a href="event_categories.php?event_id=<?= $selected_event_id ?>" class="alert-link">Add categories first</a>
                                            </div>
                                        <?php else: ?>
                                            <div class="border-0 shadow-none card">
                                                <div class="border-0 shadow-none card-header py-2">
                                                    <div class="form-check">
                                                        <input class="form-check-input" type="checkbox" 
                                                               id="all_categories_modal_<?= $h ?>"
                                                               <?= empty($setting['categories']) ? 'checked' : '' ?>
                                                               onchange="toggleAllCategoriesModal(<?= $h ?>)">
                                                        <label class="form-check-label fw-semibold" for="all_categories_modal_<?= $h ?>">
                                                            All Divisions
                                                        </label>
                                                    </div>
                                                </div>
                                                <div class="border-0 shadow-none card-body" style="max-height: 200px; overflow-y: auto;">
                                                    <div class="row">
                                                        <?php foreach ($event_categories as $category): ?>
                                                            <div class="col-md-6">
                                                                <div class="form-check">
                                                                    <input class="form-check-input category-checkbox" type="checkbox" 
                                                                           name="categories[<?= $h ?>][]" 
                                                                           value="<?= $category['id'] ?>"
                                                                           id="cat_modal_<?= $h ?>_<?= $category['id'] ?>"
                                                                           data-heat="<?= $h ?>"
                                                                           <?= in_array($category['id'], $setting['categories']) ? 'checked' : '' ?>
                                                                           <?= empty($setting['categories']) ? 'disabled' : '' ?>
                                                                           onchange="updateCategorySelectionModal(<?= $h ?>)">
                                                                    <label class="form-check-label" for="cat_modal_<?= $h ?>_<?= $category['id'] ?>">
                                                                        <?= htmlspecialchars($category['category_name']) ?>
                                                                    </label>
                                                                </div>
                                                            </div>
                                                        <?php endforeach; ?>
                                                    </div>
                                                </div>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                
                                <!-- Timing & Runs Tab -->
                                <div class="tab-pane fade" id="timing-<?= $h ?>" role="tabpanel">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label fw-semibold">Start Time</label>
                                                <input type="time" class="form-control" 
                                                       name="time_start[<?= $h ?>]" 
                                                       value="<?= htmlspecialchars($setting['time_start'] ?? '') ?>"
                                                       onchange="calculateEstimatedFinishModal(<?= $h ?>)">
                                                <div class="form-text">When this heat is scheduled to begin</div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label fw-semibold">Time per Athlete</label>
                                                <div class="input-group">
                                                    <input type="number" class="form-control" 
                                                           name="estimate_time_per_participant[<?= $h ?>]" 
                                                           value="<?= $setting['estimate_time_per_participant'] ?>"
                                                           min="0" step="5"
                                                           onchange="calculateEstimatedFinishModal(<?= $h ?>)">
                                                    <span class="input-group-text">seconds</span>
                                                </div>
                                                <div class="form-text">Estimated time allocation per athlete</div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label fw-semibold">Number of Runs</label>
                                                <select class="form-select" name="runs_count[<?= $h ?>]" 
                                                        onchange="updateRunOptionsModal(<?= $h ?>)">
                                                    <?php for ($r = 1; $r <= 5; $r++): ?>
                                                        <option value="<?= $r ?>" <?= $setting['runs_count'] == $r ? 'selected' : '' ?>>
                                                            <?= $r ?> Run<?= $r > 1 ? 's' : '' ?>
                                                        </option>
                                                    <?php endfor; ?>
                                                </select>
                                                <div class="form-text">How many runs each participant gets</div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-6">
                                            <div class="mb-3">
                                                <label class="form-label fw-semibold">Scoring Method</label>
                                                <select class="form-select" name="runs_scoring_method[<?= $h ?>]" 
                                                        id="runs_scoring_modal_<?= $h ?>"
                                                        onchange="updateScoringExplanationModal(<?= $h ?>)">
                                                    <!-- Options populated by JavaScript -->
                                                </select>
                                                <div class="form-text">
                                                    <span id="scoring_explanation_modal_<?= $h ?>">How to calculate final score from multiple runs</span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <!-- Estimated Finish -->
                                    <div id="estimated_finish_modal_<?= $h ?>" class="alert alert-light border d-none">
                                        <i class="fas fa-clock text-primary me-2"></i>
                                        <strong>Estimated Finish:</strong>
                                        <span id="finish_time_modal_<?= $h ?>"></span>
                                    </div>
                                </div>
                                
                                <!-- Format Settings Tab -->
                                <div class="tab-pane fade" id="format-<?= $h ?>" role="tabpanel">
                                    <div class="row">
                                        <div class="col-lg-6">
                                            <h6 class="text-muted mb-3">Competition Format</h6>
                                            
                                            <div class="mb-3">
                                                <label class="form-label fw-semibold">Format Name</label>
                                                <input type="text" class="form-control" 
                                                       name="format_name[<?= $h ?>]" 
                                                       value="<?= htmlspecialchars($setting['format_name'] ?? '') ?>"
                                                       placeholder="e.g., Street Style, Park Competition">
                                                <div class="form-text">Override event format name for this heat</div>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label class="form-label fw-semibold">Discipline</label>
                                                <select class="form-select" name="discipline[<?= $h ?>]">
                                                    <option value="">-- Use Event Default --</option>
                                                    <option value="freestyle_scooter" <?= $setting['discipline'] == 'freestyle_scooter' ? 'selected' : '' ?>>Freestyle Scooter</option>
                                                    <option value="street_scooter" <?= $setting['discipline'] == 'street_scooter' ? 'selected' : '' ?>>Street Scooter</option>
                                                    <option value="park_scooter" <?= $setting['discipline'] == 'park_scooter' ? 'selected' : '' ?>>Park Scooter</option>
                                                    <option value="freestyle_skiing" <?= $setting['discipline'] == 'freestyle_skiing' ? 'selected' : '' ?>>Freestyle Skiing</option>
                                                    <option value="alpine_skiing" <?= $setting['discipline'] == 'alpine_skiing' ? 'selected' : '' ?>>Alpine Skiing</option>
                                                    <option value="snowboard_freestyle" <?= $setting['discipline'] == 'snowboard_freestyle' ? 'selected' : '' ?>>Snowboard Freestyle</option>
                                                    <option value="skateboard_street" <?= $setting['discipline'] == 'skateboard_street' ? 'selected' : '' ?>>Skateboard Street</option>
                                                    <option value="skateboard_park" <?= $setting['discipline'] == 'skateboard_park' ? 'selected' : '' ?>>Skateboard Park</option>
                                                    <option value="bmx_freestyle" <?= $setting['discipline'] == 'bmx_freestyle' ? 'selected' : '' ?>>BMX Freestyle</option>
                                                    <option value="breaking" <?= $setting['discipline'] == 'breaking' ? 'selected' : '' ?>>Breaking</option>
                                                    <option value="parkour" <?= $setting['discipline'] == 'parkour' ? 'selected' : '' ?>>Parkour</option>
                                                </select>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label class="form-label fw-semibold">Difficulty Level</label>
                                                <select class="form-select" name="difficulty_level[<?= $h ?>]">
                                                    <option value="">-- No Specific Level --</option>
                                                    <option value="beginner" <?= $setting['difficulty_level'] == 'beginner' ? 'selected' : '' ?>>Beginner</option>
                                                    <option value="intermediate" <?= $setting['difficulty_level'] == 'intermediate' ? 'selected' : '' ?>>Intermediate</option>
                                                    <option value="advanced" <?= $setting['difficulty_level'] == 'advanced' ? 'selected' : '' ?>>Advanced</option>
                                                    <option value="professional" <?= $setting['difficulty_level'] == 'professional' ? 'selected' : '' ?>>Professional</option>
                                                    <option value="elite" <?= $setting['difficulty_level'] == 'elite' ? 'selected' : '' ?>>Elite</option>
                                                </select>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label class="form-label fw-semibold">Scoring Method</label>
                                                <select class="form-select" name="scoring_method[<?= $h ?>]">
                                                    <option value="standard" <?= $setting['scoring_method'] == 'standard' ? 'selected' : '' ?>>Standard Scoring</option>
                                                    <option value="comparative" <?= $setting['scoring_method'] == 'comparative' ? 'selected' : '' ?>>Comparative Scoring</option>
                                                    <option value="elimination" <?= $setting['scoring_method'] == 'elimination' ? 'selected' : '' ?>>Elimination</option>
                                                    <option value="head_to_head" <?= $setting['scoring_method'] == 'head_to_head' ? 'selected' : '' ?>>Head to Head</option>
                                                    <option value="battle" <?= $setting['scoring_method'] == 'battle' ? 'selected' : '' ?>>Battle Format</option>
                                                </select>
                                            </div>
                                        </div>
                                        
                                        <div class="col-lg-6">
                                            <h6 class="text-muted mb-3">Course & Judging</h6>
                                            
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <label class="form-label fw-semibold">Course Length</label>
                                                        <div class="input-group">
                                                            <input type="number" class="form-control" 
                                                                   name="course_length[<?= $h ?>]" 
                                                                   value="<?= $setting['course_length'] ?>"
                                                                   min="0" step="1">
                                                            <span class="input-group-text">meters</span>
                                                        </div>
                                                        <div class="form-text">0 = not applicable</div>
                                                    </div>
                                                </div>
                                                
                                                <div class="col-md-6">
                                                    <div class="mb-3">
                                                        <label class="form-label fw-semibold">Time Limit</label>
                                                        <div class="input-group">
                                                            <input type="number" class="form-control" 
                                                                   name="time_limit[<?= $h ?>]" 
                                                                   value="<?= $setting['time_limit'] ?>"
                                                                   min="0" step="1">
                                                            <span class="input-group-text">seconds</span>
                                                        </div>
                                                        <div class="form-text">0 = no limit</div>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label class="form-label fw-semibold">Required Figures</label>
                                                <textarea class="form-control" rows="3"
                                                          name="required_figures[<?= $h ?>]" 
                                                          placeholder="List required figures/maneuvers..."><?= htmlspecialchars(is_array($setting['required_figures']) ? implode("\n", $setting['required_figures']) : $setting['required_figures']) ?></textarea>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label class="form-label fw-semibold">Judges Required</label>
                                                <input type="number" class="form-control" 
                                                       name="judges_required[<?= $h ?>]" 
                                                       value="<?= $setting['judges_required'] ?>"
                                                       min="1" max="15">
                                                <div class="form-text">Number of judges needed</div>
                                            </div>
                                            
                                            <div class="row">
                                                <div class="col-md-4">
                                                    <div class="mb-3">
                                                        <label class="form-label fw-semibold">Score Min</label>
                                                        <input type="number" class="form-control" 
                                                               name="scale_min[<?= $h ?>]" 
                                                               value="<?= $setting['scale_min'] ?>"
                                                               step="0.01">
                                                    </div>
                                                </div>
                                                
                                                <div class="col-md-4">
                                                    <div class="mb-3">
                                                        <label class="form-label fw-semibold">Score Max</label>
                                                        <input type="number" class="form-control" 
                                                               name="scale_max[<?= $h ?>]" 
                                                               value="<?= $setting['scale_max'] ?>"
                                                               step="0.01">
                                                    </div>
                                                </div>
                                                
                                                <div class="col-md-4">
                                                    <div class="mb-3">
                                                        <label class="form-label fw-semibold">Precision</label>
                                                        <select class="form-select" name="precision_decimal[<?= $h ?>]">
                                                            <option value="0" <?= $setting['precision_decimal'] == 0 ? 'selected' : '' ?>>Whole Numbers</option>
                                                            <option value="1" <?= $setting['precision_decimal'] == 1 ? 'selected' : '' ?>>1 Decimal</option>
                                                            <option value="2" <?= $setting['precision_decimal'] == 2 ? 'selected' : '' ?>>2 Decimals</option>
                                                        </select>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label class="form-label fw-semibold">
                                                    <i class="fas fa-filter me-1"></i>Judge Score Drop Rule
                                                </label>
                                                <select class="form-select" name="drop_rule[<?= $h ?>]" id="drop_rule_<?= $h ?>">
                                                    <option value="" <?= empty($setting['drop_rule']) ? 'selected' : '' ?>>-- Use Event Default --</option>
                                                    <option value="none" <?= ($setting['drop_rule'] ?? '') === 'none' ? 'selected' : '' ?>>None - Use All Judge Scores</option>
                                                    <option value="lowest" <?= ($setting['drop_rule'] ?? '') === 'lowest' ? 'selected' : '' ?>>Drop Lowest Score</option>
                                                    <option value="highest" <?= ($setting['drop_rule'] ?? '') === 'highest' ? 'selected' : '' ?>>Drop Highest Score</option>
                                                    <option value="highest_and_lowest" <?= ($setting['drop_rule'] ?? '') === 'highest_and_lowest' ? 'selected' : '' ?>>Drop Highest & Lowest Scores</option>
                                                </select>
                                                <div class="form-text">
                                                    <i class="fas fa-info-circle me-1"></i>Override which judge scores to exclude when calculating run scores. 
                                                    Leave as "Use Event Default" to inherit from event format settings.
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-12">
                                            <div class="mb-3">
                                                <div class="form-check form-switch">
                                                    <input class="form-check-input" type="checkbox" 
                                                           name="weather_dependent[<?= $h ?>]" value="1"
                                                           id="weather_modal_<?= $h ?>"
                                                           <?= $setting['weather_dependent'] ? 'checked' : '' ?>>
                                                    <label class="form-check-label" for="weather_modal_<?= $h ?>">
                                                        <i class="fas fa-cloud-sun me-1"></i>Weather Dependent Competition
                                                    </label>
                                                </div>
                                                <div class="form-text">Check if this heat is affected by weather conditions</div>
                                            </div>
                                            
                                            <div class="mb-3">
                                                <label class="form-label fw-semibold">Format Description</label>
                                                <textarea class="form-control" rows="3"
                                                          name="format_description[<?= $h ?>]" 
                                                          placeholder="Additional format-specific details, rules, or requirements for this heat"><?= htmlspecialchars($setting['format_description'] ?? '') ?></textarea>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Flow Control Tab -->
                                <div class="tab-pane fade" id="flow-<?= $h ?>" role="tabpanel">
                                    <div class="row">
                                        <div class="col-12 col-sm-12 col-lg-8">
                                            <div class="border-0 shadow-none card">
                                                <div class="border-0 shadow-none card-header">
                                                    <h6 class="mb-0">
                                                        <i class="fas fa-sitemap me-2"></i>Flow Configuration
                                                    </h6>
                                                </div>
                                                <div class="border-0 shadow-none card-body">
                                                    <div class="mb-3">
                                                        <label class="form-label fw-semibold">Flow Type</label>
                                                        <select class="form-select" name="flow_type[<?= $h ?>]" 
                                                                onchange="toggleFlowConfigModal(<?= $h ?>)">
                                                            <option value="none" <?= $setting['flow_type'] == 'none' ? 'selected' : '' ?>>
                                                                No Flow (Manual Entry)
                                                            </option>
                                                            <option value="promotion" <?= $setting['flow_type'] == 'promotion' ? 'selected' : '' ?>>
                                                                Promotion from Previous Heat
                                                            </option>
                                                            <option value="qualifying" <?= $setting['flow_type'] == 'qualifying' ? 'selected' : '' ?>>
                                                                Qualifying Heat (Best from Categories)
                                                            </option>
                                                        </select>
                                                    </div>
                                                    
                                                    <div id="flow_config_modal_<?= $h ?>" class="<?= $setting['flow_type'] !== 'none' ? '' : 'd-none' ?>">
                                                        <?php if ($h > 1): ?>
                                                            <div class="mb-3">
                                                                <label class="form-label fw-semibold">Source Heat(s) <span class="badge bg-info small">Multi-select</span></label>
                                                                <select class="form-select" name="flow_source_heat[<?= $h ?>][]" multiple size="<?= min($h - 1, 5) ?>">
                                                                    <?php 
                                                                    // Parse existing source heats (could be JSON array or single value)
                                                                    $selected_heats = [];
                                                                    if (!empty($setting['flow_source_heat'])) {
                                                                        $decoded = json_decode($setting['flow_source_heat'], true);
                                                                        $selected_heats = is_array($decoded) ? $decoded : [$setting['flow_source_heat']];
                                                                    }
                                                                    ?>
                                                                    <?php for ($source_h = 1; $source_h < $h; $source_h++): ?>
                                                                        <option value="<?= $source_h ?>" 
                                                                                data-heat-number="<?= $source_h ?>"
                                                                                <?= in_array($source_h, $selected_heats) ? 'selected' : '' ?>>
                                                                            Heat <?= $source_h ?><?= !empty($heat_settings[$source_h]['heat_name']) ? ' (' . htmlspecialchars($heat_settings[$source_h]['heat_name']) . ')' : '' ?>
                                                                        </option>
                                                                    <?php endfor; ?>
                                                                </select>
                                                                <div class="form-text">
                                                                    <i class="fas fa-info-circle me-1"></i>Hold Ctrl (Cmd on Mac) to select multiple source heats
                                                                </div>
                                                            </div>
                                                        <?php endif; ?>
                                                        
                                                        <div class="mb-3">
                                                            <label class="form-label fw-semibold">Athletes to Promote per Division</label>
                                                            <input type="number" class="form-control" 
                                                                   name="flow_participants_per_category[<?= $h ?>]"
                                                                   value="<?= $setting['flow_participants_per_category'] ?>"
                                                                   min="1"
                                                                   data-heat="<?= $h ?>">
                                                            <div class="form-text">
                                                                Number of top participants from each category to advance to this heat
                                                            </div>
                                                        </div>
                                                        
                                                        <div class="mb-3">
                                                            <label class="form-label fw-semibold">
                                                                Position Range 
                                                                <span class="badge bg-secondary ms-1">Optional</span>
                                                            </label>
                                                            <input type="text" class="form-control" 
                                                                   name="flow_position_range[<?= $h ?>]"
                                                                   value="<?= htmlspecialchars($setting['flow_position_range'] ?? '') ?>"
                                                                   placeholder="e.g., 1-3, 4-8, 9-12"
                                                                   pattern="\d+-\d+"
                                                                   data-heat="<?= $h ?>">
                                                            <div class="form-text">
                                                                <i class="fas fa-info-circle me-1"></i>
                                                                Specify position range from source heat (e.g., "3-4" for 3rd and 4th place, "4-15" for positions 4-15).
                                                                <br>Leave empty to use top N participants per category.
                                                            </div>
                                                        </div>
                                                        
                                                        <div class="alert alert-info">
                                                            <i class="fas fa-info-circle me-2"></i>
                                                            Flow configuration will automatically populate participants based on results from the source heat.
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-12 col-lg-4 col-sm-12">
                                            <!-- Diversity Rules Configuration -->
                                            <div class="border-0 shadow-none card">
                                                <div class="border-0 shadow-none card-header">
                                                    <h6 class="mb-0">
                                                        <i class="fas fa-shapes me-2"></i>Diversity Rules
                                                    </h6>
                                                </div>
                                                <div class="border-0 shadow-none card-body">
                                                    <div class="form-check form-switch mb-3">
                                                        <input class="form-check-input" type="checkbox" 
                                                               name="diversity_rules_enabled[<?= $h ?>]" 
                                                               id="diversity_rules_<?= $h ?>"
                                                               value="1"
                                                               <?= ($setting['diversity_rules_enabled'] ?? 1) ? 'checked' : '' ?>>
                                                        <label class="form-check-label fw-semibold" for="diversity_rules_<?= $h ?>">
                                                            Apply Diversity Rules
                                                        </label>
                                                    </div>
                                                    
                                                    <div class="alert alert-info">
                                                        <i class="fas fa-info-circle me-2"></i>
                                                        <small>
                                                            When enabled, diversity rules defined in the scoring format will be applied to this heat.
                                                            Disable to ignore diversity requirements for this specific heat.
                                                        </small>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="border-0 shadow-none card">
                                                <div class="border-0 shadow-none card-header">
                                                    <h6 class="mb-0">
                                                        <i class="fas fa-info-circle me-2"></i>Flow Info
                                                    </h6>
                                                </div>
                                                <div class="border-0 shadow-none card-body">
                                                    <div class="text-center">
                                                        <div class="mb-2">
                                                            <i class="fas fa-arrow-right fa-2x text-primary"></i>
                                                        </div>
                                                        <h6 class="text-primary">Automatic</h6>
                                                        <small class="text-muted">
                                                            Athlets will be automatically populated based on your flow configuration
                                                        </small>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Activation Tab -->
                                <div class="tab-pane fade" id="activation-<?= $h ?>" role="tabpanel">
                                    <div class="row">
                                        <div class="col-md-8">
                                            <!-- Active Heat Selection -->
                                            <div class="border-0 shadow-none card">
                                                <div class="border-0 shadow-none card-header">
                                                    <h6 class="mb-0">
                                                        <i class="fas fa-power-off me-2"></i>Heat Activation
                                                    </h6>
                                                </div>
                                                <div class="border-0 shadow-none card-body">
                                                    <div class="form-check form-switch mb-3">
                                                        <input class="form-check-input" type="radio" name="is_active" 
                                                               value="<?= $h ?>" id="active_modal_<?= $h ?>"
                                                               <?= $setting['is_active'] ? 'checked' : '' ?>
                                                               onchange="toggleActiveHeatModal(<?= $h ?>)">
                                                        <label class="form-check-label fw-semibold" for="active_modal_<?= $h ?>">
                                                            Set as Active Heat
                                                        </label>
                                                    </div>
                                                    
                                                    <?php if ($setting['is_active']): ?>
                                                        <div class="alert alert-success">
                                                            <i class="fas fa-check-circle me-2"></i>
                                                            This heat is currently active and receiving all scoring activity
                                                        </div>
                                                    <?php else: ?>
                                                        <div class="alert alert-light">
                                                            <i class="fas fa-info-circle me-2"></i>
                                                            This heat is inactive. Select to activate for scoring.
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                            
                                            <!-- Active Run Selection -->
                                            <div id="run_selector_modal_<?= $h ?>" class="card mt-3 <?= $setting['is_active'] ? '' : 'd-none' ?>" data-active-run="<?= max(1, intval($setting['active_run'] ?? 1)) ?>">
                                                <div class="border-0 shadow-none card-header">
                                                    <h6 class="mb-0">
                                                        <i class="fas fa-play me-2"></i>Active Run Selection
                                                    </h6>
                                                </div>
                                                <div class="border-0 shadow-none card-body">
                                                    <div class="row">
                                                        <?php for ($r = 1; $r <= max(1, $setting['runs_count']); $r++): ?>
                                                            <div class="col-auto">
                                                                <div class="form-check">
                                                                    <input class="form-check-input" type="radio" 
                                                                           name="active_run[<?= $h ?>]" value="<?= $r ?>" 
                                                                           id="run_modal_<?= $h ?>_<?= $r ?>" 
                                                                           <?= $r === $setting['active_run'] ? 'checked' : '' ?>>
                                                                    <label class="form-check-label" for="run_modal_<?= $h ?>_<?= $r ?>">
                                                                        Run <?= $r ?>
                                                                    </label>
                                                                </div>
                                                            </div>
                                                        <?php endfor; ?>
                                                    </div>
                                                    <div class="form-text mt-2">
                                                        Select which run is currently active for scoring
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4">
                                            <div class="border-0 shadow-none card">
                                                <div class="border-0 shadow-none card-header">
                                                    <h6 class="mb-0">
                                                        <i class="fas fa-chart-line me-2"></i>Heat Status
                                                    </h6>
                                                </div>
                                                <div class="border-0 shadow-none card-body">
                                                    <div class="text-center">
                                                        <?php if ($setting['is_active']): ?>
                                                            <div class="text-success mb-2">
                                                                <i class="fas fa-fire fa-2x"></i>
                                                            </div>
                                                            <h6 class="text-success">ACTIVE</h6>
                                                            <small class="text-muted">Receiving all scores</small>
                                                        <?php else: ?>
                                                            <div class="text-muted mb-2">
                                                                <i class="fas fa-pause fa-2x"></i>
                                                            </div>
                                                            <h6 class="text-muted">INACTIVE</h6>
                                                            <small class="text-muted">Not receiving scores</small>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Scoring Layout Tab -->
                                <div class="tab-pane fade" id="scoring-layout-<?= $h ?>" role="tabpanel">
                                    <?php
                                        // Get current heat-specific scoring layout config
                                        $heatLayoutConfig = [];
                                        if (!empty($setting['scoring_layout_config'])) {
                                            $heatLayoutConfig = json_decode($setting['scoring_layout_config'], true) ?? [];
                                        }
                                        
                                        // Get event-level default config
                                        $eventLayoutConfig = [];
                                        if (!empty($event_data['scoring_layout_config'])) {
                                            $eventLayoutConfig = json_decode($event_data['scoring_layout_config'], true) ?? [];
                                        }
                                        
                                        // Define available components for judge scoring panel
                                        $scoringComponents = [
                                            'criteriaInputGrid' => [
                                                'title' => 'Criteria Scoring Grid',
                                                'description' => 'Individual criteria input fields with sliders for detailed scoring',
                                                'category' => 'scoring'
                                            ],
                                            'finalCalculatedScore' => [
                                                'title' => 'Final Score Display',
                                                'description' => 'Shows the final calculated score from criteria or manual input',
                                                'category' => 'scoring'
                                            ],
                                            'figuresCard' => [
                                                'title' => 'Figures Selection Card',
                                                'description' => 'Figure selection chips for snowboard/freestyle sports',
                                                'category' => 'figures'
                                            ],
                                            'diversityValidation' => [
                                                'title' => 'Diversity Validation',
                                                'description' => 'Real-time figure diversity rule checking and warnings',
                                                'category' => 'figures'
                                            ],
                                            'figureHistory' => [
                                                'title' => 'Figure History',
                                                'description' => 'Previous figure selections across runs for reference',
                                                'category' => 'figures'
                                            ],
                                            'latestScores' => [
                                                'title' => 'Latest Scores Table',
                                                'description' => 'Recent scores with edit/overwrite functionality',
                                                'category' => 'history'
                                            ],
                                            'otherHeatScores' => [
                                                'title' => 'Other Judges Scores',
                                                'description' => 'Scores from other judges for the same participant',
                                                'category' => 'history'
                                            ],
                                            'formatInfo' => [
                                                'title' => 'Format Information Card',
                                                'description' => 'Scoring format details, rules, and metadata',
                                                'category' => 'info'
                                            ],
                                            'drawingBoardButton' => [
                                                'title' => 'Drawing Board Button',
                                                'description' => 'Button to open the drawing board for adding situation sketches',
                                                'category' => 'info'
                                            ]
                                        ];
                                        
                                        // Group components by category
                                        $componentsByCategory = [];
                                        foreach ($scoringComponents as $key => $component) {
                                            $componentsByCategory[$component['category']][$key] = $component;
                                        }
                                    ?>
                                    
                                    <div class="row">
                                        <div class="col-lg-8">
                                            <div class="alert alert-info mb-4">
                                                <i class="fas fa-info-circle me-2"></i>
                                                <strong>Heat-Specific Layout:</strong> Configure which components are visible on the judge scoring panel for this heat only. 
                                                Leave unchecked to inherit from event-level settings.
                                            </div>
                                            
                                            <?php foreach ($componentsByCategory as $categoryName => $components): ?>
                                                <div class="card mb-3">
                                                    <div class="card-header">
                                                        <h6 class="mb-0">
                                                            <?php
                                                                $categoryIcons = [
                                                                    'scoring' => 'fas fa-calculator',
                                                                    'figures' => 'fas fa-snowflake',
                                                                    'history' => 'fas fa-history', 
                                                                    'info' => 'fas fa-info-circle'
                                                                ];
                                                                $categoryTitles = [
                                                                    'scoring' => 'Scoring Components',
                                                                    'figures' => 'Figure Selection',
                                                                    'history' => 'Score History',
                                                                    'info' => 'Information'
                                                                ];
                                                            ?>
                                                            <i class="<?= $categoryIcons[$categoryName] ?? 'fas fa-cube' ?> me-2"></i>
                                                            <?= $categoryTitles[$categoryName] ?? ucfirst($categoryName) ?>
                                                        </h6>
                                                    </div>
                                                    <div class="card-body">
                                                        <?php foreach ($components as $key => $component): ?>
                                                            <?php
                                                                // Check if heat has override, otherwise use event default
                                                                $heatEnabled = $heatLayoutConfig['cards'][$key]['enabled'] ?? null;
                                                                $eventEnabled = $eventLayoutConfig['cards'][$key]['enabled'] ?? true;
                                                                $hasOverride = $heatEnabled !== null;
                                                                $isEnabled = $heatEnabled ?? $eventEnabled;
                                                            ?>
                                                            <div class="mb-3 p-2 <?= $hasOverride ? 'bg-light border rounded' : '' ?>">
                                                                <div class="form-check">
                                                                    <input class="form-check-input" type="hidden" 
                                                                           name="heat_scoring_layout[<?= $h ?>][cards][<?= $key ?>][enabled]" value="0">
                                                                    <input class="form-check-input" type="checkbox" 
                                                                           name="heat_scoring_layout[<?= $h ?>][cards][<?= $key ?>][enabled]" 
                                                                           id="heat_layout_<?= $h ?>_<?= $key ?>" 
                                                                           value="1" <?= $isEnabled ? 'checked' : '' ?>>
                                                                    <label class="form-check-label" for="heat_layout_<?= $h ?>_<?= $key ?>">
                                                                        <strong><?= htmlspecialchars($component['title']) ?></strong>
                                                                        <?php if ($hasOverride): ?>
                                                                            <span class="badge bg-primary ms-2">Heat Override</span>
                                                                        <?php else: ?>
                                                                            <span class="badge bg-secondary ms-2">From Event</span>
                                                                        <?php endif; ?>
                                                                        <br>
                                                                        <small class="text-muted"><?= htmlspecialchars($component['description']) ?></small>
                                                                    </label>
                                                                </div>
                                                                
                                                                <?php if ($key === 'figuresCard'): ?>
                                                                    <!-- Figure Categories Selection (conditional on figuresCard enabled) -->
                                                                    <div class="ms-4 mt-3 p-3 border rounded bg-white" id="heat_figureCategoriesConfig_<?= $h ?>" style="display: <?= $isEnabled ? 'block' : 'none' ?>;">
                                                                        <h6 class="mb-2 text-primary"><i class="fas fa-layer-group me-1"></i>Visible Figure Categories</h6>
                                                                        <p class="small text-muted mb-2">Select which figure categories judges can see for this heat (overrides event settings)</p>
                                                                        
                                                                        <?php
                                                                        // Get categories for the event's sport
                                                                        $categoriesStmt = $pdo->prepare("
                                                                            SELECT category_name, sort_order 
                                                                            FROM figure_categories 
                                                                            WHERE sport_name = ? AND is_active = 1 
                                                                            ORDER BY sort_order, category_name
                                                                        ");
                                                                        $categoriesStmt->execute([$event_data['sport_discipline'] ?? '']);
                                                                        $availableCategories = $categoriesStmt->fetchAll(PDO::FETCH_ASSOC);
                                                                        
                                                                        // Get heat-specific selected categories
                                                                        $heatSelectedCategories = $heatLayoutConfig['cards']['figuresCard']['visibleCategories'] ?? [];
                                                                        
                                                                        // Get event-level selected categories
                                                                        $eventSelectedCategories = $eventLayoutConfig['cards']['figuresCard']['visibleCategories'] ?? [];
                                                                        ?>
                                                                        
                                                                        <?php if (empty($availableCategories)): ?>
                                                                            <div class="alert alert-warning alert-sm mb-0">
                                                                                <i class="fas fa-info-circle me-1"></i>
                                                                                <strong>No categories found.</strong> Please select a sport discipline for the event, or add categories in 
                                                                                <a href="figures_management.php" target="_blank">Figures Management</a>.
                                                                            </div>
                                                                        <?php else: ?>
                                                                            <div class="form-check mb-2">
                                                                                <input class="form-check-input" type="checkbox" 
                                                                                       id="heat_use_event_categories_<?= $h ?>"
                                                                                       <?= empty($heatSelectedCategories) ? 'checked' : '' ?>
                                                                                       onchange="toggleHeatCategoryOverride(<?= $h ?>, this.checked)">
                                                                                <label class="form-check-label fw-semibold" for="heat_use_event_categories_<?= $h ?>">
                                                                                    <span class="badge bg-secondary">Use Event Settings</span>
                                                                                    <?php if (!empty($eventSelectedCategories)): ?>
                                                                                        <small class="text-muted ms-2">(<?= count($eventSelectedCategories) ?> categories)</small>
                                                                                    <?php else: ?>
                                                                                        <small class="text-muted ms-2">(All categories)</small>
                                                                                    <?php endif; ?>
                                                                                </label>
                                                                            </div>
                                                                            
                                                                            <div id="heat_category_override_<?= $h ?>" style="display: <?= !empty($heatSelectedCategories) ? 'block' : 'none' ?>;">
                                                                                <hr class="my-2">
                                                                                <p class="small fw-semibold mb-2"><i class="fas fa-layer-group me-1"></i>Heat-Specific Categories:</p>
                                                                                <div class="row g-2">
                                                                                    <?php foreach ($availableCategories as $category): ?>
                                                                                        <?php 
                                                                                        $categoryName = $category['category_name'];
                                                                                        // If heat has override, use heat selection; otherwise show event default
                                                                                        $isSelected = !empty($heatSelectedCategories) 
                                                                                            ? in_array($categoryName, $heatSelectedCategories)
                                                                                            : (empty($eventSelectedCategories) || in_array($categoryName, $eventSelectedCategories));
                                                                                        ?>
                                                                                        <div class="col-md-6">
                                                                                            <div class="form-check">
                                                                                                <input class="form-check-input heat-figure-category-checkbox" type="checkbox" 
                                                                                                       name="heat_scoring_layout[<?= $h ?>][cards][figuresCard][visibleCategories][]" 
                                                                                                       id="heat_category_<?= $h ?>_<?= htmlspecialchars($categoryName) ?>" 
                                                                                                       value="<?= htmlspecialchars($categoryName) ?>" 
                                                                                                       data-heat="<?= $h ?>"
                                                                                                       <?= $isSelected ? 'checked' : '' ?>>
                                                                                                <label class="form-check-label" for="heat_category_<?= $h ?>_<?= htmlspecialchars($categoryName) ?>">
                                                                                                    <span class="badge bg-secondary"><?= htmlspecialchars($categoryName) ?></span>
                                                                                                </label>
                                                                                            </div>
                                                                                        </div>
                                                                                    <?php endforeach; ?>
                                                                                </div>
                                                                                <div class="mt-2">
                                                                                    <button type="button" class="btn btn-sm btn-outline-primary" onclick="toggleAllHeatCategories(<?= $h ?>, true)">
                                                                                        <i class="fas fa-check-square me-1"></i>Select All
                                                                                    </button>
                                                                                    <button type="button" class="btn btn-sm btn-outline-secondary" onclick="toggleAllHeatCategories(<?= $h ?>, false)">
                                                                                        <i class="fas fa-square me-1"></i>Deselect All
                                                                                    </button>
                                                                                </div>
                                                                            </div>
                                                                            
                                                                            <div class="form-text mt-2">
                                                                                <i class="fas fa-info-circle me-1"></i>
                                                                                Leave "Use Event Settings" checked to inherit from event configuration, or uncheck to set heat-specific categories.
                                                                            </div>
                                                                        <?php endif; ?>
                                                                    </div>
                                                                <?php endif; ?>
                                                            </div>
                                                        <?php endforeach; ?>
                                                    </div>
                                                </div>
                                            <?php endforeach; ?>
                                        </div>
                                        
                                        <div class="col-lg-4">
                                            <div class="card">
                                                <div class="card-header">
                                                    <h6 class="mb-0">
                                                        <i class="fas fa-cogs me-2"></i>Layout Options
                                                    </h6>
                                                </div>
                                                <div class="card-body">
                                                    <div class="mb-3">
                                                        <label class="form-label">Display Mode</label>
                                                        <select class="form-select form-select-sm" 
                                                                name="heat_scoring_layout[<?= $h ?>][layout][compactMode]">
                                                            <option value="">Use Event Default</option>
                                                            <option value="0" <?= ($heatLayoutConfig['layout']['compactMode'] ?? null) === false ? 'selected' : '' ?>>Full Size Cards</option>
                                                            <option value="1" <?= ($heatLayoutConfig['layout']['compactMode'] ?? null) === true ? 'selected' : '' ?>>Compact Mode</option>
                                                        </select>
                                                    </div>
                                                    
                                                    <div class="mb-3">
                                                        <div class="form-check">
                                                            <input class="form-check-input" type="checkbox" 
                                                                   name="heat_scoring_layout[<?= $h ?>][layout][hideEmptyCards]" 
                                                                   id="heat_hideEmpty_<?= $h ?>" value="1" 
                                                                   <?= ($heatLayoutConfig['layout']['hideEmptyCards'] ?? true) ? 'checked' : '' ?>>
                                                            <label class="form-check-label" for="heat_hideEmpty_<?= $h ?>">
                                                                Hide Empty Cards
                                                            </label>
                                                        </div>
                                                    </div>
                                                    
                                                    <hr>
                                                    
                                                    <div class="alert alert-warning">
                                                        <i class="fas fa-exclamation-triangle me-2"></i>
                                                        <strong>Note:</strong> Heat-specific settings override event defaults only for this heat.
                                                    </div>
                                                    
                                                    <button type="button" class="btn btn-outline-secondary btn-sm w-100" 
                                                            onclick="resetHeatLayout(<?= $h ?>)">
                                                        <i class="fas fa-undo me-1"></i>Reset to Event Defaults
                                                    </button>
                                                </div>
                                            </div>
                                            
                                            <div class="card mt-3">
                                                <div class="card-header">
                                                    <h6 class="mb-0">
                                                        <i class="fas fa-eye me-2"></i>Preview
                                                    </h6>
                                                </div>
                                                <div class="card-body">
                                                    <p class="text-muted mb-2">Enabled components for judges:</p>
                                                    <div id="layoutPreview_<?= $h ?>" class="small">
                                                        <!-- Will be populated by JavaScript -->
                                                    </div>
                                                    <hr>
                                                    <a href="../judge/score.php?event_id=<?= $selected_event_id ?>&heat=<?= $h ?>" 
                                                       target="_blank" class="btn btn-outline-primary btn-sm w-100">
                                                        <i class="fas fa-external-link-alt me-1"></i>
                                                        Test Judge Panel
                                                    </a>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                                <i class="fas fa-times me-1"></i>Cancel
                            </button>
                            <button type="submit" class="btn btn-success">
                                <i class="fas fa-save me-1"></i>Save Heat Configuration
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    <?php endforeach; ?>

    <?php elseif ($selected_event_id): ?>
        <div class="alert alert-warning">
            <i class="fas fa-exclamation-triangle me-2"></i>
            This event has no heats defined (heats_total = 0). 
            <a href="events.php" class="alert-link">Edit the event</a> to set the number of heats.
        </div>
    <?php endif; ?>
        </div>

</div></div>
        

</div>

<script>
const HEAT_CARD_LAYOUT_KEY = 'heatCardLayoutMode';
const HEAT_CARD_LAYOUT_OPTIONS = ['full', 'compact', 'monitor', 'monitor-row', 'minimal', 'judges'];

function getHeatCardLayoutPreference() {
    const saved = localStorage.getItem(HEAT_CARD_LAYOUT_KEY);
    return HEAT_CARD_LAYOUT_OPTIONS.includes(saved) ? saved : 'full';
}

window.setHeatCardLayoutPreference = function(layout) {
    if (!HEAT_CARD_LAYOUT_OPTIONS.includes(layout)) {
        return;
    }
    localStorage.setItem(HEAT_CARD_LAYOUT_KEY, layout);
    // Update body class
    document.body.className = document.body.className.replace(/\bheat-layout-\S+/g, '') + ' heat-layout-' + layout;
    
    // Update heat-buttons-container class based on layout
    const container = document.getElementById('heat-buttons-container');
    if (container) {
        if (['monitor', 'monitor-row', 'minimal', 'judges'].includes(layout)) {
            container.className = 'overflow-scroll p-3 row';
        } else if (['full', 'compact'].includes(layout)) {
            container.className = 'd-flex gap-4 overflow-scroll p-3';
        }
        // Add layout class
        container.className = container.className.replace(/\bheat-layout-\S+/g, '') + ' heat-layout-' + layout;
    }
    highlightHeatLayoutSelector();
    loadHeatCards(false);
};

function highlightHeatLayoutSelector() {
    const current = getHeatCardLayoutPreference();
    document.querySelectorAll('[data-heat-layout]').forEach(btn => {
        btn.classList.toggle('active', btn.dataset.heatLayout === current);
    });
}

// Modal functions
function testModal() {
    //console.log('Test modal function called');
    alert('Test function works! Bootstrap available: ' + (typeof bootstrap !== 'undefined'));
    
    if (typeof bootstrap !== 'undefined') {
        // Try to open the first heat modal
        const firstModal = document.querySelector('[id^="heatModal"]');
        if (firstModal) {
            //console.log('Found modal:', firstModal.id);
            try {
                const modal = new bootstrap.Modal(firstModal);
                modal.show();
                //console.log('Modal opened successfully');
            } catch (error) {
                console.error('Error opening modal:', error);
                alert('Error: ' + error.message);
            }
        } else {
            alert('No modal found');
        }
    }
}

function openHeatModal(heatNumber) {
    //console.log('Opening modal for heat:', heatNumber);
    
    // Wait a bit for Bootstrap to be fully loaded if needed
    setTimeout(() => {
        // Check if Bootstrap is available
        if (typeof bootstrap === 'undefined') {
            console.error('Bootstrap is not loaded!');
            alert('Bootstrap is not loaded. Please refresh the page.');
            return;
        }
        
        const modalElement = document.getElementById('heatModal' + heatNumber);
        if (!modalElement) {
            console.error('Modal element not found:', 'heatModal' + heatNumber);
            alert('Modal not found for heat ' + heatNumber);
            return;
        }
        
        try {
            const modal = new bootstrap.Modal(modalElement);
            modal.show();
            
            // Initialize the modal content
            setTimeout(() => {
                updateRunOptionsModal(heatNumber);
                calculateEstimatedFinishModal(heatNumber);
                toggleFlowConfigModal(heatNumber);
                
                // Update button text in main card when modal opens
                updateHeatButtonDisplay(heatNumber);
            }, 100);
            
            //console.log('Modal opened successfully for heat:', heatNumber);
        } catch (error) {
            console.error('Error opening modal:', error);
            alert('Error opening modal: ' + error.message);
        }
    }, 50);
}

function activateHeat(heatNumber) {
    // Store the heat number for later use
    window.pendingActivateHeat = heatNumber;
    
    // Get heat settings
    const heat = window.heatSettings[heatNumber] || {};
    const runsCount = heat.runs_count || 1;
    
    // Update modal titles
    document.getElementById('activateHeatNumber').textContent = heatNumber;
    document.getElementById('confirmHeatNumber').textContent = heatNumber;
    
    // Check for currently active heat
    let activeHeat = null;
    for (const heatNum in window.heatSettings) {
        if (window.heatSettings[heatNum].status === 'active') {
            activeHeat = {
                number: heatNum,
                name: window.heatSettings[heatNum].heat_name || 'Unnamed Heat'
            };
            break;
        }
    }
    
    // Show/hide warning about active heat
    const warningDiv = document.getElementById('activeHeatWarning');
    if (activeHeat && activeHeat.number != heatNumber) {
        document.getElementById('currentActiveHeatNumber').textContent = activeHeat.number;
        document.getElementById('currentActiveHeatName').textContent = activeHeat.name;
        warningDiv.classList.remove('d-none');
    } else {
        warningDiv.classList.add('d-none');
    }
    
    // Generate run selection buttons
    const runSelectionDiv = document.getElementById('runSelectionOptions');
    let runButtons = '';
    
    for (let i = 1; i <= runsCount; i++) {
        const isFirst = i === 1;
        runButtons += `
            <button type="button" class="btn btn-outline-success ${isFirst ? 'active' : ''}" 
                    onclick="selectActivateRun(${i})" 
                    data-run="${i}" 
                    id="activateRunBtn_${i}">
                <i class="fas fa-play-circle me-2"></i>Run ${i}
            </button>
        `;
    }
    
    runSelectionDiv.innerHTML = runButtons;
    
    // Set default selected run
    window.selectedActivateRun = 1;
    
    // Show the modal
    const modal = new bootstrap.Modal(document.getElementById('activateHeatModal'));
    modal.show();
}

function selectActivateRun(runNumber) {
    // Update selected run
    window.selectedActivateRun = runNumber;
    
    // Update button states
    document.querySelectorAll('#runSelectionOptions button').forEach(btn => {
        btn.classList.remove('active');
    });
    document.getElementById('activateRunBtn_' + runNumber).classList.add('active');
}

function confirmActivateHeat() {
    const heatNumber = window.pendingActivateHeat;
    const runNumber = window.selectedActivateRun || 1;
    const eventId = <?= json_encode($selected_event_id) ?>;
    
    if (!eventId || !heatNumber) {
        alert('Invalid heat selection');
        return;
    }
    
    // Create form and submit
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = window.location.href;
    
    // Add event_id
    const eventInput = document.createElement('input');
    eventInput.type = 'hidden';
    eventInput.name = 'event_id';
    eventInput.value = eventId;
    form.appendChild(eventInput);
    
    // Add save_settings flag
    const saveInput = document.createElement('input');
    saveInput.type = 'hidden';
    saveInput.name = 'save_settings';
    saveInput.value = '1';
    form.appendChild(saveInput);
    
    // Add is_active (which heat to activate)
    const activeInput = document.createElement('input');
    activeInput.type = 'hidden';
    activeInput.name = 'is_active';
    activeInput.value = heatNumber;
    form.appendChild(activeInput);
    
    // Add active_run for the heat being activated
    const runInput = document.createElement('input');
    runInput.type = 'hidden';
    runInput.name = 'active_run[' + heatNumber + ']';
    runInput.value = runNumber;
    form.appendChild(runInput);
    
    // Add all required heat data to maintain existing settings
    const heat = window.heatSettings[heatNumber] || {};
    
    // Add heat_name
    const nameInput = document.createElement('input');
    nameInput.type = 'hidden';
    nameInput.name = 'heat_name[' + heatNumber + ']';
    nameInput.value = heat.heat_name || '';
    form.appendChild(nameInput);
    
    // Add other required fields
    const fields = {
        'scoring_type': heat.scoring_type || 'Points',
        'runs_count': heat.runs_count || 1,
        'runs_scoring_method': heat.runs_scoring_method || 'best_from_all',
        'time_start': heat.time_start || '',
        'estimate_time_per_participant': heat.estimate_time_per_participant || 0,
        'flow_type': heat.flow_type || 'none',
        'flow_source_heat': heat.flow_source_heat || '',
        'flow_participants_per_category': heat.flow_participants_per_category || 0,
        'flow_position_range': heat.flow_position_range || ''
    };
    
    for (const [fieldName, fieldValue] of Object.entries(fields)) {
        const input = document.createElement('input');
        input.type = 'hidden';
        input.name = fieldName + '[' + heatNumber + ']';
        input.value = fieldValue;
        form.appendChild(input);
    }
    
    // Submit the form
    document.body.appendChild(form);
    form.submit();
}

// Heat Cards AJAX Loading
let heatCardsRefreshTimer = null;
let currentRefreshInterval = 30; // Default 30 seconds

// Add new heat function
function addNewHeat() {
    const eventId = <?= json_encode($selected_event_id) ?>;
    
    if (!eventId) {
        showToast('No event selected', 'warning');
        return;
    }
    
    // Check token price first
    const price = <?= TokenSystem::price('add_heat') ?? 'null' ?>;
    
    if (price === null) {
        showToast('Token price not configured for this action', 'danger');
        return;
    }
    
    // Show Bootstrap modal for confirmation
    const modal = new bootstrap.Modal(document.getElementById('addHeatConfirmModal'));
    document.getElementById('addHeatTokenCost').textContent = price;
    modal.show();
}

// Confirm add heat from modal
function confirmAddHeat() {
    const eventId = <?= json_encode($selected_event_id) ?>;
    
    // Close the modal
    const modal = bootstrap.Modal.getInstance(document.getElementById('addHeatConfirmModal'));
    if (modal) {
        modal.hide();
    }
    
    // Disable button and show loading
    const addBtn = document.getElementById('addHeatBtn');
    if (addBtn) {
        addBtn.disabled = true;
        addBtn.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i>Adding...';
    }
    
    // Send request
    const formData = new FormData();
    formData.append('add_heat', '1');
    formData.append('event_id', eventId);
    
    fetch(window.location.pathname + '?event_id=' + eventId, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast(data.message, 'success');
            
            // Reload page to show new heat
            setTimeout(() => {
                window.location.reload();
            }, 1500);
        } else {
            showToast(data.message || 'Failed to add heat', 'danger');
            
            // Re-enable button
            if (addBtn) {
                addBtn.disabled = false;
                addBtn.innerHTML = '<i class="fas fa-plus me-1"></i>Add Heat';
            }
        }
    })
    .catch(error => {
        console.error('Error adding heat:', error);
        showToast('Network error: ' + error.message, 'danger');
        
        // Re-enable button
        if (addBtn) {
            addBtn.disabled = false;
            addBtn.innerHTML = '<i class="fas fa-plus me-1"></i>Add Heat';
        }
    });
}

// Toast notification helper
function showToast(message, type = 'info') {
    const alertDiv = document.createElement('div');
    alertDiv.className = `alert alert-${type} alert-dismissible fade show position-fixed top-0 start-50 translate-middle-x mt-3`;
    alertDiv.style.zIndex = '9999';
    alertDiv.style.minWidth = '300px';
    alertDiv.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    `;
    document.body.appendChild(alertDiv);
    
    setTimeout(() => {
        alertDiv.remove();
    }, 5000);
}

function loadHeatCards(showLoading = true) {
    const eventId = <?= json_encode($selected_event_id) ?>;
    const container = document.getElementById('heat-buttons-container');
    const layout = getHeatCardLayoutPreference();
    
    if (!eventId) {
        container.innerHTML = '<div class="col-12"><div class="alert alert-warning">No event selected</div></div>';
        return;
    }
    
    if (showLoading) {
        const refreshBtn = document.querySelector('[onclick="loadHeatCards()"]');
        if (refreshBtn) {
            const icon = refreshBtn.querySelector('i');
            if (icon) {
                icon.classList.add('fa-spin');
            }
        }
    }
    
    fetch(`heat_cards_api.php?event_id=${eventId}&layout=${layout}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                container.innerHTML = data.data;
                
                // Update last refresh time
                const now = new Date();
                const timeStr = now.toLocaleTimeString();
                const lastRefreshEl = document.getElementById('lastRefreshTime');
                if (lastRefreshEl) {
                    lastRefreshEl.textContent = 'Last: ' + timeStr;
                }
                
                //console.log('Heat cards loaded:', data.heat_count, 'heats at', data.timestamp);
            } else {
                container.innerHTML = '<div class="col-12"><div class="alert alert-danger">Error: ' + (data.message || 'Failed to load heat cards') + '</div></div>';
            }
        })
        .catch(error => {
            console.error('Error loading heat cards:', error);
            container.innerHTML = '<div class="col-12"><div class="alert alert-danger">Network error loading heat cards</div></div>';
        })
        .finally(() => {
            // Stop spinning icon
            const refreshBtn = document.querySelector('[onclick="loadHeatCards()"]');
            if (refreshBtn) {
                const icon = refreshBtn.querySelector('i');
                if (icon) {
                    icon.classList.remove('fa-spin');
                }
            }
        });
}

function setRefreshInterval() {
    const select = document.getElementById('refreshInterval');
    const interval = parseInt(select.value);
    
    // Clear existing timer
    if (heatCardsRefreshTimer) {
        clearInterval(heatCardsRefreshTimer);
        heatCardsRefreshTimer = null;
    }
    
    currentRefreshInterval = interval;
    
    if (interval > 0) {
        // Set up auto-refresh
        heatCardsRefreshTimer = setInterval(() => {
            loadHeatCards(false); // Load without showing loading state
            refreshFlowEditorNodes(); // Update flow editor nodes
        }, interval * 1000);
        
        //console.log('Auto-refresh enabled:', interval, 'seconds');
    } else {
        //console.log('Auto-refresh disabled (manual mode)');
    }
}

// Refresh flow editor nodes status
function refreshFlowEditorNodes() {
    const eventId = <?= json_encode($selected_event_id) ?>;
    
    if (!eventId || !window.flowEditor) {
        return;
    }
    
    fetch('heat_flow_api.php?event_id=' + eventId)
        .then(response => response.json())
        .then(data => {
            if (!data.flow_chains) {
                console.warn('No flow chains data received');
                return;
            }
            
            // Flatten flow chains to get all heats
            const heatsData = [];
            data.flow_chains.forEach(chain => {
                chain.forEach(heatInfo => {
                    heatsData.push(heatInfo);
                });
            });
            
            // Update each node with new data
            heatsData.forEach(heatInfo => {
                const heatNumber = heatInfo.heat_number;
                const nodeData = window.flowEditor.nodes.get(heatNumber);
                
                if (!nodeData) {
                    return; // Node doesn't exist
                }
                
                // Determine status based on heat state and progress
                let status = 'pending';
                if (heatInfo.is_active) {
                    status = 'active';
                } else if (heatInfo.progress) {
                    // Check completion percentage
                    const completion = heatInfo.progress.completion_percentage || 0;
                    const scoringCompletion = heatInfo.progress.scoring_completion_percentage || 0;
                    
                    // Heat is completed if either:
                    // 1. All participants completed (completion >= 100)
                    // 2. All participants have scores (scoring_completion >= 100)
                    if (completion >= 100 || scoringCompletion >= 100) {
                        status = 'completed';
                    } else if (heatInfo.bib_latest_on_run > 0) {
                        // Heat has started but not completed
                        status = 'in_progress';
                    }
                }
                
                // Update node data with all available information
                nodeData.data.is_active = heatInfo.is_active || false;
                nodeData.data.status = status;
                nodeData.data.participant_count = heatInfo.participant_count || 0;
                nodeData.data.active_run = heatInfo.active_run || 1;
                nodeData.data.runs_count = heatInfo.runs_count || 1;
                nodeData.data.time_start = heatInfo.time_start || '';
                nodeData.data.estimate_time_per_participant = heatInfo.estimate_time_per_participant || 0;
                nodeData.data.bib_on_start = heatInfo.bib_on_start || null;
                nodeData.data.bib_latest_on_run = heatInfo.bib_latest_on_run || null;
                
                // Update node element HTML to reflect new status
                updateNodeElement(nodeData);
            });
            
            //console.log('Flow editor nodes refreshed at', new Date().toLocaleTimeString());
        })
        .catch(error => {
            console.error('Error refreshing flow editor nodes:', error);
        });
}

// Build node HTML from data (shared template for create and update)
function buildNodeHTML(nodeData) {
    // Calculate estimated end time and duration
    let timeHtml = '';
    if (nodeData.time_start) {
        const startTime = nodeData.time_start;
        const estimateMinutes = nodeData.participant_count * (nodeData.estimate_time_per_participant || 0);
        const estimateSeconds = estimateMinutes * 60;
        
        // Calculate end time
        const startParts = startTime.split(':');
        const startDate = new Date();
        startDate.setHours(parseInt(startParts[0]), parseInt(startParts[1]), parseInt(startParts[2] || 0));
        const endDate = new Date(startDate.getTime() + estimateSeconds * 1000);
        
        const endTime = endDate.getHours().toString().padStart(2, '0') + ':' + 
                       endDate.getMinutes().toString().padStart(2, '0');
        const duration = Math.round(estimateMinutes) + 'min';
        
        timeHtml = ''
    }
    
    // Status badge
    let statusBadge = '';
    let statusIcon = '';
    let statusClass = 'bg-secondary';
    
    if (nodeData.is_active) {
        statusIcon = 'fa-play';
        statusClass = 'bg-success';
        statusBadge = 'ACTIVE';
    } else if (nodeData.status === 'completed' || nodeData.status === 'finished') {
        statusIcon = 'fa-check';
        statusClass = 'bg-info';
        statusBadge = 'COMPLETED';
    } else if (nodeData.status === 'in_progress') {
        statusIcon = 'fa-hourglass-half';
        statusClass = 'bg-warning';
        statusBadge = 'IN PROGRESS';
    } else {
        statusIcon = 'fa-pause';
        statusClass = 'bg-secondary';
        statusBadge = 'PENDING';
    }
    
    return '<div class="card-header bg-light border-0 p-3">' +
        '<div class="d-flex justify-content-between align-items-start">' +
            '<div class="d-flex flex-column text-primary">' +
                '<h5 class="mb-1 fw-bold">' + nodeData.heat_name +  '</h5>' +
                (nodeData.heat_name ? '<small class="mb-0 fw-bold text-muted small"><i class="fas fa-fire me-2 text-muted"></i>Heat ' + nodeData.heat_number + '</small>' : '') +
            '</div>' +
            '<div class="d-flex flex-column">' +
                '<div class="row g-2">' +
                    '<div class="d-flex flex-column gap-1 align-items-end">' +
                        '<small class="badge small ' + statusClass + '">' +
                            '<i class="fas ' + statusIcon + ' me-1"></i>' + statusBadge +
                        '</small>' +
                    '</div>' +
                '</div>' +
                '<div class="row g-1">' +
                    '<div class="col-6">' +
                        '<div class="text-center p-1 bg-light bg-opacity-25 rounded d-flex flex-row align-items-center justify-content-center gap-1 small" title="Athletes">' +
                            '<i class="fas fa-users text-primary mb-1"></i>' +
                            '<div class="fw-bold">' + nodeData.participant_count + '</div>' +
                        '</div>' +
                    '</div>' +
                    '<div class="col-6">' +
                        '<div class="text-center p-1 bg-light bg-opacity-25 rounded d-flex flex-row align-items-center justify-content-center gap-1 small" title="Runs">' +
                            '<i class="fas fa-redo text-info mb-1"></i>' +
                            '<div class="fw-bold">' + nodeData.runs_count + '</div>' +
                        '</div>' +
                    '</div>' +
                '</div>' +
            '</div>' +
        '</div>' +
        '<small class="fw-bold text-primary-emphasis"><i class="fas fa-play-circle me-1 small"></i>Current Run</small> ' +
        '<span class="badge small ' + (nodeData.is_active ? 'bg-success' : 'bg-secondary') + '">' + 
            nodeData.active_run + '/' + nodeData.runs_count + 
        '</span>' +
    '</div>' +
    timeHtml +
    '<div class="flow-ports">' +
        '<div class="text-center">' +
            '<div class="flow-port flow-port-in" data-port="in" data-heat="' + nodeData.heat_number + '"></div>' +
            '' +
        '</div>' +
        '<div class="text-center">' +
            '<div class="flow-port flow-port-out" data-port="out" data-heat="' + nodeData.heat_number + '"></div>' +
            '' +
        '</div>' +
    '</div>' +
    '<div class="position-absolute bottom-0 end-0 m-2" style="z-index: 10;">' +
        '<button class="btn btn-sm btn-light border-0 shadow-sm rounded-circle p-1 " ' +
            'onclick="event.stopPropagation(); openHeatModal(' + nodeData.heat_number + ')" ' +
            'title="Configure heat settings" ' +
            'style="width: 32px; height: 32px; display: flex; align-items: center; justify-content: center;">' +
            '<i class="fas fa-cog text-primary"></i>' +
        '</button>' +
    '</div>';
}

// Update node element with current data
function updateNodeElement(nodeData) {
    const element = nodeData.element;
    const data = nodeData.data;
    
    // Update active class
    if (data.is_active) {
        element.classList.add('active-heat');
    } else {
        element.classList.remove('active-heat');
    }
    
    // Rebuild HTML using shared template
    element.innerHTML = buildNodeHTML(data);
    
    // Re-attach port click handlers
    const ports = element.querySelectorAll('.flow-port');
    ports.forEach(port => {
        port.addEventListener('click', window.flowEditor.handlePortClick);
    });
}

function showHeatResults(heatNumber) {
    const eventId = <?= json_encode($selected_event_id) ?>;
    
    if (!eventId) {
        alert('No event selected');
        return;
    }
    
    // Get heat name from heat settings
    const heat = window.heatSettings[heatNumber] || {};
    const heatName = heat.heat_name || 'Unnamed Heat';
    
    // Update modal title with heat number and name
    document.getElementById('resultsHeatNumber').textContent = heatNumber;
    document.getElementById('resultsHeatName').textContent = heatName;
    
    // Show the modal
    const modal = new bootstrap.Modal(document.getElementById('heatResultsModal'));
    modal.show();
    
    // Load configurations for this heat
    loadHeatConfigurations(eventId, heatNumber);
}

function loadHeatConfigurations(eventId, heatNumber) {
    const configsList = document.getElementById('savedConfigsList');
    
    // Show loading state
    configsList.innerHTML = `
        <div class="text-center py-3 text-muted">
            <div class="spinner-border spinner-border-sm" role="status">
                <span class="visually-hidden">Loading...</span>
            </div>
            <p class="mt-2 small">Loading configurations...</p>
        </div>
    `;
    
    // Fetch all configurations for the event
    fetch(`../api/public_dashboard_api.php?action=get_configurations&event_id=${eventId}`)
        .then(response => response.json())
        .then(data => {
            if (!data.success || !data.configurations || data.configurations.length === 0) {
                configsList.innerHTML = `
                    <div class="alert alert-warning mb-0 small">
                        <i class="fas fa-exclamation-triangle me-2"></i>
                        No saved reports found for this event.
                    </div>
                `;
                return;
            }
            
            // Use all configurations (no filtering)
            const allConfigs = data.configurations;
            
            // Build configuration list
            let html = '';
            allConfigs.forEach((config, index) => {
                const isActive = index === 0 ? 'active' : '';
                const viewTypeIcon = config.view_type === 'start_list' ? 'fa-list-ol' : 'fa-table';
                const viewTypeLabel = config.view_type === 'start_list' ? 'Start List' : 'Summary';
                const heatLabel = config.heat_number ? `Heat ${config.heat_number}` : 'All Heats';
                
                html += `
                    <button type="button" 
                            class="list-group-item list-group-item-action config-button ${isActive}"
                            data-config-id="${config.id}"
                            onclick="loadHeatConfiguration(${config.id}, '${config.view_type}')"
                            title="Click to load">
                        <div class="d-flex flex-column">
                            <div class="d-flex justify-content-between align-items-start mb-1">
                                <span class="me-2 text-start flex-grow-1">
                                    <i class="fas ${viewTypeIcon} me-2"></i>${escapeHtml(config.name)}
                                </span>
                                <span class="badge bg-light text-dark">${viewTypeLabel}</span>
                            </div>
                            <small class="text-muted">
                                <i class="fas fa-fire me-1"></i>${heatLabel}
                            </small>
                        </div>
                    </button>
                `;
            });
            
            configsList.innerHTML = html;
            
            // Auto-load first configuration
            if (allConfigs.length > 0) {
                loadHeatConfiguration(allConfigs[0].id, allConfigs[0].view_type);
            }
        })
        .catch(error => {
            console.error('Error loading configurations:', error);
            configsList.innerHTML = `
                <div class="alert alert-danger mb-0 small">
                    <i class="fas fa-times-circle me-2"></i>
                    Error loading configurations: ${escapeHtml(error.message)}
                </div>
            `;
        });
}

function loadHeatConfiguration(configId, viewType) {
    const contentDisplay = document.getElementById('resultsContentDisplay');
    
    // Update active state
    document.querySelectorAll('#savedConfigsList .config-button').forEach(btn => {
        btn.classList.remove('active');
    });
    document.querySelector(`[data-config-id="${configId}"]`)?.classList.add('active');
    
    // Show loading
    contentDisplay.innerHTML = `
        <div class="text-center py-5">
            <div class="spinner-border text-primary" role="status">
                <span class="visually-hidden">Loading...</span>
            </div>
            <p class="mt-2 text-muted">Loading results...</p>
        </div>
    `;
    
    // Determine API endpoint based on view type
    const apiEndpoint = viewType === 'start_list' 
        ? '../api/start_list_api.php' 
        : '../api/summary_table_api.php';
    
    // Load configuration
    fetch(`${apiEndpoint}?config_id=${configId}&format=html`)
        .then(response => {
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}`);
            }
            return response.text();
        })
        .then(html => {
            contentDisplay.innerHTML = html;
        })
        .catch(error => {
            console.error('Error loading configuration:', error);
            contentDisplay.innerHTML = `
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    Error loading results: ${error.message}
                </div>
            `;
        });
}

function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// ============================================================================
// CONFIRMATION MODAL HELPERS
// ============================================================================

let actionConfirmModalInstance = null;
let actionConfirmResolveFn = null;

function getActionConfirmModal() {
    const modalEl = document.getElementById('actionConfirmModal');
    if (!modalEl) return null;
    if (!actionConfirmModalInstance) {
        actionConfirmModalInstance = new bootstrap.Modal(modalEl);
        modalEl.addEventListener('hidden.bs.modal', () => {
            if (actionConfirmResolveFn) {
                actionConfirmResolveFn(false);
                actionConfirmResolveFn = null;
            }
        });
    }
    return actionConfirmModalInstance;
}

function showActionConfirm(options = {}) {
    return new Promise((resolve) => {
        const modalEl = document.getElementById('actionConfirmModal');
        const modalInstance = getActionConfirmModal();
        if (!modalEl || !modalInstance) {
            resolve(window.confirm(options.message || 'Are you sure?'));
            return;
        }
        const {
            title = 'Confirm Action',
            message = 'Are you sure you want to continue?',
            confirmLabel = 'Confirm',
            confirmVariant = 'primary',
            iconClass = 'fas fa-question-circle text-primary'
        } = options;
        modalEl.querySelector('.action-confirm-title').textContent = title;
        modalEl.querySelector('.action-confirm-message').textContent = message;
        const iconEl = modalEl.querySelector('.action-confirm-icon');
        if (iconEl) {
            iconEl.className = `action-confirm-icon ${iconClass}`;
        }
        const confirmBtn = modalEl.querySelector('.action-confirm-approve');
        const cancelBtn = modalEl.querySelector('.action-confirm-cancel');
        if (confirmBtn) {
            confirmBtn.textContent = confirmLabel;
            confirmBtn.className = `btn action-confirm-approve btn-${confirmVariant}`;
            confirmBtn.onclick = () => {
                resolve(true);
                actionConfirmResolveFn = null;
                modalInstance.hide();
            };
        }
        if (cancelBtn) {
            cancelBtn.onclick = () => {
                resolve(false);
                actionConfirmResolveFn = null;
                modalInstance.hide();
            };
        }
        actionConfirmResolveFn = resolve;
        modalInstance.show();
    });
}

async function changeActiveRun(heatNumber, direction) {
    const eventId = <?= json_encode($selected_event_id) ?>;
    
    if (!eventId) {
        alert('No event selected');
        return;
    }
    
    const action = direction > 0 ? 'next' : 'previous';
    const confirmed = await showActionConfirm({
        title: `Change Active Run`,
        message: `Switch to the ${action.toUpperCase()} run for Heat ${heatNumber}?`,
        confirmLabel: direction > 0 ? 'Next Run' : 'Previous Run',
        confirmVariant: 'primary',
        iconClass: 'fas fa-shuffle text-primary'
    });
    if (!confirmed) return;
    
    // Submit via form
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = window.location.href;
    
    const eventInput = document.createElement('input');
    eventInput.type = 'hidden';
    eventInput.name = 'event_id';
    eventInput.value = eventId;
    form.appendChild(eventInput);
    
    const actionInput = document.createElement('input');
    actionInput.type = 'hidden';
    actionInput.name = 'change_active_run';
    actionInput.value = '1';
    form.appendChild(actionInput);
    
    const heatInput = document.createElement('input');
    heatInput.type = 'hidden';
    heatInput.name = 'heat_number';
    heatInput.value = heatNumber;
    form.appendChild(heatInput);
    
    const directionInput = document.createElement('input');
    directionInput.type = 'hidden';
    directionInput.name = 'direction';
    directionInput.value = direction;
    form.appendChild(directionInput);
    
    document.body.appendChild(form);
    form.submit();
}

async function closeHeat(heatNumber) {
    const eventId = <?= json_encode($selected_event_id) ?>;
    
    if (!eventId) {
        alert('No event selected');
        return;
    }
    
    const confirmed = await showActionConfirm({
        title: `Finish Heat ${heatNumber}`,
        message: 'Mark this heat as finished? This will end active scoring.',
        confirmLabel: 'Yes, Finish Heat',
        confirmVariant: 'success',
        iconClass: 'fas fa-flag-checkered text-success'
    });
    if (!confirmed) return;
    
    // Submit via form
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = window.location.href;
    
    const eventInput = document.createElement('input');
    eventInput.type = 'hidden';
    eventInput.name = 'event_id';
    eventInput.value = eventId;
    form.appendChild(eventInput);
    
    const actionInput = document.createElement('input');
    actionInput.type = 'hidden';
    actionInput.name = 'close_heat';
    actionInput.value = '1';
    form.appendChild(actionInput);
    
    const heatInput = document.createElement('input');
    heatInput.type = 'hidden';
    heatInput.name = 'heat_number';
    heatInput.value = heatNumber;
    form.appendChild(heatInput);
    
    document.body.appendChild(form);
    form.submit();
}

async function cancelHeat(heatNumber) {
    const eventId = <?= json_encode($selected_event_id) ?>;
    
    if (!eventId) {
        alert('No event selected');
        return;
    }
    
    const confirmed = await showActionConfirm({
        title: `Cancel Heat ${heatNumber}`,
        message: 'Cancel this heat? Athletes will no longer compete in it.',
        confirmLabel: 'Yes, Cancel Heat',
        confirmVariant: 'danger',
        iconClass: 'fas fa-ban text-danger'
    });
    if (!confirmed) return;
    
    // Submit via form
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = window.location.href;
    
    const eventInput = document.createElement('input');
    eventInput.type = 'hidden';
    eventInput.name = 'event_id';
    eventInput.value = eventId;
    form.appendChild(eventInput);
    
    const actionInput = document.createElement('input');
    actionInput.type = 'hidden';
    actionInput.name = 'cancel_heat';
    actionInput.value = '1';
    form.appendChild(actionInput);
    
    const heatInput = document.createElement('input');
    heatInput.type = 'hidden';
    heatInput.name = 'heat_number';
    heatInput.value = heatNumber;
    form.appendChild(heatInput);
    
    document.body.appendChild(form);
    form.submit();
}

async function rescheduleHeat(heatNumber) {
    const eventId = <?= json_encode($selected_event_id) ?>;
    
    if (!eventId) {
        alert('No event selected');
        return;
    }
    
    const confirmed = await showActionConfirm({
        title: `Reschedule Heat ${heatNumber}`,
        message: 'Mark this heat for rescheduling? It will move out of the live queue.',
        confirmLabel: 'Yes, Reschedule',
        confirmVariant: 'warning',
        iconClass: 'fas fa-calendar-alt text-warning'
    });
    if (!confirmed) return;
    
    // Submit via form
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = window.location.href;
    
    const eventInput = document.createElement('input');
    eventInput.type = 'hidden';
    eventInput.name = 'event_id';
    eventInput.value = eventId;
    form.appendChild(eventInput);
    
    const actionInput = document.createElement('input');
    actionInput.type = 'hidden';
    actionInput.name = 'reschedule_heat';
    actionInput.value = '1';
    form.appendChild(actionInput);
    
    const heatInput = document.createElement('input');
    heatInput.type = 'hidden';
    heatInput.name = 'heat_number';
    heatInput.value = heatNumber;
    form.appendChild(heatInput);
    
    document.body.appendChild(form);
    form.submit();
}

async function resetParticipantStatus(heatNumber) {
    const eventId = <?= json_encode($selected_event_id) ?>;
    
    if (!eventId) {
        alert('No event selected');
        return;
    }
    
    const confirmed = await showActionConfirm({
        title: `Reset Participant Status - Heat ${heatNumber}`,
        message: 'Reset all participants in this heat to "initial" status? This will clear their current status.',
        confirmLabel: 'Yes, Reset Status',
        confirmVariant: 'warning',
        iconClass: 'fas fa-redo-alt text-warning'
    });
    if (!confirmed) return;
    
    try {
        const formData = new FormData();
        formData.append('event_id', eventId);
        formData.append('heat_number', heatNumber);
        
        const response = await fetch('reset_participant_status_api.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            showSuccessToast('✅ Status Reset', result.message);
            // Refresh heat cards to show updated status
            if (typeof loadHeatCards === 'function') {
                loadHeatCards();
            }
        } else {
            showErrorToast('❌ Reset Failed', result.message || 'Failed to reset participant status');
        }
    } catch (error) {
        console.error('Error resetting participant status:', error);
        showErrorToast('❌ Error', 'An error occurred while resetting participant status');
    }
}

async function setPendingHeat(heatNumber) {
    const eventId = <?= json_encode($selected_event_id) ?>;
    
    if (!eventId) {
        alert('No event selected');
        return;
    }
    
    const confirmed = await showActionConfirm({
        title: `Set Heat ${heatNumber} to Pending`,
        message: 'Return this heat to pending state? It will be removed from active view.',
        confirmLabel: 'Yes, Set Pending',
        confirmVariant: 'secondary',
        iconClass: 'fas fa-pause-circle text-secondary'
    });
    if (!confirmed) return;
    
    // Submit via form
    const form = document.createElement('form');
    form.method = 'POST';
    form.action = window.location.href;
    
    const eventInput = document.createElement('input');
    eventInput.type = 'hidden';
    eventInput.name = 'event_id';
    eventInput.value = eventId;
    form.appendChild(eventInput);
    
    const actionInput = document.createElement('input');
    actionInput.type = 'hidden';
    actionInput.name = 'set_pending_heat';
    actionInput.value = '1';
    form.appendChild(actionInput);
    
    const heatInput = document.createElement('input');
    heatInput.type = 'hidden';
    heatInput.name = 'heat_number';
    heatInput.value = heatNumber;
    form.appendChild(heatInput);
    
    document.body.appendChild(form);
    form.submit();
}

function updateHeatButtonDisplay(heatNumber) {
    // Update the button text and heat name display in the main cards when data changes
    const heatNameInput = document.querySelector(`input[name="heat_name[${heatNumber}]"]`);
    if (heatNameInput) {
        // Update card display based on modal input
        heatNameInput.addEventListener('input', function() {
            const cardText = document.querySelector(`#heat-buttons-container .card:nth-child(${heatNumber}) .card-text`);
            if (cardText) {
                cardText.textContent = this.value.trim() || 'Unnamed Heat';
            }
        });
    }
}

// Modal-specific functions (copies of existing functions with modal suffix)
function toggleActiveHeatModal(heatNumber) {
    // Show/hide run selector in modal
    const runSelector = document.getElementById(`run_selector_modal_${heatNumber}`);
    const isActive = document.getElementById(`active_modal_${heatNumber}`).checked;
    
    if (runSelector) {
        if (isActive) {
            runSelector.classList.remove('d-none');
        } else {
            runSelector.classList.add('d-none');
        }
    }
}

function toggleAllCategoriesModal(heatNumber) {
    const allCheckbox = document.getElementById(`all_categories_modal_${heatNumber}`);
    const categoryCheckboxes = document.querySelectorAll(`input[data-heat="${heatNumber}"].category-checkbox`);
    
    if (allCheckbox.checked) {
        categoryCheckboxes.forEach(checkbox => {
            checkbox.checked = false;
            checkbox.disabled = true;
        });
    } else {
        categoryCheckboxes.forEach(checkbox => {
            checkbox.disabled = false;
        });
    }
}

function updateCategorySelectionModal(heatNumber) {
    const allCheckbox = document.getElementById(`all_categories_modal_${heatNumber}`);
    const categoryCheckboxes = document.querySelectorAll(`input[data-heat="${heatNumber}"].category-checkbox:checked`);
    
    if (categoryCheckboxes.length > 0) {
        allCheckbox.checked = false;
    }
}

function toggleFlowConfigModal(heatNumber) {
    const flowType = document.querySelector(`select[name="flow_type[${heatNumber}]"]`).value;
    const flowConfig = document.getElementById(`flow_config_modal_${heatNumber}`);
    const participantsInput = document.querySelector(`input[name="flow_participants_per_category[${heatNumber}]"]`);
    const sourceHeatSelect = document.querySelector(`select[name="flow_source_heat[${heatNumber}]"]`);
    
    if (flowType !== 'none') {
        flowConfig.classList.remove('d-none');
        if (participantsInput) {
            participantsInput.removeAttribute('disabled');
            participantsInput.setAttribute('min', '1');
        }
        if (sourceHeatSelect) {
            sourceHeatSelect.removeAttribute('disabled');
        }
    } else {
        flowConfig.classList.add('d-none');
        if (participantsInput) {
            participantsInput.setAttribute('disabled', 'disabled');
            participantsInput.removeAttribute('min');
        }
        if (sourceHeatSelect) {
            sourceHeatSelect.setAttribute('disabled', 'disabled');
        }
    }
}

function calculateEstimatedFinishModal(heatNumber) {
    const startTime = document.querySelector(`input[name="time_start[${heatNumber}]"]`).value;
    const timePerParticipant = parseInt(document.querySelector(`input[name="estimate_time_per_participant[${heatNumber}]"]`).value) || 0;
    
    if (startTime && timePerParticipant > 0) {
        // Get actual participant count for this heat
        const participantCount = window.heatParticipantCounts ? window.heatParticipantCounts[heatNumber] || 0 : 0;
        
        if (participantCount > 0) {
            const totalSeconds = timePerParticipant * participantCount;
            const totalMinutes = Math.ceil(totalSeconds / 60);
            
            const [hours, minutes] = startTime.split(':').map(Number);
            const startDateTime = new Date();
            startDateTime.setHours(hours, minutes, 0, 0);
            
            const finishDateTime = new Date(startDateTime.getTime() + (totalMinutes * 60000));
            const finishTime = finishDateTime.toTimeString().slice(0, 5);
            
            const finishElement = document.getElementById(`finish_time_modal_${heatNumber}`);
            if (finishElement) {
                finishElement.textContent = `${finishTime} (${totalMinutes} minutes for ${participantCount} participants)`;
                document.getElementById(`estimated_finish_modal_${heatNumber}`).classList.remove('d-none');
            }
        } else {
            const finishElement = document.getElementById(`finish_time_modal_${heatNumber}`);
            if (finishElement) {
                finishElement.textContent = 'No participants assigned to this heat yet';
                document.getElementById(`estimated_finish_modal_${heatNumber}`).classList.remove('d-none');
            }
        }
    } else {
        const estimatedElement = document.getElementById(`estimated_finish_modal_${heatNumber}`);
        if (estimatedElement) {
            estimatedElement.classList.add('d-none');
        }
    }
}

function updateRunOptionsModal(heatNumber) {
    const runsCount = parseInt(document.querySelector(`select[name="runs_count[${heatNumber}]"]`).value);
    const scoringSelect = document.getElementById(`runs_scoring_modal_${heatNumber}`);
    
    if (!scoringSelect) return;
    
    // Store current selection to try to maintain it
    const currentValue = scoringSelect.value;
    
    // Clear existing options
    scoringSelect.innerHTML = '';
    
    // Generate scoring options based on runs count
    const scoringOptions = generateScoringOptions(runsCount);
    
    // Get the saved value from PHP for this heat
    const savedValue = 'best_from_all'; // Default fallback
    
    scoringOptions.forEach(option => {
        const optionElement = document.createElement('option');
        optionElement.value = option.value;
        optionElement.textContent = option.text;
        
        // Set selected if it matches saved value or current value
        if (option.value === savedValue || option.value === currentValue) {
            optionElement.selected = true;
        }
        
        scoringSelect.appendChild(optionElement);
    });
    
    // If no option was selected and we have options, select the first one
    if (!scoringSelect.value && scoringOptions.length > 0) {
        scoringSelect.value = scoringOptions[0].value;
    }
    
    // Update explanation
    updateScoringExplanationModal(heatNumber);
    
    // Update run selector for active heat selection
    updateRunSelectorModal(heatNumber);
}

function updateScoringExplanationModal(heatNumber) {
    const scoringSelect = document.getElementById(`runs_scoring_modal_${heatNumber}`);
    const explanationSpan = document.getElementById(`scoring_explanation_modal_${heatNumber}`);
    
    if (!scoringSelect || !explanationSpan) return;
    
    const selectedValue = scoringSelect.value;
    
    const explanations = {
        'single_run': 'Score from the single run attempt',
        'best_from_2': 'Highest score from 2 runs',
        'average_from_2': 'Average score from both runs',
        'best_from_3': 'Highest score from 3 runs',
        'average_best_2_from_3': 'Average of the 2 highest scores from 3 runs',
        'average_from_3': 'Average score from all 3 runs',
        'best_from_4': 'Highest score from 4 runs',
        'average_best_2_from_4': 'Average of the 2 highest scores from 4 runs',
        'average_best_3_from_4': 'Average of the 3 highest scores from 4 runs',
        'average_from_4': 'Average score from all 4 runs',
        'best_from_5': 'Highest score from 5 runs',
        'average_best_2_from_5': 'Average of the 2 highest scores from 5 runs',
        'average_best_3_from_5': 'Average of the 3 highest scores from 5 runs',
        'average_best_4_from_5': 'Average of the 4 highest scores from 5 runs',
        'average_from_5': 'Average score from all 5 runs'
    };
    
    explanationSpan.textContent = explanations[selectedValue] || 'How to calculate final score from multiple runs';
}

function getPreferredActiveRunSelection(heatNumber, runSelector, runsCount) {
    let preferredRun = null;
    const normalizedRunsCount = runsCount && !Number.isNaN(runsCount) ? runsCount : 1;
    if (runSelector) {
        const existingChecked = runSelector.querySelector('input[name="active_run[' + heatNumber + ']"]:checked');
        if (existingChecked) {
            preferredRun = parseInt(existingChecked.value, 10);
        } else if (runSelector.dataset && runSelector.dataset.activeRun) {
            preferredRun = parseInt(runSelector.dataset.activeRun, 10);
        }
    }
    if ((!preferredRun || Number.isNaN(preferredRun)) && window.heatSettings) {
        const heatData = window.heatSettings[heatNumber] || window.heatSettings[String(heatNumber)];
        if (heatData && heatData.active_run) {
            preferredRun = parseInt(heatData.active_run, 10);
        }
    }
    if (!preferredRun || Number.isNaN(preferredRun)) {
        preferredRun = 1;
    }
    if (preferredRun > normalizedRunsCount) {
        preferredRun = normalizedRunsCount;
    }
    if (preferredRun < 1) {
        preferredRun = 1;
    }
    return preferredRun;
}

function updateRunSelectorModal(heatNumber) {
    const runsCount = parseInt(document.querySelector(`select[name="runs_count[${heatNumber}]"]`).value) || 1;
    const runSelector = document.getElementById(`run_selector_modal_${heatNumber}`);
    
    if (!runSelector) return;
    const preferredRun = getPreferredActiveRunSelection(heatNumber, runSelector, runsCount);
    
    // Rebuild run radio buttons
    const runOptions = Array.from({length: runsCount}, (_, i) => {
        const runNum = i + 1;
        return `
            <div class="col-auto">
                <div class="form-check">
                    <input class="form-check-input" type="radio" 
                           name="active_run[${heatNumber}]" value="${runNum}" 
                           id="run_modal_${heatNumber}_${runNum}" ${runNum === preferredRun ? 'checked' : ''}>
                    <label class="form-check-label" for="run_modal_${heatNumber}_${runNum}">
                        Run ${runNum}
                    </label>
                </div>
            </div>
        `;
    }).join('');
    
    const runsContainer = runSelector.querySelector('.row');
    if (runsContainer) {
        runsContainer.innerHTML = runOptions;
        runSelector.dataset.activeRun = preferredRun;
        runsContainer.querySelectorAll(`input[name="active_run[${heatNumber}]"]`).forEach(input => {
            input.addEventListener('change', (event) => {
                runSelector.dataset.activeRun = event.target.value;
            });
        });
    }
}

function validateHeatForm(heatNumber) {
    let isValid = true;
    const errors = [];
    
    // Check flow configuration
    const flowType = document.querySelector(`select[name="flow_type[${heatNumber}]"]`);
    if (flowType) {
        const flowTypeValue = flowType.value;
        
        if (flowTypeValue !== 'none') {
            const participantsInput = document.querySelector(`input[name="flow_participants_per_category[${heatNumber}]"]`);
            const participantsValue = participantsInput ? parseInt(participantsInput.value) || 0 : 0;
            
            if (participantsValue < 1) {
                errors.push(`Please specify number of participants to promote (minimum 1)`);
                isValid = false;
            }
            
            if (flowTypeValue === 'promotion') {
                const sourceHeat = document.querySelector(`select[name="flow_source_heat[${heatNumber}]"]`);
                const sourceHeatValue = sourceHeat ? sourceHeat.value : '';
                if (!sourceHeatValue) {
                    errors.push(`Please select a source heat for promotion flow`);
                    isValid = false;
                }
            }
        }
    }
    
    if (!isValid) {
        alert('Please fix the following errors:\n\n' + errors.join('\n'));
        return false;
    }
    
    return true;
}

function generateScoringOptions(runsCount) {
    const options = [];
    
    if (runsCount === 1) {
        options.push({
            value: 'single_run',
            text: 'Single Run Score'
        });
    } else if (runsCount === 2) {
        options.push(
            {
                value: 'best_from_2',
                text: 'Best from 2 Runs'
            },
            {
                value: 'average_from_2',
                text: 'Average from 2 Runs'
            }
        );
    } else if (runsCount === 3) {
        options.push(
            {
                value: 'best_from_3',
                text: 'Best from 3 Runs'
            },
            {
                value: 'average_best_2_from_3',
                text: 'Average of Best 2 from 3 Runs'
            },
            {
                value: 'average_from_3',
                text: 'Average from 3 Runs'
            }
        );
    } else if (runsCount === 4) {
        options.push(
            {
                value: 'best_from_4',
                text: 'Best from 4 Runs'
            },
            {
                value: 'average_best_2_from_4',
                text: 'Average of Best 2 from 4 Runs'
            },
            {
                value: 'average_best_3_from_4',
                text: 'Average of Best 3 from 4 Runs'
            },
            {
                value: 'average_from_4',
                text: 'Average from 4 Runs'
            }
        );
    } else if (runsCount === 5) {
        options.push(
            {
                value: 'best_from_5',
                text: 'Best from 5 Runs'
            },
            {
                value: 'average_best_2_from_5',
                text: 'Average of Best 2 from 5 Runs'
            },
            {
                value: 'average_best_3_from_5',
                text: 'Average of Best 3 from 5 Runs'
            },
            {
                value: 'average_best_4_from_5',
                text: 'Average of Best 4 from 5 Runs'
            },
            {
                value: 'average_from_5',
                text: 'Average from 5 Runs'
            }
        );
    }
    
    return options;
}

function updateScoringExplanation(heatNumber) {
    const scoringSelect = document.getElementById(`runs_scoring_${heatNumber}`);
    const explanationSpan = document.getElementById(`scoring_explanation_${heatNumber}`);
    
    if (!scoringSelect || !explanationSpan) return;
    
    const selectedValue = scoringSelect.value;
    
    const explanations = {
        'single_run': 'Score from the single run attempt',
        'best_from_2': 'Highest score from 2 runs',
        'average_from_2': 'Average score from both runs',
        'best_from_3': 'Highest score from 3 runs',
        'average_best_2_from_3': 'Average of the 2 highest scores from 3 runs',
        'average_from_3': 'Average score from all 3 runs',
        'best_from_4': 'Highest score from 4 runs',
        'average_best_2_from_4': 'Average of the 2 highest scores from 4 runs',
        'average_best_3_from_4': 'Average of the 3 highest scores from 4 runs',
        'average_from_4': 'Average score from all 4 runs',
        'best_from_5': 'Highest score from 5 runs',
        'average_best_2_from_5': 'Average of the 2 highest scores from 5 runs',
        'average_best_3_from_5': 'Average of the 3 highest scores from 5 runs',
        'average_best_4_from_5': 'Average of the 4 highest scores from 5 runs',
        'average_from_5': 'Average score from all 5 runs'
    };
    
    explanationSpan.textContent = explanations[selectedValue] || 'How to calculate final score from multiple runs';
}

function updateRunSelector(heatNumber) {
    const runsCount = parseInt(document.querySelector(`select[name="runs_count[${heatNumber}]"]`).value) || 1;
    const runSelector = document.getElementById(`run_selector_${heatNumber}`);
    
    if (!runSelector) return;
    const preferredRun = getPreferredActiveRunSelection(heatNumber, runSelector, runsCount);
    
    // Rebuild run radio buttons
    const runOptions = Array.from({length: runsCount}, (_, i) => {
        const runNum = i + 1;
        return `
            <div class="col-auto">
                <div class="form-check">
                    <input class="form-check-input" type="radio" 
                           name="active_run[${heatNumber}]" value="${runNum}" 
                           id="run_${heatNumber}_${runNum}" ${runNum === preferredRun ? 'checked' : ''}>
                    <label class="form-check-label" for="run_${heatNumber}_${runNum}">
                        Run ${runNum}
                    </label>
                </div>
            </div>
        `;
    }).join('');
    
    const runsContainer = runSelector.querySelector('.row');
    if (runsContainer) {
        runsContainer.innerHTML = runOptions;
        runSelector.dataset.activeRun = preferredRun;
        runsContainer.querySelectorAll(`input[name="active_run[${heatNumber}]"]`).forEach(input => {
            input.addEventListener('change', (event) => {
                runSelector.dataset.activeRun = event.target.value;
            });
        });
    }
}

function updateRunOptions(heatNumber) {
    const runsCount = parseInt(document.querySelector(`select[name="runs_count[${heatNumber}]"]`).value);
    const scoringSelect = document.getElementById(`runs_scoring_${heatNumber}`);
    
    if (!scoringSelect) return;
    
    // Store current selection to try to maintain it
    const currentValue = scoringSelect.value;
    
    // Clear existing options
    scoringSelect.innerHTML = '';
    
    // Generate scoring options based on runs count
    const scoringOptions = generateScoringOptions(runsCount);
    
    // Get the saved value from PHP for this heat
    const savedValue = '<?= $setting['runs_scoring_method'] ?? 'best_from_all' ?>';
    
    scoringOptions.forEach(option => {
        const optionElement = document.createElement('option');
        optionElement.value = option.value;
        optionElement.textContent = option.text;
        
        // Set selected if it matches saved value or current value
        if (option.value === savedValue || option.value === currentValue) {
            optionElement.selected = true;
        }
        
        scoringSelect.appendChild(optionElement);
    });
    
    // If no option was selected and we have options, select the first one
    if (!scoringSelect.value && scoringOptions.length > 0) {
        scoringSelect.value = scoringOptions[0].value;
    }
    
    // Update explanation
    updateScoringExplanation(heatNumber);
    
    // Update run selector for active heat selection
    updateRunSelector(heatNumber);
}

function attachScoringMethodListeners() {
    <?php foreach ($heat_order as $h): ?>
        const scoringSelect_<?= $h ?> = document.getElementById('runs_scoring_<?= $h ?>');
        if (scoringSelect_<?= $h ?>) {
            scoringSelect_<?= $h ?>.addEventListener('change', function() {
                updateScoringExplanation(<?= $h ?>);
            });
        }
    <?php endforeach; ?>
}

function toggleActiveHeat(heatNumber) {
    // Hide all run selectors
    document.querySelectorAll('.run-selector').forEach(selector => {
        selector.classList.add('d-none');
    });
    
    // Remove active styles from all heat cards
    document.querySelectorAll('.heat-card').forEach(card => {
        card.classList.remove('border-success', 'bg-success-subtle' , 'active-heat-card');
        card.classList.add('opacity-75');
    });
    
    // Show run selector for active heat
    const runSelector = document.getElementById(`run_selector_${heatNumber}`);
    if (runSelector) {
        runSelector.classList.remove('d-none');
    }
    
    // Add active class to selected heat card
    const activeCard = document.querySelector(`input[value="${heatNumber}"]`).closest('.heat-card');
    if (activeCard) {
        activeCard.classList.add('border-success', 'bg-success-subtle', 'active-heat-card');
        activeCard.classList.remove('opacity-75');
    }
}

function toggleAllCategories(heatNumber) {
    const allCheckbox = document.getElementById(`all_categories_${heatNumber}`);
    if (!allCheckbox) return; // Element doesn't exist
    
    const categoryCheckboxes = document.querySelectorAll(`input[data-heat="${heatNumber}"].category-checkbox`);
    
    if (allCheckbox.checked) {
        categoryCheckboxes.forEach(checkbox => {
            checkbox.checked = false;
            checkbox.disabled = true;
        });
    } else {
        categoryCheckboxes.forEach(checkbox => {
            checkbox.disabled = false;
        });
    }
}

function updateCategorySelection(heatNumber) {
    const allCheckbox = document.getElementById(`all_categories_${heatNumber}`);
    if (!allCheckbox) return; // Element doesn't exist
    
    const categoryCheckboxes = document.querySelectorAll(`input[data-heat="${heatNumber}"].category-checkbox:checked`);
    
    if (categoryCheckboxes.length > 0) {
        allCheckbox.checked = false;
    }
}

function toggleFlowConfig(heatNumber) {
    const flowType = document.querySelector(`select[name="flow_type[${heatNumber}]"]`);
    if (!flowType) return;
    
    const flowTypeValue = flowType.value;
    const flowConfig = document.getElementById(`flow_config_${heatNumber}`);
    const participantsInput = document.querySelector(`input[name="flow_participants_per_category[${heatNumber}]"]`);
    const sourceHeatSelect = document.querySelector(`select[name="flow_source_heat[${heatNumber}]"]`);
    
    if (flowTypeValue !== 'none') {
        if (flowConfig) flowConfig.classList.remove('d-none');
        if (participantsInput) {
            participantsInput.removeAttribute('disabled');
            participantsInput.setAttribute('min', '1');
        }
        if (sourceHeatSelect) {
            sourceHeatSelect.removeAttribute('disabled');
        }
    } else {
        if (flowConfig) flowConfig.classList.add('d-none');
        if (participantsInput) {
            participantsInput.setAttribute('disabled', 'disabled');
            participantsInput.removeAttribute('min');
        }
        if (sourceHeatSelect) {
            sourceHeatSelect.setAttribute('disabled', 'disabled');
        }
    }
}

function calculateEstimatedFinish(heatNumber) {
    const startTime = document.querySelector(`input[name="time_start[${heatNumber}]"]`).value;
    const timePerParticipant = parseInt(document.querySelector(`input[name="estimate_time_per_participant[${heatNumber}]"]`).value) || 0;
    
    if (startTime && timePerParticipant > 0) {
        // Get actual participant count for this heat
        const participantCount = window.heatParticipantCounts ? window.heatParticipantCounts[heatNumber] || 0 : 0;
        
        if (participantCount > 0) {
            const totalSeconds = timePerParticipant * participantCount;
            const totalMinutes = Math.ceil(totalSeconds / 60);
            
            const [hours, minutes] = startTime.split(':').map(Number);
            const startDateTime = new Date();
            startDateTime.setHours(hours, minutes, 0, 0);
            
            const finishDateTime = new Date(startDateTime.getTime() + (totalMinutes * 60000));
            const finishTime = finishDateTime.toTimeString().slice(0, 5);
            
            const finishElement = document.getElementById(`finish_time_${heatNumber}`);
            if (finishElement) {
                finishElement.textContent = `${finishTime} (${totalMinutes} minutes for ${participantCount} participants)`;
                document.getElementById(`estimated_finish_${heatNumber}`).classList.remove('d-none');
            }
        } else {
            const finishElement = document.getElementById(`finish_time_${heatNumber}`);
            if (finishElement) {
                finishElement.textContent = 'No participants assigned to this heat yet';
                document.getElementById(`estimated_finish_${heatNumber}`).classList.remove('d-none');
            }
        }
    } else {
        const estimatedElement = document.getElementById('estimated_finish_' + heatNumber);
        if (estimatedElement) {
            estimatedElement.classList.add('d-none');
        }
    }
}

function updateHeatNameDisplay(heatNumber) {
    const nameInput = document.querySelector('input[name="heat_name[' + heatNumber + ']"]');
    const nameDisplay = document.getElementById('heat_name_display_' + heatNumber);
    
    if (nameInput && nameDisplay) {
        const name = nameInput.value.trim();
        nameDisplay.textContent = name ? '(' + name + ')' : '';
    }
}

function updateHeatNameReferences(changedHeatNumber) {
    const nameInput = document.querySelector('input[name="heat_name[' + changedHeatNumber + ']"]');
    const newName = nameInput ? nameInput.value.trim() : '';
    
    // Update all source heat dropdowns that reference this heat
    document.querySelectorAll('option[data-heat-number="' + changedHeatNumber + '"]').forEach(option => {
        const heatNumber = changedHeatNumber;
        const baseText = 'Heat ' + heatNumber;
        option.textContent = newName ? baseText + ' (' + newName + ')' : baseText;
    });
    
    // Update the heat name display
    updateHeatNameDisplay(changedHeatNumber);
    
    // Update flow visualization if visible
    updateFlowDisplay();
}


function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `alert alert-${type} alert-dismissible fade show position-fixed top-0 end-0 mt-3 me-3 shadow w-auto`;
    notification.innerHTML = `
        <i class="fas fa-${type === 'success' ? 'check-circle' : 'info-circle'} me-2"></i>
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(notification);
    
    // Auto-remove after 3 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 3000);
}

function updateFlowDisplay() {
    // This would update the flow visualization section if it's visible
    //console.log('Flow display updated');
}

function validateForm() {
    let isValid = true;
    const errors = [];
    
    // Check each heat's flow configuration
    <?php foreach ($heat_order as $h): ?>
        const flowType_<?= $h ?> = document.querySelector('select[name="flow_type[<?= $h ?>]"]');
        if (flowType_<?= $h ?>) {
            const flowTypeValue = flowType_<?= $h ?>.value;
            
            if (flowTypeValue !== 'none') {
                const participantsInput_<?= $h ?> = document.querySelector('input[name="flow_participants_per_category[<?= $h ?>]"]');
                const participantsValue_<?= $h ?> = participantsInput_<?= $h ?> ? parseInt(participantsInput_<?= $h ?>.value) || 0 : 0;
                
                if (participantsValue_<?= $h ?> < 1) {
                    errors.push('Heat <?= $h ?>: Please specify number of participants to promote (minimum 1)');
                    isValid = false;
                }
                
                <?php if ($h > 1): ?>
                    if (flowTypeValue === 'promotion') {
                        const sourceHeat_<?= $h ?> = document.querySelector('select[name="flow_source_heat[<?= $h ?>]"]');
                        const sourceHeatValue = sourceHeat_<?= $h ?> ? sourceHeat_<?= $h ?>.value : '';
                        if (!sourceHeatValue) {
                            errors.push('Heat <?= $h ?>: Please select a source heat for promotion flow');
                            isValid = false;
                        }
                    }
                <?php endif; ?>
            }
        }
    <?php endforeach; ?>
    
    if (!isValid) {
        alert('Please fix the following errors:\n\n' + errors.join('\n'));
        return false;
    }
    
    return true;
}

// Drag-to-scroll functionality for heat buttons container
function initializeDragScroll() {
    const container = document.getElementById('heat-buttons-container');
    if (!container) return;
    
    let isDown = false;
    let startX;
    let scrollLeft;
    
    container.addEventListener('mousedown', (e) => {
        // Only enable drag on the container itself, not on buttons or interactive elements
        if (e.target.closest('button, a, input, select, textarea')) {
            return;
        }
        
        isDown = true;
        container.style.cursor = 'grabbing';
        startX = e.pageX - container.offsetLeft;
        scrollLeft = container.scrollLeft;
        e.preventDefault();
    });
    
    container.addEventListener('mouseleave', () => {
        isDown = false;
        container.style.cursor = 'grab';
    });
    
    container.addEventListener('mouseup', () => {
        isDown = false;
        container.style.cursor = 'grab';
    });
    
    container.addEventListener('mousemove', (e) => {
        if (!isDown) return;
        e.preventDefault();
        const x = e.pageX - container.offsetLeft;
        const walk = (x - startX) * 2; // Scroll speed multiplier
        container.scrollLeft = scrollLeft - walk;
    });
    
    // Touch support for mobile
    let touchStartX;
    let touchScrollLeft;
    
    container.addEventListener('touchstart', (e) => {
        if (e.target.closest('button, a, input, select, textarea')) {
            return;
        }
        touchStartX = e.touches[0].pageX - container.offsetLeft;
        touchScrollLeft = container.scrollLeft;
    }, { passive: true });
    
    container.addEventListener('touchmove', (e) => {
        if (!touchStartX) return;
        const x = e.touches[0].pageX - container.offsetLeft;
        const walk = (x - touchStartX) * 2;
        container.scrollLeft = touchScrollLeft - walk;
    }, { passive: true });
    
    container.addEventListener('touchend', () => {
        touchStartX = null;
    });
    
    //console.log('Drag-to-scroll initialized for heat-buttons-container');
}

// Enhanced initialization
document.addEventListener('DOMContentLoaded', function() {
    //console.log('DOM Content Loaded');
    //console.log('Bootstrap available:', typeof bootstrap !== 'undefined');

    highlightHeatLayoutSelector();
    
    // Initialize drag-to-scroll for heat buttons container
    initializeDragScroll();
    
    // Load heat configuration modals via API
    <?php if ($selected_event_id && $heats_total > 0): ?>
    loadHeatConfigModals();
    loadHeatCards();
        
        // Set up auto-refresh with default interval (30s)
        setRefreshInterval();
    <?php endif; ?>
    
    // Test function - add simple click listener
    document.querySelectorAll('[onclick*="openHeatModal"]').forEach(button => {
        //console.log('Found button with openHeatModal:', button);
    });
    
    // Make participant counts available to JavaScript
    const heatParticipantCounts = <?= json_encode($heat_participant_counts ?? []) ?>;
    window.heatParticipantCounts = heatParticipantCounts;
    
    // Make heat settings available to JavaScript
    const heatSettings = <?= json_encode($heat_settings ?? []) ?>;
    window.heatSettings = heatSettings;
    
    // Initialize run options and scoring methods for all heats
    <?php foreach ($heat_order as $h): ?>
        // Initialize each heat with proper values
        (function(heatNum) {
            const savedScoringMethod = '<?= $heat_settings[$h]['runs_scoring_method'] ?? 'best_from_all' ?>';
            
            // Update run options first
            updateRunOptions(heatNum);
            
            // Set the saved scoring method
            const scoringSelect = document.getElementById(`runs_scoring_${heatNum}`);
            if (scoringSelect && savedScoringMethod) {
                scoringSelect.value = savedScoringMethod;
                updateScoringExplanation(heatNum);
            }
            
            calculateEstimatedFinish(heatNum);
            toggleFlowConfig(heatNum);
            
            // Initialize category selection
            <?php if (empty($heat_settings[$h]['categories'] ?? [])): ?>
                toggleAllCategories(heatNum);
            <?php endif; ?>
            
            // Attach heat name change listeners
            const nameInput = document.querySelector(`input[name="heat_name[${heatNum}]"]`);
            if (nameInput) {
                nameInput.addEventListener('input', function() {
                    updateHeatNameDisplay(heatNum);
                });
                nameInput.addEventListener('change', function() {
                    updateHeatNameReferences(heatNum);
                });
            }
        })(<?= $h ?>);
    <?php endforeach; ?>
    
    // Attach scoring method listeners
    attachScoringMethodListeners();
    
    // Show run selector for active heat
    const activeHeatRadio = document.querySelector('input[name="is_active"]:checked');
    if (activeHeatRadio) {
        toggleActiveHeat(activeHeatRadio.value);
    }
    
    // Add form validation
    const form = document.querySelector('form');
    if (form) {
        form.addEventListener('submit', function(e) {
            
            if (!validateForm()) {
                e.preventDefault();
            } else {
                // Update URL with event_id parameter before submission
                c
            }
        });
    }
    
   
    // Clean up refresh timer on page unload
    window.addEventListener('beforeunload', function() {
        if (heatCardsRefreshTimer) {
            clearInterval(heatCardsRefreshTimer);
        }
    });
});

// Load Heat Configuration Modals via API
async function loadHeatConfigModals() {
    const eventId = <?= json_encode($selected_event_id) ?>;
    const container = document.getElementById('heatConfigModalsContainer');
    const loadingIndicator = document.getElementById('modalsLoading');
    
    if (!eventId) {
        console.error('No event ID provided');
        return;
    }
    
    try {
        const response = await fetch(`heat_config_modals_api.php?event_id=${eventId}&heat_numbers=all`);
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const data = await response.json();
        
        if (data.success && data.html) {
            // Hide loading indicator
            if (loadingIndicator) {
                loadingIndicator.style.display = 'none';
            }
            
            // Inject modals HTML
            container.innerHTML = data.html;
            
            // Update global heat settings if provided
            if (data.heat_settings) {
                window.heatSettings = data.heat_settings;
            }
            
            if (data.heat_participant_counts) {
                window.heatParticipantCounts = data.heat_participant_counts;
            }
            
            //console.log('Heat configuration modals loaded successfully:', data.heat_numbers);
            
            // Initialize run options and scoring methods for all loaded heats
            if (data.heat_numbers && Array.isArray(data.heat_numbers)) {
                data.heat_numbers.forEach(heatNum => {
                    if (typeof updateRunOptionsModal === 'function') {
                        updateRunOptionsModal(heatNum);
                    }
                });
            }
        } else {
            throw new Error(data.error || 'Failed to load heat configuration modals');
        }
    } catch (error) {
        console.error('Error loading heat configuration modals:', error);
        container.innerHTML = `
            <div class="alert alert-danger">
                <i class="fas fa-exclamation-triangle me-2"></i>
                <strong>Error loading configuration panels:</strong> ${error.message}
                <button type="button" class="btn btn-sm btn-outline-danger ms-3" onclick="loadHeatConfigModals()">
                    <i class="fas fa-redo me-1"></i>Retry
                </button>
            </div>
        `;
    }
}

// Scoring Layout Functions
function resetHeatLayout(heatNumber) {
    if (confirm('Reset this heat\'s scoring layout to use event defaults?')) {
        // Uncheck all checkboxes for this heat
        const checkboxes = document.querySelectorAll(`input[name^="heat_scoring_layout[${heatNumber}]"]`);
        checkboxes.forEach(checkbox => {
            if (checkbox.type === 'checkbox') {
                checkbox.checked = false;
            }
        });
        
        // Reset select boxes
        const selects = document.querySelectorAll(`select[name^="heat_scoring_layout[${heatNumber}]"]`);
        selects.forEach(select => {
            select.selectedIndex = 0; // Select first option (usually "Use Event Default")
        });
        
        updateLayoutPreview(heatNumber);
        
        // Show feedback
        const alert = document.createElement('div');
        alert.className = 'alert alert-success alert-dismissible fade show mt-2';
        alert.innerHTML = `
            <i class="fas fa-check me-2"></i>
            Heat ${heatNumber} layout reset to event defaults.
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        
        const targetTab = document.getElementById(`scoring-layout-${heatNumber}`);
        if (targetTab) {
            targetTab.insertBefore(alert, targetTab.firstChild);
            
            // Auto dismiss after 3 seconds
            setTimeout(() => {
                if (alert.parentNode) {
                    alert.remove();
                }
            }, 3000);
        }
    }
}

function updateLayoutPreview(heatNumber) {
    const previewDiv = document.getElementById(`layoutPreview_${heatNumber}`);
    if (!previewDiv) return;
    
    const enabledComponents = [];
    const checkboxes = document.querySelectorAll(`input[name^="heat_scoring_layout[${heatNumber}][cards]"][name$="[enabled]"]:checked`);
    
    checkboxes.forEach(checkbox => {
        const match = checkbox.name.match(/\[cards\]\[([^\]]+)\]/);
        if (match) {
            const componentKey = match[1];
            const componentNames = {
                'criteriaInputGrid': 'Criteria Grid',
                'finalCalculatedScore': 'Final Score',
                'figuresCard': 'Figures',
                'diversityValidation': 'Diversity Check',
                'figureHistory': 'Figure History',
                'latestScores': 'Latest Scores',
                'otherHeatScores': 'Other Judges',
                'formatInfo': 'Format Info'
            };
            
            enabledComponents.push(componentNames[componentKey] || componentKey);
        }
    });
    
    if (enabledComponents.length > 0) {
        previewDiv.innerHTML = enabledComponents.map(name => 
            `<span class="badge bg-success me-1 mb-1">${name}</span>`
        ).join('');
    } else {
        previewDiv.innerHTML = '<em class="text-muted">All components disabled</em>';
    }
}

// Update previews when checkboxes change
document.addEventListener('DOMContentLoaded', function() {
    // Initialize layout previews for all heats
    const layoutCheckboxes = document.querySelectorAll('input[name*="heat_scoring_layout"][name*="[enabled]"]');
    layoutCheckboxes.forEach(checkbox => {
        const heatMatch = checkbox.name.match(/\[(\d+)\]/);
        if (heatMatch) {
            const heatNumber = heatMatch[1];
            
            // Update preview when checkbox changes
            checkbox.addEventListener('change', () => updateLayoutPreview(heatNumber));
            
            // Initialize preview
            updateLayoutPreview(heatNumber);
        }
    });
    
    // Add event listeners for figuresCard checkboxes to show/hide category config
    const figuresCardCheckboxes = document.querySelectorAll('input[id^="heat_layout_"][id*="_figuresCard"]');
    figuresCardCheckboxes.forEach(checkbox => {
        const heatMatch = checkbox.id.match(/heat_layout_(\d+)_figuresCard/);
        if (heatMatch) {
            const heatNumber = heatMatch[1];
            const categoryConfig = document.getElementById(`heat_figureCategoriesConfig_${heatNumber}`);
            
            if (categoryConfig) {
                checkbox.addEventListener('change', function() {
                    categoryConfig.style.display = this.checked ? 'block' : 'none';
                });
            }
        }
    });
});

// Toggle heat category override (use event settings vs custom)
function toggleHeatCategoryOverride(heatNumber, useEventSettings) {
    const overrideDiv = document.getElementById(`heat_category_override_${heatNumber}`);
    if (overrideDiv) {
        overrideDiv.style.display = useEventSettings ? 'none' : 'block';
        
        // If using event settings, uncheck all heat-specific categories
        if (useEventSettings) {
            const categoryCheckboxes = document.querySelectorAll(
                `input[name="heat_scoring_layout[${heatNumber}][cards][figuresCard][visibleCategories][]"]`
            );
            categoryCheckboxes.forEach(checkbox => {
                checkbox.checked = false;
            });
        }
    }
}

// Toggle all category checkboxes for a specific heat
function toggleAllHeatCategories(heatNumber, selectAll) {
    const checkboxes = document.querySelectorAll(
        `input[name="heat_scoring_layout[${heatNumber}][cards][figuresCard][visibleCategories][]"]`
    );
    checkboxes.forEach(checkbox => {
        checkbox.checked = selectAll;
    });
}
</script>

    <!-- Floating Action Button -->
    <?php if ($selected_event_id): ?>
    <div id="heatConfigFAB" class="fab-container">
        <button class="fab-main" onclick="toggleFAB()" title="Quick Actions">
            <i class="fas fa-plus"></i>
        </button>
        <div class="fab-menu">
            <a href="active_heat_panel.php?event_id=<?= $selected_event_id ?>" class="fab-action" title="Start Gate">
                <i class="fas fa-users"></i>
            </a>
            <a href="heats_configure.php?event_id=<?= $selected_event_id ?>" class="fab-action" title="Manage Athletes">
                <i class="fas fa-user-cog"></i>
            </a>
            <a href="heat_flow_preview.php?event_id=<?= $selected_event_id ?>" class="fab-action" title="Preview Flow">
                <i class="fas fa-sitemap"></i>
            </a>
            <button class="fab-action" onclick="window.scrollTo({top: 0, behavior: 'smooth'})" title="Scroll to Top">
                <i class="fas fa-arrow-up"></i>
            </button>
        </div>
    </div>
    
    <style>
        .fab-container {
            position: fixed;
            bottom: 2rem;
            right: 2rem;
            z-index: 1000;
        }
        
        .fab-main {
            width: 56px;
            height: 56px;
            border-radius: 50%;
            background: linear-gradient(135deg, #0f3460 0%, #16213e 100%);
            border: none;
            color: white;
            font-size: 1.5rem;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
            cursor: pointer;
            transition: all 0.3s cubic-bezier(0.68, -0.55, 0.265, 1.55);
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .fab-main:hover {
            transform: scale(1.1);
            box-shadow: 0 6px 20px rgba(0, 0, 0, 0.4);
        }
        
        .fab-main.open {
            transform: rotate(45deg);
            background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
        }
        
        .fab-menu {
            position: absolute;
            bottom: 70px;
            right: 0;
            display: flex;
            flex-direction: column;
            gap: 0.75rem;
            opacity: 0;
            pointer-events: none;
            transition: all 0.3s;
        }
        
        .fab-menu.open {
            opacity: 1;
            pointer-events: all;
            bottom: 75px;
        }
        
        .fab-action {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            background: linear-gradient(135deg, #28a745 0%, #218838 100%);
            border: none;
            color: white;
            font-size: 1.25rem;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.2);
            cursor: pointer;
            transition: all 0.2s;
            display: flex;
            align-items: center;
            justify-content: center;
            text-decoration: none;
            transform: scale(0);
        }
        
        .fab-menu.open .fab-action {
            transform: scale(1);
        }
        
        .fab-menu.open .fab-action:nth-child(1) { transition-delay: 0.05s; }
        .fab-menu.open .fab-action:nth-child(2) { transition-delay: 0.1s; }
        .fab-menu.open .fab-action:nth-child(3) { transition-delay: 0.15s; }
        .fab-menu.open .fab-action:nth-child(4) { transition-delay: 0.2s; }
        
        .fab-action:hover {
            transform: scale(1.15);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
        }
    </style>
    
    <script>
        function toggleFAB() {
            const fabMain = document.querySelector('.fab-main');
            const fabMenu = document.querySelector('.fab-menu');
            fabMain.classList.toggle('open');
            fabMenu.classList.toggle('open');
        }
        
        // Close FAB when clicking outside
        document.addEventListener('click', function(event) {
            const fabContainer = document.getElementById('heatConfigFAB');
            if (fabContainer && !fabContainer.contains(event.target)) {
                const fabMain = document.querySelector('.fab-main');
                const fabMenu = document.querySelector('.fab-menu');
                if (fabMain && fabMenu) {
                    fabMain.classList.remove('open');
                    fabMenu.classList.remove('open');
                }
            }
        });
    </script>
    <?php endif; ?>

    <?php include '../admin/footer.php'; ?>
</body>
</html>