<?php
/**
 * Active Heat Panel - Standalone Component
 * Displays real-time heat status, judge panel status, and results
 * 
 * Usage:
 * 1. Include directly: <?php include 'active_heat_panel.php'; ?>
 * 2. Standalone: active_heat_panel.php?event_id=3&heat=1
 * 3. AJAX load: Load into a container via fetch/AJAX
 */

include_once '../includes/auth.php';
include_once '../includes/db.php';
include_once '../includes/validate_event_access.php';

$event_id = $_GET['event_id'] ?? $_POST['event_id'] ?? null;
$heat_number = $_GET['heat'] ?? $_POST['heat'] ?? null;
$standalone = isset($_GET['standalone']) || !headers_sent();

if ($standalone): ?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Active Heat Panel</title>
    <?php include_once '../includes/stylesheets.php'; ?>
</head>
<body class="body-bg-aurora-bright">
    <?php if (!isset($_GET['embed'])): ?>
        <?php include '../menu.php'; ?>
    <?php endif; ?>
    <div class="container-StyleScore p-4">
<?php endif; ?>

<!-- Active Heat Panel Container -->
<div id="activeHeatPanelContainer" class="border-0 shadow-none">
    <h5 class="mb-0 fw-bold text-primary">
            <i class="fas fa-fire me-2"></i>Active Heat Monitor
        </h5>
    <!-- Panel Header -->
    <div class="d-flex justify-content-between align-items-center mb-3">
        
        <div class="d-flex gap-2 align-items-center flex-wrap">
            <!-- Heat Card Layout Selector -->
            <div class="d-flex flex-column gap-1">
                <small class="text-muted fw-bold" style="font-size: 0.7rem;">Heat Cards:</small>
                <div class="btn-group btn-group-sm" role="group" aria-label="Heat card layout">
                    <button type="button" class="btn btn-outline-secondary active" onclick="setHeatCardLayout('judges', this)" title="Judges status only">
                        <i class="fas fa-user-tie"></i> Judges
                    </button>
                    <button type="button" class="btn btn-outline-secondary" onclick="setHeatCardLayout('monitor-row', this)" title="Monitor row layout">
                        <i class="fas fa-align-justify"></i> Row
                    </button>
                    <button type="button" class="btn btn-outline-secondary" onclick="setHeatCardLayout('monitor', this)" title="Monitor layout">
                        <i class="fas fa-desktop"></i> Monitor
                    </button>
                    <button type="button" class="btn btn-outline-secondary" onclick="setHeatCardLayout('full', this)" title="Full details">
                        <i class="fas fa-expand"></i> Full
                    </button>
                </div>
            </div>
            
            <!-- Participant Card Layout Selector -->
            <div class="d-flex flex-column gap-1" style="display: none !important;">
                <small class="text-muted fw-bold" style="font-size: 0.7rem;">Participant Cards:</small>
                <div class="btn-group btn-group-sm" role="group" aria-label="Participant card layout">
                    <button type="button" class="btn btn-outline-primary" onclick="setParticipantCardLayout('full', this)" title="Full details with controls and info">
                        <i class="fas fa-expand-alt"></i> Full
                    </button>
                    <button type="button" class="btn btn-outline-primary" onclick="setParticipantCardLayout('compact', this)" title="Compact view">
                        <i class="fas fa-compress-alt"></i> Compact
                    </button>
                    <button type="button" class="btn btn-outline-primary active" onclick="setParticipantCardLayout('controls', this)" title="With position controls">
                        <i class="fas fa-arrows-alt-v"></i> Controls
                    </button>
                    <button type="button" class="btn btn-outline-primary" onclick="setParticipantCardLayout('info', this)" title="Info only, no controls">
                        <i class="fas fa-info-circle"></i> Info
                    </button>
                </div>
            </div>
            
            
        </div>
        <div>
            <!-- Refresh Interval Controls -->
            <div class="btn-group btn-group-sm" role="group" aria-label="Refresh interval">
                <input type="radio" class="btn-check" name="heatCardRefresh" id="refresh2s" value="2" autocomplete="off">
                <label class="btn btn-outline-primary" for="refresh2s" title="Refresh every 2 seconds">2s</label>
                
                <input type="radio" class="btn-check" name="heatCardRefresh" id="refresh5s" value="5" autocomplete="off" checked>
                <label class="btn btn-outline-primary" for="refresh5s" title="Refresh every 5 seconds">5s</label>
                
                <input type="radio" class="btn-check" name="heatCardRefresh" id="refresh10s" value="10" autocomplete="off">
                <label class="btn btn-outline-primary" for="refresh10s" title="Refresh every 10 seconds">10s</label>
                
                <input type="radio" class="btn-check" name="heatCardRefresh" id="refresh20s" value="20" autocomplete="off">
                <label class="btn btn-outline-primary" for="refresh20s" title="Refresh every 20 seconds">20s</label>
                
                <input type="radio" class="btn-check" name="heatCardRefresh" id="refreshPause" value="0" autocomplete="off">
                <label class="btn btn-outline-warning" for="refreshPause" title="Pause refresh">
                    <i class="fas fa-pause"></i>
                </label>
            </div>
            
            <!-- Last Update Time -->
            <span class="badge bg-info" id="activeHeatUpdateTime">
                <i class="fas fa-refresh me-1"></i> --
            </span>
            
            <!-- Manual Refresh Button -->
            <button class="btn btn-sm btn-outline-primary" onclick="loadActiveHeatCard()" title="Refresh now">
                <i class="fas fa-sync-alt"></i>
            </button>
        </div>

    </div>
   
<div class="col-12 position-sticky sticky-top top-0">
                <!-- Heat Card Display Area -->
                <div id="activeHeatCardContainer" class="row">
                    <!-- Active heat card will be loaded here -->
                    <div class="col-12 text-center text-muted py-5">
                        <i class="fas fa-fire fa-3x mb-3 opacity-25"></i>
                        <p class="mb-0">No active heat selected</p>
                        <p class="small text-muted">Heat status and judge panels will appear here</p>
                    </div>
                </div>
            </div>
    <div class="row g-4 mt-1 align-items-start heat-monitor-grid">
        

        <div class="col-12 col-xl-4">
            

            <div class="card queue-summary-card shadow-sm border-0 mb-3">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-start">
                        <div>
                            <h6 class="mb-0 fw-bold text-primary">
                                <i class="fas fa-people-arrows me-2"></i>Performing Queue
                            </h6>
                            <small class="text-muted" id="queueSummarySubtitle">Awaiting heat data…</small>
                        </div>
                        <div class="d-flex align-items-center gap-2">
                            <span class="badge bg-light text-muted" id="queueUpdatedAt">--</span>
                            <button class="btn btn-sm btn-outline-primary" onclick="loadParticipantList()" title="Refresh queue">
                                <i class="fas fa-sync-alt"></i>
                            </button>
                        </div>
                    </div>
                    <div class="queue-status-grid mt-3" id="queueStatusGrid">
                        <div class="queue-stat">
                            <span class="queue-stat-label">Participants</span>
                            <span class="queue-stat-value" id="queueTotalCount">--</span>
                        </div>
                        <div class="queue-stat">
                            <span class="queue-stat-label">Performing</span>
                            <span class="queue-stat-value text-warning" id="queuePerformingCount">0</span>
                        </div>
                        <div class="queue-stat">
                            <span class="queue-stat-label">Waiting</span>
                            <span class="queue-stat-value text-primary" id="queueWaitingCount">0</span>
                        </div>
                        <div class="queue-stat">
                            <span class="queue-stat-label">Completed</span>
                            <span class="queue-stat-value text-success" id="queueCompletedCount">0</span>
                        </div>
                        <div class="queue-stat">
                            <span class="queue-stat-label">Run</span>
                            <span class="queue-stat-value" id="queueRunLabel">--</span>
                        </div>
                    </div>
                    <div class="performing-slots mt-3" id="queuePerformingSlots">
                        <div class="performing-slot current">
                            <span class="slot-label">Now Performing</span>
                            <div class="slot-content text-muted small">Waiting for heat data…</div>
                        </div>
                        <div class="performing-slot next">
                            <span class="slot-label">Up Next</span>
                            <div class="slot-content text-muted small">Queue will appear here.</div>
                        </div>
                    </div>
                    <div class="queue-chips mt-3" id="queueChipsList">
                        <div class="text-muted small">Queue will appear once participants load.</div>
                    </div>
                </div>
            </div>

            <div class="card heat-alerts-card shadow-sm border-0">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-start">
                        <div>
                            <h6 class="mb-0 fw-bold text-danger">
                                <i class="fas fa-bell me-2"></i>Heat Alerts
                            </h6>
                            <small class="text-muted" id="alertsStatusLabel">Not connected</small>
                        </div>
                        <div class="d-flex align-items-center gap-2">
                            <span class="badge badge-soft-danger" id="alertsCountBadge">0</span>
                            <button class="btn btn-sm btn-outline-secondary" onclick="loadHeatNotifications(true)" title="Refresh alerts">
                                <i class="fas fa-sync-alt"></i>
                            </button>
                        </div>
                    </div>
                    <div class="alerts-list mt-3" id="alertsList">
                        <div class="text-muted small text-center py-3">
                            Alerts will stream in automatically once available.
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-12 col-xl-8">
            <!-- Participant List Section -->
            <div id="participantListSection" class="" style="display: none;">
                
                <!-- Participant Cards Grid -->
                <div id="participantCardsGrid" class="row g-3">
                    <!-- Participant cards will be loaded here -->
                    <div class="col-12 text-center text-muted py-3">
                        <i class="fas fa-spinner fa-spin fa-2x mb-2"></i>
                        <p class="mb-0">Loading participants...</p>
                    </div>
                </div>
                <div class="card-body bg-white mb-3 p-2  position-sticky sticky-bottom">

                        <h6 class="mb-0 fw-bold text-primary mb-2">
                                <i class="fas fa-users me-2"></i>Participants & Scores
                        </h6>

              
                        <div class="d-flex justify-content-between align-items-center">
                            
                            <div class="row">
                                <div class="col-3">
                                <!-- Filter by category -->
                                <select id="categoryFilter" class="form-select form-select-sm" >
                                    <option value="">All Categories</option>
                                </select>
                                </div>
                                <div class="col-2">
                                    <!-- Filter scored participants -->
                                    <select id="scoredFilter" class="form-select form-select-sm" >
                                        <option value="all">All Participants</option>
                                        <option value="unscored" selected>Unscored Only</option>
                                        <option value="scored_bottom">Scored at Bottom</option>
                                        <option value="performing_scoring">Performing / Scoring</option>
                                        <option value="complete">Complete</option>
                                    </select>
                                </div>
                                <div class="col-2">
                                    <!-- Sort options -->
                                    <select id="sortOrder" class="form-select form-select-sm">
                                        <option value="sort_order" selected>Start Order</option>
                                        <option value="bib">BIB Order</option>
                                        <option value="name">Name (A-Z)</option>
                                        <option value="score_desc">Score (High-Low)</option>
                                        <option value="score_asc">Score (Low-High)</option>
                                        <option value="update_newest">Update (Newest)</option>
                                        <option value="update_oldest">Update (Oldest)</option>
                                    </select>
                                </div>
                                <div class="col-auto">
                                    <div class="form-check form-switch form-switch-sm align-self-center d-flex gap-2" title="Toggle to show or hide currently performing athletes">
                                        <input class="form-check-input" type="checkbox" id="performingToggle" checked>
                                        <label class="form-check-label small text-nowrap" for="performingToggle">
                                            <i class="fas fa-thumbtack text-warning me-1"></i>Show Performing
                                        </label>
                                    </div>
                                </div>
                                <div class="col-auto">
                                    <div class="form-check form-switch form-switch-sm align-self-center d-flex gap-2" title="Toggle to show or hide participant scores">
                                        <input class="form-check-input" type="checkbox" id="scoresToggle">
                                        <label class="form-check-label small text-nowrap" for="scoresToggle">
                                            <i class="fas fa-eye me-1"></i>Show Scores
                                        </label>
                                    </div>
                                </div>
                                <div class="col-auto">
                                    <button class="btn btn-sm btn-outline-primary" onclick="loadParticipantList()" title="Refresh list">
                                        <i class="fas fa-sync-alt"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                  </div>
            </div>
        </div>
    </div>    
    <!-- Results Modal Integration -->
    <div class="modal fade" id="heatResultsModal" tabindex="-1" aria-labelledby="heatResultsModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-fullscreen p-5">
            <div class="modal-content">
                <div class="modal-header bg-primary text-white">
                    <h5 class="modal-title" id="heatResultsModalLabel">
                        <i class="fas fa-trophy me-2"></i>
                        <span id="resultsHeatName">Heat Results</span>
                    </h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body p-0">
                    <div class="row g-0">
                        <!-- Sidebar: Saved Configurations -->
                        <div class="col-md-3 bg-light border-end">
                            <div class="p-3">
                                <h6 class="fw-bold mb-3">
                                    <i class="fas fa-list me-2"></i>Available Reports
                                </h6>
                                <div id="savedConfigsList" class="d-flex flex-column gap-2">
                                    <!-- Configurations will be loaded here -->
                                    <div class="text-center text-muted py-3">
                                        <i class="fas fa-spinner fa-spin"></i>
                                        <p class="small mb-0 mt-2">Loading configurations...</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Main Content: Results Display -->
                        <div class="col-md-9">
                            <div class="p-4">
                                <div id="resultsContentDisplay">
                                    <!-- Results content will be loaded here -->
                                    <div class="text-center text-muted py-5">
                                        <i class="fas fa-chart-bar fa-3x mb-3 opacity-25"></i>
                                        <p class="mb-0">Select a report from the sidebar</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
        <!-- Generic Confirmation Modal -->
        <div class="modal fade" id="actionConfirmModal" tabindex="-1" aria-labelledby="actionConfirmModalLabel" aria-hidden="true">
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title action-confirm-title" id="actionConfirmModalLabel">Confirm Action</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    <div class="modal-body">
                        <div class="d-flex align-items-start gap-3">
                            <div class="fs-2 text-primary">
                                <i class="action-confirm-icon fas fa-question-circle"></i>
                            </div>
                            <p class="mb-0 action-confirm-message">Are you sure?</p>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-outline-secondary action-confirm-cancel" data-bs-dismiss="modal">Cancel</button>
                        <button type="button" class="btn btn-primary action-confirm-approve">Confirm</button>
                    </div>
                </div>
            </div>
        </div>
    
</div>

<style>
/* Active Heat Panel Styles */

#activeHeatCardContainer .heat-card {
    transition: all 0.3s ease;
}

#activeHeatCardContainer .heat-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

.judge-panel-status {
    animation: pulse-status 2s infinite;
}

@keyframes pulse-status {
    0% { opacity: 1; }
    50% { opacity: 0.7; }
    100% { opacity: 1; }
}

@keyframes scorePulse {
    0% { transform: scale(1); color: inherit; }
    50% { transform: scale(1.2); color: #0d6efd; font-weight: bold; }
    100% { transform: scale(1); color: inherit; }
}

/* Results Modal Styles */
.config-button {
    transition: all 0.2s ease;
    cursor: pointer;
  
    padding: 12px;
    border: 2px solid transparent;
}

.config-button:hover {
    background-color: #e9ecef;
    border-color: #0d6efd;
}

.config-button.active {
    background-color: #0d6efd;
    color: white;
    border-color: #0d6efd;
}

#resultsContentDisplay {
    min-height: 400px;
}

/* Participant List Styles */
.participant-score-card {
    background: white;
    border: 2px solid #dee2e6;
   
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.participant-score-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.1);
    border-color: #0d6efd;
}


.participant-score-card.performing {
    border-color: #ffc107;
    background: linear-gradient(135deg, rgba(255, 193, 7, 0.1), white);
}



.participant-info {
    flex-grow: 1;
    min-width: 0;
}

.participant-name {
    font-weight: 600;
    font-size: 1.1rem;
    color: #212529;
    margin-bottom: 4px;
}

.participant-details {
    font-size: 0.85rem;
    color: #6c757d;
}

.score-display {
    text-align: right;
}

.score-value {
    font-size: 1.5rem;
    font-weight: bold;
    color: #198754;
}

.score-label {
    font-size: 0.75rem;
    color: #6c757d;
    text-transform: uppercase;
}

.status-badge {
    position: absolute;
    top: 8px;
    right: 8px;
    font-size: 0.7rem;
    padding: 4px 8px;
}

.badge-sm {
    font-size: 0.65rem;
    padding: 2px 6px;
    vertical-align: middle;
    text-transform: uppercase;
    font-weight: 600;
}

.criteria-scores {
    display: flex;
    gap: 8px;
    margin-top: 8px;
    flex-wrap: wrap;
}

.criteria-score-item {
    background: #f8f9fa;
    padding: 4px 8px;
    
    font-size: 0.75rem;
    border: 1px solid #dee2e6;
}

.criteria-score-item strong {
    color: #495057;
    margin-right: 4px;
}

/* Run Indicators (O/X pattern) */
.run-indicators-container {
    margin-top: 6px;
}

.run-indicators {
    display: flex;
    align-items: center;
    gap: 8px;
}

.run-label {
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    color: #6c757d;
}

.run-indicator {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 24px;
    height: 24px;
    border-radius: 50%;
    font-size: 0.85rem;
    font-weight: bold;
    border: 2px solid #dee2e6;
    background: #f8f9fa;
    color: #dc3545;
    transition: all 0.2s ease;
}

.run-indicator.completed {
    background: #d1e7dd;
    border-color: #198754;
    color: #198754;
}

.run-indicator.current {
    box-shadow: 0 0 0 3px rgba(13, 110, 253, 0.25);
    border-color: #0d6efd;
}

.run-indicator.completed.current {
    box-shadow: 0 0 0 3px rgba(25, 135, 84, 0.25);
}

.run-summary {
    font-size: 0.75rem;
    font-weight: 600;
}

.right-0 {
    right: 0;
}
/* Position Controls */
.position-controls {
    min-width: 50px;
}

.position-controls .btn:hover {
    transform: scale(1.1);
}

.performing-hold-btn {
    position: relative;
    overflow: hidden;
    transition: transform 0.2s ease;
}

.performing-hold-btn::after {
    content: '';
    position: absolute;
    left: 0;
    top: 0;
    height: 100%;
    width: var(--hold-progress, 0%);
    background: rgba(25, 135, 84, 0.2);
    pointer-events: none;
    transition: width 0.1s linear, opacity 0.2s ease;
    opacity: 0;
}

.performing-hold-btn.hold-arming::after {
    opacity: 1;
}

.performing-hold-btn.hold-triggered {
    animation: holdSuccessPulse 0.4s ease;
}

@keyframes holdSuccessPulse {
    0% { box-shadow: 0 0 0 rgba(25, 135, 84, 0.6); }
    70% { box-shadow: 0 0 12px rgba(25, 135, 84, 0.5); }
    100% { box-shadow: 0 0 0 rgba(25, 135, 84, 0); }
}

.participant-score-card.flash-move-up {
    animation: flashMovePrimary 0.9s ease;
}

.participant-score-card.flash-move-down {
    animation: flashMoveInfo 0.9s ease;
}

@keyframes flashMovePrimary {
    0% {
        box-shadow: 0 0 0 rgba(13, 110, 253, 0.6);
        background-color: rgba(13, 110, 253, 0.12);
    }
    60% {
        box-shadow: 0 0 18px rgba(13, 110, 253, 0.4);
        background-color: rgba(13, 110, 253, 0.2);
    }
    100% {
        box-shadow: 0 0 0 rgba(13, 110, 253, 0);
        background-color: inherit;
    }
}

@keyframes flashMoveInfo {
    0% {
        box-shadow: 0 0 0 rgba(13, 202, 240, 0.6);
        background-color: rgba(13, 202, 240, 0.12);
    }
    60% {
        box-shadow: 0 0 18px rgba(13, 202, 240, 0.4);
        background-color: rgba(13, 202, 240, 0.2);
    }
    100% {
        box-shadow: 0 0 0 rgba(13, 202, 240, 0);
        background-color: inherit;
    }
}

/* Toast notifications */
.toast-container {
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: 9999;
}

.notification-toast {
    min-width: 300px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.3);
}
#activeHeatCardContainer .col-md-4.col-lg-3.mb-3 {
    width: 100%;
}

/* Score visibility toggle - hidden by default */
.score-display {
    display: none;
}

.scores-visible .score-display {
    display: block;
}

/* Queue + alert column */

.queue-status-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
    gap: 12px;
}

.queue-stat {
    background: #f8f9fa;
    
    padding: 12px;
    border: 1px solid #eef1f4;
}

.queue-stat-label {
    font-size: 0.7rem;
    text-transform: uppercase;
    letter-spacing: 0.08em;
    color: #77838f;
    display: block;
}

.queue-stat-value {
    font-size: 1.35rem;
    font-weight: 700;
    color: #182026;
}

.performing-slots {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.performing-slot {
    
    border: 1px dashed rgba(13,110,253,0.25);
    padding: 12px;
    background: linear-gradient(135deg, rgba(13,110,253,0.05), rgba(255,255,255,0.9));
}

.performing-slot.next {
    border-color: rgba(40,167,69,0.35);
    background: linear-gradient(135deg, rgba(40,167,69,0.05), rgba(255,255,255,0.9));
}

.slot-label {
    font-size: 0.75rem;
    text-transform: uppercase;
    letter-spacing: 0.08em;
    color: #6c757d;
    display: block;
}

.slot-content {
    margin-top: 6px;
    display: flex;
    flex-wrap: wrap;
    gap: 8px;
}

.slot-chip {
    background: #fff;
   
    padding: 6px 10px;
    border: 1px solid rgba(0,0,0,0.05);
    display: inline-flex;
    gap: 6px;
    align-items: center;
    font-size: 0.85rem;
}

.slot-bib {
    font-weight: 700;
    color: #0d6efd;
}

.queue-chips {
    display: flex;
    flex-direction: column;
    gap: 10px;
}

.queue-chip {
   
    border: 1px solid #e9ecef;
    padding: 12px;
    display: flex;
    gap: 12px;
    align-items: center;
    background: #fff;
    box-shadow: 0 5px 15px rgba(0,0,0,0.05);
}

.queue-chip-rank {
    width: 32px;
    height: 32px;
   
    background: #f1f3f5;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 700;
    color: #495057;
}

.queue-chip-current {
    border-color: #fdc453;
    background: #fff6e6;
}

.queue-chip-next {
    border-color: #8be28b;
    background: #f1fff1;
}

.queue-chip-completed {
    opacity: 0.7;
}

.queue-chip-name {
    font-weight: 600;
    color: #1d2533;
}

.queue-chip-meta {
    font-size: 0.75rem;
    color: #6c757d;
}

.heat-alerts-card .alerts-list {
    max-height: 420px;
    overflow-y: auto;
}

.alert-item {
    display: flex;
    gap: 12px;
    padding: 12px;
   
    border: 1px solid #eef1f4;
    background: #fff;
    margin-bottom: 12px;
    box-shadow: 0 6px 18px rgba(0,0,0,0.06);
}

.alert-icon {
    width: 44px;
    height: 44px;
   
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.1rem;
    color: #fff;
}

.alert-type-info {
    background: #0dcaf0;
}

.alert-type-warning {
    background: #ffc107;
    color: #212529;
}

.alert-type-success {
    background: #198754;
}

.alert-type-danger {
    background: #dc3545;
}

.alert-body {
    flex: 1;
}

.alert-title {
    font-weight: 600;
    margin-bottom: 4px;
}

.alert-message {
    font-size: 0.9rem;
    color: #495057;
}

.alert-time {
    font-size: 0.75rem;
}

.alert-meta {
    font-size: 0.75rem;
    color: #868e96;
}

.badge-soft-danger,
.badge-soft-warning,
.badge-soft-info,
.badge-soft-success {
    border-radius: 999px;
    font-size: 0.7rem;
    font-weight: 600;
    padding: 4px 10px;
}

.badge-soft-danger {
    background: rgba(220,53,69,0.15);
    color: #c92a2a;
}

.badge-soft-warning {
    background: rgba(255,193,7,0.18);
    color: #ad7d00;
}

.badge-soft-info {
    background: rgba(13,202,240,0.18);
    color: #0b7285;
}

.badge-soft-success {
    background: rgba(25,135,84,0.18);
    color: #0f5132;
}

.alerts-list::-webkit-scrollbar {
    width: 6px;
}

.alerts-list::-webkit-scrollbar-thumb {
    background: rgba(0,0,0,0.15);
    border-radius: 999px;
}

.alert-placeholder {
   
    padding: 12px;
    background: rgba(220,53,69,0.08);
}

@media (max-width: 1199px) {
    .queue-status-grid {
        grid-template-columns: repeat(auto-fit, minmax(140px, 1fr));
    }
    .heat-monitor-grid {
        flex-direction: column-reverse;
    }
}
</style>

<script>
// Active Heat Panel JavaScript
let activeHeatCardInterval = null;
let activeHeatCardRefreshRate = 5; // Default 10 seconds

// Initialize event_id - use URL param or fall back to localStorage
let currentEventId = <?= json_encode($event_id) ?>;
if (!currentEventId) {
    // Try to get from localStorage (saved by EventSelectionManager)
    const storedEventId = localStorage.getItem('selectedEventId');
    if (storedEventId) {
        currentEventId = storedEventId;
        console.log('📦 Loaded event_id from localStorage:', currentEventId);
    }
}

let currentHeatNumber = <?= json_encode($heat_number) ?>;
let participantCardLayout = 'controls'; // Layout mode: 'full', 'compact', 'controls', 'info'
let heatCardLayout = 'judges'; // Heat card layout: 'full', 'compact', 'monitor', 'minimal', 'monitor-row', 'judges'
const LONG_PRESS_DURATION = 2000; // ms
const longPressStates = new WeakMap();
const PLAY_BUTTON_VISIBLE_ROWS = 5; // Set to a number or 'all' to allow everyone

// Queue + alerts state
let heatNotificationsTimer = null;
let heatNotifications = [];
let latestNotificationId = 0;
let notificationsLoading = false;
let heatNotificationsInitialized = false;
const HEAT_NOTIFICATION_POLL_INTERVAL = 15000; // 15 seconds

function canShowPlayButtonForIndex(positionIndex) {
    if (PLAY_BUTTON_VISIBLE_ROWS === 'all') {
        return true;
    }
    const limit = parseInt(PLAY_BUTTON_VISIBLE_ROWS, 10);
    if (Number.isNaN(limit) || limit < 0) {
        return true;
    }
    if (typeof positionIndex !== 'number' || positionIndex < 0) {
        return false;
    }
    return positionIndex < limit;
}

/**
 * Switch heat card layout mode
 */
function setHeatCardLayout(layout, btn) {
    console.log('🎨 Switching heat card layout to:', layout);
    heatCardLayout = layout;
    
    // Update active button state
    document.querySelectorAll('[onclick^="setHeatCardLayout"]').forEach(b => {
        b.classList.remove('active');
    });
    if (btn) {
        btn.classList.add('active');
    } else if (event && event.target) {
        event.target.closest('button').classList.add('active');
    }
    
    // Reload heat card with new layout
    loadActiveHeatCard();
}

/**
 * Switch participant card layout mode
 */
function setParticipantCardLayout(layout, btn) {
    console.log('🎨 Switching participant card layout to:', layout);
    participantCardLayout = layout;
    
    // Update active button state
    document.querySelectorAll('[onclick^="setParticipantCardLayout"]').forEach(b => {
        b.classList.remove('active');
    });
    if (btn) {
        btn.classList.add('active');
    } else if (event && event.target) {
        event.target.closest('button').classList.add('active');
    }
    
    // Check if participant data is loaded
    if (!participantListData || participantListData.length === 0) {
        console.log('⚠️ No participant data loaded yet');
        return;
    }
    
    // Re-render participant list with new layout
    console.log('📋 Re-rendering', participantListData.length, 'participants with layout:', layout);
    renderParticipantList();
}

document.addEventListener('DOMContentLoaded', function() {
    console.log('🚀 Active Heat Panel v2.3 - Heat & Participant Layouts');
    console.log('Event ID:', currentEventId);
    console.log('Heat Number:', currentHeatNumber);
    console.log('Participant Card Layout:', participantCardLayout);
    console.log('Heat Card Layout:', heatCardLayout);
    
    // Setup event listeners
    setupRefreshControls();
    
    // Initial load
    loadActiveHeatCard();
    
    // Start auto-refresh
    restartActiveHeatCardInterval();
    
    // Setup filter/sort listeners
    setupParticipantListControls();

    // Prime queue + alerts panels
    updateHeatQueuePanel();
    initializeHeatNotifications();
});

function setupParticipantListControls() {
    const categoryFilter = document.getElementById('categoryFilter');
    const scoredFilter = document.getElementById('scoredFilter');
    const sortOrder = document.getElementById('sortOrder');
    const performingToggle = document.getElementById('performingToggle');
    const scoresToggle = document.getElementById('scoresToggle');
    
    if (categoryFilter) {
        categoryFilter.addEventListener('change', () => {
            renderParticipantList();
        });
    }
    
    if (scoredFilter) {
        scoredFilter.addEventListener('change', () => {
            renderParticipantList();
        });
    }
    
    if (sortOrder) {
        sortOrder.addEventListener('change', () => {
            renderParticipantList();
        });
    }

    if (performingToggle) {
        performingToggle.addEventListener('change', () => {
            renderParticipantList();
        });
    }

    if (scoresToggle) {
        scoresToggle.addEventListener('change', () => {
            const grid = document.getElementById('participantCardsGrid');
            if (scoresToggle.checked) {
                grid.classList.add('scores-visible');
            } else {
                grid.classList.remove('scores-visible');
            }
        });
    }
}

function isHoldActivationKey(event) {
    if (!event) return false;
    const key = event.key || event.code;
    return key === ' ' || key === 'Spacebar' || key === 'Space' || key === 'Enter';
}

function executeLongPressAction(action, bibNumber) {
    switch (action) {
        case 'perform':
            return setParticipantPerforming(bibNumber);
        case 'stop':
            return setParticipantStopped(bibNumber);
        case 'reset':
            return resetParticipantStatus(bibNumber);
        default:
            console.warn('Unknown long-press action:', action);
    }
}

function startLongPressAction(bibNumber, button, event, action = 'perform') {
    if (!button || button.disabled) {
        return false;
    }
    if (event) {
        event.preventDefault();
        event.stopPropagation();
    }
    delete button.dataset.holdCompleted;
    cancelLongPressAction(button);
    button.classList.add('hold-arming');
    button.style.setProperty('--hold-progress', '0%');
    const state = {
        bibNumber,
        action,
        start: performance.now(),
        timeoutId: null,
        rafId: null
    };
    const updateProgress = (timestamp) => {
        const currentState = longPressStates.get(button);
        if (!currentState) {
            return;
        }
        const elapsed = timestamp - currentState.start;
        const progress = Math.min(1, elapsed / LONG_PRESS_DURATION);
        button.style.setProperty('--hold-progress', `${(progress * 100).toFixed(1)}%`);
        if (progress < 1) {
            currentState.rafId = requestAnimationFrame(updateProgress);
        }
    };
    state.rafId = requestAnimationFrame(updateProgress);
    state.timeoutId = setTimeout(() => finalizeLongPressAction(button, state), LONG_PRESS_DURATION);
    longPressStates.set(button, state);
    return false;
}

function finalizeLongPressAction(button, state) {
    if (!button) {
        return;
    }
    if (state?.timeoutId) {
        clearTimeout(state.timeoutId);
    }
    if (state?.rafId) {
        cancelAnimationFrame(state.rafId);
    }
    longPressStates.delete(button);
    button.classList.remove('hold-arming');
    button.classList.add('hold-triggered');
    button.dataset.holdCompleted = 'true';
    button.style.setProperty('--hold-progress', '100%');
    executeLongPressAction(state.action, state.bibNumber);
    setTimeout(() => {
        button.classList.remove('hold-triggered');
        button.style.removeProperty('--hold-progress');
        delete button.dataset.holdCompleted;
    }, 500);
}

function cancelLongPressAction(button) {
    if (!button) {
        return false;
    }
    const state = longPressStates.get(button);
    if (state) {
        clearTimeout(state.timeoutId);
        cancelAnimationFrame(state.rafId);
        longPressStates.delete(button);
    }
    if (button.dataset.holdCompleted === 'true') {
        return false;
    }
    button.classList.remove('hold-arming');
    button.style.removeProperty('--hold-progress');
    return false;
}

function handleLongPressKeyDown(event, bibNumber, button, action = 'perform') {
    if (!isHoldActivationKey(event)) {
        return true;
    }
    if (event.repeat) {
        event.preventDefault();
        return false;
    }
    startLongPressAction(bibNumber, button, event, action);
    return false;
}

function handleLongPressKeyUp(event, button) {
    if (!isHoldActivationKey(event)) {
        return true;
    }
    event.preventDefault();
    cancelLongPressAction(button);
    return false;
}

function setupRefreshControls() {
    // Heat card refresh interval listeners
    document.querySelectorAll('input[name="heatCardRefresh"]').forEach(radio => {
        radio.addEventListener('change', (e) => {
            activeHeatCardRefreshRate = parseInt(e.target.value);
            console.log('Refresh interval changed to:', activeHeatCardRefreshRate);
            
            if (activeHeatCardRefreshRate === 0) {
                showInfoToast('⏸️ Auto-refresh Paused', 'Heat card will not auto-update');
            } else {
                showSuccessToast(`⏱️ Refresh: ${activeHeatCardRefreshRate}s`, 'Heat card will update automatically');
            }
            
            restartActiveHeatCardInterval();
        });
    });
}

async function loadActiveHeatCard(heatNumber = null) {
    const container = document.getElementById('activeHeatCardContainer');
    const updateTime = document.getElementById('activeHeatUpdateTime');
    
    if (!currentEventId) {
        container.innerHTML = `
            <div class="col-12 text-center text-muted py-5">
                <i class="fas fa-exclamation-triangle fa-3x mb-3 text-warning"></i>
                <p class="mb-0">No event ID provided</p>
            </div>
        `;
        updateTime.innerHTML = '<i class="fas fa-refresh me-1"></i> --';
        return;
    }
    
    try {
        // Step 1: Get the active heat number from database
        let heatToLoad = heatNumber || currentHeatNumber;
        
        if (!heatToLoad) {
            const activeHeatResponse = await fetch(`get_active_heat_api.php?event_id=${currentEventId}`);
            const activeHeatData = await activeHeatResponse.json();
            
            if (activeHeatData.success && activeHeatData.data) {
                heatToLoad = activeHeatData.data.heat_number;
                console.log('🔥 Active heat from DB:', heatToLoad, 'Status:', activeHeatData.data.status);
            } else {
                console.warn('⚠️ No active heat found:', activeHeatData.message);
            }
        }
        
        if (!heatToLoad) {
            container.innerHTML = `
                <div class="col-12 text-center text-muted py-5">
                    <i class="fas fa-fire fa-3x mb-3 opacity-25"></i>
                    <p class="mb-0">No heats configured</p>
                    <p class="small text-muted">Configure heats for this event first</p>
                </div>
            `;
            updateTime.innerHTML = '<i class="fas fa-clock me-1"></i>No heats';
            return;
        }
        
        // Step 2: Load the heat card HTML for this specific heat
        const response = await fetch(`heat_cards_api.php?event_id=${currentEventId}&layout=${heatCardLayout}`);
        const data = await response.json();
        
        if (!response.ok || !data.success) {
            throw new Error(data.error || 'Failed to load heat cards');
        }
        
        // Parse the HTML and find the specific heat card
        const parser = new DOMParser();
        const doc = parser.parseFromString(data.data, 'text/html');
        const allCards = doc.querySelectorAll('[class*="col-"]');
        
        let activeCard = null;
        
        // Find the card for our heat
        allCards.forEach(card => {
            const heatText = card.textContent;
            if (heatText.includes(`Heat ${heatToLoad}`)) {
                activeCard = card;
            }
        });
        
        if (activeCard) {
            // Update current heat number for future reference
            currentHeatNumber = heatToLoad;
            
            // Display the heat card
            container.innerHTML = activeCard.outerHTML;
            
            // Update timestamp
            const now = new Date();
            updateTime.innerHTML = '<i class="fas fa-refresh me-1"></i> ' + now.toLocaleTimeString();
            
            console.log('✅ Loaded heat card for Heat:', heatToLoad);
            
            // Load participant list for this heat
            loadParticipantList(heatToLoad);
        } else {
            container.innerHTML = `
                <div class="col-12 text-center text-muted py-5">
                    <i class="fas fa-fire fa-3x mb-3 opacity-25"></i>
                    <p class="mb-0">Heat ${heatToLoad} not found</p>
                    <p class="small text-muted">Heat card could not be generated</p>
                </div>
            `;
            updateTime.innerHTML = '<i class="fas fa-clock me-1"></i>Heat not found';
            console.warn('⚠️ Heat card not found for heat:', heatToLoad);
        }
        
    } catch (error) {
        console.error('Error loading active heat card:', error);
        container.innerHTML = `
            <div class="col-12">
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <strong>Error loading heat data:</strong> ${error.message}
                </div>
            </div>
        `;
        updateTime.innerHTML = '<i class="fas fa-exclamation-triangle me-1"></i>Error';
    }
}

function restartActiveHeatCardInterval() {
    // Clear existing interval
    if (activeHeatCardInterval) {
        clearInterval(activeHeatCardInterval);
        activeHeatCardInterval = null;
    }
    
    // Only start if not paused (rate > 0)
    if (activeHeatCardRefreshRate > 0) {
        activeHeatCardInterval = setInterval(() => {
            loadActiveHeatCard();
        }, activeHeatCardRefreshRate * 1000);
        console.log(`Active heat card refresh interval set to ${activeHeatCardRefreshRate}s`);
    } else {
        console.log('Active heat card auto-refresh paused');
    }
}

// ============================================================================
// PARTICIPANT LIST FUNCTIONS
// ============================================================================

let participantListData = [];
let participantGridInitialized = false;

async function loadParticipantList(heatNumber = null) {
    const targetHeat = heatNumber || currentHeatNumber;
    
    if (!currentEventId || !targetHeat) {
        console.log('Cannot load participants: missing event_id or heat_number');
        return;
    }
    
    const section = document.getElementById('participantListSection');
    const grid = document.getElementById('participantCardsGrid');
    
    // Show loading placeholder only before the first render to avoid flicker
    if (!participantGridInitialized || grid.children.length === 0) {
        grid.innerHTML = `
            <div class="col-12 text-center text-muted py-3">
                <i class="fas fa-spinner fa-spin fa-2x mb-2"></i>
                <p class="mb-0">Loading participants...</p>
            </div>
        `;
    }
    
    try {
        // Load participants from heat management API
        const response = await fetch(`heat_management_api.php?action=get_queue&event_id=${currentEventId}&heat_number=${targetHeat}&mode=sort_order`);
        const data = await response.json();
        
        if (!response.ok) {
            throw new Error('API request failed');
        }
        
        // The API returns { queue: [], heat_settings: {}, mode: '' }
        participantListData = data.queue || [];
        
        // Store heat_settings globally for access by loadParticipantScores()
        window.currentHeatSettings = data.heat_settings || {};
        
        // Debug: Check participant data structure
        if (participantListData.length > 0) {
            console.log('🔍 First participant object keys:', Object.keys(participantListData[0]));
            console.log('🔍 First participant sample:', participantListData[0]);
            console.log('🔍 Sort orders:', participantListData.slice(0, 5).map(p => `BIB ${p.bib_number}: sort_order=${p.sort_order}`));
        }
        
        // Load scores for participants
        await loadParticipantScores(targetHeat);
        
        // Populate category filter
        populateCategoryFilter();
        
        // Note: renderParticipantList() is called by loadParticipantScores() via updateParticipantCards()
        // No need to call it again here
        
        // Show the section
        section.style.display = 'block';
        
        //console.log(`Loaded ${participantListData.length} participants for Heat ${targetHeat}`);
        
    } catch (error) {
        //console.error('Error loading participant list:', error);
        grid.innerHTML = `
            <div class="col-12">
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <strong>Error loading participants:</strong> ${error.message}
                </div>
            </div>
        `;
        updateHeatQueuePanel([], window.currentHeatSettings);
    }
}

async function loadParticipantScores(heatNumber) {
    // Load scores from scoring_results table for current run AND runs completion data
    try {
        // Get total runs from heat_settings (reliable source) instead of parsing HTML badge
        let totalRuns = window.currentHeatSettings?.runs_count;
        if (!Number.isFinite(totalRuns) || totalRuns < 1) {
            totalRuns = 1;
        }
        
        // Get current run from heat_settings first, fall back to heat card badge (UI state)
        let currentRun = window.currentHeatSettings?.active_run || 1;
        const runBadge = document.querySelector('#activeHeatCardContainer .badge.bg-success');
        if (runBadge && runBadge.textContent.includes('/')) {
            const match = runBadge.textContent.match(/(\d+)\/(\d+)/);
            if (match) {
                currentRun = parseInt(match[1]);
                if (match[2]) {
                    const badgeTotal = parseInt(match[2]);
                    if (Number.isFinite(badgeTotal) && badgeTotal >= currentRun) {
                        totalRuns = badgeTotal;
                    }
                }
            }
        }
        
        console.log(`🔍 Loading scores for Heat ${heatNumber}, Current Run: ${currentRun}/${totalRuns} (from heat_settings)`);
        
        // Load current run scores from scoring_results
        const scoresUrl = `heat_management_api.php?action=get_heat_run_scores&event_id=${currentEventId}&heat_number=${heatNumber}&run_number=${currentRun}`;
        const scoresResponse = await fetch(scoresUrl);
        const scoresData = await scoresResponse.json();
        
        // Load ALL runs completion data from runs table
        const runsUrl = `heat_management_api.php?action=get_participant_runs&event_id=${currentEventId}&heat_number=${heatNumber}`;
        const runsResponse = await fetch(runsUrl);
        const runsData = await runsResponse.json();
        
        console.log('📥 Scores Response:', scoresData);
        console.log('📥 Runs Response:', runsData);
        
        // Build a map: event_participant_id -> array of run records
        const runsByParticipant = new Map();
        if (runsData.success && Array.isArray(runsData.runs)) {
            console.log(`📊 Processing ${runsData.runs.length} run records from database`);
            runsData.runs.forEach(run => {
                const epId = String(run.event_participant_id);
                if (!runsByParticipant.has(epId)) {
                    runsByParticipant.set(epId, []);
                }
                runsByParticipant.get(epId).push(run);
            });
            console.log(`📊 Runs mapped to ${runsByParticipant.size} participants`);
        }
        
        // Merge data into participant list
        participantListData.forEach(participant => {
            const epId = String(participant.id || participant.event_participant_id);
            
            // Get runs from runs table for this participant
            const participantRuns = runsByParticipant.get(epId) || [];
            
            // Build run indicators data: which runs are completed (exist in runs table)
            participant.total_runs = totalRuns;
            participant.current_run = currentRun;
            participant.completed_runs = []; // Array of run numbers that are completed
            
            participantRuns.forEach(run => {
                const runNum = parseInt(run.run_number);
                if (runNum >= 1 && runNum <= totalRuns) {
                    participant.completed_runs.push(runNum);
                }
            });
            
            console.log(`👤 BIB ${participant.bib_number} (ID: ${epId}): ${participant.completed_runs.length}/${totalRuns} runs completed [${participant.completed_runs.join(', ')}]`);
            
            // Get current run scores from scoring_results
            if (scoresData.success && Array.isArray(scoresData.scores)) {
                const participantScores = scoresData.scores.filter(s => String(s.event_participant_id) === epId);
                
                if (participantScores.length > 0) {
                    participant.has_score = true;
                    participant.judge_count = participantScores.length;
                    
                    // Calculate average score
                    const totalScore = participantScores.reduce((sum, s) => sum + parseFloat(s.score_value || 0), 0);
                    participant.total_score = (totalScore / participantScores.length).toFixed(2);
                    
                    // Store first judge's data as representative
                    const firstScore = participantScores[0];
                    participant.figures_json = firstScore.figures_json;
                    participant.status = firstScore.status;
                    participant.run_number = currentRun;
                } else {
                    participant.has_score = false;
                    participant.judge_count = 0;
                    participant.total_score = null;
                }
            } else {
                participant.has_score = false;
                participant.judge_count = 0;
                participant.total_score = null;
            }
        });
        
        console.log('✅ Loaded scores and run completion data');
        renderParticipantList();
        updateHeatQueuePanel(participantListData, window.currentHeatSettings);
        
    } catch (error) {
        console.error('❌ Error loading scores:', error);
    }
}

function populateCategoryFilter() {
    const categoryFilter = document.getElementById('categoryFilter');
    if (!categoryFilter) return;
    
    // Get unique categories
    const categories = [...new Set(participantListData
        .map(p => p.display_category || p.event_category || p.participant_category)
        .filter(c => c))];
    
    categoryFilter.innerHTML = '<option value="">All Categories</option>';
    categories.forEach(cat => {
        const option = document.createElement('option');
        option.value = cat;
        option.textContent = cat;
        categoryFilter.appendChild(option);
    });
}

/**
 * Build run indicators showing O (completed) or X (missing) for each run
 * Based on runs table: SELECT * FROM runs WHERE event_participant_id = ?
 */
function buildRunIndicatorsHTML(participant) {
    const totalRuns = participant.total_runs || 0;
    if (!totalRuns) {
        return '';
    }
    
    const completedRuns = participant.completed_runs || [];
    const currentRun = participant.current_run || 1;
    
    console.log(`🔨 Building indicators for BIB ${participant.bib_number}: ${completedRuns.length}/${totalRuns} runs, completed: [${completedRuns.join(', ')}]`);
    
    let html = '<div class="run-indicators-container col-auto">';
    html += '<div class="run-indicators align-items-center">';
    
    // Label
   
    html += '<div class="d-flex gap-2 align-items-center">';
    // Build indicators for each run
    for (let run = 1; run <= totalRuns; run++) {
        const isCompleted = completedRuns.includes(run);
        const isCurrent = run === currentRun;
        
        let className = 'run-indicator';
        let symbol = '';
        let title = `Run ${run}: Not completed`;
        
        if (isCompleted) {
            className += ' completed';
            symbol = 'O';
            title = `Run ${run}: Completed`;
        }
        
        if (isCurrent) {
            className += ' current';
            title += ' (current)';
        }
        
        html += `<span class="${className}" title="${title}" data-run="${run}">${symbol}</span>`;
    }
     html += '</div>';
    // Summary count
    const completedCount = completedRuns.length;
    //html += `<span class="run-summary text-muted small ms-2">(${completedCount}/${totalRuns})</span>`;
    
    html += '</div></div>';
    return html;
}

// Update existing cards without rebuilding - prevents flickering
function updateParticipantCardsWithData(dataToRender) {
    const grid = document.getElementById('participantCardsGrid');
    
    // Check if we have any participant cards (not error/empty messages)
    const existingWrappers = grid.querySelectorAll('.col-md-12[data-bib]');
    const hasCards = existingWrappers.length > 0;
    
    // If no existing cards AND grid is empty or has placeholder, do initial full render
    if (!hasCards) {
        console.log('📝 Initial render: building ' + dataToRender.length + ' cards');
        const html = dataToRender.map((participant, index) => {
            participant._renderIndex = index;
            return renderParticipantCard(participant, index);
        }).join('');
        grid.innerHTML = html;
        participantGridInitialized = true;
        return;
    }
    
    participantGridInitialized = true;
    console.log('🔄 Smart update: ' + existingWrappers.length + ' existing, ' + dataToRender.length + ' new');
    
    // Create a map of existing wrappers by BIB number
    const wrapperMap = new Map();
    existingWrappers.forEach(wrapper => {
        const bib = wrapper.getAttribute('data-bib');
        if (bib) {
            wrapperMap.set(bib, wrapper);
        }
    });
    
    // Track which BIBs are in the new data
    const newBibs = new Set(dataToRender.map(p => p.bib_number.toString()));
    
    // Remove wrappers that no longer exist in filtered data
    wrapperMap.forEach((wrapper, bib) => {
        if (!newBibs.has(bib)) {
            console.log('➖ Removing BIB ' + bib);
            wrapper.style.transition = 'opacity 0.3s ease, transform 0.3s ease';
            wrapper.style.opacity = '0';
            wrapper.style.transform = 'scale(0.9)';
            setTimeout(() => wrapper.remove(), 300);
        }
    });
    
    // Update existing cards and add new ones
    dataToRender.forEach((participant, index) => {
        participant._renderIndex = index;
        const bibStr = participant.bib_number.toString();
        const existingWrapper = wrapperMap.get(bibStr);
        
        if (existingWrapper) {
            // Check if data has changed by comparing data attributes
            const hasScore = participant.has_score && participant.total_score !== null;
            const score = hasScore ? parseFloat(participant.total_score).toFixed(2) : '--';
            const judgeCount = participant.judge_count || 0;
            const status = participant.status || 'none';
            const hasScoreFlag = hasScore ? '1' : '0';
            const figuresData = participant.figures_json ? JSON.stringify(participant.figures_json) : '';
            const participantStatus = participant.participant_status || '';
            const completedRunsData = JSON.stringify(participant.completed_runs || []);
            
            // Compare current data with stored data attributes
            const scoreChanged = existingWrapper.getAttribute('data-score') !== score;
            const judgeCountChanged = existingWrapper.getAttribute('data-judge-count') !== judgeCount.toString();
            const statusChanged = existingWrapper.getAttribute('data-status') !== status;
            const hasScoreChanged = existingWrapper.getAttribute('data-has-score') !== hasScoreFlag;
            const figuresChanged = existingWrapper.getAttribute('data-figures') !== figuresData;
            const participantStatusChanged = existingWrapper.getAttribute('data-participant-status') !== participantStatus;
            const completedRunsChanged = existingWrapper.getAttribute('data-completed-runs') !== completedRunsData;
            const previousIndexAttr = existingWrapper.getAttribute('data-position-index') ?? '';
            const positionIndexChanged = previousIndexAttr !== index.toString();
            
            const dataChanged = scoreChanged || judgeCountChanged || statusChanged || hasScoreChanged || figuresChanged || participantStatusChanged || completedRunsChanged;
            
            if (dataChanged || positionIndexChanged) {
                console.log('🔄 Updating BIB ' + bibStr + ':', {
                    scoreChanged,
                    judgeCountChanged,
                    statusChanged,
                    hasScoreChanged,
                    figuresChanged,
                    participantStatusChanged,
                    completedRunsChanged,
                    positionIndexChanged
                });
                
                // Get the inner card element
                const card = existingWrapper.querySelector('.participant-score-card');
                
                // Update the card with new data
                updateSingleCardFromData(card, participant, {
                    scoreChanged,
                    judgeCountChanged,
                    statusChanged,
                    hasScoreChanged,
                    figuresChanged,
                    participantStatusChanged,
                    completedRunsChanged,
                    positionIndex: index,
                    positionIndexChanged
                });
                
                // Update data attributes on wrapper
                existingWrapper.setAttribute('data-score', score);
                existingWrapper.setAttribute('data-judge-count', judgeCount);
                existingWrapper.setAttribute('data-status', status);
                existingWrapper.setAttribute('data-has-score', hasScoreFlag);
                existingWrapper.setAttribute('data-figures', figuresData);
                existingWrapper.setAttribute('data-participant-status', participantStatus);
                existingWrapper.setAttribute('data-completed-runs', completedRunsData);
            }
            
            // Check if card needs to be repositioned
            const currentIndex = Array.from(grid.children).indexOf(existingWrapper);
            if (currentIndex !== index) {
                // Move card to correct position
                if (index === 0) {
                    grid.insertBefore(existingWrapper, grid.firstChild);
                } else if (index >= grid.children.length) {
                    grid.appendChild(existingWrapper);
                } else {
                    grid.insertBefore(existingWrapper, grid.children[index]);
                }
            }

            existingWrapper.setAttribute('data-position-index', index);
        } else {
            // Add new card with fade-in animation
            console.log('➕ Adding new BIB ' + participant.bib_number);
            const newCardHTML = renderParticipantCard(participant, index);
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = newCardHTML;
            const newWrapper = tempDiv.firstElementChild;
            
            // Insert in correct position based on index
            if (index === 0) {
                grid.insertBefore(newWrapper, grid.firstChild);
            } else if (index >= grid.children.length) {
                grid.appendChild(newWrapper);
            } else {
                grid.insertBefore(newWrapper, grid.children[index]);
            }
            
            // Fade in animation
            newWrapper.style.opacity = '0';
            newWrapper.style.transform = 'scale(0.9)';
            requestAnimationFrame(() => {
                newWrapper.style.transition = 'opacity 0.3s ease, transform 0.3s ease';
                newWrapper.style.opacity = '1';
                newWrapper.style.transform = 'scale(1)';
            });
        }
    });
}

// Legacy function for backward compatibility
function updateParticipantCards() {
    updateParticipantCardsWithData(participantListData);
}

// Update only the specific elements that changed in a card
function updateSingleCardFromData(card, participant, changes) {
    const hasScore = participant.has_score && participant.total_score !== null;
    const score = hasScore ? parseFloat(participant.total_score).toFixed(2) : '';
    const judgeCount = participant.judge_count || 0;
    
    // Update score value if it changed
    if (changes.scoreChanged) {
        const scoreValueEl = card.querySelector('.score-value');
        if (scoreValueEl) {
            scoreValueEl.textContent = score;
            // Add pulse animation when score changes
            scoreValueEl.style.animation = 'none';
            requestAnimationFrame(() => {
                scoreValueEl.style.animation = 'scorePulse 0.5s ease-out';
            });
        }
    }
    
    // Update judge count in label if it changed
    if (changes.judgeCountChanged || changes.hasScoreChanged) {
        const scoreLabelEl = card.querySelector('.score-label');
        if (scoreLabelEl) {
            const newLabel = hasScore && judgeCount > 1 ? 'Avg (' + judgeCount + ' judges)' : '';
            scoreLabelEl.textContent = newLabel;
        }
    }
    
    // Update status badge and card class if status changed
    if (changes.statusChanged || changes.hasScoreChanged || changes.judgeCountChanged) {
        const existingBadge = card.querySelector('.status-badge');
        
        // Determine what badge should be shown
        let newBadgeHTML = '';
        if (participant.status === 'current') {
            newBadgeHTML = '<span class="status-badge badge bg-warning text-dark">Performing</span>';
        } else if (hasScore) {
           // newBadgeHTML = '<span class="status-badge badge bg-success">Scored (' + judgeCount + ' judges)</span>';
        }
        
        if (newBadgeHTML) {
            if (!existingBadge) {
                // Add badge if it doesn't exist
                const tempDiv = document.createElement('div');
                tempDiv.innerHTML = newBadgeHTML;
                card.insertBefore(tempDiv.firstChild, card.firstChild);
            } else {
                // Replace badge if it exists
                const tempDiv = document.createElement('div');
                tempDiv.innerHTML = newBadgeHTML;
                existingBadge.replaceWith(tempDiv.firstChild);
            }
        } else if (existingBadge) {
            // Remove badge if it shouldn't exist
            existingBadge.remove();
        }
        
        // Update card class
        const statusClass = participant.status === 'current' ? 'performing' : (hasScore ? 'has-score' : '');
        card.className = 'participant-score-card p-2 participant-score-card rounded-3 shadow-lg ' + statusClass + ' ' + Object.keys(card.dataset).map(k => 'data-' + k).join(' ');
        
        // Restore data attributes (class change removes them)
        card.setAttribute('data-bib', card.dataset.bib);
        card.setAttribute('data-score', card.dataset.score);
        card.setAttribute('data-judge-count', card.dataset.judgeCount);
        card.setAttribute('data-status', card.dataset.status);
        card.setAttribute('data-has-score', card.dataset.hasScore);
        card.setAttribute('data-figures', card.dataset.figures);
    }

    if (changes.participantStatusChanged) {
        updateParticipantStatusBadgeElement(card, participant.participant_status || '');
    }

    // Update control buttons if status, participant status, layout, or position changed
    if (changes.statusChanged || changes.participantStatusChanged || changes.positionIndexChanged) {
        const derivedPositionIndex = typeof changes.positionIndex === 'number'
            ? changes.positionIndex
            : (typeof participant._renderIndex === 'number' ? participant._renderIndex : null);
        replaceCardControls(card, participant, derivedPositionIndex);
    }
    
    // Update run indicators if completed runs changed
    if (changes.completedRunsChanged) {
        const participantInfoEl = card.querySelector('.participant-info');
        if (participantInfoEl) {
            const existingIndicators = participantInfoEl.querySelector('.run-indicators-container');
            const newIndicatorsHTML = buildRunIndicatorsHTML(participant).trim();
            
            if (newIndicatorsHTML) {
                const tempDiv = document.createElement('div');
                tempDiv.innerHTML = newIndicatorsHTML;
                const newIndicators = tempDiv.firstElementChild;
                
                if (existingIndicators) {
                    existingIndicators.replaceWith(newIndicators);
                } else {
                    // Append after participant details
                    participantInfoEl.appendChild(newIndicators);
                }
            } else if (existingIndicators) {
                existingIndicators.remove();
            }
        }
    }
    
    // Update figures if they changed
    if (changes.figuresChanged) {
        const existingFigures = card.querySelector('.criteria-scores');
        
        let newFiguresHTML = '';
        if (participant.figures_json) {
            try {
                const figures = typeof participant.figures_json === 'string' 
                    ? JSON.parse(participant.figures_json) 
                    : participant.figures_json;
                
                if (figures && typeof figures === 'object') {
                    newFiguresHTML = '<div class="criteria-scores">';
                    const figureData = figures.figures || figures;
                    
                    for (const [key, value] of Object.entries(figureData)) {
                        if (key !== 'validation' && value !== null && value !== '') {
                            const displayValue = Array.isArray(value) ? value.join(', ') : value;
                            newFiguresHTML += '<div class="criteria-score-item"><strong>' + key + ':</strong> ' + displayValue + '</div>';
                        }
                    }
                    newFiguresHTML += '</div>';
                }
            } catch (e) {
                console.error('Error parsing figures:', e);
            }
        }
        
        if (newFiguresHTML) {
            if (!existingFigures) {
                // Add figures if they don't exist
                const tempDiv = document.createElement('div');
                tempDiv.innerHTML = newFiguresHTML;
                card.appendChild(tempDiv.firstChild);
            } else {
                // Replace figures if they exist
                const tempDiv = document.createElement('div');
                tempDiv.innerHTML = newFiguresHTML;
                existingFigures.replaceWith(tempDiv.firstChild);
            }
        } else if (existingFigures) {
            // Remove figures if they shouldn't exist
            existingFigures.remove();
        }
    }
    
    // Add subtle highlight flash when anything updates
    card.style.transition = 'background-color 0.3s ease';
    const originalBg = card.style.backgroundColor;
    card.style.backgroundColor = '#e7f3ff';
    setTimeout(() => {
        card.style.backgroundColor = originalBg || '';
    }, 300);
}

// Legacy function - now replaced by updateSingleCardFromData
function updateSingleCard(card, participant) {
    const hasScore = participant.has_score && participant.total_score !== null;
    const score = hasScore ? parseFloat(participant.total_score).toFixed(2) : '';
    const judgeCount = participant.judge_count || 0;
    
    let updated = false;
    
    // Update score value only if changed
    const scoreValueEl = card.querySelector('.score-value');
    if (scoreValueEl && scoreValueEl.textContent !== score) {
        scoreValueEl.textContent = score;
        // Add pulse animation when score changes
        scoreValueEl.style.animation = 'none';
        requestAnimationFrame(() => {
            scoreValueEl.style.animation = 'scorePulse 0.5s ease-out';
        });
        updated = true;
    }
    
    // Update score label only if changed
    const scoreLabelEl = card.querySelector('.score-label');
    if (scoreLabelEl) {
        const newLabel = hasScore && judgeCount > 1 ? 'Avg (' + judgeCount + ' judges)' : '';
        if (scoreLabelEl.textContent !== newLabel) {
            scoreLabelEl.textContent = newLabel;
            updated = true;
        }
    }
    
    // Update status badge only if needed
    const existingBadge = card.querySelector('.status-badge');
    let shouldHaveBadge = false;
    let newBadgeHTML = '';
    
    if (participant.status === 'current') {
        shouldHaveBadge = true;
        newBadgeHTML = '<span class="status-badge badge bg-warning text-dark">Performing</span>';
    } else if (hasScore) {
        shouldHaveBadge = true;
       // newBadgeHTML = '<span class="status-badge badge bg-success">Scored (' + judgeCount + ' judges)</span>';
    }
    
    if (shouldHaveBadge) {
        if (!existingBadge) {
            // Add badge if it doesn't exist
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = newBadgeHTML;
            card.insertBefore(tempDiv.firstChild, card.firstChild);
            updated = true;
        } else if (existingBadge.textContent !== tempDiv.textContent) {
            // Update badge if text changed
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = newBadgeHTML;
            if (existingBadge.outerHTML !== tempDiv.innerHTML) {
                existingBadge.replaceWith(tempDiv.firstChild);
                updated = true;
            }
        }
    } else if (existingBadge) {
        // Remove badge if it shouldn't exist
        existingBadge.remove();
        updated = true;
    }

    updateParticipantStatusBadgeElement(card, participant.participant_status || '');
    
    // Update control buttons based on status
    const participantId = participant.event_participant_id || participant.id;
    const isPerforming = participant.status === 'current' || participantStatus === 'performing';
    const participantStatus = participant.participant_status || '';
    const isStopped = participantStatus === 'stop';
    
    const getHoldEventAttributes = (action) => `
        onmousedown="return startLongPressAction(${participant.bib_number}, this, event, '${action}')"
        onmouseup="return cancelLongPressAction(this)"
        onmouseleave="return cancelLongPressAction(this)"
        ontouchstart="return startLongPressAction(${participant.bib_number}, this, event, '${action}')"
        ontouchend="return cancelLongPressAction(this)"
        ontouchcancel="return cancelLongPressAction(this)"
        onkeydown="return handleLongPressKeyDown(event, ${participant.bib_number}, this, '${action}')"
        onkeyup="return handleLongPressKeyUp(event, this)"
        onblur="return cancelLongPressAction(this)"
    `;

    let actionButtonsHTML = '';
    if (isPerforming) {
        actionButtonsHTML = `
            <button class="btn btn-lg btn-danger performing-hold-btn px-3 px-md-5" title="Hold 2s to stop performing" ${getHoldEventAttributes('stop')}>
                <i class="fas fa-stop"></i>
            </button>
        `;
    } else {
        actionButtonsHTML = `
            <button class="btn btn-lg btn-outline-success performing-hold-btn px-3 px-md-5" title="Hold 2s to mark as performing" ${getHoldEventAttributes('perform')}>
                <i class="fas fa-play"></i>
            </button>
        `;

        if (isStopped) {
            actionButtonsHTML += `
                <button class="btn btn-lg btn-outline-secondary performing-hold-btn px-2 px-md-3" title="Hold 2s to reset status" ${getHoldEventAttributes('reset')}>
                    <i class="fas fa-undo"></i>
                </button>
            `;
        }
    }

    const newControlsHTML = `
        <div class="position-controls d-flex flex-row gap-1">
            <button class="btn btn-lg btn-outline-primary" onclick="moveParticipantUp(${participantId})" title="Move up in start order">
                <i class="fas fa-arrow-up"></i>
            </button>
            <button class="btn btn-lg btn-outline-primary" onclick="moveParticipantDown(${participantId})" title="Move down in start order">
                <i class="fas fa-arrow-down"></i>
            </button>
            ${actionButtonsHTML}
        </div>
    `;

    const flexRow = card.querySelector('.d-flex.align-items-center');
    if (flexRow) {
        const existingControls = flexRow.querySelector('.position-controls');
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = newControlsHTML.trim();
        const newControls = tempDiv.firstElementChild;
        
        if (existingControls && newControls) {
            existingControls.replaceWith(newControls);
            updated = true;
        } else if (!existingControls && newControls) {
            flexRow.appendChild(newControls);
            updated = true;
        }
    }
    
    // Update card styling only if status changed
    const statusClass = participant.status === 'current' ? 'performing' : (hasScore ? 'has-score' : '');
    const expectedClassName = 'participant-score-card ' + statusClass;
    if (card.className !== expectedClassName) {
        card.className = expectedClassName;
        updated = true;
    }
    
    // Update figures only if changed
    const existingFigures = card.querySelector('.criteria-scores');
    let newFiguresHTML = '';
    
    if (participant.figures_json) {
        try {
            const figures = typeof participant.figures_json === 'string' 
                ? JSON.parse(participant.figures_json) 
                : participant.figures_json;
            
            if (figures && typeof figures === 'object') {
                newFiguresHTML = '<div class="criteria-scores">';
                const figureData = figures.figures || figures;
                
                for (const [key, value] of Object.entries(figureData)) {
                    if (key !== 'validation' && value !== null && value !== '') {
                        const displayValue = Array.isArray(value) ? value.join(', ') : value;
                        newFiguresHTML += '<div class="criteria-score-item"><strong>' + key + ':</strong> ' + displayValue + '</div>';
                    }
                }
                newFiguresHTML += '</div>';
            }
        } catch (e) {
            console.error('Error parsing figures:', e);
        }
    }
    
    if (newFiguresHTML) {
        if (!existingFigures) {
            // Add figures if they don't exist
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = newFiguresHTML;
            card.appendChild(tempDiv.firstChild);
            updated = true;
        } else if (existingFigures.innerHTML !== newFiguresHTML.replace('<div class="criteria-scores">', '').replace('</div>', '')) {
            // Update figures if content changed
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = newFiguresHTML;
            existingFigures.replaceWith(tempDiv.firstChild);
            updated = true;
        }
    } else if (existingFigures) {
        // Remove figures if they shouldn't exist
        existingFigures.remove();
        updated = true;
    }
    
    // Optional: Add subtle highlight when card updates
    if (updated) {
        card.style.transition = 'background-color 0.3s ease';
        const originalBg = card.style.backgroundColor;
        card.style.backgroundColor = '#e7f3ff';
        setTimeout(() => {
            card.style.backgroundColor = originalBg;
        }, 300);
    }
}

function renderParticipantList() {
    const grid = document.getElementById('participantCardsGrid');
    const categoryFilter = document.getElementById('categoryFilter');
    const scoredFilter = document.getElementById('scoredFilter');
    const sortOrder = document.getElementById('sortOrder');
    const performingToggle = document.getElementById('performingToggle');
    const showPerformingParticipants = performingToggle ? performingToggle.checked : true;
    const isPerformingParticipant = (participant) => {
        const queueStatus = (participant.status || '').toLowerCase();
        const participantStatus = (participant.participant_status || '').toLowerCase();
        return queueStatus === 'current' || participantStatus === 'performing';
    };
    
    if (!participantListData || participantListData.length === 0) {
        grid.innerHTML = `
            <div class="col-12 text-center text-muted py-3">
                <i class="fas fa-users fa-2x mb-2 opacity-25"></i>
                <p class="mb-0">No participants in this heat</p>
            </div>
        `;
        return;
    }
    
    // Filter by category
    let filtered = participantListData;
    const currentRunFilter = window.currentHeatSettings?.active_run || participantListData[0]?.current_run || 1;
    const normalizedCurrentRun = Number.isFinite(currentRunFilter) && currentRunFilter > 0 ? currentRunFilter : 1;
    const normalizedRunsCount = Number.isFinite(window.currentHeatSettings?.runs_count) && window.currentHeatSettings.runs_count > 0
        ? window.currentHeatSettings.runs_count
        : (participantListData[0]?.total_runs || normalizedCurrentRun);

    const getParticipantRunTotal = (participant) => {
        const totalRuns = Number.isFinite(participant.total_runs) && participant.total_runs > 0
            ? participant.total_runs
            : normalizedRunsCount;
        return Math.max(totalRuns, normalizedCurrentRun);
    };

    const isExpectedThisRun = (participant) => normalizedCurrentRun <= getParticipantRunTotal(participant);
    const hasCurrentRunScore = (participant) => participant.has_score && participant.total_score !== null;
    const isUnscoredCurrentRun = (participant) => isExpectedThisRun(participant) && !hasCurrentRunScore(participant);
    const selectedCategory = categoryFilter?.value;
    if (selectedCategory) {
        filtered = filtered.filter(p => 
            (p.display_category === selectedCategory) ||
            (p.event_category === selectedCategory) ||
            (p.participant_category === selectedCategory)
        );
    }

    if (!showPerformingParticipants) {
        filtered = filtered.filter(p => !isPerformingParticipant(p));
    }
    
    // Filter by scored status
    const scoredFilterValue = scoredFilter?.value || 'all';
    let unscored = [];
    let scored = [];
    
    if (scoredFilterValue === 'unscored') {
        // Only show participants who have activity in the current run but no scores yet
        filtered = filtered.filter(isUnscoredCurrentRun);
    } else if (scoredFilterValue === 'scored_bottom') {
        // Separate scored and unscored
        unscored = filtered.filter(isUnscoredCurrentRun);
        scored = filtered.filter(p => !isUnscoredCurrentRun(p));
    } else if (scoredFilterValue === 'performing_scoring') {
        // Show only participants with status 'performing' or 'scoring'
        filtered = filtered.filter(p => 
            p.participant_status === 'performing' || p.participant_status === 'scoring'
        );
    } else if (scoredFilterValue === 'complete') {
        // Show only participants with status 'complete'
        filtered = filtered.filter(p => p.participant_status === 'complete');
    }
    
    // Sort
    const sortBy = sortOrder?.value || 'bib';
    const sortFunction = (a, b) => {
        switch (sortBy) {
            case 'bib':
                return parseInt(a.bib_number) - parseInt(b.bib_number);
            case 'sort_order':
                const result = (a.sort_order || 0) - (b.sort_order || 0);
                return result;
            case 'name':
                const nameA = `${a.display_first_name} ${a.display_last_name}`.toLowerCase();
                const nameB = `${b.display_first_name} ${b.display_last_name}`.toLowerCase();
                return nameA.localeCompare(nameB);
            case 'score_desc':
                return (b.total_score || 0) - (a.total_score || 0);
            case 'score_asc':
                return (a.total_score || 0) - (b.total_score || 0);
            case 'update_newest':
                // Sort by updated_at timestamp, newest first (null values last)
                const timeA = a.updated_at ? new Date(a.updated_at).getTime() : 0;
                const timeB = b.updated_at ? new Date(b.updated_at).getTime() : 0;
                return timeB - timeA;
            case 'update_oldest':
                // Sort by updated_at timestamp, oldest first (null values last)
                const timeA_old = a.updated_at ? new Date(a.updated_at).getTime() : Number.MAX_SAFE_INTEGER;
                const timeB_old = b.updated_at ? new Date(b.updated_at).getTime() : Number.MAX_SAFE_INTEGER;
                return timeA_old - timeB_old;
            default:
                return 0;
        }
    };
    
    if (scoredFilterValue === 'scored_bottom') {
        // Sort both groups independently
        unscored.sort(sortFunction);
        scored.sort(sortFunction);
        // Combine: unscored first, then scored
        filtered = [...unscored, ...scored];
    } else {
        filtered.sort(sortFunction);
    }

    if (showPerformingParticipants) {
        const performing = filtered.filter(isPerformingParticipant);
        const others = filtered.filter(p => !isPerformingParticipant(p));
        if (performing.length) {
            filtered = [...performing, ...others];
        }
    }
    
    // Debug: Log the sorted order
    if (sortBy === 'sort_order') {
        console.log('📋 Rendering order (sort_order):');
        filtered.slice(0, 10).forEach((p, idx) => {
            console.log(`  ${idx + 1}. BIB ${p.bib_number} - sort_order: ${p.sort_order}`);
        });
    }
    
    // Render cards
    if (filtered.length === 0) {
        grid.innerHTML = `
            <div class="col-12 text-center text-muted py-3">
                <i class="fas fa-filter fa-2x mb-2 opacity-25"></i>
                <p class="mb-0">No participants match the selected filter</p>
            </div>
        `;
        return;
    }
    
    // Use smart update with filtered data
    updateParticipantCardsWithData(filtered);
}

function getParticipantStatusBadgeColor(status) {
    const statusColors = {
        'approved': 'success',
        'pending': 'warning',
        'rejected': 'danger',
        'withdrawn': 'secondary',
        'performing': 'warning',
        'scoring': 'info',
        'complete': 'success'
    };
    return statusColors[status] || 'secondary';
}

function updateParticipantStatusBadgeElement(card, participantStatus) {
    const nameEl = card.querySelector('.participant-name');
    if (!nameEl) return;
    const existingBadge = nameEl.querySelector('.participantStatusBadge');
    if (participantStatus) {
        const badgeColor = getParticipantStatusBadgeColor(participantStatus);
        if (existingBadge) {
            existingBadge.className = `participantStatusBadge position-absolute right-0 top-0 w-100 badge bg-${badgeColor} bg-opacity-50 badge-sm ms-2`;
            existingBadge.textContent = participantStatus;
        } else {
            const badge = document.createElement('span');
            badge.className = `participantStatusBadge position-absolute right-0 top-0 w-100 badge bg-${badgeColor} bg-opacity-50 badge-sm ms-2`;
            badge.textContent = participantStatus;
            nameEl.appendChild(badge);
        }
    } else if (existingBadge) {
        existingBadge.remove();
    }
}

function buildParticipantControlsHTML(participant, positionIndex = null) {
    const showControls = ['full', 'compact', 'controls'].includes(participantCardLayout);
    if (!showControls) {
        return '';
    }

    const participantId = participant.event_participant_id || participant.id;
    const isPerforming = participant.status === 'current';
    const participantStatus = participant.participant_status || '';
    const isStopped = participantStatus === 'stop';
    const allowPlayButton = canShowPlayButtonForIndex(positionIndex ?? participant._renderIndex ?? null);

    const getHoldEventAttributes = (action) => `
        onmousedown="return startLongPressAction(${participant.bib_number}, this, event, '${action}')"
        onmouseup="return cancelLongPressAction(this)"
        onmouseleave="return cancelLongPressAction(this)"
        ontouchstart="return startLongPressAction(${participant.bib_number}, this, event, '${action}')"
        ontouchend="return cancelLongPressAction(this)"
        ontouchcancel="return cancelLongPressAction(this)"
        onkeydown="return handleLongPressKeyDown(event, ${participant.bib_number}, this, '${action}')"
        onkeyup="return handleLongPressKeyUp(event, this)"
        onblur="return cancelLongPressAction(this)"
    `;

    let actionButtonsHTML = '';
    if (isPerforming) {
        actionButtonsHTML = `
            <button class="btn btn-lg btn-danger performing-hold-btn px-3 px-md-5" title="Hold 2s to stop performing" ${getHoldEventAttributes('stop')}>
                <i class="fas fa-stop"></i>
            </button>
        `;
    } else if (allowPlayButton) {
        actionButtonsHTML = `
            <button class="btn btn-lg btn-outline-success performing-hold-btn px-3 px-md-5" title="Hold 2s to mark as performing" ${getHoldEventAttributes('perform')}>
                <i class="fas fa-play"></i>
            </button>
        `;

        if (isStopped) {
            actionButtonsHTML += `
                <button class="btn btn-lg btn-outline-secondary performing-hold-btn px-4" title="Hold 2s to reset status" ${getHoldEventAttributes('reset')}>
                    <i class="fas fa-undo"></i>
                </button>
            `;
        }
    } else if (isStopped) {
        actionButtonsHTML = `
            <button class="btn btn-lg btn-outline-secondary performing-hold-btn px-4" title="Hold 2s to reset status" ${getHoldEventAttributes('reset')}>
                <i class="fas fa-undo"></i>
            </button>
        `;
    }

    return `
        <div class="position-controls d-flex flex-row gap-1">
            <button class="btn btn-lg btn-outline-primary" onclick="moveParticipantUp(${participantId})" title="Move up in start order">
                <i class="fas fa-arrow-up"></i>
            </button>
            <button class="btn btn-lg btn-outline-primary" onclick="moveParticipantDown(${participantId})" title="Move down in start order">
                <i class="fas fa-arrow-down"></i>
            </button>
            ${actionButtonsHTML}
        </div>
    `;
}

function replaceCardControls(card, participant, positionIndex = null) {
    const controlsHTML = buildParticipantControlsHTML(participant, positionIndex);
    const flexRow = card.querySelector('.d-flex.align-items-center');
    if (!flexRow) {
        return;
    }
    const existingControls = flexRow.querySelector('.position-controls');
    if (!controlsHTML) {
        if (existingControls) {
            existingControls.remove();
        }
        return;
    }
    const tempDiv = document.createElement('div');
    tempDiv.innerHTML = controlsHTML.trim();
    const newControls = tempDiv.firstElementChild;
    if (!newControls) {
        if (existingControls) {
            existingControls.remove();
        }
        return;
    }
    if (existingControls) {
        existingControls.replaceWith(newControls);
    } else {
        flexRow.appendChild(newControls);
    }
}

function renderParticipantCard(participant, positionIndex = null) {
    const name = `${participant.display_first_name || ''} ${participant.display_last_name || ''}`.trim() || 'Unnamed';
    const category = participant.display_category || participant.event_category || participant.participant_category || 'N/A';
    const country = participant.country || participant.display_country || '';
    const club = participant.club || participant.display_club || '';
    
    const hasScore = participant.has_score && participant.total_score !== null;
    const score = hasScore ? parseFloat(participant.total_score).toFixed(2) : '';
    const judgeCount = participant.judge_count || 0;
    
    const statusClass = participant.status === 'current' ? 'performing' : (hasScore ? 'has-score' : '');
    
    // Determine what to show based on layout mode v2\admin\test_participant_layouts.html
    const showControls = ['full', 'compact', 'controls'].includes(participantCardLayout);
    const showDetails = ['full', 'info'].includes(participantCardLayout);
    const showFigures = ['full', 'info'].includes(participantCardLayout);

    const renderIndex = typeof positionIndex === 'number'
        ? positionIndex
        : (typeof participant._renderIndex === 'number' ? participant._renderIndex : null);
    
    // Create data attributes for state tracking
    const dataAttrs = {
        'data-bib': participant.bib_number,
        'data-score': score,
        'data-judge-count': judgeCount,
        'data-status': participant.status || 'none',
        'data-has-score': hasScore ? '1' : '0',
        'data-figures': participant.figures_json ? JSON.stringify(participant.figures_json) : '',
        'data-participant-status': participant.participant_status || '',
        'data-position-index': renderIndex !== null ? renderIndex : '',
        'data-completed-runs': JSON.stringify(participant.completed_runs || [])
    };
    
    const dataAttrString = Object.entries(dataAttrs)
        .map(([key, value]) => `${key}="${String(value).replace(/"/g, '&quot;')}"`)
        .join(' ');
    
    let statusBadge = '';
    if (participant.status === 'current') {
        statusBadge = '<span class="status-badge badge bg-warning text-dark">Performing</span>';
    } else if (hasScore) {
        //statusBadge = '<span class="status-badge badge bg-success">Scored (' + judgeCount + ' judges)</span>';
    }
    
    // Participant status badge (approved/pending/etc)
    let participantStatusBadge = '';
    if (participant.participant_status) {
        const bgColor = getParticipantStatusBadgeColor(participant.participant_status);
        participantStatusBadge = '<span class="participantStatusBadge position-absolute right-0 top-0 w-100  badge bg-' + bgColor + ' bg-opacity-50 badge-sm ms-2">' + participant.participant_status + '</span>';
    }
    
    let figuresHTML = '';
    if (showFigures && participant.figures_json) {
        try {
            const figures = typeof participant.figures_json === 'string' 
                ? JSON.parse(participant.figures_json) 
                : participant.figures_json;
            
            if (figures && typeof figures === 'object') {
                figuresHTML = '<div class="criteria-scores">';
                
                // Check if figures has a 'figures' property (nested structure)
                const figureData = figures.figures || figures;
                
                for (const [key, value] of Object.entries(figureData)) {
                    if (key !== 'validation' && value !== null && value !== '') {
                        const displayValue = Array.isArray(value) ? value.join(', ') : value;
                        figuresHTML += '<div class="criteria-score-item"><strong>' + key + ':</strong> ' + displayValue + '</div>';
                    }
                }
                figuresHTML += '</div>';
            }
        } catch (e) {
            console.error('Error parsing figures:', e);
        }
    }
    
    // Build participant details section
    let detailsHTML = '';
    if (showDetails) {
        detailsHTML = `
            <div class="participant-details">
                <div><i class="fas fa-tag me-1"></i>${category}</div>
                ${country ? '<div><i class="fas fa-flag me-1"></i>' + country + '</div>' : ''}
                ${club ? '<div><i class="fas fa-building me-1"></i>' + club + '</div>' : ''}
            </div>
        `;
    } else {
        // Compact mode: just category
        detailsHTML = '<div class="participant-details"><small class="text-muted">' + category + '</small></div>';
    }
    
    // Build run indicators (O/X pattern)
    const runIndicatorsHTML = buildRunIndicatorsHTML(participant);
    
    const controlsHTML = showControls ? buildParticipantControlsHTML(participant, renderIndex) : '';
    
    return `
        <div class="col-md-12" ${dataAttrString}>
            <div class="participant-score-card p-2 participant-score-card rounded-3 shadow-lg ${statusClass}">
                ${statusBadge}
                
                <div class="d-flex align-items-center gap-3">
                    <div class="col-1 css-bib"> <div class="badge participant-bib-badge ss-f-number">
                       ${participant.bib_number}</div>
                    </div>
                    
                    <div class="col-2 col-sm-4 flex-grow-1 participant-info">
                        <div class="participant-name">${name}${participantStatusBadge}</div>
                        ${detailsHTML}
                       
                    </div>
                     ${runIndicatorsHTML}
                    <div class="score-display">
                        <div class="score-value">${score}</div>
                        <div class="score-label">${hasScore && judgeCount > 1 ? 'Avg (' + judgeCount + ' judges)' : ''}</div>
                    </div>
                    
                    ${controlsHTML}
                </div>
                
                ${figuresHTML}
            </div>
        </div>
    `;
}

function flashParticipantCard(bibNumber, variant = 'primary') {
    if (!bibNumber) {
        return;
    }
    const className = variant === 'info' ? 'flash-move-down' : 'flash-move-up';
    const card = document.querySelector(`[data-bib="${bibNumber}"] .participant-score-card`) 
        || document.querySelector(`.participant-score-card[data-bib="${bibNumber}"]`);
    if (!card) {
        return;
    }
    card.classList.remove('flash-move-up', 'flash-move-down');
    // Force reflow so animation can restart
    void card.offsetWidth;
    card.classList.add(className);
    setTimeout(() => {
        card.classList.remove(className);
    }, 900);
}

// ============================================================================
// PARTICIPANT POSITION MANAGEMENT
// ============================================================================

async function setParticipantPerforming(bibNumber) {
    try {
        console.log('▶️ Setting participant BIB', bibNumber, 'as performing');
        
        const formData = new FormData();
        formData.append('action', 'update_participant_status');
        formData.append('event_id', currentEventId);
        formData.append('bib_number', bibNumber);
        formData.append('status', 'started');
        
        const response = await fetch('heat_management_api.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            showSuccessToast('▶️ Performing', `BIB ${bibNumber} is now performing`);
            
            // Reload participant list and heat card to reflect status change
            await loadParticipantList(currentHeatNumber);
            await loadActiveHeatCard();
        } else {
            showErrorToast('Error', result.message || 'Failed to update status');
        }
    } catch (error) {
        console.error('❌ Error setting participant as performing:', error);
        showErrorToast('Error', 'Failed to update participant status');
    }
}

async function setParticipantStopped(bibNumber) {
    try {
        console.log('⏹️ Stopping participant BIB', bibNumber);
    const formData = new FormData();
    formData.append('action', 'update_participant_status');
    formData.append('event_id', currentEventId);
    formData.append('bib_number', bibNumber);
    formData.append('status', 'initial');

        const response = await fetch('heat_management_api.php', {
            method: 'POST',
            body: formData
        });

        const result = await response.json();

        if (result.success) {
            showWarningToast('⏹️ Stopped', `BIB ${bibNumber} has been stopped`);
            await loadParticipantList(currentHeatNumber);
            await loadActiveHeatCard();
        } else {
            showErrorToast('Error', result.message || 'Failed to stop participant');
        }
    } catch (error) {
        console.error('❌ Error stopping participant:', error);
        showErrorToast('Error', 'Failed to stop participant');
    }
}

async function resetParticipantStatus(bibNumber) {
    try {
        console.log('🔁 Resetting participant BIB', bibNumber, 'to initial');
        const formData = new FormData();
        formData.append('action', 'update_participant_status');
        formData.append('event_id', currentEventId);
        formData.append('bib_number', bibNumber);
        formData.append('status', 'reset');

        const response = await fetch('heat_management_api.php', {
            method: 'POST',
            body: formData
        });

        const result = await response.json();

        if (result.success) {
            showInfoToast('🔁 Reset', `BIB ${bibNumber} reset to initial state`);
            await loadParticipantList(currentHeatNumber);
            await loadActiveHeatCard();
        } else {
            showErrorToast('Error', result.message || 'Failed to reset participant');
        }
    } catch (error) {
        console.error('❌ Error resetting participant:', error);
        showErrorToast('Error', 'Failed to reset participant');
    }
}

async function moveParticipantUp(participantId) {
    // participantId could be event_participant_id or id - find by either
    const participant = participantListData.find(p => 
        p.event_participant_id === participantId || p.id === participantId
    );
    if (!participant) {
        showErrorToast('Error', 'Participant not found');
        return;
    }
    
    // Ensure we have the correct ID field
    const epId = participant.event_participant_id || participant.id;
    console.log('⬆️ Moving up - Participant ID:', epId, 'BIB:', participant.bib_number);
    
    // Find the participant above (with next lower sort_order)
    const sortedList = [...participantListData].sort((a, b) => (a.sort_order || 0) - (b.sort_order || 0));
    const currentIndex = sortedList.findIndex(p => 
        (p.event_participant_id || p.id) === epId
    );
    
    console.log('  Current index in sorted list:', currentIndex);
    console.log('  Total participants:', sortedList.length);
    
    if (currentIndex <= 0) {
        showInfoToast('Info', 'Participant is already at the top');
        return;
    }
    
    const prevParticipant = sortedList[currentIndex - 1];
    const prevEpId = prevParticipant.event_participant_id || prevParticipant.id;
    console.log('  Previous participant ID:', prevEpId, 'BIB:', prevParticipant.bib_number);
    
    if (epId === prevEpId) {
        console.error('❌ ERROR: Trying to swap participant with itself!');
        showErrorToast('Error', 'Cannot swap participant with itself');
        return;
    }
    
    flashParticipantCard(participant.bib_number, 'primary');
    await swapParticipantPositions(participant, prevParticipant);
}

async function moveParticipantDown(participantId) {
    // participantId could be event_participant_id or id - find by either
    const participant = participantListData.find(p => 
        p.event_participant_id === participantId || p.id === participantId
    );
    if (!participant) {
        showErrorToast('Error', 'Participant not found');
        return;
    }
    
    // Ensure we have the correct ID field
    const epId = participant.event_participant_id || participant.id;
    console.log('⬇️ Moving down - Participant ID:', epId, 'BIB:', participant.bib_number);
    
    // Find the participant below (with next higher sort_order)
    const sortedList = [...participantListData].sort((a, b) => (a.sort_order || 0) - (b.sort_order || 0));
    const currentIndex = sortedList.findIndex(p => 
        (p.event_participant_id || p.id) === epId
    );
    
    console.log('  Current index in sorted list:', currentIndex);
    console.log('  Total participants:', sortedList.length);
    
    if (currentIndex >= sortedList.length - 1) {
        showInfoToast('Info', 'Participant is already at the bottom');
        return;
    }
    
    const nextParticipant = sortedList[currentIndex + 1];
    const nextEpId = nextParticipant.event_participant_id || nextParticipant.id;
    console.log('  Next participant ID:', nextEpId, 'BIB:', nextParticipant.bib_number);
    
    if (epId === nextEpId) {
        console.error('❌ ERROR: Trying to swap participant with itself!');
        showErrorToast('Error', 'Cannot swap participant with itself');
        return;
    }
    
    flashParticipantCard(participant.bib_number, 'info');
    await swapParticipantPositions(participant, nextParticipant);
}

async function swapParticipantPositions(participant1, participant2) {
    try {
        const ep1Id = participant1.event_participant_id || participant1.id;
        const ep2Id = participant2.event_participant_id || participant2.id;
        
        console.log('🔄 Swapping positions:');
        console.log('  Participant 1 ID:', ep1Id, 'BIB:', participant1.bib_number, 'current sort_order:', participant1.sort_order);
        console.log('  Participant 2 ID:', ep2Id, 'BIB:', participant2.bib_number, 'current sort_order:', participant2.sort_order);
        
        // Store original values
        const originalOrder1 = participant1.sort_order;
        const originalOrder2 = participant2.sort_order;
        
        // Update in database - send the NEW order for each participant (swapped)
        const formData = new FormData();
        formData.append('action', 'swap_positions');
        formData.append('event_id', currentEventId);
        formData.append('participant1_id', ep1Id);
        formData.append('participant2_id', ep2Id);
        formData.append('participant1_order', originalOrder2); // participant1 gets participant2's order
        formData.append('participant2_order', originalOrder1); // participant2 gets participant1's order
        
        const response = await fetch('heat_management_api.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        console.log('📥 Swap result:', result);
        
        if (result.success) {
            // Update local data with new values
            participant1.sort_order = originalOrder2;
            participant2.sort_order = originalOrder1;
            
            console.log('✅ Updated local data:');
            console.log('  Participant 1:', participant1.bib_number, 'new sort_order:', participant1.sort_order);
            console.log('  Participant 2:', participant2.bib_number, 'new sort_order:', participant2.sort_order);
            
            // Reload participant list to get fresh data from database
            console.log('🔄 Reloading participant list from database...');
            await loadParticipantList(currentHeatNumber);
            
            showSuccessToast('Success', 'Position updated');
        } else {
            showErrorToast('Error', result.message || 'Failed to update position');
        }
    } catch (error) {
        console.error('❌ Error swapping positions:', error);
        showErrorToast('Error', 'Failed to update position');
    }
}

// ============================================================================
// RESULTS MODAL FUNCTIONS
// ============================================================================

function showHeatResults(heatNumber) {
    if (!currentEventId) {
        showErrorToast('❌ Error', 'No event ID available');
        return;
    }
    
    // Update modal title with heat number
    document.getElementById('resultsHeatName').textContent = `Heat ${heatNumber} Results`;
    
    // Show the modal
    const modal = new bootstrap.Modal(document.getElementById('heatResultsModal'));
    modal.show();
    
    // Load configurations for this heat
    loadHeatConfigurations(currentEventId, heatNumber);
}

function loadHeatConfigurations(eventId, heatNumber) {
    const configsList = document.getElementById('savedConfigsList');
    
    // Show loading state
    configsList.innerHTML = `
        <div class="text-center text-muted py-3">
            <i class="fas fa-spinner fa-spin"></i>
            <p class="small mb-0 mt-2">Loading reports...</p>
        </div>
    `;
    
    // Fetch all configurations for the event
    fetch(`../api/public_dashboard_api.php?action=get_configurations&event_id=${eventId}`)
        .then(response => response.json())
        .then(data => {
            if (!data.success || !data.configurations || data.configurations.length === 0) {
                configsList.innerHTML = `
                    <div class="text-center text-muted py-3">
                        <i class="fas fa-info-circle fa-2x mb-2"></i>
                        <p class="small mb-0">No reports configured</p>
                    </div>
                `;
                return;
            }
            
            // Filter configurations for this heat or general (null heat_number)
            const relevantConfigs = data.configurations.filter(config => 
                config.heat_number === null || config.heat_number == heatNumber
            );
            
            if (relevantConfigs.length === 0) {
                configsList.innerHTML = `
                    <div class="text-center text-muted py-3">
                        <i class="fas fa-info-circle fa-2x mb-2"></i>
                        <p class="small mb-0">No reports for this heat</p>
                    </div>
                `;
                return;
            }
            
            // Render configuration buttons
            configsList.innerHTML = relevantConfigs.map(config => {
                const icon = config.view_type === 'start_list' 
                    ? 'fa-list-ol' 
                    : 'fa-trophy';
                
                const label = config.view_type === 'start_list' 
                    ? 'Start List' 
                    : 'Results';
                
                const heatLabel = config.heat_number 
                    ? `Heat ${config.heat_number}` 
                    : 'All Heats';
                
                return `
                    <button class="config-button btn btn-outline-primary text-start" 
                            data-config-id="${config.config_id}"
                            data-view-type="${config.view_type}"
                            onclick="loadHeatConfiguration(${config.config_id}, '${config.view_type}')">
                        <i class="fas ${icon} me-2"></i>
                        <div>
                            <div class="fw-bold">${config.config_name || label}</div>
                            <small class="text-muted">${heatLabel}</small>
                        </div>
                    </button>
                `;
            }).join('');
            
            // Auto-load first configuration
            if (relevantConfigs.length > 0) {
                loadHeatConfiguration(relevantConfigs[0].config_id, relevantConfigs[0].view_type);
            }
        })
        .catch(error => {
            console.error('Error loading configurations:', error);
            configsList.innerHTML = `
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    Error loading reports
                </div>
            `;
        });
}

function loadHeatConfiguration(configId, viewType) {
    const contentDisplay = document.getElementById('resultsContentDisplay');
    
    // Update active state
    document.querySelectorAll('#savedConfigsList .config-button').forEach(btn => {
        btn.classList.remove('active');
    });
    const activeBtn = document.querySelector(`[data-config-id="${configId}"]`);
    if (activeBtn) activeBtn.classList.add('active');
    
    // Show loading
    contentDisplay.innerHTML = `
        <div class="text-center text-muted py-5">
            <i class="fas fa-spinner fa-spin fa-3x mb-3"></i>
            <p class="mb-0">Loading report...</p>
        </div>
    `;
    
    // Determine API endpoint based on view type
    const apiEndpoint = viewType === 'start_list' 
        ? '../api/public_dashboard_api.php?action=get_start_list'
        : '../api/summary_table_api.php';
    
    // Load configuration
    fetch(`${apiEndpoint}&config_id=${configId}&format=html`)
        .then(response => response.text())
        .then(html => {
            contentDisplay.innerHTML = html;
        })
        .catch(error => {
            console.error('Error loading configuration:', error);
            contentDisplay.innerHTML = `
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    <strong>Error loading report:</strong> ${error.message}
                </div>
            `;
        });
}

// ============================================================================
// HEAT CARD ACTION FUNCTIONS (called from heat card buttons)
// ============================================================================

async function changeActiveRun(heatNumber, direction) {
    const directionLabel = direction > 0 ? 'next' : 'previous';
    const confirmed = await showActionConfirm({
        title: 'Change Active Run',
        message: `Move Heat ${heatNumber} to the ${directionLabel} run?`,
        confirmLabel: 'Change Run',
        confirmVariant: 'primary',
        iconClass: 'fas fa-shuffle text-primary'
    });
    if (!confirmed) return;
    showInfoToast('🔄 Changing Run', `Switching run for Heat ${heatNumber}...`);
    // This would need integration with heat management API
    console.log('Change run:', heatNumber, direction);
}

async function activateHeat(heatNumber) {
    showInfoToast('🔥 Activating Heat', `Setting Heat ${heatNumber} as active...`);
    currentHeatNumber = heatNumber;
    loadActiveHeatCard(heatNumber);
}

async function closeHeat(heatNumber) {
    const confirmed = await showActionConfirm({
        title: `Finish Heat ${heatNumber}`,
        message: 'Mark this heat as finished for all panels?',
        confirmLabel: 'Yes, Finish Heat',
        confirmVariant: 'success',
        iconClass: 'fas fa-flag-checkered text-success'
    });
    if (!confirmed) return;
    showInfoToast('✅ Closing Heat', `Marking Heat ${heatNumber} as finished...`);
}

async function setPendingHeat(heatNumber) {
    const confirmed = await showActionConfirm({
        title: `Set Heat ${heatNumber} Pending`,
        message: 'Move this heat back to the pending state?',
        confirmLabel: 'Yes, Set Pending',
        confirmVariant: 'secondary',
        iconClass: 'fas fa-pause-circle text-secondary'
    });
    if (!confirmed) return;
    showInfoToast('⏳ Setting Pending', `Marking Heat ${heatNumber} as pending...`);
}

async function cancelHeat(heatNumber) {
    const confirmed = await showActionConfirm({
        title: `Cancel Heat ${heatNumber}`,
        message: 'Cancel this heat? This action notifies panels and dashboards.',
        confirmLabel: 'Yes, Cancel Heat',
        confirmVariant: 'danger',
        iconClass: 'fas fa-ban text-danger'
    });
    if (!confirmed) return;
    showWarningToast('❌ Cancelling Heat', `Heat ${heatNumber} cancelled`);
}

async function rescheduleHeat(heatNumber) {
    const confirmed = await showActionConfirm({
        title: `Reschedule Heat ${heatNumber}`,
        message: 'Mark this heat for rescheduling?',
        confirmLabel: 'Yes, Reschedule',
        confirmVariant: 'warning',
        iconClass: 'fas fa-calendar-alt text-warning'
    });
    if (!confirmed) return;
    showInfoToast('📅 Rescheduling', `Heat ${heatNumber} marked for rescheduling`);
}

async function resetParticipantStatus(heatNumber) {
    const confirmed = await showActionConfirm({
        title: `Reset Participant Status - Heat ${heatNumber}`,
        message: 'Reset all participants in this heat to "initial" status? This will clear their current status.',
        confirmLabel: 'Yes, Reset Status',
        confirmVariant: 'warning',
        iconClass: 'fas fa-redo-alt text-warning'
    });
    if (!confirmed) return;
    
    try {
        const formData = new FormData();
        formData.append('event_id', currentEventId);
        formData.append('heat_number', heatNumber);
        
        const response = await fetch('reset_participant_status_api.php', {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (result.success) {
            showSuccessToast('✅ Status Reset', result.message);
            // Refresh heat cards to show updated status
            if (typeof loadHeatCards === 'function') {
                loadHeatCards();
            }
        } else {
            showErrorToast('❌ Reset Failed', result.message || 'Failed to reset participant status');
        }
    } catch (error) {
        console.error('Error resetting participant status:', error);
        showErrorToast('❌ Error', 'An error occurred while resetting participant status');
    }
}

function openHeatModal(heatNumber) {
    window.location.href = `heats_configure.php?event_id=${currentEventId}&heat=${heatNumber}`;
}

// ============================================================================
// CONFIRMATION / UTILITY FUNCTIONS
// ============================================================================

let actionConfirmModalInstance = null;
let actionConfirmResolveFn = null;

function getActionConfirmModal() {
    const modalEl = document.getElementById('actionConfirmModal');
    if (!modalEl) return null;
    if (!actionConfirmModalInstance) {
        actionConfirmModalInstance = new bootstrap.Modal(modalEl);
        modalEl.addEventListener('hidden.bs.modal', () => {
            if (actionConfirmResolveFn) {
                actionConfirmResolveFn(false);
                actionConfirmResolveFn = null;
            }
        });
    }
    return actionConfirmModalInstance;
}

function showActionConfirm(options = {}) {
    return new Promise((resolve) => {
        const modalEl = document.getElementById('actionConfirmModal');
        const modalInstance = getActionConfirmModal();
        if (!modalEl || !modalInstance) {
            resolve(window.confirm(options.message || 'Are you sure?'));
            return;
        }
        const {
            title = 'Confirm Action',
            message = 'Are you sure you want to continue?',
            confirmLabel = 'Confirm',
            confirmVariant = 'primary',
            iconClass = 'fas fa-question-circle text-primary'
        } = options;
        modalEl.querySelector('.action-confirm-title').textContent = title;
        modalEl.querySelector('.action-confirm-message').textContent = message;
        const iconEl = modalEl.querySelector('.action-confirm-icon');
        if (iconEl) {
            iconEl.className = `action-confirm-icon ${iconClass}`;
        }
        const confirmBtn = modalEl.querySelector('.action-confirm-approve');
        const cancelBtn = modalEl.querySelector('.action-confirm-cancel');
        if (confirmBtn) {
            confirmBtn.textContent = confirmLabel;
            confirmBtn.className = `btn action-confirm-approve btn-${confirmVariant}`;
            confirmBtn.onclick = () => {
                resolve(true);
                actionConfirmResolveFn = null;
                modalInstance.hide();
            };
        }
        if (cancelBtn) {
            cancelBtn.onclick = () => {
                resolve(false);
                actionConfirmResolveFn = null;
                modalInstance.hide();
            };
        }
        actionConfirmResolveFn = resolve;
        modalInstance.show();
    });
}

// ============================================================================
// TOAST HELPERS
// ============================================================================

function showSuccessToast(title, message = '') {
    createToast('success', title, message);
}

function showErrorToast(title, message = '') {
    createToast('danger', title, message);
}

function showInfoToast(title, message = '') {
    createToast('info', title, message);
}

function showWarningToast(title, message = '') {
    createToast('warning', title, message);
}

function createToast(type, title, message) {
    // Create toast container if it doesn't exist
    let container = document.getElementById('toastContainer');
    if (!container) {
        container = document.createElement('div');
        container.id = 'toastContainer';
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        document.body.appendChild(container);
    }
    
    // Create toast element
    const toastId = 'toast_' + Date.now();
    const toast = document.createElement('div');
    toast.id = toastId;
    toast.className = `toast notification-toast align-items-center text-white bg-${type} border-0`;
    toast.setAttribute('role', 'alert');
    
    const iconMap = {
        success: 'fa-check-circle',
        danger: 'fa-exclamation-circle',
        warning: 'fa-exclamation-triangle',
        info: 'fa-info-circle'
    };
    
    toast.innerHTML = `
        <div class="d-flex">
            <div class="toast-body">
                <i class="fas ${iconMap[type]} me-2"></i>
                <strong>${title}</strong>
                ${message ? `<div class="mt-1 small">${message}</div>` : ''}
            </div>
            <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
        </div>
    `;
    
    container.appendChild(toast);
    
    // Auto-remove after 5 seconds
    setTimeout(() => {
        toast.remove();
    }, 5000);
    
    // Animate in
    toast.style.opacity = '0';
    toast.style.transform = 'translateX(100%)';
    toast.style.transition = 'all 0.3s ease';
    
    setTimeout(() => {
        toast.style.opacity = '1';
        toast.style.transform = 'translateX(0)';
    }, 10);
}

// ============================================================================
// QUEUE INSIGHTS + ALERT STREAM HELPERS
// ============================================================================

function updateHeatQueuePanel(queue = [], heatSettings = {}) {
    const subtitleEl = document.getElementById('queueSummarySubtitle');
    if (!subtitleEl) {
        return;
    }

    const totalEl = document.getElementById('queueTotalCount');
    const performingEl = document.getElementById('queuePerformingCount');
    const waitingEl = document.getElementById('queueWaitingCount');
    const completedEl = document.getElementById('queueCompletedCount');
    const runLabelEl = document.getElementById('queueRunLabel');
    const updatedAtEl = document.getElementById('queueUpdatedAt');

    const safeQueue = Array.isArray(queue) ? queue.slice() : [];
    const resolvedHeatSettings = Object.keys(heatSettings || {}).length ? heatSettings : (window.currentHeatSettings || {});

    const activeRun = Number(resolvedHeatSettings?.active_run) || Number(safeQueue?.[0]?.current_run) || 1;
    const totalRuns = Number(resolvedHeatSettings?.runs_count) || Number(safeQueue?.[0]?.total_runs) || activeRun;
    const heatNumber = resolvedHeatSettings?.heat_number || currentHeatNumber || '--';
    const heatStatus = (resolvedHeatSettings?.status || 'pending').replace(/_/g, ' ');

    subtitleEl.textContent = safeQueue.length
        ? `Heat ${heatNumber} · ${heatStatus.toUpperCase()}`
        : 'Waiting for participants…';

    if (totalEl) {
        totalEl.textContent = safeQueue.length ? safeQueue.length : '--';
    }

    const performingParticipants = safeQueue.filter(isParticipantPerforming);
    const completedParticipants = safeQueue.filter(isParticipantCompleted);
    const nextParticipants = getNextParticipants(safeQueue);
    const waitingParticipants = safeQueue.filter(p => !performingParticipants.includes(p) && !completedParticipants.includes(p));

    if (performingEl) performingEl.textContent = performingParticipants.length;
    if (waitingEl) waitingEl.textContent = waitingParticipants.length;
    if (completedEl) completedEl.textContent = completedParticipants.length;
    if (runLabelEl) runLabelEl.textContent = `${activeRun}/${Math.max(activeRun, totalRuns)}`;
    if (updatedAtEl) updatedAtEl.textContent = safeQueue.length ? new Date().toLocaleTimeString() : '--';

    renderQueueSlots(performingParticipants, nextParticipants);
    renderQueueChips(waitingParticipants);
}

function isParticipantPerforming(participant) {
    const queueStatus = (participant?.status || '').toLowerCase();
    const participantStatus = (participant?.participant_status || '').toLowerCase();
    return queueStatus === 'current' || participantStatus === 'performing';
}

function isParticipantCompleted(participant) {
    const queueStatus = (participant?.status || '').toLowerCase();
    const participantStatus = (participant?.participant_status || '').toLowerCase();
    return queueStatus === 'completed' || queueStatus === 'done' || participantStatus === 'complete' || participant.queue_position === -1;
}

function getNextParticipants(queue = []) {
    return queue
        .filter(participant => {
            const queueStatus = (participant?.status || '').toLowerCase();
            return queueStatus === 'next' || participant.queue_position === 1;
        })
        .slice(0, 2);
}

function renderQueueSlots(performing = [], next = []) {
    const container = document.getElementById('queuePerformingSlots');
    if (!container) return;

    const performingHtml = buildSlotContent(performing, 'No athlete on course');
    const nextHtml = buildSlotContent(next, 'Queue forming soon');

    container.innerHTML = `
        <div class="performing-slot current">
            <span class="slot-label">Now Performing</span>
            <div class="slot-content">${performingHtml}</div>
        </div>
        <div class="performing-slot next">
            <span class="slot-label">Up Next</span>
            <div class="slot-content">${nextHtml}</div>
        </div>
    `;
}

function buildSlotContent(list = [], emptyText = '') {
    if (!Array.isArray(list) || list.length === 0) {
        return `<span class="text-muted small">${escapeHtml(emptyText)}</span>`;
    }
    return list.slice(0, 2).map(participant => {
        const name = formatQueueParticipantName(participant);
        const bib = participant?.bib_number ? `#${escapeHtml(participant.bib_number)}` : '';
        return `
            <span class="slot-chip">
                <span class="slot-bib">${bib}</span>
                <span>${name}</span>
            </span>
        `;
    }).join('');
}

function renderQueueChips(participants = []) {
    const container = document.getElementById('queueChipsList');
    if (!container) return;

    if (!participants.length) {
        container.innerHTML = '<div class="text-muted small">All caught up – no one waiting.</div>';
        return;
    }

    container.innerHTML = participants.slice(0, 8).map((participant, index) => {
        const queueStatus = (participant?.status || 'waiting').toLowerCase();
        const statusClass = `queue-chip-${queueStatus}`;
        const name = formatQueueParticipantName(participant);
        const category = escapeHtml(participant?.display_category || participant?.event_category || '');
        const bib = participant?.bib_number ? `BIB ${escapeHtml(participant.bib_number)}` : '';
        return `
            <div class="queue-chip ${statusClass}" data-bib="${escapeHtml(participant?.bib_number || '')}">
                <div class="queue-chip-rank">${index + 1}</div>
                <div>
                    <div class="queue-chip-name">${name}</div>
                    <div class="queue-chip-meta">
                        ${bib ? `<span class="badge bg-light text-dark me-1">${bib}</span>` : ''}
                        ${category}
                    </div>
                </div>
            </div>
        `;
    }).join('');
}

function formatQueueParticipantName(participant) {
    const first = participant?.display_first_name || participant?.first_name || '';
    const last = participant?.display_last_name || participant?.last_name || '';
    const combined = `${first} ${last}`.trim();
    if (combined) return escapeHtml(combined);
    if (participant?.name) return escapeHtml(participant.name);
    if (participant?.bib_number) return `BIB ${escapeHtml(participant.bib_number)}`;
    return 'Unknown Athlete';
}

function escapeHtml(value) {
    if (value === null || value === undefined) {
        return '';
    }
    return String(value)
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#39;');
}

function initializeHeatNotifications() {
    if (heatNotificationsInitialized) {
        return;
    }
    const alertsList = document.getElementById('alertsList');
    if (!alertsList) {
        return;
    }
    if (!currentEventId) {
        const statusLabel = document.getElementById('alertsStatusLabel');
        if (statusLabel) {
            statusLabel.textContent = 'Select an event to monitor alerts';
            statusLabel.classList.add('text-warning');
        }
        return;
    }

    heatNotificationsInitialized = true;
    loadHeatNotifications(true);

    if (heatNotificationsTimer) {
        clearInterval(heatNotificationsTimer);
    }

    heatNotificationsTimer = setInterval(() => {
        if (document.hidden) {
            return;
        }
        loadHeatNotifications();
    }, HEAT_NOTIFICATION_POLL_INTERVAL);
}

async function loadHeatNotifications(forceFullReload = false) {
    const alertsList = document.getElementById('alertsList');
    if (!alertsList || !currentEventId) {
        return;
    }
    if (notificationsLoading && !forceFullReload) {
        return;
    }

    const statusLabel = document.getElementById('alertsStatusLabel');
    if (statusLabel) {
        statusLabel.textContent = 'Updating…';
        statusLabel.classList.remove('text-danger');
    }

    notificationsLoading = true;

    try {
        const params = new URLSearchParams({
            action: 'get_notifications',
            event_id: currentEventId,
            category: 'heat',
            limit: '8'
        });
        if (forceFullReload || !latestNotificationId) {
            params.append('since', '0');
        } else {
            params.append('since', String(latestNotificationId));
        }

        const response = await fetch(`../api/notification_api.php?${params.toString()}`, {
            credentials: 'same-origin'
        });
        const data = await response.json();

        if (!data.success) {
            throw new Error(data.error || 'Failed to load alerts');
        }

        latestNotificationId = Math.max(latestNotificationId, data.latest_id || 0);
        mergeHeatNotifications(data.notifications || [], forceFullReload);
        renderHeatNotificationsList();

        if (statusLabel) {
            statusLabel.textContent = `Updated ${new Date().toLocaleTimeString()}`;
        }
    } catch (error) {
        console.error('Heat alerts error:', error);
        if (statusLabel) {
            statusLabel.textContent = 'Alerts unavailable';
            statusLabel.classList.add('text-danger');
        }
        if (!heatNotifications.length) {
            alertsList.innerHTML = `
                <div class="alert-placeholder text-danger small">
                    <i class="fas fa-exclamation-triangle me-1"></i>${escapeHtml(error.message)}
                </div>
            `;
        }
    } finally {
        notificationsLoading = false;
    }
}

function mergeHeatNotifications(newNotifications = [], reset = false) {
    if (reset) {
        heatNotifications = (newNotifications || []).slice().sort((a, b) => (Number(b?.id) || 0) - (Number(a?.id) || 0));
        return;
    }

    const existingIds = new Set(heatNotifications.map(notification => Number(notification?.id)));
    (newNotifications || []).forEach(notification => {
        const id = Number(notification?.id);
        if (!existingIds.has(id)) {
            heatNotifications.unshift(notification);
            existingIds.add(id);
        }
    });

    heatNotifications.sort((a, b) => (Number(b?.id) || 0) - (Number(a?.id) || 0));
    heatNotifications = heatNotifications.slice(0, 8);
}

function renderHeatNotificationsList() {
    const alertsList = document.getElementById('alertsList');
    const countBadge = document.getElementById('alertsCountBadge');
    if (!alertsList) return;

    if (!heatNotifications.length) {
        alertsList.innerHTML = '<div class="text-muted small text-center py-3">No alerts yet for this heat.</div>';
        if (countBadge) {
            countBadge.textContent = '0';
        }
        return;
    }

    alertsList.innerHTML = heatNotifications.map(buildNotificationHtml).join('');
    if (countBadge) {
        countBadge.textContent = heatNotifications.length;
    }
}

function buildNotificationHtml(notification) {
    const type = (notification?.type || 'info').toLowerCase();
    const iconClass = notificationTypeToClass(type);
    const icon = notificationTypeToIcon(type);
    const title = escapeHtml(notification?.title || 'System Alert');
    const message = escapeHtml(notification?.message || '');
    const category = escapeHtml(notification?.category || 'general');
    const timeAgo = formatRelativeTime(notification?.created_at || notification?.timestamp);
    const notificationId = escapeHtml(notification?.id || '');

    const badgeVariant = type === 'danger' ? 'danger' : type === 'warning' ? 'warning' : type === 'success' ? 'success' : 'info';

    return `
        <div class="alert-item" data-notification-id="${notificationId}">
            <div class="alert-icon ${iconClass}">
                <i class="${icon}"></i>
            </div>
            <div class="alert-body">
                <div class="d-flex justify-content-between gap-3 align-items-start">
                    <div>
                        <div class="alert-title">${title}</div>
                        ${message ? `<p class="alert-message mb-1">${message}</p>` : ''}
                    </div>
                    <div class="text-end d-flex flex-column align-items-end gap-1">
                        <span class="alert-time">${timeAgo}</span>
                        <button type="button" class="btn btn-link btn-sm p-0 text-decoration-none text-muted" title="Dismiss" onclick="dismissHeatNotification('${notificationId}', this)">
                            <i class="fas fa-times"></i>
                        </button>
                    </div>
                </div>
                <div class="alert-meta d-flex gap-2 align-items-center flex-wrap mt-2">
                    <span class="badge badge-soft-${badgeVariant}">${category}</span>
                    ${notification?.user_id ? `<span class="text-muted small">${escapeHtml(notification.user_id)}</span>` : ''}
                </div>
            </div>
        </div>
    `;
}

function notificationTypeToClass(type) {
    switch (type) {
        case 'danger':
        case 'error':
            return 'alert-type-danger';
        case 'warning':
            return 'alert-type-warning';
        case 'success':
            return 'alert-type-success';
        default:
            return 'alert-type-info';
    }
}

function notificationTypeToIcon(type) {
    switch (type) {
        case 'danger':
        case 'error':
            return 'fas fa-exclamation-circle';
        case 'warning':
            return 'fas fa-exclamation-triangle';
        case 'success':
            return 'fas fa-check-circle';
        default:
            return 'fas fa-info-circle';
    }
}

function formatRelativeTime(timestamp) {
    if (!timestamp) return '--';
    const parsed = new Date(timestamp).getTime();
    if (Number.isNaN(parsed)) {
        return '--';
    }
    const diffMs = Date.now() - parsed;
    const diffSeconds = Math.floor(diffMs / 1000);
    if (diffSeconds < 60) {
        return `${diffSeconds}s ago`;
    }
    const diffMinutes = Math.floor(diffSeconds / 60);
    if (diffMinutes < 60) {
        return `${diffMinutes}m ago`;
    }
    const diffHours = Math.floor(diffMinutes / 60);
    if (diffHours < 24) {
        return `${diffHours}h ago`;
    }
    const diffDays = Math.floor(diffHours / 24);
    return `${diffDays}d ago`;
}

async function dismissHeatNotification(notificationId, triggerButton) {
    if (!notificationId) {
        return;
    }
    try {
        if (triggerButton) {
            triggerButton.disabled = true;
        }
        const response = await fetch('../api/notification_api.php?action=dismiss_notification', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded'
            },
            body: new URLSearchParams({ notification_id: notificationId }),
            credentials: 'same-origin'
        });
        const data = await response.json();
        if (!data.success) {
            throw new Error(data.error || 'Failed to dismiss notification');
        }
        heatNotifications = heatNotifications.filter(notification => String(notification?.id) !== String(notificationId));
        renderHeatNotificationsList();
    } catch (error) {
        console.error('Dismiss notification failed:', error);
    } finally {
        if (triggerButton) {
            triggerButton.disabled = false;
        }
    }
}

document.addEventListener('visibilitychange', () => {
    if (!document.hidden && heatNotificationsInitialized) {
        loadHeatNotifications();
    }
});
</script>

<!-- Bootstrap JS Bundle (required for modals) -->
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<?php if ($standalone): ?>
    </div>
    <?php include '../admin/footer.php'; ?>
</body>
</html>
<?php endif; ?>
