<?php
/**
 * Migration: Create user_assigned_events join table
 * 
 * This creates a normalized many-to-many relationship between users and events.
 * It also migrates data from the users.assigned_events column if it exists.
 *
 * Usage: php add_user_assigned_events_table_migration.php
 */

require_once 'includes/db.php';

try {
    echo "Starting migration: Create user_assigned_events table...\n";
    
    // Check if table already exists
    $check_stmt = $pdo->query("SHOW TABLES LIKE 'user_assigned_events'");
    if ($check_stmt->rowCount() > 0) {
        echo "✓ Table 'user_assigned_events' already exists.\n";
        exit(0);
    }
    
    // Create the join table
    echo "Creating 'user_assigned_events' table...\n";
    $pdo->exec("
        CREATE TABLE `user_assigned_events` (
          `id` int(11) NOT NULL AUTO_INCREMENT,
          `user_id` int(11) NOT NULL,
          `event_id` int(11) NOT NULL,
          `assigned_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
          `assigned_by` int(11) DEFAULT NULL,
          PRIMARY KEY (`id`),
          UNIQUE KEY `unique_user_event` (`user_id`, `event_id`),
          KEY `idx_user_id` (`user_id`),
          KEY `idx_event_id` (`event_id`),
          CONSTRAINT `fk_uae_user` FOREIGN KEY (`user_id`) REFERENCES `users` (`id`) ON DELETE CASCADE,
          CONSTRAINT `fk_uae_event` FOREIGN KEY (`event_id`) REFERENCES `events` (`id`) ON DELETE CASCADE,
          CONSTRAINT `fk_uae_assigned_by` FOREIGN KEY (`assigned_by`) REFERENCES `users` (`id`) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    ");
    echo "✓ Table 'user_assigned_events' created successfully.\n";
    
    // Migrate existing data from users.assigned_events column if it exists
    echo "\nMigrating existing data from users.assigned_events...\n";
    
    $check_column = $pdo->query("SHOW COLUMNS FROM `users` LIKE 'assigned_events'");
    if ($check_column->rowCount() > 0) {
        // Get users with assigned events
        $users_stmt = $pdo->query("
            SELECT id, assigned_events 
            FROM users 
            WHERE assigned_events IS NOT NULL 
            AND assigned_events != ''
        ");
        $users = $users_stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $migrated_count = 0;
        foreach ($users as $user) {
            // Parse the assigned_events field (could be CSV or JSON)
            $event_ids = [];
            $assigned_events = trim($user['assigned_events']);
            
            // Try JSON first
            if ($assigned_events[0] === '[') {
                $event_ids = json_decode($assigned_events, true);
            } else {
                // Assume CSV
                $event_ids = array_map('trim', explode(',', $assigned_events));
            }
            
            // Insert into join table
            $insert_stmt = $pdo->prepare("
                INSERT IGNORE INTO user_assigned_events (user_id, event_id)
                VALUES (?, ?)
            ");
            
            foreach ($event_ids as $event_id) {
                if (is_numeric($event_id) && $event_id > 0) {
                    $insert_stmt->execute([$user['id'], $event_id]);
                    $migrated_count++;
                }
            }
        }
        
        echo "✓ Migrated $migrated_count event assignments from " . count($users) . " users.\n";
    } else {
        echo "ℹ No 'assigned_events' column found in users table - skipping data migration.\n";
    }
    
    echo "\n✅ Migration completed successfully!\n";
    echo "You can now use the user_assigned_events table for managing user event assignments.\n";
    
} catch (PDOException $e) {
    echo "❌ Migration failed: " . $e->getMessage() . "\n";
    exit(1);
}
