-- Add custom_fields JSON column to participants table
-- This column will store dynamic custom field data as JSON

ALTER TABLE `participants` 
ADD COLUMN `custom_fields` JSON DEFAULT NULL 
COMMENT 'Dynamic custom fields stored as JSON' 
AFTER `photo`;

-- Note: MariaDB 10.4+ supports JSON but indexing is different than MySQL 8.0
-- For better search performance, you can add virtual columns for specific fields later
-- Example: ALTER TABLE participants ADD COLUMN team_name_virtual VARCHAR(255) AS (JSON_UNQUOTE(JSON_EXTRACT(custom_fields, '$.team_name'))) VIRTUAL;
-- Then: CREATE INDEX idx_team_name ON participants(team_name_virtual);

-- Create a new table to store custom field definitions
CREATE TABLE IF NOT EXISTS `participant_custom_field_definitions` (
  `id` INT(11) NOT NULL AUTO_INCREMENT,
  `field_key` VARCHAR(100) NOT NULL COMMENT 'Unique key for the field (e.g., team_name, coach)',
  `field_label` VARCHAR(255) NOT NULL COMMENT 'Display label for the field',
  `field_type` ENUM('text', 'number', 'date', 'select', 'textarea') DEFAULT 'text',
  `field_options` JSON DEFAULT NULL COMMENT 'Options for select fields',
  `is_searchable` TINYINT(1) DEFAULT 1,
  `is_required` TINYINT(1) DEFAULT 0,
  `display_order` INT(11) DEFAULT 0,
  `is_active` TINYINT(1) DEFAULT 1,
  `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  `updated_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  UNIQUE KEY `unique_field_key` (`field_key`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
COMMENT='Definitions for custom participant fields';

-- Insert some example custom fields
INSERT INTO `participant_custom_field_definitions` 
(`field_key`, `field_label`, `field_type`, `field_options`, `is_searchable`, `display_order`) 
VALUES
('team_name', 'Team Name', 'text', NULL, 1, 1),
('coach_name', 'Coach Name', 'text', NULL, 1, 2),
('sponsor', 'Sponsor', 'text', NULL, 1, 3),
('skill_level', 'Skill Level', 'select', '["Beginner", "Intermediate", "Advanced", "Professional"]', 1, 4),
('notes', 'Notes', 'textarea', NULL, 1, 5);
