# Event Participants Updated At Column

## Overview
Added automatic timestamp tracking to the `event_participants` table to monitor when participant records are modified.

## Database Changes

### New Column
- **Table**: `event_participants`
- **Column**: `updated_at`
- **Type**: `TIMESTAMP`
- **Nullable**: `YES`
- **Default**: `NULL`
- **On Update**: `CURRENT_TIMESTAMP`

### Behavior
The `updated_at` column automatically updates to the current timestamp whenever any field in the participant record is modified. This includes updates to:
- `participant_status` (approved → scoring → performing → complete)
- `bib_number`
- `heat_number`
- `sort_order`
- Any other participant data

## Usage Examples

### Tracking Recent Updates
```sql
-- Find participants updated in the last hour
SELECT * FROM event_participants 
WHERE updated_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR);

-- Find participants by status change time
SELECT bib_number, participant_status, updated_at 
FROM event_participants 
WHERE event_id = 3 
ORDER BY updated_at DESC;
```

### Workflow Monitoring
```sql
-- Check when participants moved to 'performing' status
SELECT bib_number, participant_status, updated_at 
FROM event_participants 
WHERE participant_status = 'performing' 
AND event_id = 3;

-- Track completion times
SELECT bib_number, updated_at as completed_at 
FROM event_participants 
WHERE participant_status = 'complete' 
ORDER BY updated_at;
```

## Integration Points

### Automatic Updates
The column updates automatically when:
1. **Score Creation** (`judge/score.php`, `judge/head_judge.php`)
   - Status changes to 'scoring'

2. **Play Button** (`admin/active_heat_panel.php`)
   - Status changes to 'performing'

3. **Approval Complete** (`judge/head_judge.php`)
   - Status changes to 'complete'
   - `bib_performing` cleared

### No Code Changes Required
Since the column uses MySQL's `ON UPDATE CURRENT_TIMESTAMP`, all existing UPDATE queries will automatically trigger the timestamp update without requiring code modifications.

## Migration Files

### SQL Migration
**File**: `add_updated_at_to_event_participants.sql`
```sql
ALTER TABLE `event_participants` 
ADD COLUMN `updated_at` TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP
AFTER `participant_status`;
```

### PHP Migration
**File**: `add_updated_at_migration.php`
- Checks if column exists
- Adds column if missing
- Sets initial timestamps for existing records
- Run with: `php add_updated_at_migration.php`

## Benefits

1. **Audit Trail**: Track when participant records were last modified
2. **Workflow Monitoring**: Identify bottlenecks in the scoring process
3. **Real-time Updates**: Filter for recently changed participants in dashboards
4. **Performance**: Monitor how long participants spend in each status
5. **Debugging**: Trace timing issues in the scoring workflow

## Status Workflow Timeline

With `updated_at`, you can now track the complete participant journey:

```
Registration → approved (initial timestamp)
              ↓
First Score → scoring (updated_at changes)
              ↓
Play Button → performing (updated_at changes)
              ↓
All Approved → complete (updated_at changes)
```

## Future Enhancements

Potential uses for the `updated_at` column:
- Dashboard filtering: "Show participants updated in last 5 minutes"
- Heat progress tracking: Average time from 'performing' to 'complete'
- Performance metrics: Time spent in each status
- Live updates: Poll for participants where `updated_at > last_check_time`
- Activity monitoring: Detect inactive participants stuck in a status

## Version
- **Added**: November 14, 2025
- **Database Version**: v2.5
- **Related Features**: Participant Status Workflow, Active Heat Panel
