# 🚀 Permissions System - Quick Reference

## Installation Complete ✅

**Database Tables:**
- ✅ `permissions` (42 default permissions)
- ✅ `role_permissions` (role-to-permission mappings)

**Files Created:**
- ✅ `includes/permissions.php` - Helper functions
- ✅ `admin/permissions_manager.php` - Visual admin panel
- ✅ `admin/permissions_api.php` - AJAX API
- ✅ `create_permissions_tables.sql` - SQL schema
- ✅ `install_permissions_system.php` - Migration script

## 🎯 Quick Access

**Admin Panel:** `http://localhost/v2/admin/permissions_manager.php`
- Requires: super_admin login
- Features: Visual checkbox matrix, add/edit/delete permissions
- Auto-save: Click checkbox to toggle permission instantly

## 💻 Code Usage (Copy-Paste Ready)

### Basic Check
```php
require_once '../includes/permissions.php';

if (hasPermission('create_events')) {
    // User can create events
}
```

### Require or Exit
```php
requirePermission('manage_permissions');
```

### Conditional Button
```php
<?php if (hasPermission('delete_users')): ?>
    <button class="btn btn-danger">Delete</button>
<?php endif; ?>
```

### API Protection
```php
if (!hasPermission('submit_scores')) {
    echo json_encode(['success' => false, 'message' => 'Permission denied']);
    exit;
}
```

## 📊 Default Permission Categories

| Category | Permissions | Example |
|----------|-------------|---------|
| **User Management** | 14 | create_admin, view_all_users, delete_users |
| **Event Management** | 7 | create_events, edit_events, publish_events |
| **Judge Management** | 4 | assign_judges, manage_control_points |
| **Scoring** | 4 | submit_scores, approve_scores |
| **Participant Management** | 5 | create_participants, assign_bib_numbers |
| **Dashboard & Reports** | 4 | view_admin_dashboard, export_data |
| **System** | 3 | manage_permissions, view_system_logs |

## 🔑 Key Functions

| Function | Purpose | Returns |
|----------|---------|---------|
| `hasPermission($name)` | Check single permission | bool |
| `hasAnyPermission($array)` | Check if has ANY permission | bool |
| `hasAllPermissions($array)` | Check if has ALL permissions | bool |
| `requirePermission($name)` | Exit with 403 if no permission | void |
| `getRolePermissions($role)` | Get all permissions for role | array |
| `clearPermissionsCache()` | Clear permission cache | void |

## 🎨 Current Role Permissions

| Role | Can Create Users | Can Manage Events | Can Score | Can Manage Permissions |
|------|-----------------|-------------------|-----------|----------------------|
| **super_admin** | ALL | ✅ All | ✅ All | ✅ Yes |
| **admin** | All except super_admin | ✅ All | ✅ All | ❌ No |
| **event_organizer** | office, media, judge, participant | ✅ Assigned only | ✅ View/Approve | ❌ No |
| **office** | media, participant | ❌ View only | ❌ No | ❌ No |
| **head_judge** | participant | ❌ View only | ✅ Submit/Edit/Approve | ❌ No |
| **judge** | participant | ❌ View only | ✅ Submit/View | ❌ No |
| **media** | participant | ❌ View only | ❌ View only | ❌ No |
| **participant** | - | ❌ View only | ❌ No | ❌ No |

## 🔄 Quick Actions

### Add Custom Permission
1. Go to `admin/permissions_manager.php`
2. Click "Add New Permission"
3. Enter: name (e.g., `export_reports`), display name, description, category
4. Click "Save" - auto-assigned to super_admin
5. Toggle checkboxes to assign to other roles
6. Use in code: `hasPermission('export_reports')`

### Grant Permission to Role
1. Go to permissions manager
2. Find permission row
3. Find role column
4. Click checkbox to toggle ON
5. Auto-saved instantly

### Revoke Permission
1. Find checked checkbox in matrix
2. Click to uncheck
3. Auto-saved instantly

### Edit Permission Details
1. Click edit button (pencil icon) on permission row
2. Modify name/description/category
3. Save
4. All role assignments preserved

### Delete Permission
1. Click trash icon on permission row
2. Confirm deletion
3. Cascades to remove from all roles

## 🚨 Important Notes

- ✅ **super_admin** ALWAYS has ALL permissions (cannot be unchecked)
- ✅ Changes save **instantly** (no save button needed)
- ✅ Permission checks are **cached** (cleared automatically)
- ✅ All existing role hierarchy **preserved** in default setup
- ⚠️ Deleting permission removes it from **all roles** (cannot undo)
- ⚠️ Permission names are **case-sensitive** in code

## 🎯 Integration Priority

**High Priority (Start Here):**
1. ✅ `admin/user_management.php` - User CRUD operations
2. ✅ `admin/events.php` - Event management
3. ✅ `api/judge_score_api.php` - Scoring APIs
4. ✅ `admin/judge_manager.php` - Judge assignments

**Medium Priority:**
5. Participant management pages
6. Dashboard access control
7. Report generation

**Low Priority:**
8. Public-facing pages (already restricted)
9. Read-only views

## 📖 Full Documentation

See `PERMISSIONS_IMPLEMENTATION_GUIDE.md` for:
- Complete code examples
- Migration strategies
- Troubleshooting guide
- Performance tips
- Security best practices

---

**Status:** ✅ READY TO USE

Access panel now: `admin/permissions_manager.php` (super_admin only)
