# Permissions System Implementation Guide

## 🚀 Quick Start

### Installation (COMPLETED ✅)

The permissions system has been installed with:
- ✅ `permissions` table - 42 default permissions across 7 categories
- ✅ `role_permissions` table - Role-to-permission mappings
- ✅ Helper functions in `includes/permissions.php`
- ✅ Admin panel at `admin/permissions_manager.php`
- ✅ API at `admin/permissions_api.php`

### Access the Panel

**URL:** `http://localhost/v2/admin/permissions_manager.php`

**Login Required:** super_admin role only

## 📊 What You Get

### Visual Permission Matrix
- **Checkbox Grid**: Roles (columns) × Permissions (rows)
- **Live Toggle**: Click any checkbox to grant/revoke permission instantly
- **Auto-save**: Changes save automatically with visual confirmation
- **Color-coded Roles**: Each role has distinct color badge
- **Categorized View**: Permissions grouped by function (User Management, Events, Scoring, etc.)

### Permission Management
- **Add New Permissions**: Create custom permissions on the fly
- **Edit Permissions**: Update names, descriptions, categories
- **Delete Permissions**: Remove unused permissions (cascades to roles)
- **Statistics Dashboard**: View total roles, permissions, assignments

### Built-in Permissions (42 Total)

#### User Management (14)
- `create_super_admin`, `create_admin`, `create_event_organizer`, `create_office`, `create_media`, `create_judge`, `create_head_judge`, `create_participant`
- `view_all_users`, `view_created_users`, `edit_all_users`, `edit_created_users`, `delete_users`, `manage_user_groups`

#### Event Management (7)
- `view_all_events`, `view_assigned_events`, `create_events`, `edit_events`, `delete_events`, `publish_events`, `manage_event_categories`, `manage_heat_settings`

#### Judge Management (4)
- `view_all_judges`, `view_event_judges`, `assign_judges`, `manage_control_points`

#### Scoring (4)
- `submit_scores`, `view_scores`, `edit_scores`, `approve_scores`

#### Participant Management (5)
- `view_participants`, `create_participants`, `edit_participants`, `delete_participants`, `assign_bib_numbers`

#### Dashboard & Reports (4)
- `view_public_dashboard`, `view_admin_dashboard`, `generate_reports`, `export_data`

#### System Administration (3)
- `manage_permissions`, `view_system_logs`, `manage_settings`

## 💻 How to Use in Code

### Basic Permission Check

```php
// Include at top of file
require_once '../includes/permissions.php';

// Check single permission
if (hasPermission('create_events')) {
    // User can create events
    echo '<button onclick="createEvent()">New Event</button>';
}

// Check multiple permissions (ANY)
if (hasAnyPermission(['edit_events', 'delete_events'])) {
    // User can edit OR delete events
    echo '<button onclick="manageEvent()">Manage</button>';
}

// Check multiple permissions (ALL)
if (hasAllPermissions(['view_scores', 'approve_scores'])) {
    // User can view AND approve scores
    echo '<button onclick="approveScores()">Approve All</button>';
}
```

### Require Permission or Exit

```php
// At top of protected page
require_once '../includes/permissions.php';
requirePermission('manage_permissions', 'Only super admins can access this page');

// In API endpoint
if (!hasPermission('submit_scores')) {
    http_response_code(403);
    echo json_encode(['success' => false, 'message' => 'No permission to submit scores']);
    exit;
}
```

### Conditional UI Elements

```php
<!-- In HTML -->
<?php if (hasPermission('delete_users')): ?>
    <button class="btn btn-danger" onclick="deleteUser(<?= $user_id ?>)">
        <i class="fas fa-trash"></i> Delete
    </button>
<?php endif; ?>

<?php if (hasPermission('edit_events')): ?>
    <a href="edit_event.php?id=<?= $event_id ?>" class="btn btn-primary">Edit Event</a>
<?php endif; ?>
```

### Form Validation

```php
// Check permission before processing form
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['create_user'])) {
    $role_to_create = $_POST['role'];
    
    // Check if user has permission to create this role
    $permission_map = [
        'super_admin' => 'create_super_admin',
        'admin' => 'create_admin',
        'event_organizer' => 'create_event_organizer',
        'judge' => 'create_judge',
        // ... etc
    ];
    
    $required_permission = $permission_map[$role_to_create] ?? 'create_participant';
    
    if (!hasPermission($required_permission)) {
        die(json_encode([
            'success' => false,
            'message' => "You don't have permission to create {$role_to_create} users"
        ]));
    }
    
    // Proceed with user creation
}
```

### Replace Role-Based Checks

**BEFORE (Old Way):**
```php
if ($_SESSION['role'] === 'super_admin' || $_SESSION['role'] === 'admin') {
    // Allow access
}
```

**AFTER (New Way):**
```php
if (hasPermission('manage_settings')) {
    // Allow access - works for any role with this permission
}
```

## 🔧 Integration Examples

### Example 1: Update Event Creation Page

```php
// admin/events.php
require_once '../includes/permissions.php';

// Check permission at page level
requirePermission('view_all_events');

// Conditional create button
if (hasPermission('create_events')): ?>
    <button class="btn btn-success" onclick="createNewEvent()">
        <i class="fas fa-plus"></i> New Event
    </button>
<?php endif; ?>

// Conditional edit/delete buttons in table
foreach ($events as $event) {
    echo '<tr>';
    echo '<td>' . $event['name'] . '</td>';
    echo '<td>';
    
    if (hasPermission('edit_events')) {
        echo '<button onclick="editEvent(' . $event['id'] . ')">Edit</button>';
    }
    
    if (hasPermission('delete_events')) {
        echo '<button onclick="deleteEvent(' . $event['id'] . ')">Delete</button>';
    }
    
    echo '</td>';
    echo '</tr>';
}
```

### Example 2: Update Judge Assignment API

```php
// api/judge_assignment_api.php
require_once '../includes/permissions.php';

$action = $_POST['action'] ?? '';

switch ($action) {
    case 'assign_judge':
        requirePermission('assign_judges');
        // Process assignment
        break;
        
    case 'remove_judge':
        requirePermission('assign_judges');
        // Process removal
        break;
        
    case 'view_judges':
        requirePermission('view_event_judges');
        // Return judge list
        break;
}
```

### Example 3: Update User Management

```php
// admin/user_management.php
require_once '../includes/permissions.php';

// Page-level check
requirePermission('view_all_users');

// Create button - only show if has ANY create permission
$create_permissions = [
    'create_super_admin', 'create_admin', 'create_event_organizer',
    'create_judge', 'create_office', 'create_media', 'create_participant'
];

if (hasAnyPermission($create_permissions)): ?>
    <button onclick="openCreateUserModal()">Create User</button>
<?php endif; ?>

// Delete button per user
foreach ($users as $user) {
    if (hasPermission('delete_users')): ?>
        <button onclick="deleteUser(<?= $user['id'] ?>)">Delete</button>
    <?php endif;
}
```

## 🎯 Migration Strategy

### Phase 1: Install (DONE ✅)
- ✅ Run `install_permissions_system.php`
- ✅ Verify tables created
- ✅ Access admin panel

### Phase 2: Review & Customize
1. Open `admin/permissions_manager.php`
2. Review default permission assignments
3. Adjust checkboxes as needed for your organization
4. Add custom permissions if needed

### Phase 3: Code Integration (Gradual)

**Start with high-priority pages:**
1. User management (`admin/user_management.php`)
2. Event management (`admin/events.php`)
3. Judge management (`admin/judge_manager.php`)
4. Scoring APIs (`api/judge_score_api.php`)

**Replace role checks:**
```bash
# Search for old role checks
grep -r "SESSION\['role'\]" admin/
grep -r "in_array.*role" admin/

# Replace with permission checks
# OLD: if ($_SESSION['role'] === 'admin')
# NEW: if (hasPermission('specific_action'))
```

**Test thoroughly:**
- Login as different roles
- Verify buttons show/hide correctly
- Test API endpoints with permission checks
- Check error messages are helpful

### Phase 4: Extend

**Add custom permissions:**
1. Go to permissions panel
2. Click "Add New Permission"
3. Fill in: name, display name, description, category
4. Assign to appropriate roles via checkboxes
5. Use in code with `hasPermission('your_new_permission')`

**Example custom permissions:**
- `export_financial_reports`
- `manage_sponsors`
- `send_notifications`
- `view_analytics`
- `manage_merchandise`

## 🔐 Security Benefits

### Fine-Grained Control
- **OLD**: All admins can do everything
- **NEW**: Admins can have different permission sets

### Easy Auditing
```sql
-- See all permissions for a role
SELECT p.display_name, p.description
FROM role_permissions rp
JOIN permissions p ON rp.permission_id = p.id
WHERE rp.role = 'event_organizer';

-- See who granted permissions
SELECT rp.*, u.username as granted_by_user
FROM role_permissions rp
JOIN users u ON rp.granted_by = u.id
WHERE rp.role = 'judge';
```

### Flexible Updates
- Change permissions without code changes
- Temporarily revoke permissions via admin panel
- Create temporary elevated permissions for specific tasks

## 📈 Performance

### Caching
- Permission checks are cached per role
- Cache cleared automatically on permission changes
- Minimal database queries (1 per unique permission per request)

### Optimization Tips
```php
// GOOD: Check once, use many times
$can_edit = hasPermission('edit_events');
foreach ($events as $event) {
    if ($can_edit) {
        echo '<button>Edit</button>';
    }
}

// BAD: Check inside loop
foreach ($events as $event) {
    if (hasPermission('edit_events')) { // Repeated check
        echo '<button>Edit</button>';
    }
}
```

## 🐛 Troubleshooting

### Permission Check Returns False
1. Check user is logged in: `var_dump($_SESSION['user_id'])`
2. Check permission exists: Query `permissions` table
3. Check role has permission: Check `role_permissions` table
4. Check permission name matches exactly (case-sensitive)
5. Clear cache: Call `clearPermissionsCache()`

### Changes Not Reflecting
1. Clear permissions cache: `clearPermissionsCache()`
2. Log out and log back in
3. Check browser console for errors
4. Verify database changes: `SELECT * FROM role_permissions WHERE role = 'your_role'`

### SQL Errors
1. Verify tables exist: `SHOW TABLES LIKE 'permissions'`
2. Check foreign keys: `SHOW CREATE TABLE role_permissions`
3. Re-run migration if needed: `php install_permissions_system.php`

## 📚 API Reference

### hasPermission($permission_name, $user_id = null)
Returns `true` if user has permission, `false` otherwise.
- super_admin always returns `true`
- Uses session data by default
- Cached for performance

### hasAnyPermission($permission_names, $user_id = null)
Returns `true` if user has at least one of the specified permissions.

### hasAllPermissions($permission_names, $user_id = null)
Returns `true` if user has all of the specified permissions.

### requirePermission($permission_name, $message = null)
Exits with 403 error if user lacks permission.
- Use at top of protected pages
- Custom error message optional

### getRolePermissions($role)
Returns array of all permissions for a role.
- Useful for displaying role capabilities
- Returns full permission objects with descriptions

### getAllPermissionsGrouped()
Returns all permissions grouped by category.
- Used by admin panel
- Good for permission overview pages

### clearPermissionsCache()
Clears the permission cache.
- Called automatically on permission changes
- Manually call after bulk updates

## 🎨 UI Components

### Permission Badge
```php
<?php if (hasPermission('manage_permissions')): ?>
    <span class="badge bg-warning">
        <i class="fas fa-shield-alt"></i> Admin
    </span>
<?php endif; ?>
```

### Conditional Section
```php
<?php if (hasPermission('view_system_logs')): ?>
    <div class="card">
        <div class="card-header">System Logs</div>
        <div class="card-body">
            <!-- Log content -->
        </div>
    </div>
<?php endif; ?>
```

### Multi-Level Permissions
```php
<?php if (hasPermission('view_scores')): ?>
    <!-- Everyone can view -->
    <div class="scores-list">...</div>
    
    <?php if (hasPermission('edit_scores')): ?>
        <!-- Some can edit -->
        <button class="btn-edit">Edit</button>
    <?php endif; ?>
    
    <?php if (hasPermission('approve_scores')): ?>
        <!-- Few can approve -->
        <button class="btn-approve">Approve</button>
    <?php endif; ?>
<?php endif; ?>
```

## ✅ Next Steps

1. **Access Panel**: Visit `admin/permissions_manager.php`
2. **Review Defaults**: Check permission assignments make sense
3. **Customize**: Toggle checkboxes to adjust permissions
4. **Test**: Login as different roles to verify permissions
5. **Integrate**: Start replacing role checks with permission checks
6. **Expand**: Add custom permissions as needed

## 🎯 Best Practices

1. **Descriptive Names**: Use clear permission names like `create_events` not `ce`
2. **Granular Permissions**: Better to have specific permissions than generic ones
3. **Check Early**: Validate permissions at page/API entry point
4. **Fail Secure**: Default to denying access if uncertain
5. **User Feedback**: Show helpful error messages when permission denied
6. **Document Custom**: Keep track of custom permissions you add
7. **Regular Audit**: Periodically review permission assignments
8. **Test All Roles**: Always test with different user roles

---

**System Status:** ✅ FULLY OPERATIONAL

The permissions system is installed and ready to use. Access the admin panel to start managing permissions!
