# Password System Documentation

## Overview
The StyleScore user authentication system has been updated to support secure password hashing using bcrypt while maintaining backward compatibility with existing plain text passwords.

## Current State

### Password Storage
- **New Users**: Passwords are automatically hashed using `password_hash()` with `PASSWORD_DEFAULT` (bcrypt)
- **Legacy Users**: Some existing users may still have plain text passwords in the database
- **Hash Format**: Bcrypt hashes are 60 characters long and start with `$2y$`

### Login Process (`includes/process_login.php`)
The login system automatically detects password format and uses the appropriate verification method:

```php
// Check if password is hashed (bcrypt starts with $2y$ and is 60 chars)
if (strlen($user['password']) === 60 && strpos($user['password'], '$2y$') === 0) {
    // Use password_verify for hashed passwords
    $password_valid = password_verify($password, $user['password']);
} else {
    // Legacy plain text comparison
    $password_valid = ($password === $user['password']);
}
```

## User Management

### Creating New Users (`admin/user_management.php`)
All new users created through the user management interface automatically have their passwords hashed:

```php
$stmt = $pdo->prepare("INSERT INTO users (username, password, role, created_at) VALUES (?, ?, ?, NOW())");
$stmt->execute([$username, password_hash($password, PASSWORD_DEFAULT), $role]);
```

### Password Reset/Change
Currently, there is no password change functionality in the UI. Users need to be managed through:
1. Database direct update (for administrators)
2. Future password reset feature (to be implemented)

## Security Recommendations

### 1. Migrate All Passwords to Hashed Format
Run the migration script to convert all plain text passwords to hashed format:

```bash
C:\xampp\php\php.exe migrate_passwords_to_hash.php
```

**IMPORTANT**: Before running migration, note down current passwords as they will be hashed!

### 2. Implement Password Change Feature
Add functionality for users to change their own passwords through the UI.

### 3. Implement Password Reset
Add secure password reset flow via email or admin approval.

### 4. Password Policies
Consider implementing:
- Minimum password length (currently no validation)
- Password complexity requirements
- Password expiration/rotation
- Failed login attempt tracking

## Testing

### Check Current Password Format
```bash
C:\xampp\php\php.exe check_passwords.php
```

This will show which users have plain text vs hashed passwords.

### Test Login Verification
```bash
C:\xampp\php\php.exe test_login_passwords.php
```

This tests the password verification logic for both formats.

## Database Schema

### Users Table
```sql
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,  -- Stores both plain text (legacy) and hashed passwords
    role ENUM('super_admin', 'admin', 'head_judge', 'judge', 'office', 'media', 'participant'),
    judge_id INT DEFAULT NULL,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_login TIMESTAMP NULL
);
```

## Migration Path

### Current Status
- ✅ Login supports both plain text and hashed passwords
- ✅ New users automatically get hashed passwords
- ✅ User creation through UI uses password hashing
- ⚠️ Some legacy users may have plain text passwords
- ❌ No password change functionality
- ❌ No password reset functionality

### Recommended Steps
1. **Run password migration** to hash all existing plain text passwords
2. **Test login** with both old and new users
3. **Implement password change** feature for users
4. **Implement password reset** feature (email or admin-based)
5. **Add password policies** (minimum length, complexity)
6. **Remove plain text fallback** after confirming all passwords are hashed

## Files Modified

### `includes/process_login.php`
- Added password format detection
- Added bcrypt verification using `password_verify()`
- Maintained backward compatibility with plain text passwords

### `admin/user_management.php`
- Already uses `password_hash()` for new users (line 85)
- No changes needed

### Migration Scripts Created
- `check_passwords.php` - Check password format in database
- `test_login_passwords.php` - Test login verification logic
- `migrate_passwords_to_hash.php` - Migrate all plain text passwords to hashed format

## Security Notes

### Bcrypt Advantages
- Automatically salted
- Computationally expensive (protects against brute force)
- Industry standard for password hashing
- Built-in to PHP via `password_hash()` and `password_verify()`

### Why Backward Compatibility?
- Prevents locking out existing users
- Allows gradual migration
- Can be removed once all passwords are confirmed hashed

### Future Considerations
- Implement password history (prevent reuse)
- Add two-factor authentication (2FA)
- Implement account lockout after failed attempts
- Add session timeout and security logging
