# Participant & Heat Card Layout System

## Overview
The StyleScore system now supports configurable layouts for both **participant cards** and **heat cards** to provide different levels of detail and functionality depending on the context.

---

## Participant Card Layouts

### Layout Modes

### 1. `full` - Complete View
**Use case**: Heat management configuration pages where admins need full control

**Features**:
- ✅ Position controls (move up/down arrows)
- ✅ Full participant details (category, country, club)
- ✅ Score breakdown with figures/criteria
- ✅ Judge count display
- ✅ Status badges (Performing, Scored)

**Example**:
```javascript
participantCardLayout = 'full';
```

### 2. `compact` - Minimal with Controls
**Use case**: Tight spaces where controls are needed but details can be condensed

**Features**:
- ✅ Position controls (move up/down arrows)
- ⚠️ Minimal details (category only, inline)
- ❌ No figures/criteria breakdown
- ✅ Judge count display
- ✅ Status badges

**Example**:
```javascript
participantCardLayout = 'compact';
```

### 3. `controls` - Standard with Controls
**Use case**: General heat management where reordering is needed

**Features**:
- ✅ Position controls (move up/down arrows)
- ⚠️ Minimal details (category only, inline)
- ❌ No figures/criteria breakdown
- ✅ Judge count display
- ✅ Status badges

**Example**:
```javascript
participantCardLayout = 'controls';
```

### 4. `info` - Read-Only Display (Default for Active Heat Panel)
**Use case**: Monitoring dashboards, public displays, judge panels

**Features**:
- ❌ No position controls
- ✅ Full participant details (category, country, club)
- ✅ Score breakdown with figures/criteria
- ✅ Judge count display
- ✅ Status badges (Performing, Scored)

**Example**:
```javascript
participantCardLayout = 'info'; // Default for active_heat_panel.php
```

## Implementation

### API Endpoint
```
GET heat_management_api.php?action=get_participant_cards&event_id=X&heat_number=Y&layout=MODE
```

**Parameters**:
- `event_id` (required): Event ID
- `heat_number` (required): Heat number
- `layout` (optional): One of `full`, `compact`, `controls`, `info` (default: `full`)
- `mode` (optional): Sort mode - `sort_order` or `bib` (default: `sort_order`)

**Response**:
```json
{
  "success": true,
  "participants": [...],
  "layout": "info",
  "mode": "sort_order",
  "heat_settings": {...},
  "metadata": {
    "include_controls": false,
    "include_details": true,
    "include_figures": true,
    "total_participants": 12
  }
}
```

### Client-Side Configuration

In `active_heat_panel.php`:
```javascript
// Set layout mode globally
let participantCardLayout = 'info'; // Change to 'full', 'compact', or 'controls' as needed
```

The `renderParticipantCard()` function automatically reads this variable and adjusts the output accordingly.

### Dynamic Layout Switching

You can add UI controls to switch layouts on the fly:

```html
<select id="layoutSelector" class="form-select">
  <option value="info">Info Only</option>
  <option value="controls">With Controls</option>
  <option value="compact">Compact</option>
  <option value="full">Full Details</option>
</select>
```

```javascript
document.getElementById('layoutSelector').addEventListener('change', (e) => {
  participantCardLayout = e.target.value;
  renderParticipantList(); // Re-render with new layout
});
```

## Visual Differences

### Info Layout (Default)
```
┌─────────────────────────────────────────────────────────┐
│ [Performing Badge]                                      │
│                                                         │
│  ┌───┐  John Smith               Score                │
│  │ 5 │  ⚙ Solo Male              98.50                │
│  └───┘  🏴 USA                   Avg (3 judges)        │
│         🏛️ Dance Club                                   │
│                                                         │
│  [Rotation: 720, 900] [Difficulty: 8.5]                │
└─────────────────────────────────────────────────────────┘
```

### Compact Layout
```
┌────────────────────────────────────────┐
│ [Scored Badge]                         │
│  ┌───┐  Jane Doe      95.25      ↑    │
│  │ 7 │  Solo Female   Avg (3)    ↓    │
│  └───┘                                 │
└────────────────────────────────────────┘
```

## Best Practices

1. **Use `info` layout** for:
   - Active heat monitoring panels
   - Judge displays
   - Public scoreboards
   - Embed panels in dashboards

2. **Use `controls` layout** for:
   - Heat configuration pages
   - Admin management interfaces
   - Start list reordering

3. **Use `compact` layout** for:
   - Mobile views
   - Split-screen interfaces
   - Quick overview panels

4. **Use `full` layout** for:
   - Detailed heat management
   - Score verification screens
   - Administrative review pages

## Migration Notes

**Existing Code**: All existing implementations continue to work. The default behavior (without specifying layout) renders all features for backward compatibility.

**Active Heat Panel**: Now uses `info` layout by default to provide a cleaner, read-only monitoring experience.

**To Enable Controls**: Simply change `participantCardLayout = 'info'` to `participantCardLayout = 'controls'` in the JavaScript initialization.

---

## Heat Card Layouts

Heat cards display the overall status, progress, and management controls for each heat. Different layouts provide varying levels of detail and interactivity.

### Layout Modes

#### 1. `full` - Complete Management View (Default)
**Use case**: Heat configuration pages, admin dashboards with full control

**Features**:
- ✅ Run change controls (Prev/Next buttons for active heats)
- ✅ All action buttons (Activate, Finish, Cancel, Reschedule, etc.)
- ✅ Judge status panel with detailed information
- ✅ Progress bars and statistics
- ✅ Time schedule (start, end, duration)

**Example**:
```javascript
heatCardLayout = 'full';
```

#### 2. `compact` - Reduced View with Controls
**Use case**: Tight layouts where controls are needed but space is limited

**Features**:
- ❌ No run change controls
- ✅ All action buttons
- ❌ No judge status panel
- ✅ Progress bars and statistics
- ✅ Time schedule

**Example**:
```javascript
heatCardLayout = 'compact';
```

#### 3. `monitor` - Read-Only Monitoring View ⭐
**Use case**: Active heat monitoring panels, public displays, real-time dashboards

**Features**:
- ❌ No run change controls
- ❌ No action buttons
- ✅ Judge status panel (shows judge readiness)
- ✅ Progress bars and statistics
- ❌ No time schedule

**Perfect for**: Active Heat Panel, judge coordination screens, live monitoring

**Example**:
```javascript
heatCardLayout = 'monitor'; // Default for active_heat_panel.php
```

#### 4. `minimal` - Essential Information Only
**Use case**: Compact displays, mobile views, overview screens

**Features**:
- ❌ No run change controls
- ❌ No action buttons
- ❌ No judge status panel
- ❌ No progress bars
- ❌ No time schedule

**Shows only**: Heat number, name, status badge, participant count, run count

**Example**:
```javascript
heatCardLayout = 'minimal';
```

#### 5. `monitor-row` - Horizontal Compact Monitor ⭐
**Use case**: Ultra-compact monitoring, horizontal displays, dashboard widgets

**Features**:
- ❌ No run change controls
- ❌ No action buttons
- ✅ All info in single row (heat number, status, run progress, participants, judges)
- ✅ Compact judge status icons only
- ✅ Progress bar (inline)

**Perfect for**: Dashboard widgets, horizontal monitor strips, multi-heat overviews

**Example**:
```javascript
heatCardLayout = 'monitor-row';
```

**Visual**:
```
┌──────────────────────────────────────────────────────────────────────┐
│ Heat 1  [ACTIVE]  Run 2/3  👥 12 (10 done)  [████████░░] ✓ ✓ ⏸ ✓ ✓ │
└──────────────────────────────────────────────────────────────────────┘
```

#### 6. `judges` - Judge Status Only with Color Coding 🎨
**Use case**: Judge coordination panels, status monitoring, head judge dashboard

**Features**:
- ❌ No run change controls
- ❌ No action buttons
- ✅ **ONLY** judge status badges
- ✅ Parent background color indicates overall readiness:
  - 🟢 **Green** (`bg-success`): All judges ready
  - 🔵 **Blue** (`bg-info`): Mixed status (some not ready)
- ✅ Compact judge badges with status icons

**Perfect for**: Judge coordination, head judge panel, status-at-a-glance

**Example**:
```javascript
heatCardLayout = 'judges';
```

**Visual (All Ready)**:
```
┌────────────────────────────────┐
│  🔥 Heat 1                     │ <- Green background
│  👔 Judges (5)                 │
│   ✓ Smith  ✓ Jones  ✓ Davis   │
│   ✓ Brown  ✓ Wilson            │
└────────────────────────────────┘
```

**Visual (Mixed Status)**:
```
┌────────────────────────────────┐
│  🔥 Heat 1                     │ <- Blue background
│  👔 Judges (5)                 │
│   ✓ Smith  ✓ Jones  ⏸ Davis   │
│   ⚠ Brown  ✓ Wilson            │
└────────────────────────────────┘
```

### Heat Card Feature Comparison

| Feature | FULL | COMPACT | MONITOR | MINIMAL | MONITOR-ROW | JUDGES |
|---------|------|---------|---------|---------|-------------|--------|
| **Run Controls** (Prev/Next) | ✅ | ❌ | ❌ | ❌ | ❌ | ❌ |
| **Action Buttons** | ✅ | ✅ | ❌ | ❌ | ❌ | ❌ |
| **Judge Status Panel** | ✅ | ❌ | ✅ | ❌ | ✅ (icons) | ✅ (only) |
| **Progress Bars** | ✅ | ✅ | ✅ | ❌ | ✅ (inline) | ❌ |
| **Time Schedule** | ✅ | ✅ | ❌ | ❌ | ❌ | ❌ |
| **Layout** | Vertical | Vertical | Vertical | Vertical | **Horizontal** | Vertical |
| **Bg Color Coding** | ❌ | ❌ | ❌ | ❌ | ❌ | ✅ (readiness) |
| **Basic Info** (name, status) | ✅ | ✅ | ✅ | ✅ | ✅ | ✅ |
| **Best Use Case** | Admin | Compact Admin | Monitor | Overview | Dashboard Widget | Judge Coordination |

### Heat Card API Endpoint
```
GET heat_cards_api.php?event_id=X&layout=MODE
```

**Parameters**:
- `event_id` (required): Event ID
- `layout` (optional): One of `full`, `compact`, `monitor`, `minimal`, `monitor-row`, `judges` (default: `full`)

**Response**:
```json
{
  "success": true,
  "data": "<HTML content>",
  "heat_count": 3,
  "timestamp": "2025-11-14 10:30:00"
}
```

### Client-Side Configuration

In `active_heat_panel.php`:
```javascript
// Set layout modes globally
let participantCardLayout = 'info';    // Participant cards layout
let heatCardLayout = 'monitor';        // Heat cards layout
```

Both layouts are automatically applied when the respective functions load data.

---

## Combined Usage Example

**Active Heat Panel** (monitoring configuration):
```javascript
let participantCardLayout = 'info';    // Read-only participant cards
let heatCardLayout = 'monitor';        // Judge status visible, no action buttons
```

**Heat Configuration Page** (full management):
```javascript
let participantCardLayout = 'controls'; // Position controls enabled
let heatCardLayout = 'full';           // All management features
```

**Public Display** (minimal information):
```javascript
let participantCardLayout = 'info';    // Clean participant info
let heatCardLayout = 'minimal';        // Minimal heat overview
```

---

## Visual Examples

### Monitor Layout (Active Heat Panel Default)
```
┌───────────────────────────────────────────┐
│ 🔥 Heat 1                    [ACTIVE] 🔥  │
│ Finals - Solo Male                        │
│ 👥 12    🔄 3                              │
├───────────────────────────────────────────┤
│ [Current Run: 2/3]                        │
│                                           │
│ 📊 Scoring Progress            80%       │
│ [████████████████░░░░] ✓ 10  ✏ 12       │
│                                           │
│ 👔 Judges (5)         ✓ 3  ⏸ 1  ⚠ 1    │
│  ✓ Judge Smith        H1R2   2m ago      │
│  ✓ Judge Jones        H1R2   now         │
│  ⏸ Judge Williams     H1R2   5m ago      │
│  ⚠ Judge Davis        H1R1   10m ago     │
│  ✓ Judge Brown        H1R2   1m ago      │
└───────────────────────────────────────────┘
```

### Full Layout (Complete Management)
```
┌───────────────────────────────────────────┐
│ 🔥 Heat 1                    [ACTIVE] 🔥  │
│ Finals - Solo Male                        │
│ 👥 12    🔄 3                              │
├───────────────────────────────────────────┤
│ [Current Run: 2/3]                        │
│  [< Prev]  [Next >]                       │
│                                           │
│ 🕐 Schedule                                │
│  Start: 14:00  End: 15:30  Duration: 90m │
│                                           │
│ 📊 Scoring Progress            80%       │
│ [████████████████░░░░] ✓ 10  ✏ 12       │
│                                           │
│ 👔 Judges (5)         ✓ 3  ⏸ 1  ⚠ 1    │
│  ✓ Judge Smith        H1R2   2m ago      │
│  ... (full list) ...                      │
├───────────────────────────────────────────┤
│ [✓ Finish]  [⏸ Pending]                  │
│ [🚫 Cancel] [🕐 Reschedule]               │
│ [⚙ Config]  [🏆 Results]                  │
└───────────────────────────────────────────┘
```

### Minimal Layout
```
┌────────────────────────┐
│ 🔥 Heat 1   [ACTIVE] 🔥│
│ Finals - Solo Male     │
│ 👥 12    🔄 3           │
└────────────────────────┘
```

---

## Migration Notes

**Existing Code**: All existing implementations continue to work. The default behavior (without specifying layout) renders all features for backward compatibility.

**Active Heat Panel**: Now uses:
- `heatCardLayout = 'monitor'` - Shows judge status but no action buttons
- `participantCardLayout = 'info'` - Read-only participant information

**Heat Configuration Pages**: Use `heatCardLayout = 'full'` for complete management capabilities.

**To Customize**: Simply change the layout variables in the JavaScript initialization section of your page.

---

## Best Practices

### Heat Card Layouts

1. **Use `full` layout** for:
   - Heat configuration pages
   - Admin management interfaces
   - Control panels with run management

2. **Use `compact` layout** for:
   - Multi-heat overview pages
   - Mobile admin views
   - Quick management panels

3. **Use `monitor` layout** for:
   - Active heat monitoring
   - Judge coordination screens
   - Live event dashboards

4. **Use `minimal` layout** for:
   - Public displays
   - Heat selection lists
   - Overview grids with many heats

### Participant Card Layouts

(See participant card sections above for detailed best practices)
