# Global Notification Enable/Disable Feature

## Overview
The notification system now includes a **global enable/disable toggle** that allows administrators to completely turn off notifications for all users at the event level.

## Purpose
- **Event Control**: Disable notifications during setup, testing, or off-competition periods
- **User Experience**: Prevent notification spam when the event is not actively running
- **System Management**: Reduce server load when real-time updates are not needed
- **Privacy**: Ensure notifications only appear when the event is officially live

## How It Works

### Database Structure
```sql
-- event_notification_settings table includes:
enabled BOOLEAN DEFAULT TRUE  -- Controls notification system for the event
```

When `enabled = FALSE`:
- No notifications are sent to ANY user (admin, judge, staff, viewer)
- Polling stops automatically
- Live status indicator shows "DISABLED"
- Notification panel is visually disabled (greyed out)

### User Interface

#### Admin Panel (`admin/notifications.php`)

**Primary Toggle (Top of Page)**
```
┌─────────────────────────────────────────────────────┐
│ 🔌 Notification System Status                      │
│                                                     │
│ Enable or disable notifications for all users      │
│                                    [●====] ENABLED │
└─────────────────────────────────────────────────────┘
```

Features:
- Large, prominent toggle at the top of notifications page
- Shows current status with color-coded badge (green = enabled, red = disabled)
- Requires confirmation before toggling
- Updates status text dynamically
- Greys out notification content when disabled

**Settings Modal**
- Also includes toggle in advanced settings modal
- Both toggles stay synchronized
- Preserved when saving other settings

### API Behavior

#### `GET /api/notification_api.php?action=get_notifications`
```json
{
  "success": true,
  "notifications": [],
  "latest_id": 0,
  "count": 0,
  "disabled": true,
  "message": "Notifications are currently disabled for this event"
}
```

When disabled, returns empty notification array with `disabled: true` flag.

#### `POST /api/notification_api.php?action=save_settings`
```javascript
formData.append('enabled', true/false);
```

Saves the enabled state along with other settings.

### Client-Side Behavior

**NotificationManager.js**
```javascript
// In pollNotifications()
if (data.disabled === true) {
    console.log('Notifications are disabled - stopping polling');
    this.stopPolling();
    this.connectionStatus = 'disabled';
    this.updateConnectionStatus();
    return;
}
```

**Status Indicator**
```
LIVE        → Green badge with pulse animation (enabled & connected)
DISABLED    → Grey badge (notifications disabled by admin)
DISCONNECTED → Red badge (connection lost)
LOCAL       → Yellow badge (local mode, no API)
```

## Usage Workflow

### Enable/Disable from Admin Panel

1. Navigate to `admin/notifications.php`
2. Select the event from dropdown
3. Toggle the switch at top of page
4. Confirm the action in dialog:
   - **Enable**: "Enable notifications for all users?"
   - **Disable**: "Disable notifications for ALL users?"
5. System updates immediately for all connected users

### Check Status via JavaScript
```javascript
// Check if notifications are enabled
fetch(`../api/notification_api.php?action=get_settings&event_id=${eventId}`)
  .then(response => response.json())
  .then(data => {
    if (data.settings.enabled) {
      console.log('Notifications are enabled');
    } else {
      console.log('Notifications are DISABLED');
    }
  });
```

### Typical Use Cases

**Before Event Starts**
```
Status: DISABLED
Reason: Setup and testing phase, no live competition yet
Action: Admin disables notifications to prevent confusion
```

**During Competition**
```
Status: ENABLED
Reason: Live event, real-time updates needed
Action: Admin enables notifications for all users
```

**Between Heats**
```
Status: Optional (can disable for breaks)
Reason: Reduce notification noise during downtime
Action: Admin can toggle based on preference
```

**Post-Event**
```
Status: DISABLED
Reason: Event concluded, no more live updates
Action: Admin disables to stop polling
```

## Technical Implementation

### Files Modified

1. **api/notification_api.php**
   - Added `enabled` column to settings table
   - Modified `getNotifications()` to check enabled status
   - Modified `saveNotificationSettings()` to store enabled flag
   - Modified `getNotificationSettings()` to return enabled flag

2. **admin/notifications.php**
   - Added prominent toggle UI at page top
   - Added toggle in settings modal
   - Added `toggleNotificationSystem()` JavaScript function
   - Added `updateNotificationStatusUI()` to update visual state
   - Synchronized both toggles

3. **assets/js/notification_manager.js**
   - Modified `pollNotifications()` to detect disabled state
   - Added 'disabled' connection status
   - Modified `updateConnectionStatus()` to show disabled badge
   - Stops polling when disabled

4. **admin/sql/add_notification_enabled_column.sql**
   - Migration SQL to add `enabled` column
   - Creates index for performance
   - Safe for existing databases (uses ALTER TABLE IF NOT EXISTS)

### Database Migration

Run the migration to add the column:
```bash
mysql -u root -p pzls_livescore_398dsk < admin/sql/add_notification_enabled_column.sql
```

Or execute in phpMyAdmin:
```sql
ALTER TABLE event_notification_settings 
ADD COLUMN enabled BOOLEAN DEFAULT TRUE AFTER event_id;
```

### Default Behavior
- **New events**: Notifications ENABLED by default
- **Existing events**: Notifications ENABLED (column defaults to TRUE)
- **No settings record**: Falls back to enabled=true

## Security Considerations

- Only admins can toggle notification status
- Regular session validation required
- No permission bypass for disabled notifications
- All users affected equally (no role exceptions)

## Performance Benefits

When disabled:
- **Zero polling requests** from all clients
- **No database queries** for notifications
- **Reduced server load** during non-event periods
- **Bandwidth savings** from stopped polling

## Future Enhancements

Potential improvements:
- [ ] Schedule notifications (auto-enable at event start time)
- [ ] Event-wide notification announcement before enabling
- [ ] Notification history/audit log
- [ ] Per-role enable/disable (e.g., disable for viewers only)
- [ ] API endpoint to toggle remotely
- [ ] Webhook support when status changes

## Troubleshooting

**Notifications not appearing despite being enabled**
1. Check browser console for errors
2. Verify event ID is correct
3. Test with "Test Live System" button
4. Check network tab for API responses

**Toggle not saving**
1. Check database connection
2. Verify event_notification_settings table exists
3. Run migration SQL if needed
4. Check browser console for errors

**Users still seeing notifications when disabled**
1. Hard refresh page (Ctrl+F5)
2. Check settings in database directly
3. Verify API is returning disabled=true
4. Clear browser cache

## Related Documentation
- [NOTIFICATION_SYSTEM.md](./NOTIFICATION_SYSTEM.md) - Complete notification system docs
- [API_ARCHITECTURE.md](./API_ARCHITECTURE.md) - API design patterns
- [ENHANCED_FEATURES.md](./ENHANCED_FEATURES.md) - Feature overview

## Version History
- **v1.0** (2025-11-17): Initial implementation of global toggle
- Database schema updated
- API modified to check enabled status
- UI updated with prominent toggle
- Documentation created
