# MySQL Migration Guide - Strict Mode Compatibility

## Overview
When moving the StyleScore application and database to a new server with MySQL strict mode enabled, you may encounter compatibility issues. This guide helps you resolve them.

## Common Errors

### Error 1: BLOB/TEXT/JSON Default Value
```
SQL Error (1101): BLOB, TEXT, GEOMETRY or JSON column 'show_categories' can't have a default value
```

**Cause**: MySQL strict mode doesn't allow TEXT/BLOB/JSON columns to have non-NULL default values.

**Solution**: The column default has been changed from `DEFAULT '[]'` to `DEFAULT NULL`.

---

### Error 2: Invalid Default Value for DATE Column
```
SQL Error (1067): Invalid default value for 'created_at'
```

**Cause**: `DATE` column type cannot use `current_timestamp()` as default (it returns a datetime value).

**Solution**: Changed column type from `DATE` to `DATETIME`.

---

## Migration Steps

### Option 1: Fresh Installation (Recommended)
If you're setting up a new database:

```bash
mysql -u root -p < freestyle_db.sql
```

The `freestyle_db.sql` file has been updated with all fixes.

---

### Option 2: Existing Database Migration
If you're migrating an existing database:

**Step 1: Backup your database**
```bash
mysqldump -u root -p freestyle_db > backup_before_migration.sql
```

**Step 2: Apply fixes**
```bash
mysql -u root -p freestyle_db < fix_mysql_strict_mode.sql
```

Or via PowerShell:
```powershell
Get-Content "fix_mysql_strict_mode.sql" | C:\xampp\mysql\bin\mysql.exe -u root -p freestyle_db
```

**Step 3: Verify**
The migration script includes optional verification queries (uncomment them in the SQL file).

---

## What Was Fixed

### 1. `event_notification_settings` table
- **Column**: `show_categories`
- **Change**: `DEFAULT '[]'` → `DEFAULT NULL`
- **Impact**: Existing NULL values are updated to `'[]'` for consistency

### 2. `participants` table
- **Column**: `created_at`
- **Change**: `date NOT NULL` → `datetime NOT NULL`
- **Impact**: No data loss, date values are preserved

### 3. Missing columns added
- `participants.photo` - varchar(255), stores photo file path
- `participants.motion_video` - varchar(255), stores video file path
- These are added safely (only if they don't already exist)

---

## PHP Application Changes

### Handling NULL vs Empty Array
Make sure your PHP code handles NULL values properly:

```php
// Before
$categories = json_decode($row['show_categories'], true);

// After (recommended)
$categories = json_decode($row['show_categories'] ?? '[]', true);
```

### Date Handling
The `created_at` change from DATE to DATETIME is backward compatible:

```php
// Both work the same
$created_date = date('Y-m-d', strtotime($row['created_at'])); // Just date
$created_datetime = $row['created_at']; // Full datetime
```

---

## Checking MySQL Strict Mode

To check if strict mode is enabled on your server:

```sql
SELECT @@sql_mode;
```

Look for `STRICT_TRANS_TABLES` or `STRICT_ALL_TABLES` in the output.

To temporarily disable (NOT recommended for production):
```sql
SET sql_mode = '';
```

---

## Troubleshooting

### Still getting errors after migration?
1. Make sure you ran the migration on the correct database
2. Check if there are custom tables not covered by the migration
3. Review MySQL error log for specific column/table names

### Migration script fails?
- Ensure you have ALTER TABLE privileges
- Check if columns already exist (script handles this)
- Verify database connection settings

---

## Files Modified

1. **freestyle_db.sql** - Updated base schema
2. **fix_mysql_strict_mode.sql** - Migration script for existing databases

---

## Support

For issues or questions, check:
- MySQL error logs
- PHP error logs
- Application logs in `/v2/logs/` (if applicable)

---

**Last Updated**: November 16, 2025
