# Judge Status Modal Overlay Implementation

## Overview
The judge status card now displays as a **modal overlay on top of all panels** when the status is set to "Pending". This ensures maximum visibility and prevents judges from missing the "Ready for next performance?" prompt.

## Features

### Modal Overlay Display
- **When Status = Pending**: Card appears as centered modal with semi-transparent backdrop
- **When Status ≠ Pending**: Card appears as normal inline element
- **Backdrop**: Semi-transparent dark overlay (50% black with 4px blur)
- **Centering**: Flexbox-based vertical and horizontal centering
- **Z-Index**: 10000 to ensure it appears above all other content
- **Animation**: Smooth fade-in for backdrop, slide-up for modal content

### Enhanced Pending State
When status is "Pending", the modal displays:
- **Large Icon**: 2rem clock icon
- **Prominent Message**: "Ready for next performance?" (1.5rem, bold)
- **Large READY Button**: Green, btn-lg with padding (primary action)
- **Options Button**: Gray btn-lg with ellipsis icon (secondary action)
- **Expanded Padding**: p-4 instead of p-2 for more breathing room
- **Shadow & Border**: Warning border and shadow for emphasis

### User Interaction
- **Click READY**: Sets status to "Ready", closes modal, shows 3-second toast
- **Click ... Button**: Expands collapse panel with all status options
- **Click Backdrop**: Toggles the collapse panel (convenience feature)
- **Click Inside Card**: No action (event.stopPropagation prevents backdrop click)

## Technical Implementation

### HTML Structure
```html
<div id="judge-status-modal-overlay" class="judge-status-modal-overlay" 
     onclick="event.target === this && toggleJudgeStatusCollapse()">
    <div class="judge-status-modal-content" onclick="event.stopPropagation()">
        <div class="card mb-0" id="judge-status-card">
            <!-- Card header with status display -->
            <!-- Collapsible section with status options -->
        </div>
    </div>
</div>
```

### CSS Classes

**Modal Overlay Container** (`.judge-status-modal-overlay`):
```css
position: fixed;
top: 0; left: 0; right: 0; bottom: 0;
background: rgba(0, 0, 0, 0.5);
backdrop-filter: blur(4px);
z-index: 10000;
display: none;
align-items: center;
justify-content: center;
animation: fadeIn 0.3s ease-out;
```

**Show State** (`.judge-status-modal-overlay.show`):
```css
display: flex;
```

**Modal Content** (`.judge-status-modal-content`):
```css
background: white;
border-radius: 12px;
box-shadow: 0 10px 40px rgba(0, 0, 0, 0.3);
max-width: 600px;
width: 90%;
animation: slideUpModal 0.3s ease-out;
```

**Dark Mode Support**:
- Content background: #2d3748
- Content text: #e2e8f0
- Overlay background: rgba(0, 0, 0, 0.7)

### JavaScript Functions

**updateJudgeStatusUI(status)**:
```javascript
const modalOverlay = document.getElementById('judge-status-modal-overlay');

if (status === 'pending') {
    // Show modal overlay
    modalOverlay.classList.add('show');
    
    // Enhanced pending state styling
    // ... (larger card, prominent message, big buttons)
} else {
    // Hide modal overlay
    modalOverlay.classList.remove('show');
    
    // Normal inline card styling
}
```

**toggleJudgeStatusCollapse()**:
```javascript
function toggleJudgeStatusCollapse() {
    if (judgeStatusCollapse) {
        judgeStatusCollapse.toggle();
    }
}
```

## Auto-Pending Triggers
When any of these events occur, status automatically switches to "Pending" (which triggers modal display):

1. **handleHeatChange()** - Manual heat selection change
2. **handleRunChange()** - Manual run selection change
3. **applyActiveHeatRun()** - Server-triggered heat/run updates
4. **Participant Select** - New participant selected from dropdown
5. **showHeatRunChangeOverlay()** - Heat/Run change notification displayed
6. **showParticipantSelectedOverlay()** - Participant selection notification displayed

## Workflow

### Typical Judge Workflow:
1. **Event Update** → Status auto-switches to "Pending"
2. **Modal Appears** → Centered overlay with "Ready for next performance?"
3. **Judge Reviews** → Context (heat, run, participant)
4. **Judge Clicks READY** → Modal disappears, status changes to "Ready"
5. **Judge Scores** → Performance with confidence in correct context

### Alternative Workflow:
1. **Modal Appears** (Pending state)
2. **Judge Clicks ...** → Collapse panel expands with all status options
3. **Judge Selects Different Status** → Modal disappears, new status shows inline
4. **Or Judge Clicks Backdrop** → Same as clicking ... button

## Browser Compatibility
- **Modern Browsers**: Full support (Chrome, Firefox, Edge, Safari)
- **Backdrop Blur**: Supported in all modern browsers
- **Animations**: CSS keyframe animations (widely supported)
- **Flexbox**: Full support for centering

## File Modified
- **judge/score.php** (8601 lines)
  - Lines 2063-2139: CSS for modal overlay + animations
  - Lines 2426-2467: HTML structure with modal wrapper
  - Lines 6458-6563: Updated updateJudgeStatusUI() with modal logic
  - Lines 6624-6630: Added toggleJudgeStatusCollapse() helper

## Testing Checklist
- [ ] Modal appears when status switches to Pending
- [ ] Modal disappears when status changes to Ready (or other)
- [ ] Backdrop shows semi-transparent dark overlay
- [ ] Card is centered vertically and horizontally
- [ ] READY button changes status and closes modal
- [ ] ... button expands collapse panel with options
- [ ] Backdrop click toggles collapse panel
- [ ] Animations play smoothly (fade-in, slide-up)
- [ ] Dark mode displays correctly
- [ ] Responsive on mobile devices
- [ ] Auto-pending triggers work (heat/run/participant changes)
- [ ] Toast notification shows for 3 seconds on status change

## Future Enhancements
- Add keyboard shortcut (e.g., Enter key) to click READY button
- Add escape key to close collapse panel
- Add sound notification when modal appears
- Add vibration on mobile devices when status changes to Pending
- Add configurable backdrop opacity in settings
