# Quick Start: HTML Table Publishing

## For Event Organizers

### Step 1: Create Summary Table Configuration

1. Go to **Admin → Results Configuration**
2. Create a new configuration:
   - Type: **Summary Table**
   - Status: **Public** ← Important!
   - Configure columns, filters, sorting as needed
3. Save the configuration

### Step 2: Publish to Server

1. Go to **Admin → Publish Servers**
2. Select your event
3. Select publish target (FTP/SFTP server)
4. Click **Publish Now**

HTML tables will be automatically generated and uploaded to the `tables/` directory.

### Step 3: Access HTML Tables

Your tables will be available at:
```
https://your-public-server.com/tables/summary_table_[ID].html
```

To find the ID, check the configuration list in **Results Configuration**.

---

## For Developers

### Quick Test

```php
<?php
require_once __DIR__ . '/../includes/db.php';
require_once __DIR__ . '/publish_generators.php';

$event_id = 3;
$temp_dir = sys_get_temp_dir() . '/test_html';
mkdir($temp_dir, 0755, true);

$files = generateSummaryTableHTMLs($event_id, $temp_dir, $pdo);

foreach ($files as $file) {
    echo "Generated: " . $file['remote'] . "\n";
    echo "Size: " . filesize($file['local']) . " bytes\n";
}
?>
```

### Direct API Call

```bash
# Generate HTML table directly from API
curl "http://localhost/v2/api/summary_table_api.php?event_id=3&format=html&styling=partial&show_rank=true&show_bib=true&show_participant=true" > table.html
```

---

## Embedding in External Websites

### Option 1: IFrame (Recommended)

```html
<iframe 
    src="https://your-server.com/tables/summary_table_89.html" 
    width="100%" 
    height="600" 
    frameborder="0"
    title="Event Results">
</iframe>
```

### Option 2: Direct Link

```html
<a href="https://your-server.com/tables/summary_table_89.html" 
   target="_blank">
    View Full Results
</a>
```

### Option 3: AJAX Load

```javascript
fetch('https://your-server.com/tables/summary_table_89.html')
    .then(response => response.text())
    .then(html => {
        document.getElementById('results-container').innerHTML = html;
    });
```

---

## Tips

### Multiple Tables on One Page

Create different configurations for different views:
- Config 74: Overall results
- Config 75: Heat 1 results only
- Config 76: Female participants only

Then embed multiple iframes or create a tabbed interface.

### Auto-Refresh for Live Events

Add to your embedding page:

```html
<script>
// Refresh every 30 seconds during live event
setInterval(function() {
    document.getElementById('results-iframe').src = 
        document.getElementById('results-iframe').src;
}, 30000);
</script>
```

### Custom Styling

The HTML includes basic Bootstrap styles. To override:

```html
<iframe id="results-iframe" src="..."></iframe>
<style>
    #results-iframe {
        border: 2px solid #0066cc;
        border-radius: 8px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    }
</style>
```

---

## Common Issues

**Problem:** Table shows "No data"
- **Solution:** Ensure event has participants and scores

**Problem:** HTML not generated
- **Solution:** Check configuration status is "Public"

**Problem:** Old data showing
- **Solution:** Republish the event to update HTML files

**Problem:** Styling looks different
- **Solution:** Check result styling configuration in admin

---

## Example Configuration JSON

For reference, here's what a configuration looks like:

```json
{
    "heat_number": [1, 2, 3, 4, 5, 6],
    "showRank": true,
    "showBib": true,
    "showParticipant": true,
    "showCategory": true,
    "showClub": true,
    "showGender": true,
    "showRuns": true,
    "showHeatBest": true,
    "heatDirection": "row",
    "sortBy": "sort_order",
    "sortDirection": "desc",
    "gender": "all",
    "columnRenames": {
        "participant": "Name",
        "heat_best": "Best.Rr"
    }
}
```

---

## Support

For issues or questions:
1. Check `HTML_TABLE_PUBLISHING.md` for detailed documentation
2. Run test: `http://localhost/v2/admin/test_html_table_generation.php`
3. Check PHP error logs
