# Heat Configuration Panel - Design Reference
**Modern Two-Pane Bootstrap 5 Interface**

## Overview
This document serves as a reference for the modern, professional panel design implemented in `admin/heats_configure.php`. Use this as a template for creating consistent, user-friendly admin panels throughout the application.

---

## Layout Architecture

### Two-Pane Responsive Grid
```html
<div class="row g-3">
    <!-- LEFT PANE: Source/Input Panel -->
    <div class="col-lg-5 col-xxl-4">
        <!-- Sticky header with filters/tools -->
        <!-- Scrollable content area -->
    </div>
    
    <!-- RIGHT PANE: Destination/Output Panel -->
    <div class="col-lg-7 col-xxl-8">
        <!-- Sticky header with filters/actions -->
        <!-- Scrollable content area -->
    </div>
</div>
```

**Breakpoints:**
- `col-lg-5/7`: Large screens (≥992px) - 5/7 split
- `col-xxl-4/8`: Extra large screens (≥1400px) - 4/8 split for more work space
- Below `lg`: Stack vertically (mobile-first)

---

## Component Structure

### 1. Workspace Metrics (Summary Cards)
```html
<div class="row g-2 mb-3">
    <div class="col-md-4">
        <div class="card border-0 shadow-sm h-100">
            <div class="card-body p-3">
                <div class="d-flex align-items-center">
                    <div class="flex-shrink-0">
                        <i class="fas fa-users fa-2x text-primary opacity-50"></i>
                    </div>
                    <div class="ms-3">
                        <div class="text-muted small">Available Participants</div>
                        <div class="h4 mb-0 fw-bold">
                            <span id="participantCount">57</span>
                            <small class="text-muted">visible</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <!-- Repeat for other metrics -->
</div>
```

**Purpose:** Provide at-a-glance overview of workspace state
**Icons:** FontAwesome with opacity-50 for subtle emphasis
**Colors:** Use Bootstrap contextual colors (primary, success, info, etc.)

---

### 2. Sticky Headers with Filters
```html
<div class="card-header bg-white sticky-top" style="top: 0; z-index: 100;">
    <h6 class="mb-2">
        <i class="fas fa-filter me-2"></i>Filter &amp; Search
    </h6>
    
    <!-- Filter Palette -->
    <div class="btn-group btn-group-sm mb-2" role="group">
        <button type="button" class="btn btn-outline-primary btn-sm" 
                onclick="selectAllVisible()">
            <i class="fas fa-check-square me-1"></i>Select All
        </button>
        <button type="button" class="btn btn-outline-secondary btn-sm" 
                onclick="clearSelection()">
            <i class="fas fa-times me-1"></i>Clear
        </button>
        <button type="button" class="btn btn-outline-danger btn-sm" 
                onclick="clearAllFiltersAndSearch()">
            <i class="fas fa-filter-circle-xmark me-1"></i>Reset
        </button>
    </div>
    
    <!-- Filter Dropdowns -->
    <div class="row g-2">
        <div class="col-md-6">
            <select class="form-select form-select-sm" id="categoryFilter" 
                    onchange="filterParticipants()">
                <option value="">All Categories</option>
                <!-- Options populated by PHP -->
            </select>
        </div>
        <!-- More filters -->
    </div>
    
    <!-- Search Bar -->
    <div class="input-group input-group-sm mt-2">
        <span class="input-group-text">
            <i class="fas fa-search"></i>
        </span>
        <input type="text" class="form-control" id="participantSearch" 
               placeholder="Search by name, BIB, club..." 
               oninput="searchParticipants()">
        <button class="btn btn-outline-secondary" type="button" 
                onclick="clearSearch()">
            <i class="fas fa-times"></i>
        </button>
    </div>
    
    <!-- Status Badge -->
    <div class="mt-2 text-center">
        <span class="badge bg-info" id="participantStatusBadge">0 selected</span>
    </div>
</div>
```

**Key Features:**
- `sticky-top` with `z-index: 100` keeps header visible while scrolling
- Icon-first design for quick recognition
- Compact `btn-sm` and `form-select-sm` for space efficiency
- Real-time status badges

---

### 3. Scrollable Content Area
```html
<div class="participant-display-container position-relative" 
     style="max-height: 50vh; overflow-y: auto;">
    <table class="table table-hover table-sm mb-0" id="participantsTable">
        <thead class="table-light sticky-top" style="top: 0;">
            <tr>
                <th width="40">
                    <input type="checkbox" id="selectAllParticipants" 
                           class="form-check-input">
                </th>
                <th width="60">BIB</th>
                <th>Name</th>
                <th>Category</th>
                <th width="50">Gender</th>
                <th width="60">Year</th>
                <th width="80">Status</th>
            </tr>
        </thead>
        <tbody>
            <!-- Data rows with data-* attributes for filtering -->
            <tr class="participant-row" 
                data-id="123"
                data-category="Adult"
                data-gender="M"
                data-club="Club Name"
                data-year="1995"
                data-search-text="doe john 45 club name">
                <!-- Row content -->
            </tr>
        </tbody>
    </table>
</div>
```

**Styling:**
```css
.participant-display-container {
    max-height: 50vh;
    overflow-y: auto;
}

.table thead.sticky-top {
    top: 0;
    z-index: 10;
}
```

---

### 4. Drag-and-Drop Integration (SortableJS)

#### Participant Table (Source - Clone Mode)
```javascript
const participantTable = document.querySelector('#participantsTable tbody');
new Sortable(participantTable, {
    group: {
        name: 'participants',
        pull: 'clone',    // Clone items (don't remove from source)
        put: false        // Don't accept drops back
    },
    sort: false,          // Disable sorting in source table
    animation: 150,
    draggable: '.participant-row',
    ghostClass: 'sortable-ghost',
    dragClass: 'sortable-drag',
    chosenClass: 'sortable-chosen',
    multiDrag: true,      // Enable multi-select drag
    selectedClass: 'selected-row',
    avoidImplicitDeselect: false,
    fallbackTolerance: 3,
    onStart: function(evt) {
        console.log('Drag started:', evt.item.dataset.id);
    },
    onEnd: function(evt) {
        // Remove cloned elements from source
        if (evt.item) evt.item.remove();
        if (evt.items) evt.items.forEach(item => item.remove());
    }
});
```

#### Heat Boards (Destination - Full Sorting)
```javascript
document.querySelectorAll('.sortable-tbody').forEach(tbody => {
    new Sortable(tbody, {
        group: {
            name: 'heat-assignments',
            pull: true,
            put: ['heat-assignments', 'participants']  // Accept from both sources
        },
        animation: 150,
        draggable: 'tr:not(.empty-heat-placeholder)',
        emptyInsertThreshold: 100,  // Allow drops into empty areas
        ghostClass: 'sortable-ghost',
        dragClass: 'sortable-drag',
        chosenClass: 'sortable-chosen',
        multiDrag: true,
        selectedClass: 'selected-row',
        onAdd: function(evt) {
            // Handle new participant assignment
            const participantId = evt.item.dataset.id;
            const heatNumber = evt.to.closest('.heat-board').dataset.heat;
            assignParticipantToHeat(participantId, heatNumber);
        }
    });
});
```

---

### 5. Action Button Toolbar
```html
<div class="p-3 border-bottom bg-white">
    <div class="d-flex flex-wrap gap-1 align-items-center">
        <button type="button" id="reverseSelectedBtn" 
                class="btn btn-outline-secondary btn-sm">
            <i class="fas fa-exchange-alt me-1"></i>Reverse Selected
        </button>
        <button type="button" id="shuffleSelectedBtn" 
                class="btn btn-outline-warning btn-sm">
            <i class="fas fa-random me-1"></i>Shuffle Selected
        </button>
        <button type="button" id="sortByNameBtn" 
                class="btn btn-outline-info btn-sm">
            <i class="fas fa-sort-alpha-down me-1"></i>Sort by Name
        </button>
        
        <div class="vr mx-1"></div>  <!-- Vertical divider -->
        
        <button type="button" class="btn btn-primary btn-sm" 
                data-bs-toggle="modal" data-bs-target="#assignbibModal">
            <i class="fas fa-hashtag me-1"></i>Assign BIB
        </button>
    </div>
</div>
```

**CSS for vertical divider:**
```css
.vr {
    width: 1px;
    height: 24px;
    background-color: #dee2e6;
}
```

---

### 6. Collapsible Heat Boards
```html
<div class="accordion" id="heatAccordion">
    <div class="card heat-board mb-2" data-heat="1">
        <div class="card-header p-0" id="heading1">
            <button class="btn btn-link text-decoration-none w-100 text-start p-3 
                           d-flex justify-content-between align-items-center" 
                    type="button" data-bs-toggle="collapse" 
                    data-bs-target="#collapse1">
                <span>
                    <span class="badge heat-number-1 text-white me-2">Heat 1</span>
                    <strong>Heat Name</strong>
                    <span class="badge bg-secondary ms-2">5</span>  <!-- Count -->
                </span>
                <i class="fas fa-chevron-down"></i>
            </button>
        </div>
        
        <div id="collapse1" class="accordion-collapse collapse show" 
             data-bs-parent="#heatAccordion">
            <div class="card-body p-0">
                <table class="table table-sm table-hover mb-0">
                    <tbody class="sortable-tbody" data-heat="1">
                        <!-- Assignment rows or empty placeholder -->
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>
```

---

### 7. Bootstrap Modals (Generic)
```html
<!-- Generic Alert Modal -->
<div class="modal fade" id="genericAlertModal" tabindex="-1">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header border-0">
                <h5 class="modal-title">
                    <i class="fas fa-info-circle text-primary me-2" 
                       id="alertIcon"></i>
                    <span id="alertTitle">Notice</span>
                </h5>
                <button type="button" class="btn-close" 
                        data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="alertMessage">
                <!-- Dynamic message -->
            </div>
            <div class="modal-footer border-0">
                <button type="button" class="btn btn-primary" 
                        data-bs-dismiss="modal">OK</button>
            </div>
        </div>
    </div>
</div>

<!-- Generic Confirm Modal -->
<div class="modal fade" id="genericConfirmModal" tabindex="-1">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header border-0">
                <h5 class="modal-title">
                    <i class="fas fa-question-circle text-warning me-2" 
                       id="confirmIcon"></i>
                    <span id="confirmTitle">Confirm Action</span>
                </h5>
                <button type="button" class="btn-close" 
                        data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="confirmMessage">
                <!-- Dynamic message -->
            </div>
            <div class="modal-footer border-0">
                <button type="button" class="btn btn-secondary" 
                        data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" 
                        id="confirmButton">Confirm</button>
            </div>
        </div>
    </div>
</div>
```

**JavaScript Helpers:**
```javascript
function showAlert(message, title = 'Notice', icon = 'info-circle') {
    document.getElementById('alertMessage').textContent = message;
    document.getElementById('alertTitle').textContent = title;
    document.getElementById('alertIcon').className = `fas fa-${icon} text-primary me-2`;
    new bootstrap.Modal(document.getElementById('genericAlertModal')).show();
}

function showConfirm(message, callback, title = 'Confirm', icon = 'question-circle') {
    document.getElementById('confirmMessage').textContent = message;
    document.getElementById('confirmTitle').textContent = title;
    document.getElementById('confirmIcon').className = `fas fa-${icon} text-warning me-2`;
    
    const confirmBtn = document.getElementById('confirmButton');
    const modal = new bootstrap.Modal(document.getElementById('genericConfirmModal'));
    
    // Remove old listeners
    const newBtn = confirmBtn.cloneNode(true);
    confirmBtn.parentNode.replaceChild(newBtn, confirmBtn);
    
    // Add new listener
    newBtn.addEventListener('click', function() {
        modal.hide();
        callback();
    });
    
    modal.show();
}
```

---

## CSS Styling System

### 1. Drag-and-Drop Visual Feedback
```css
/* Participant row dragging */
#participantsTable tbody .participant-row {
    cursor: grab;
    transition: background-color 0.2s;
}

#participantsTable tbody .participant-row:hover {
    background-color: rgba(0, 123, 255, 0.05);
}

#participantsTable tbody .participant-row:active {
    cursor: grabbing;
}

#participantsTable tbody .participant-row.selected-row {
    background-color: #fff3cd !important;
    border-left: 3px solid #ffc107;
}

/* Drag handle styling */
.drag-handle {
    user-select: none;
}

.drag-handle .fa-grip-vertical {
    opacity: 0.3;
    transition: opacity 0.2s;
}

.participant-row:hover .drag-handle .fa-grip-vertical {
    opacity: 0.6;
}

.participant-row.selected-row .drag-handle .fa-grip-vertical {
    opacity: 1;
    color: #ffc107 !important;
}

/* SortableJS drag states */
.sortable-ghost {
    opacity: 0.4;
    background-color: #007bff !important;
    color: white !important;
}

.sortable-ghost td {
    background-color: #007bff !important;
    color: white !important;
    border-color: #0056b3 !important;
}

.sortable-drag {
    opacity: 1 !important;
    background-color: white !important;
    box-shadow: 0 5px 15px rgba(0, 0, 0, 0.3) !important;
    transform: rotate(3deg);
    cursor: grabbing !important;
}

.sortable-chosen {
    opacity: 0.8;
    background-color: #fff3cd;
    border-left: 3px solid #ffc107;
}

/* Multi-drag styling */
.sortable-selected {
    background-color: #fff3cd !important;
    border-left: 3px solid #ffc107 !important;
}
```

### 2. Heat Board Drop Zones
```css
.sortable-tbody {
    min-height: 100px;
    display: table-row-group;
    transition: background-color 0.3s;
    position: relative;
}

.sortable-tbody.sortable-drag-over {
    background-color: #e3f2fd;
}

.sortable-tbody.sortable-drag-over::before {
    content: "";
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    border: 3px dashed #007bff;
    pointer-events: none;
    z-index: 1;
}

/* Empty heat placeholder */
.empty-heat-placeholder {
    pointer-events: none;
    user-select: none;
    height: 100px;
}

.empty-heat-placeholder td {
    background-color: #f8f9fa !important;
    border: 2px dashed #dee2e6;
    height: 100px;
    vertical-align: middle;
}
```

### 3. Heat Number Color Coding (HSL Spectrum)
```css
.heat-number-1  { background-color: hsl(209, 100%, 50%) !important; }
.heat-number-2  { background-color: hsl(234, 100%, 50%) !important; }
.heat-number-3  { background-color: hsl(264, 100%, 50%) !important; }
.heat-number-4  { background-color: hsl(255, 100%, 50%) !important; }
.heat-number-5  { background-color: hsl(245, 100%, 50%) !important; }
.heat-number-6  { background-color: hsl(222, 100%, 50%) !important; }
.heat-number-7  { background-color: hsl(279, 100%, 50%) !important; }
.heat-number-8  { background-color: hsl(272, 100%, 50%) !important; }
.heat-number-9  { background-color: hsl(285, 100%, 50%) !important; }
.heat-number-10 { background-color: hsl(294, 100%, 50%) !important; }
/* Continue pattern... */
```

### 4. Animation Effects
```css
/* Shuffle animation */
.shuffled {
    animation: flash 0.5s ease-in-out;
}

@keyframes flash {
    0% { background-color: #fff176; }
    100% { background-color: transparent; }
}

/* Status badge pulse */
#participantStatusBadge.has-selection {
    background: linear-gradient(135deg, #28a745 0%, #20c997 100%) !important;
    animation: pulse 1s ease-in-out;
}

@keyframes pulse {
    0% { transform: scale(1); }
    50% { transform: scale(1.1); }
    100% { transform: scale(1); }
}
```

### 5. Responsive Scrollbars (Webkit)
```css
.participant-display-container::-webkit-scrollbar {
    width: 8px;
}

.participant-display-container::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 4px;
}

.participant-display-container::-webkit-scrollbar-thumb {
    background: linear-gradient(135deg, #6c757d 0%, #495057 100%);
    border-radius: 4px;
}

.participant-display-container::-webkit-scrollbar-thumb:hover {
    background: linear-gradient(135deg, #495057 0%, #343a40 100%);
}
```

---

## JavaScript Patterns

### 1. Global State Management
```javascript
// Global state for participant selection
let selectedParticipants = new Set();

function updateSelectionStatus() {
    const selectedCount = selectedParticipants.size;
    document.getElementById('selectedCount').textContent = selectedCount;
    
    const badge = document.getElementById('participantStatusBadge');
    if (selectedCount > 0) {
        badge.textContent = `${selectedCount} selected`;
        badge.className = 'badge bg-success ms-2';
    } else {
        badge.textContent = '0 selected';
        badge.className = 'badge bg-info ms-2';
    }
}
```

### 2. Filtering with data-* Attributes
```javascript
function filterParticipants() {
    const category = document.getElementById('categoryFilter').value;
    const gender = document.getElementById('genderFilter').value;
    const club = document.getElementById('clubFilter').value;
    const year = document.getElementById('yearFilter').value;
    
    const rows = document.querySelectorAll('#participantsTable tbody .participant-row');
    let visibleCount = 0;
    
    rows.forEach(row => {
        let show = true;
        
        if (category && row.dataset.category !== category) show = false;
        if (gender && row.dataset.gender !== gender) show = false;
        if (club && row.dataset.club !== club) show = false;
        if (year && row.dataset.year !== year) show = false;
        
        if (show) {
            row.classList.remove('d-none');
            visibleCount++;
        } else {
            row.classList.add('d-none');
            // Deselect hidden rows
            if (selectedParticipants.has(row.dataset.id)) {
                selectedParticipants.delete(row.dataset.id);
                row.classList.remove('selected-row');
            }
        }
    });
    
    updateSelectionStatus();
}
```

### 3. Search with Indexed Data
```javascript
function searchParticipants() {
    const searchTerm = document.getElementById('participantSearch').value.toLowerCase();
    const rows = document.querySelectorAll('#participantsTable tbody .participant-row');
    
    rows.forEach(row => {
        const searchText = row.dataset.searchText || '';  // Pre-indexed search string
        const matchesSearch = searchTerm === '' || searchText.includes(searchTerm);
        
        if (matchesSearch) {
            row.classList.remove('d-none');
        } else {
            row.classList.add('d-none');
        }
    });
}
```

### 4. Event Delegation (Avoid Inline Handlers)
```javascript
// Bad: onclick="toggleSelection(this)"
// Good: Event delegation

document.querySelector('#participantsTable tbody').addEventListener('click', function(e) {
    const row = e.target.closest('.participant-row');
    
    // Exclude checkboxes and drag handles
    if (row && !e.target.closest('input[type="checkbox"]') && 
        !e.target.closest('.drag-handle')) {
        toggleTableRowSelection(row);
    }
});
```

### 5. Keyboard Shortcuts
```javascript
document.addEventListener('keydown', function(e) {
    // Ctrl/Cmd + A to select all visible
    if ((e.ctrlKey || e.metaKey) && e.key === 'a' && 
        document.activeElement.id !== 'participantSearch') {
        e.preventDefault();
        selectAllVisible();
    }
    
    // Escape to clear selection
    if (e.key === 'Escape') {
        clearSelection();
    }
    
    // Ctrl/Cmd + F to focus search
    if ((e.ctrlKey || e.metaKey) && e.key === 'f') {
        e.preventDefault();
        document.getElementById('participantSearch').focus();
    }
});
```

---

## PHP Backend Patterns

### 1. Data Preparation with Metadata
```php
// Add search-optimized metadata to each participant
foreach ($participants as &$p) {
    $p['search_text'] = strtolower(
        $p['last_name'] . ' ' . 
        $p['first_name'] . ' ' . 
        $p['bib'] . ' ' . 
        ($p['club'] ?? '')
    );
}
```

### 2. Rendering with Data Attributes
```php
<?php foreach ($participants as $p): ?>
    <tr class="participant-row" 
        data-id="<?= $p['id'] ?>"
        data-category="<?= htmlspecialchars($p['category'] ?? '') ?>"
        data-gender="<?= htmlspecialchars($p['gender'] ?? '') ?>"
        data-club="<?= htmlspecialchars($p['club'] ?? '') ?>"
        data-year="<?= $p['date_of_birth'] ? date('Y', strtotime($p['date_of_birth'])) : '' ?>"
        data-search-text="<?= htmlspecialchars($p['search_text']) ?>">
        <!-- Row content -->
    </tr>
<?php endforeach; ?>
```

### 3. Filter Options from Data
```php
// Extract unique values for filter dropdowns
$categories = array_unique(array_filter(array_column($participants, 'category')));
$clubs = array_unique(array_filter(array_column($participants, 'club')));
$genders = array_unique(array_filter(array_column($participants, 'gender')));
$years = array_unique(array_filter(array_map(function($p) {
    return $p['date_of_birth'] ? date('Y', strtotime($p['date_of_birth'])) : null;
}, $participants)));

sort($categories);
sort($clubs);
sort($genders);
rsort($years);  // Newest first
```

---

## Design Principles

### 1. **Visual Hierarchy**
- Headers: Sticky, clear typography, icon-first
- Actions: Button groups with consistent sizing (`btn-sm`)
- Data: Table with hover states, clear row spacing
- Status: Contextual badges (success, info, warning, danger)

### 2. **Responsive Design**
- Mobile-first Bootstrap grid
- Sticky headers for table navigation
- `max-height: 50vh` for scrollable areas
- Collapsible sections for space efficiency

### 3. **User Feedback**
- Hover states on interactive elements
- Selection highlighting with left border
- Drag-and-drop ghost/drag classes
- Loading states on buttons
- Bootstrap modals replace browser alerts

### 4. **Performance**
- Event delegation over inline handlers
- CSS transitions for smooth animations
- Indexed search strings (`data-search-text`)
- Minimal DOM manipulation (add/remove classes)

### 5. **Accessibility**
- Semantic HTML5 elements
- ARIA labels where needed
- Keyboard navigation support
- Clear focus states
- Icon + text labels for clarity

---

## Dependencies

### Required Libraries
```html
<!-- Bootstrap 5.3+ -->
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<!-- FontAwesome 6+ -->
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">

<!-- SortableJS 1.15+ (for drag-drop) -->
<script src="../assets/js/vendor/Sortable.min.js"></script>

<!-- jQuery 3.6+ (legacy support only) -->
<script src="../assets/js/vendor/jquery-3.6.0.min.js"></script>
```

### Optional (If Needed)
- jQuery UI 1.13+ (for legacy table sorting)
- Chart.js (for data visualization)
- DataTables (for advanced table features)

---

## Checklist for New Panels

- [ ] Two-pane responsive layout (`col-lg-5/7` or `col-lg-4/8`)
- [ ] Workspace metrics cards at top
- [ ] Sticky headers with filters
- [ ] Scrollable content areas (`max-height: 50vh`)
- [ ] Data attributes for filtering (`data-category`, `data-gender`, etc.)
- [ ] Search functionality with indexed strings
- [ ] Multi-select support (checkbox + click)
- [ ] Bootstrap modals instead of `alert()`/`confirm()`
- [ ] Keyboard shortcuts (Ctrl+A, Escape, Ctrl+F)
- [ ] Drag-and-drop if applicable (SortableJS)
- [ ] Visual feedback (hover, selection, drag states)
- [ ] Action button toolbar with icons
- [ ] Responsive breakpoints tested
- [ ] Consistent spacing (Bootstrap `g-2`, `g-3`, `mb-2`, etc.)
- [ ] Icon-first design pattern
- [ ] Loading states on async actions

---

## Example Use Cases

1. **Participant Assignment Panel** ✅ (heats_configure.php)
2. **Score Entry Panel** - Similar two-pane: Judge selection → Score input
3. **Category Management** - Categories list → Category details/participants
4. **Report Configuration** - Report templates → Preview/settings
5. **Event Dashboard** - Events list → Event metrics/actions

---

## Notes

- Always use `htmlspecialchars()` for user data in PHP
- Prefer `classList.add/remove()` over direct style manipulation
- Use Bootstrap utility classes before custom CSS
- Test on mobile/tablet breakpoints
- Keep JavaScript vanilla unless jQuery is already required
- Document custom functions with JSDoc comments
- Use semantic variable names (`participantId` not `pid`)

---

**Last Updated:** November 17, 2025  
**Version:** 1.0  
**Author:** StyleScore Development Team
