# Heat Flow Bidirectional Sync - Documentation

## Overview
The heat flow system maintains synchronization between two interfaces:
1. **Visual Node Editor** - Drag-and-drop flow builder with connections
2. **Flow Control Tab** - Form-based configuration in heat settings modal

## Data Flow Architecture

### Database Storage
```sql
-- event_heat_settings table
flow_source_heat      TEXT      -- JSON array: [3, 4, 5] (multiple sources)
flow_type            ENUM       -- 'none', 'promotion', 'qualifying'
flow_participants_per_category INT
flow_position_range  VARCHAR
hue_offsets          TEXT      -- JSON object: {"3": 120, "4": 240}
```

### Connection Object Structure
```javascript
{
  from: 3,              // Source heat number
  to: 6,                // Destination heat number
  type: 'top',          // Flow type
  count: 5,             // Participants per category
  range: '1-3',         // Position range (optional)
  hue_offset: 120       // Color hue offset (0-360)
}
```

## Synchronization Flows

### 1. Node Editor → Form (syncConnectionsToForms)
**Trigger**: After creating/editing/deleting connections in visual editor

**Process**:
1. Group all connections by destination heat
2. Extract source heat numbers for each destination
3. Update multi-select `flow_source_heat[X][]` with selected sources
4. Clear selections for heats with no connections

**Example**:
```javascript
// If connections exist: Heat 3→6, Heat 4→6
connectionsByDest[6] = [
  {from: 3, to: 6, ...},
  {from: 4, to: 6, ...}
]

// Updates: select[name="flow_source_heat[6][]"]
// Selects options: value="3" and value="4"
```

### 2. Form → Node Editor (change event listener)
**Trigger**: When user changes source heat multi-select in Flow Control Tab

**Process**:
1. Extract heat number from select name: `flow_source_heat[6][]` → heat 6
2. Get currently selected source heats from select options
3. **Preserve existing connection data** (hue_offset, type, count, range)
4. Remove all connections to this heat
5. Re-create connections with preserved or default values
6. Re-render visual connections

**Example**:
```javascript
// User adds Heat 5 to existing Heat 3→6, Heat 4→6
// Before: [3, 4] selected
// After: [3, 4, 5] selected

// System:
// 1. Preserves hue_offset from Heat 3→6 and Heat 4→6
// 2. Removes old connections
// 3. Creates new: Heat 3→6 (preserved hue), Heat 4→6 (preserved hue), Heat 5→6 (default hue=0)
```

## Multiple Source Heats Support

### Scenario: Heat 6 with Two Sources
```
Heat 3 ──┐
         ├──> Heat 6
Heat 4 ──┘
```

### Database Representation
```json
{
  "heat_number": 6,
  "flow_source_heat": "[3, 4]",
  "flow_type": "top",
  "flow_participants_per_category": 5,
  "hue_offsets": "{\"3\": 120, \"4\": 240}"
}
```

### Visual Editor
- Two separate connection lines visible
- Each line can have different color (hue_offset)
- Clicking gear icon on each line shows that connection's config

### Flow Control Tab
```html
<select name="flow_source_heat[6][]" multiple>
  <option value="3" selected>Heat 3</option>
  <option value="4" selected>Heat 4</option>
  <option value="5">Heat 5</option>
</select>
```

## Hue Offset Storage

### Why JSON Object?
Multiple connections to same heat need individual colors:
- Heat 3→6 might need blue (hue: 0)
- Heat 4→6 might need red (hue: 120)
- Heat 5→6 might need green (hue: 240)

### Storage Format
```json
{
  "3": 120,  // Heat 3 connection uses 120° hue offset
  "4": 240,  // Heat 4 connection uses 240° hue offset
  "5": 0     // Heat 5 connection uses default colors
}
```

### Loading Process
```php
// PHP - Building connections from database
$hue_offsets_map = json_decode($heat['hue_offsets'] ?? '{}', true) ?: [];
foreach ($source_heats as $source) {
    $connections[] = [
        'from' => (int)$source,
        'to' => (int)$heat['heat_number'],
        'hue_offset' => $hue_offsets_map[$source] ?? 0
    ];
}
```

### Saving Process
```javascript
// JavaScript - Building hue_offsets JSON for this heat
const hueOffsetsForHeat = {};
flowEditor.connections.filter(c => c.to === h).forEach(conn => {
    hueOffsetsForHeat[conn.from] = conn.hue_offset || 0;
});
formData.append('hue_offsets[' + h + ']', JSON.stringify(hueOffsetsForHeat));
```

## User Workflows

### Workflow 1: Create Connections in Visual Editor
1. User clicks output port on Heat 3
2. User clicks input port on Heat 6
3. Connection created: Heat 3→6
4. `syncConnectionsToForms()` updates Flow Control Tab
5. Flow Control Tab now shows Heat 3 selected in multi-select

### Workflow 2: Add Source in Flow Control Tab
1. User opens Heat 6 modal → Flow Control tab
2. Multi-select shows Heat 3 already selected
3. User Ctrl+Click to add Heat 4
4. Change event listener fires
5. Visual editor updates to show both connections
6. Both connections visible in node editor

### Workflow 3: Customize Connection Color
1. User clicks gear icon on Heat 3→6 connection
2. Adjusts hue slider to 120°
3. Clicks "Save Configuration"
4. Connection color updates immediately
5. User clicks "Save Flow"
6. hue_offsets saved as: `{"3": 120}`

### Workflow 4: Edit After Page Refresh
1. Page loads
2. PHP builds connections from database with hue_offsets
3. Visual editor renders connections with correct colors
4. Flow Control Tab shows correct selected sources
5. User edits in either interface
6. Both interfaces stay synchronized

## Preservation Logic

### What Gets Preserved?
When syncing from Form→Visual, existing connection data is preserved:
- ✅ `hue_offset` - Custom colors
- ✅ `type` - Flow type (top/promotion/qualifying)
- ✅ `count` - Participants per category
- ✅ `range` - Position range

### Why Preserve?
Without preservation:
```
1. User sets Heat 3→6 with hue=120 (red color)
2. User adds Heat 4→6 in Flow Control Tab
3. BUG: Heat 3→6 loses red color, resets to default
```

With preservation:
```
1. User sets Heat 3→6 with hue=120 (red color)
2. User adds Heat 4→6 in Flow Control Tab
3. ✓ Heat 3→6 keeps red color (hue=120)
4. ✓ Heat 4→6 gets default color (hue=0)
```

## Debugging Tips

### Check Connection State
```javascript
// In browser console
console.log(flowEditor.connections);

// Expected output for Heat 3→6, Heat 4→6:
[
  {from: 3, to: 6, type: 'top', count: 5, range: null, hue_offset: 120},
  {from: 4, to: 6, type: 'top', count: 5, range: null, hue_offset: 240}
]
```

### Check Form State
```javascript
// Get selected source heats for Heat 6
const select = document.querySelector('select[name="flow_source_heat[6][]"]');
const selected = Array.from(select.selectedOptions).map(o => o.value);
console.log(selected); // Expected: ["3", "4"]
```

### Check Database State
```sql
SELECT 
    heat_number, 
    flow_source_heat, 
    hue_offsets 
FROM event_heat_settings 
WHERE event_id = 1 AND heat_number = 6;

-- Expected result:
-- heat_number: 6
-- flow_source_heat: [3,4]
-- hue_offsets: {"3":120,"4":240}
```

## Common Issues & Solutions

### Issue 1: Connections Disappear After Refresh
**Cause**: Not saving to database
**Solution**: Click "Save Flow" button after making changes in visual editor

### Issue 2: Colors Reset After Adding Source
**Cause**: Missing preservation logic
**Solution**: Already fixed - preservation implemented in change listener

### Issue 3: Form Doesn't Update After Visual Edit
**Cause**: `syncConnectionsToForms()` not called
**Solution**: Ensure it's called after `renderConnections()`

### Issue 4: Only Last Source Saved
**Cause**: Using single value instead of array
**Solution**: Use `flow_source_heat[X][]` (note the `[]` for array)

## Best Practices

### For Users
1. ✅ Use visual editor for quick connection creation
2. ✅ Use Flow Control Tab for precise configuration
3. ✅ Always click "Save Flow" to persist changes
4. ✅ Refresh page to verify persistence

### For Developers
1. ✅ Always preserve existing connection data when syncing
2. ✅ Store hue_offsets as JSON object (source_heat => hue)
3. ✅ Call `syncConnectionsToForms()` after connection changes
4. ✅ Call `renderConnections()` after form changes
5. ✅ Test both directions: Visual→Form and Form→Visual
