# Heat-Specific Drop Rule Override Feature

## Overview
This feature allows each heat within an event to override the event's default drop rule configuration for judge scores. This provides granular control over how scores are calculated in different phases of competition.

## Database Changes

### New Column: `event_heat_settings.drop_rule`
- **Type**: VARCHAR(100)
- **Default**: NULL
- **Location**: After `precision_decimal` column
- **Purpose**: Store heat-specific drop rule override

#### Valid Values:
- `NULL` - Use event's default drop_rule from scoring_format_judges (recommended for most cases)
- `'none'` - Use all judge scores, no drops
- `'lowest'` - Drop the lowest judge score
- `'highest'` - Drop the highest judge score  
- `'highest_and_lowest'` - Drop both highest and lowest scores

### Migration Files:
1. **SQL File**: `add_drop_rule_to_heat_settings.sql`
   - Pure SQL ALTER TABLE statement
   - Can be run directly in MySQL/phpMyAdmin

2. **PHP Migration**: `add_drop_rule_to_heat_settings_migration.php`
   - Checks if column exists before adding
   - Provides success/error feedback
   - Safe to run multiple times

3. **Test Script**: `test_add_drop_rule_column.php`
   - Quick verification script
   - Tests database connection and column creation

## UI Changes

### Admin Interface: `admin/admin_heat_config.php`

#### Location: Format Settings Tab
The drop_rule control is added in the "Format Settings" tab within each heat's configuration modal, positioned after the Score Min/Max/Precision fields.

#### Control Details:
```html
<label class="form-label fw-semibold">
    <i class="fas fa-filter me-1"></i>Judge Score Drop Rule
</label>
<select class="form-select" name="drop_rule[<?= $h ?>]">
    <option value="">-- Use Event Default --</option>
    <option value="none">None - Use All Judge Scores</option>
    <option value="lowest">Drop Lowest Score</option>
    <option value="highest">Drop Highest Score</option>
    <option value="highest_and_lowest">Drop Highest & Lowest Scores</option>
</select>
```

#### Help Text:
> Override which judge scores to exclude when calculating run scores. Leave as "Use Event Default" to inherit from event format settings.

## Backend Implementation

### Data Flow:

1. **Form Submission** (`admin/admin_heat_config.php` lines 48-56)
   ```php
   // Handle drop_rule setting (NULL means use event default)
   $drop_rule = isset($_POST['drop_rule'][$heat_number]) ? 
                trim($_POST['drop_rule'][$heat_number]) : null;
   $drop_rule = ($drop_rule === '') ? null : $drop_rule;
   ```

2. **Database Insert/Update** (lines 154-209)
   - Added `drop_rule` to INSERT column list
   - Added to VALUES list (position 30)
   - Added to ON DUPLICATE KEY UPDATE clause

3. **Data Loading** (lines 360-368)
   ```php
   SELECT heat_number, heat_name, ..., drop_rule
   FROM event_heat_settings
   ```

4. **Default Settings** (lines 1565-1581)
   ```php
   'drop_rule' => null // NULL means use event default
   ```

## Usage Examples

### Scenario 1: Qualification Heat with Stricter Judging
**Event Default**: `drop_rule = 'highest_and_lowest'` (drops extremes)
**Heat 1 Override**: `drop_rule = 'none'` (uses all scores for qualification)

**Result**: Qualification uses all judges' scores for fairness, while finals use event default.

### Scenario 2: Finals with More Conservative Scoring
**Event Default**: `drop_rule = 'none'` (all scores count)
**Heat 3 (Finals) Override**: `drop_rule = 'highest_and_lowest'` (drop extremes)

**Result**: Finals have more conservative scoring by removing outliers.

### Scenario 3: Use Event Default (Most Common)
**Event Default**: `drop_rule = 'lowest'`
**All Heats**: `drop_rule = NULL` (empty/default selection)

**Result**: All heats use the event's drop_rule setting uniformly.

## Integration with Scoring System

### Judge Panel Integration
The heat-specific drop_rule should be used by scoring APIs and judge panels:

1. **head_judge.php** - Should check heat's drop_rule first, fall back to event default
2. **judge_score_api.php** - Pass heat's drop_rule to calculation functions
3. **public_dashboard_api.php** - Display correct drop rule status per heat

### Recommended Query Pattern:
```php
// Get drop_rule with heat override support
$stmt = $pdo->prepare("
    SELECT 
        COALESCE(ehs.drop_rule, sfj.drop_rule, 'none') as effective_drop_rule,
        ehs.drop_rule as heat_override,
        sfj.drop_rule as event_default
    FROM event_heat_settings ehs
    JOIN events e ON ehs.event_id = e.id
    LEFT JOIN scoring_format_judges sfj ON e.scoring_format = sfj.format_id
    WHERE ehs.event_id = ? AND ehs.heat_number = ?
");
```

**Logic**:
- Use `ehs.drop_rule` if set (heat override)
- Else use `sfj.drop_rule` (event default)
- Else default to 'none' (safety fallback)

## User Interface Behavior

### Visual Feedback:
- **Default Selection**: "-- Use Event Default --" is pre-selected
- **Empty Value**: Treated as NULL (use event default)
- **Icon**: Filter icon (<i class="fas fa-filter">) indicates scoring modification

### Validation:
- No validation required - all options are valid
- Empty/NULL values automatically fall back to event default
- Invalid values ignored, treated as NULL

## Testing Checklist

### Database:
- [ ] Run migration script successfully
- [ ] Verify column exists: `SHOW COLUMNS FROM event_heat_settings LIKE 'drop_rule'`
- [ ] Check column allows NULL
- [ ] Verify data type VARCHAR(100)

### UI:
- [ ] Open heat configuration modal
- [ ] Navigate to "Format Settings" tab
- [ ] Verify drop_rule selector displays after precision field
- [ ] Test selecting each option
- [ ] Verify "Use Event Default" is pre-selected for new heats
- [ ] Save and reload - verify selection persists

### Backend:
- [ ] Save heat with drop_rule = NULL (use default)
- [ ] Save heat with drop_rule = 'none'
- [ ] Save heat with drop_rule = 'lowest'
- [ ] Query database to verify saved values
- [ ] Verify heat settings load correctly with drop_rule field

### Integration:
- [ ] Judge panels respect heat-specific drop_rule
- [ ] Dashboard displays correct drop rule status
- [ ] Score calculations use proper drop rule per heat
- [ ] Report cards show accurate dropped score indicators

## Troubleshooting

### Column Already Exists Error
If you see "Duplicate column name 'drop_rule'":
- Column was already added
- No action needed
- Safe to proceed

### NULL vs Empty String
- UI sends empty string `""` for "Use Event Default"
- Backend converts `""` to `NULL`
- Database stores `NULL` for default behavior

### Scoring Mismatch
If scores don't match expected drop rule:
1. Check heat's `drop_rule` value in database
2. Verify event's `scoring_format_judges.drop_rule`
3. Confirm scoring API uses COALESCE logic
4. Review judge panel calculation function

## Future Enhancements

### Potential Additions:
1. **Visual Indicator**: Show drop rule status in heat cards/buttons
2. **Bulk Edit**: Set drop_rule for multiple heats at once
3. **Preview**: Show example calculation with current drop rule
4. **History**: Track drop_rule changes per heat
5. **Validation**: Warn if heat drop_rule conflicts with run count

### API Endpoints to Update:
- `judge_score_api.php` - Include heat drop_rule in response
- `public_dashboard_api.php` - Show heat-specific drop rule status
- `heat_cards_api.php` - Display drop rule badge on cards

## Related Files

### Modified:
- `admin/admin_heat_config.php` - UI and backend logic
- Database schema: `event_heat_settings` table

### Created:
- `add_drop_rule_to_heat_settings.sql` - SQL migration
- `add_drop_rule_to_heat_settings_migration.php` - PHP migration
- `test_add_drop_rule_column.php` - Quick test script
- `HEAT_DROP_RULE_FEATURE.md` - This documentation

### To Update (recommended):
- `judge/head_judge.php` - Use heat drop_rule
- `api/judge_score_api.php` - Pass heat drop_rule
- `api/public_dashboard_api.php` - Display heat drop_rule
- `admin/heat_cards_api.php` - Show drop_rule indicator

## Summary

This feature provides heat-level granularity for drop rule configuration while maintaining backward compatibility through NULL default values. Heats without explicit drop_rule settings automatically inherit the event's default behavior, ensuring smooth operation during transition and deployment.

**Key Benefits**:
- ✅ Flexible per-heat scoring rules
- ✅ Backward compatible (NULL = use default)
- ✅ Clear UI in Format Settings tab
- ✅ Simple database schema addition
- ✅ Easy to understand and maintain
