# Head Judge Panel - Heat Drop Rule Integration

## Summary
Updated `judge/head_judge.php` to support heat-specific drop_rule overrides with priority over event format settings.

## Changes Made

### 1. Database Query Enhancement (Lines 457-472)

**Before:**
```php
// Simple query fetching only format drop_rule
$format_stmt = $pdo->prepare("
    SELECT sfj.drop_rule
    FROM events e
    JOIN scoring_format_judges sfj ON e.scoring_format = sfj.format_id
    WHERE e.id = ?
    LIMIT 1
");
$format_stmt->execute([$event_id]);
```

**After:**
```php
// Query with heat override support using COALESCE
$format_stmt = $pdo->prepare("
    SELECT 
        ehs.drop_rule as heat_drop_rule,
        sfj.drop_rule as format_drop_rule,
        COALESCE(ehs.drop_rule, sfj.drop_rule, 'none') as effective_drop_rule
    FROM events e
    LEFT JOIN event_heat_settings ehs ON e.id = ehs.event_id AND ehs.heat_number = ?
    LEFT JOIN scoring_format_judges sfj ON e.scoring_format = sfj.format_id
    WHERE e.id = ?
    LIMIT 1
");
$format_stmt->execute([$heat_number, $event_id]);
```

**Key Changes:**
- Added `LEFT JOIN event_heat_settings` to check for heat-specific drop_rule
- Uses `COALESCE()` to implement priority: Heat → Format → 'none'
- Returns all three values: `heat_drop_rule`, `format_drop_rule`, `effective_drop_rule`
- Added `$heat_number` as first parameter in execute()

### 2. API Response Enhancement (Lines 472-484)

**Before:**
```php
echo json_encode([
    'success' => true,
    'judges' => $judges,
    'participants' => $participants,
    'scores' => $scores,
    'heat_number' => $heat_number,
    'run_number' => $run_number,
    'drop_rule' => $drop_rule
]);
```

**After:**
```php
$drop_rule = $format_config ? $format_config['effective_drop_rule'] : 'none';
$heat_drop_rule = $format_config ? $format_config['heat_drop_rule'] : null;
$format_drop_rule = $format_config ? $format_config['format_drop_rule'] : null;

echo json_encode([
    'success' => true,
    'judges' => $judges,
    'participants' => $participants,
    'scores' => $scores,
    'heat_number' => $heat_number,
    'run_number' => $run_number,
    'drop_rule' => $drop_rule,
    'drop_rule_source' => $heat_drop_rule ? 'heat' : 'format',
    'heat_drop_rule' => $heat_drop_rule,
    'format_drop_rule' => $format_drop_rule
]);
```

**New Fields:**
- `drop_rule_source`: 'heat' or 'format' - indicates which setting is active
- `heat_drop_rule`: The heat-specific override (NULL if not set)
- `format_drop_rule`: The event format default
- `drop_rule`: The effective value being used (unchanged for backward compatibility)

### 3. Visual Display Enhancement (Lines 2447-2466)

**Before:**
```javascript
html += `
    <div class="alert alert-info mb-3">
        <strong><i class="fas fa-info-circle me-1"></i>Drop Rule Configuration:</strong>
        <span class="badge bg-primary ms-2">${data.drop_rule || 'none'}</span>
        ${/* description */}
    </div>
`;
```

**After:**
```javascript
const dropRuleSource = data.drop_rule_source || 'format';
const isHeatOverride = dropRuleSource === 'heat';

html += `
    <div class="alert alert-info mb-3">
        <strong><i class="fas fa-info-circle me-1"></i>Drop Rule Configuration:</strong>
        <span class="badge bg-primary ms-2">${data.drop_rule || 'none'}</span>
        ${isHeatOverride ? 
            '<span class="badge bg-warning text-dark ms-1"><i class="fas fa-fire me-1"></i>Heat Override</span>' : 
            '<span class="badge bg-secondary ms-1"><i class="fas fa-cog me-1"></i>Format Default</span>'}
        ${/* description */}
    </div>
`;
```

**Visual Indicators:**
- **Heat Override**: Yellow badge with fire icon (<i class="fas fa-fire">)
- **Format Default**: Gray badge with cog icon (<i class="fas fa-cog">)

## Priority Logic

```
Priority Chain:
1. event_heat_settings.drop_rule (Heat-specific override)
   ↓ If NULL
2. scoring_format_judges.drop_rule (Event format default)
   ↓ If NULL
3. 'none' (Fallback default)
```

## Example Scenarios

### Scenario 1: Heat Has Override
**Database:**
- `event_heat_settings.drop_rule` = 'lowest' (Heat 1)
- `scoring_format_judges.drop_rule` = 'highest_and_lowest'

**Result:**
- Effective drop_rule: 'lowest'
- Display: Primary badge "lowest" + Yellow badge "Heat Override"

### Scenario 2: Heat Uses Format Default
**Database:**
- `event_heat_settings.drop_rule` = NULL (Heat 1)
- `scoring_format_judges.drop_rule` = 'highest_and_lowest'

**Result:**
- Effective drop_rule: 'highest_and_lowest'
- Display: Primary badge "highest_and_lowest" + Gray badge "Format Default"

### Scenario 3: No Format, No Heat Override
**Database:**
- `event_heat_settings.drop_rule` = NULL
- `scoring_format_judges.drop_rule` = NULL

**Result:**
- Effective drop_rule: 'none'
- Display: Primary badge "none" + Gray badge "Format Default"

## Testing

### Manual Test Steps:
1. Open head judge panel
2. Select an event with heat settings
3. Select a heat with drop_rule override
4. Verify yellow "Heat Override" badge appears
5. Select a heat without override
6. Verify gray "Format Default" badge appears
7. Check that scores are dropped according to effective rule

### Database Test Queries:

**Check heat override:**
```sql
SELECT 
    ehs.heat_number,
    ehs.drop_rule as heat_override,
    sfj.drop_rule as format_default,
    COALESCE(ehs.drop_rule, sfj.drop_rule, 'none') as effective
FROM event_heat_settings ehs
JOIN events e ON ehs.event_id = e.id
LEFT JOIN scoring_format_judges sfj ON e.scoring_format = sfj.format_id
WHERE e.id = 3;
```

**Set heat override:**
```sql
UPDATE event_heat_settings 
SET drop_rule = 'lowest' 
WHERE event_id = 3 AND heat_number = 1;
```

**Remove heat override:**
```sql
UPDATE event_heat_settings 
SET drop_rule = NULL 
WHERE event_id = 3 AND heat_number = 1;
```

## Backward Compatibility

✅ **Fully Backward Compatible**
- Heats without `drop_rule` column: Query handles NULL gracefully
- Old API consumers: `drop_rule` field still present with same value
- New fields (`drop_rule_source`, etc.) are additive, not breaking

## Integration Points

This change integrates with:
1. ✅ `admin/admin_heat_config.php` - Heat configuration UI (already updated)
2. ✅ `judge/head_judge.php` - Head judge panel (this file)
3. ⚠️ `judge/score_panel.html` - Judge scoring panel (may need update)
4. ⚠️ `api/public_dashboard_api.php` - Public dashboard (may need update)
5. ⚠️ `api/summary_table_api.php` - Results tables (may need update)

## Performance Notes

- Added LEFT JOIN has minimal performance impact
- COALESCE() is efficient for null handling
- Query still uses indexes on event_id
- Returns single row per request (LIMIT 1)

## Related Files Modified

1. `judge/head_judge.php` - Query, API response, visual display
2. `admin/admin_heat_config.php` - Heat configuration UI (separate update)

## Documentation

See `HEAT_DROP_RULE_FEATURE.md` for complete feature documentation including:
- Database schema
- UI screenshots
- Usage examples
- API integration patterns
