# Flow Editor Node Position Persistence

## Overview
Node positions in the flow editor are now persisted to the database to prevent loss of manual layout arrangements.

## Database Changes

### New Columns in `event_heat_settings`
- `layout_x` (INT, NULL) - Flow editor node X position
- `layout_y` (INT, NULL) - Flow editor node Y position

## Migration

Run the migration by accessing:
```
http://localhost/v2/run_layout_migration.php
```

Or execute the SQL file directly:
```sql
SOURCE add_layout_positions_to_heat_settings.sql;
```

## Implementation Details

### Position Loading Priority
1. **Database positions** (highest priority) - Positions saved in `event_heat_settings.layout_x` and `layout_y`
2. **localStorage** (fallback) - Browser-local positions from previous sessions
3. **Auto-arrange** (default) - Grid layout if no saved positions exist

### Position Saving
Node positions are automatically saved when:
- User clicks "Save Flow Configuration" button
- Positions are saved alongside flow settings (connections, types, etc.)

### Code Changes

#### PHP (admin_heat_config.php)
```php
// Heat nodes now include layout positions from database
$heat_nodes[] = [
    'heat_number' => (int)$heat['heat_number'],
    // ... other fields ...
    'layout_x' => $heat['layout_x'] ?? null,
    'layout_y' => $heat['layout_y'] ?? null,
];

// POST handler saves positions
$layout_x = isset($_POST['layout_x'][$h]) ? intval($_POST['layout_x'][$h]) : null;
$layout_y = isset($_POST['layout_y'][$h]) ? intval($_POST['layout_y'][$h]) : null;

UPDATE event_heat_settings 
SET layout_x = ?, layout_y = ? 
WHERE event_id = ? AND heat_number = ?
```

#### JavaScript
```javascript
// loadNodePositions() checks database first
function loadNodePositions() {
    // First: Check database positions
    const dbPositions = {};
    let hasDbPositions = false;
    heatNodesData.forEach(node => {
        if (node.layout_x !== null && node.layout_y !== null) {
            dbPositions[node.heat_number] = {x: node.layout_x, y: node.layout_y};
            hasDbPositions = true;
        }
    });
    if (hasDbPositions) return dbPositions;
    
    // Second: Check localStorage
    const saved = localStorage.getItem('heatFlowNodePositions_' + eventId);
    if (saved) return JSON.parse(saved);
    
    // Third: Return null for auto-arrange
    return null;
}

// saveFlowConfiguration() includes positions
for (let h = 1; h <= heatsTotal; h++) {
    const nodeData = flowEditor.nodes.get(h);
    if (nodeData) {
        formData.append('layout_x[' + h + ']', Math.round(nodeData.x));
        formData.append('layout_y[' + h + ']', Math.round(nodeData.y));
    }
}
```

## Benefits

### Before
- Node positions stored only in browser localStorage
- Lost when switching browsers/devices
- Lost when clearing browser cache
- Not shared between team members

### After
- Positions persist in database
- Shared across all users/browsers/devices
- Survive browser cache clears
- Part of event configuration backup
- More reliable and professional

## User Workflow

1. User arranges flow editor nodes visually
2. User clicks "Save Flow Configuration" button
3. System saves:
   - Flow connections (source heats, types, counts)
   - Node positions (X, Y coordinates)
4. On next page load:
   - Positions load from database automatically
   - Nodes appear exactly where they were placed

## Backwards Compatibility

- Migration is optional (columns default to NULL)
- System falls back to localStorage if database positions not set
- Existing localStorage positions continue to work
- No user action required after migration

## Testing

To verify the implementation:
1. Run migration: `run_layout_migration.php`
2. Open heat configuration page
3. Arrange nodes in flow editor
4. Click "Save Flow Configuration"
5. Refresh page - positions should persist
6. Open in different browser - same positions
7. Check database: `SELECT heat_number, layout_x, layout_y FROM event_heat_settings WHERE event_id = X;`

## Files Modified

- `admin/admin_heat_config.php` - Main heat configuration page
  - POST handler updated to save positions
  - Heat nodes data includes layout_x/layout_y
  - JavaScript loadNodePositions() prioritizes database
  - JavaScript saveFlowConfiguration() includes positions

## Files Created

- `add_layout_positions_to_heat_settings.sql` - SQL migration script
- `add_layout_positions_migration.php` - CLI migration runner
- `run_layout_migration.php` - Web-based migration interface
- `FLOW_EDITOR_POSITIONS.md` - This documentation file
