# Event Metadata System - Field-Based Flexible Structure

## Overview
The event metadata system now supports **flexible field ordering and grouping**, allowing you to add custom fields anywhere in any group and control the exact display order.

## Architecture

### Data Structure
```json
{
  "metadata_fields": [
    {
      "group": "Jury Information",
      "field_key": "technical_delegate_fis",
      "field_label": "Technical Delegate FIS",
      "field_value": "John Doe",
      "field_order": 1,
      "is_system": true
    },
    {
      "group": "Jury Information",
      "field_key": "chief_observer",
      "field_label": "Chief Observer",
      "field_value": "Jane Smith",
      "field_order": 2.5,
      "is_system": false
    }
  ],
  "judges": []
}
```

### Field Properties
- **group**: Category/section name (e.g., "Jury Information", "Technical Data")
- **field_key**: Unique identifier (e.g., "technical_delegate_fis")
- **field_label**: Display name (e.g., "Technical Delegate FIS")
- **field_value**: Actual data value
- **field_order**: Sort order (supports decimals for precise positioning)
- **is_system**: Protected field flag (true = cannot rename/delete)

## Key Features

### 1. Flexible Field Ordering
**Problem**: Need to add "Chief Observer" between "Chief of Competition" (order 2) and "Head Judge" (order 3)

**Solution**: Set `field_order: 2.5`

```json
{
  "field_order": 2,
  "field_label": "Chief of Competition"
},
{
  "field_order": 2.5,
  "field_label": "Chief Observer"
},
{
  "field_order": 3,
  "field_label": "Head Judge"
}
```

### 2. Custom Fields in Existing Groups
You can add your own fields to system groups:

**Example**: Add "Assistant TD" to Jury Information
- Group: "Jury Information"
- Label: "Assistant Technical Delegate"
- Order: 1.5 (to appear after TD but before Chief)
- System: false (custom field, can be edited/removed)

### 3. Create New Groups
Create entirely new sections:

**Example**: Safety Information group
```json
{
  "group": "Safety Information",
  "field_key": "medical_officer",
  "field_label": "Medical Officer",
  "field_value": "Dr. Sarah Lee",
  "field_order": 1,
  "is_system": false
}
```

### 4. Backward Compatibility
Old format automatically migrates to new structure:
```php
// Old format
"jury": {"technical_delegate_fis": "John Doe"}

// Auto-migrates to
"metadata_fields": [{
  "group": "Jury Information",
  "field_key": "technical_delegate_fis",
  "field_label": "Technical Delegate FIS",
  "field_value": "John Doe",
  "field_order": 1,
  "is_system": true
}]
```

## Usage Examples

### Adding Custom Field via UI
1. Go to **Event Dashboard** → **Event Metadata** → **Manage**
2. Click **"Add Custom Field"**
3. Fill in:
   - **Group**: Select existing or type new
   - **Label**: Display name
   - **Value**: Initial data
   - **Order**: Position (use decimals for precise placement)
4. Click **"Add Field"**

### Inserting Field Between Existing Ones
**Scenario**: Insert "Event Secretary" between "Chief of Competition" and "Head Judge"

```
Current:
1. Technical Delegate (order: 1)
2. Chief of Competition (order: 2)
3. Head Judge (order: 3)

Action:
Add field with order: 2.5

Result:
1. Technical Delegate (order: 1)
2. Chief of Competition (order: 2)
2.5 Event Secretary (order: 2.5)  ← NEW
3. Head Judge (order: 3)
```

### Programmatic Usage
```php
// Load metadata
$metadata = getEventMetadata($pdo, $event_id);

// Add custom field to existing group
$metadata = addCustomField(
    $metadata, 
    'Jury Information',     // group
    'event_secretary',       // field_key
    'Event Secretary',       // field_label
    'Alice Johnson',         // field_value
    2.5                      // field_order (between 2 and 3)
);

// Save
saveEventMetadata($pdo, $event_id, $metadata);

// Get fields by group
$fieldsByGroup = getFieldsByGroup($metadata);
// Returns: ['Jury Information' => [...], 'Technical Data' => [...]]

// Display specific group
echo displayMetadataGroup($metadata, 'Jury Information');
```

## System vs Custom Fields

### System Fields
- Pre-defined by StyleScore
- **Cannot** be renamed or deleted
- **Can** have values changed
- Marked with 🔒 lock icon
- Gray "System" badge

**Default System Groups**:
1. **Jury Information**: TD, Chief of Competition, Head Judge, Chief of Course
2. **Technical Data**: Run Length, Pitch, Take-off to Knoll, Jump Height, Landing Pitch, Codex
3. **Venue Details**: Location, Elevation, Weather, Temperature

### Custom Fields
- Created by you
- **Can** be renamed, reordered, deleted
- **Can** be added to any group (system or custom)
- Marked with ✏️ edit icon
- Green "Custom" badge

## Benefits for Results Display

### Perfect for Reports
Fields appear in **exact order** you specify:

```
Jury Information:
1. Technical Delegate FIS: John Doe
2. Chief of Competition: Jane Smith
2.5 Chief Observer: Bob Wilson    ← Custom field inserted
3. Head Judge: Alice Johnson
4. Chief of Course: Mike Brown
```

### Dynamic Groups
Create event-specific sections:

```
Safety Information:
- Medical Officer: Dr. Sarah Lee
- Safety Coordinator: Tom White

Sponsors:
- Title Sponsor: Acme Corporation
- Official Timing: TimeTech Pro
```

## Migration Path

### Step 1: Run SQL Migration
```sql
-- Run this once on your database
SOURCE admin/sql/add_event_metadata_column.sql;
```

### Step 2: Automatic Data Migration
Old format data **automatically migrates** on first load:
- Jury fields → "Jury Information" group
- Technical data → "Technical Data" group
- Venue details → "Venue Details" group
- Custom fields → "Custom Fields" group

### Step 3: Start Using
- Existing data appears in new UI
- Add custom fields as needed
- Reorder with decimal values

## API Functions

### Core Functions
```php
// Get metadata with defaults merged
getEventMetadata($pdo, $event_id)

// Save metadata
saveEventMetadata($pdo, $event_id, $metadata)

// Organize by group with sorting
getFieldsByGroup($metadata)

// Display formatted group
displayMetadataGroup($metadata, $groupName)

// Add custom field
addCustomField($metadata, $group, $key, $label, $value, $order)

// Update field value
updateFieldValue($metadata, $fieldKey, $newValue)

// Remove custom field
removeField($metadata, $fieldKey)
```

### Legacy Functions (still work)
```php
displayJuryInfo($metadata)      // Calls displayMetadataGroup()
displayTechnicalData($metadata)  // Calls displayMetadataGroup()
displayVenueDetails($metadata)   // Calls displayMetadataGroup()
displayJudgesList($metadata)     // Separate judges array
```

## Best Practices

### Field Ordering Strategy
1. **Start with 1, 2, 3...** for initial setup
2. **Use decimals** (2.5, 2.75) when inserting between existing
3. **Leave gaps** (10, 20, 30) if you anticipate many insertions

### Group Naming
- Use clear, descriptive names
- Capitalize properly ("Jury Information" not "jury_information")
- Keep consistent across events

### Custom Field Keys
- Use lowercase with underscores
- Prefix with `custom_` to avoid conflicts
- Make descriptive: `custom_safety_officer` not `custom_field_1`

## Troubleshooting

### Q: Fields not appearing in order?
**A**: Check `field_order` values. System sorts numerically.

### Q: Can't delete a field?
**A**: System fields (`is_system: true`) are protected. Only custom fields can be deleted.

### Q: Old data missing after update?
**A**: Migration happens automatically. Check `metadata_fields` array in JSON.

### Q: How to reorder all fields?
**A**: Edit `field_order` values in UI, or update directly in database JSON.

## Files Modified

### Backend
- `includes/event_metadata_helpers.php` - Core functionality
- `admin/event_metadata.php` - Management interface
- `admin/event_dashboard.php` - Display card
- `admin/sql/add_event_metadata_column.sql` - Database migration

### Features Added
- ✅ Flexible field ordering with decimals
- ✅ Custom fields in any group
- ✅ Dynamic group creation
- ✅ Auto-migration from old format
- ✅ Protected system fields
- ✅ Drag-and-drop reordering (planned)
- ✅ Preview before save
- ✅ Field management UI

## Future Enhancements
- Drag-and-drop field reordering
- Field type validation (text, number, date)
- Required field markers
- Bulk import/export
- Template system for common configurations
- Multi-language field labels
