# Dual FTP/SFTP Protocol Implementation

## Overview
The publish_servers.php system now supports both FTP (insecure, legacy) and SFTP (secure, modern) protocols for remote file publishing. Users can select their preferred protocol when adding or editing publish servers.

## Database Schema

### Protocol Column
```sql
ALTER TABLE publish_servers 
ADD COLUMN protocol ENUM('ftp', 'sftp') DEFAULT 'sftp' AFTER port;

-- Update existing records to use SFTP
UPDATE publish_servers SET protocol = 'sftp' WHERE protocol IS NULL;
```

## Architecture

### Connection Functions

#### `testConnection($host, $port, $protocol, $username, $password)`
Routes connection test to appropriate protocol handler.

#### `testFTPConnection($host, $port, $username, $password)`
- Uses `ftp_connect()` for connection
- Uses `ftp_login()` for authentication
- Returns success/failure status

#### `testSFTPConnection($host, $port, $username, $password)`
- Checks if SSH2 extension is loaded
- Uses `ssh2_connect()` for connection
- Uses `ssh2_auth_password()` for authentication
- Initializes `ssh2_sftp()` subsystem
- Returns success/failure status

### Publishing Functions

#### `publishToServer($server, $pdo)`
Main publishing function that:
1. Determines protocol from `$server['protocol']`
2. Connects via FTP or SFTP based on protocol
3. Creates remote directories recursively
4. Generates files based on selected data types
5. Uploads files using protocol-specific upload function
6. Cleans up and closes connection

**FTP Path:**
- `ftp_connect($host, $port, 10)` - 10 second timeout
- `ftp_login($connection, $username, $password)`
- `ftp_pasv($connection, true)` - Enable passive mode
- `ftp_mkdir()` for directory creation
- `ftp_chdir()` for navigation
- `ftp_close($connection)` for cleanup

**SFTP Path:**
- Checks `function_exists('ssh2_connect')`
- `ssh2_connect($host, $port)`
- `ssh2_auth_password($connection, $username, $password)`
- `ssh2_sftp($connection)`
- `ssh2_sftp_mkdir()` for directory creation
- Connection closes automatically when variables go out of scope

### Upload Functions

#### `uploadFileFTP($connection, $remote_path, $local_file, $remote_file)`
- Creates remote directories using `ftp_mkdir()` and `ftp_chdir()`
- Uploads using `ftp_put($connection, $full_remote_path, $local_file, FTP_BINARY)`
- Returns boolean success status

#### `uploadFileSFTP($connection, $sftp, $remote_path, $local_file, $remote_file)`
- Creates remote directories using `ssh2_sftp_mkdir()`
- Checks directory existence with `ssh2_sftp_stat()`
- Uploads using `ssh2_scp_send($connection, $local_file, $full_remote_path, 0644)`
- Returns boolean success status

#### `uploadFile($connection, $sftp, $remote_path, $local_file, $remote_file)` (Legacy)
Routes to FTP or SFTP based on `$sftp` parameter:
- If `$sftp === null` → calls `uploadFileFTP()`
- If `$sftp` is set → calls `uploadFileSFTP()`

## User Interface

### Protocol Selection
Both add and edit modals feature radio button selection:

```html
<div class="btn-group w-100" role="group">
    <input type="radio" name="protocol" id="protocol_ftp_add" value="ftp">
    <label for="protocol_ftp_add">
        <i class="fas fa-network-wired me-2"></i>FTP (Port 21)
    </label>
    
    <input type="radio" name="protocol" id="protocol_sftp_add" value="sftp" checked>
    <label for="protocol_sftp_add">
        <i class="fas fa-lock me-2"></i>SFTP (Port 22) - Recommended
    </label>
</div>
```

### Automatic Port Updates
JavaScript function `updateProtocolUI(modal)` automatically:
- Changes port to 21 when FTP selected
- Changes port to 22 when SFTP selected
- Updates label from "FTP Host" to "SFTP Host" and vice versa

### Event Handlers
```javascript
// Add modal
document.getElementById('protocol_ftp_add').addEventListener('change', function() {
    if (this.checked) updateProtocolUI('add');
});

// Edit modal
document.getElementById('protocol_ftp_edit').addEventListener('change', function() {
    if (this.checked) updateProtocolUI('edit');
});
```

## Backend Processing

### Add Server
```php
case 'add_server':
    $protocol = $_POST['protocol'] ?? 'sftp';
    $port = (int)($_POST['port'] ?: ($protocol === 'ftp' ? 21 : 22));
    // ... INSERT with protocol column
```

### Edit Server
```php
case 'edit_server':
    $protocol = $_POST['protocol'] ?? 'sftp';
    $port = (int)($_POST['port'] ?: ($protocol === 'ftp' ? 21 : 22));
    // ... UPDATE with protocol column
```

### Test Connection
```php
case 'test_connection':
    $protocol = $_POST['protocol'] ?? 'sftp';
    $port = (int)($_POST['port'] ?: ($protocol === 'ftp' ? 21 : 22));
    $result = testConnection($host, $port, $protocol, $username, $password);
```

## Port Defaults
- **FTP**: Port 21 (standard)
- **SFTP**: Port 22 (SSH standard)

Auto-populated when protocol is selected, but user can override.

## Security Considerations

### FTP (Port 21)
- ⚠️ **Unencrypted** - credentials and data sent in plain text
- Susceptible to packet sniffing
- Should only be used for legacy systems
- Passive mode enabled by default for firewall compatibility

### SFTP (Port 22)
- ✅ **Encrypted** via SSH protocol
- Strong authentication
- Secure file transfer
- Recommended for all new deployments

## Extension Requirements

### FTP
- Built-in to PHP (no extension needed)
- Always available

### SFTP
- Requires `php_ssh2` extension
- Not installed by default on most systems
- See `INSTALL_SSH2_MANUAL.md` for installation instructions
- See `admin/check_ssh2.php` for web-based status checker

## Error Handling

### Missing SSH2 Extension
When SFTP is selected but SSH2 extension not available:
```
SSH2 extension is not loaded in PHP. Please install php_ssh2 extension 
or contact your server administrator.
```

### Connection Failures
- FTP: "Could not connect to FTP server: host:port"
- SFTP: "Could not connect to SFTP server: host:port"

### Authentication Failures
- FTP: "FTP authentication failed"
- SFTP: "SFTP authentication failed"

## Data Type Publishing
All data types work identically with both protocols:
- `html_dashboard` - Static HTML dashboard
- `json_data` - JSON API files
- `csv_data` - CSV exports
- `pdf_results` - PDF result sheets
- `participant_photos` - Participant images

## Migration Notes

### Existing Servers
- Existing servers default to SFTP protocol
- Database migration adds `protocol = 'sftp'` to all existing records
- No manual updates required

### Testing
1. Edit existing server configuration
2. Select FTP protocol if SSH2 not available
3. Click "Test Connection" to verify
4. Save and publish

## Future Enhancements
- [ ] FTPS (FTP over SSL) support
- [ ] SSH key authentication for SFTP
- [ ] Connection pooling for batch uploads
- [ ] Bandwidth throttling options
- [ ] Resume failed transfers
