# Judge Drawing Board Feature

## Overview
Integrated drawing board functionality for judges to document situations (DSQ/DNF/fouls) with visual sketches and notes.

## Implementation Status: ✅ Phase 1 Complete

### Completed Components

#### 1. Database Structure
- **Table**: `judge_drawings` ✅
- **Columns**:
  - `id` - Primary key
  - `judge_id` - Foreign key to users
  - `score_id` - Foreign key to scores (nullable)
  - `event_id`, `heat_number`, `run_number`
  - `participant_id` - Foreign key to event_participants
  - `background_json` - Background template filename
  - `drawing_json` - Drawing data filename
  - `canvas_width`, `canvas_height`
  - `notes` - Optional text notes
  - `status` - ENUM('draft', 'final', 'archived')
  - `created_at`, `updated_at`

#### 2. File Structure ✅
```
/drawings/
  ├── backgrounds/    # Reusable templates/grids
  ├── scores/        # Individual judge drawings
  └── .htaccess      # Protection rules
```

#### 3. User Interface ✅
- **Location**: `judge/score.php`
- **Trigger**: "Add Drawing" button below status selection (OK/DNS/DNF/DSQ)
- **Modal**: Full-screen drawing board modal
- **Features**:
  - Canvas with touch support
  - Color picker (6 colors: Black, White, Red, Green, Blue, Orange)
  - Brush size selector (Thin, Medium, Thick)
  - Drawing modes: Draw, Erase, Select
  - 2-layer system: Background (locked template) + Foreground (actual drawing)
  - Background opacity control
  - Transform tools (Scale, Rotate)
  - Undo functionality
  - Text notes field
  - Context info bar (Event, Heat/Run, Participant)

#### 4. Visual Indicators ✅
- Green dot appears on "Add Drawing" button when drawing exists
- Button classes: `.drawing-trigger-btn` with `.has-drawing` modifier

## Technical Architecture

### Drawing Data Format
```json
{
  "event_id": 3,
  "heat_number": 4,
  "run_number": 1,
  "participant_id": 42,
  "canvas_width": 800,
  "canvas_height": 600,
  "background": [
    {
      "id": "timestamp_random",
      "color": "#000000",
      "size": 2,
      "points": [{"x": 100, "y": 100}, {"x": 150, "y": 150}]
    }
  ],
  "foreground": [
    // Same structure as background
  ],
  "notes": "Participant landed outside the zone"
}
```

### JavaScript State Management
- Global `drawingBoardState` object manages all canvas state
- Real-time drawing with HTML5 Canvas API
- Touch and mouse event support
- Undo stack for stroke history

### CSS Styling
- Responsive toolbar (horizontal scroll on mobile)
- Dark mode support
- Mobile-optimized button sizes (min 44px touch targets)
- Full-screen modal for maximum drawing space

## Pending Implementation (Phase 2)

### API Endpoints (To be created)
1. **POST** `/api/judge_drawings_api.php?action=save`
   - Save new drawing
   - Update existing drawing
   - Generate unique filenames
   - Store JSON files

2. **GET** `/api/judge_drawings_api.php?action=load`
   - Load drawing by score_id
   - Load drawing by participant/heat/run

3. **GET** `/api/judge_drawings_api.php?action=list_templates`
   - List available background templates

4. **POST** `/api/judge_drawings_api.php?action=save_template`
   - Save current background as reusable template

### Template System
- Pre-made background templates (ski slopes, halfpipe, rails, etc.)
- Template selector UI
- Template preview thumbnails
- Custom template creation

### Integration Points
- **Score Submission**: Attach drawing_id to score record
- **Head Judge Panel**: View drawings in approval interface
- **Public Dashboard**: Optional display of situation drawings

### Additional Features
- Drawing thumbnail generation (PNG export)
- Drawing history/versioning
- Share drawings between judges
- Export to PDF/PNG
- Drawing annotations/labels

## Usage Instructions

### For Judges
1. Navigate to score.php
2. Select participant
3. Click "Add Drawing" button below status selection
4. Draw situation using toolbar tools
5. Add optional text notes
6. Click "Save Drawing" to store

### For Administrators
- Background templates will be managed via admin panel (future)
- Drawings viewable in head judge approval panel (future)
- Drawing files stored in `/drawings/` directory

## Files Modified/Created

### New Files
- `create_judge_drawings_table.sql` - SQL migration
- `run_judge_drawings_migration.php` - Migration script
- `DRAWING_BOARD_FEATURE.md` - This documentation

### Modified Files
- `judge/score.php`:
  - Added drawing board modal HTML
  - Added CSS styles for toolbar
  - Added JavaScript drawing engine
  - Added "Add Drawing" button

### Database
- Table `judge_drawings` created
- Foreign keys to `users`, `scores`, `events`, `event_participants`

## Browser Compatibility
- Chrome/Edge: ✅ Full support
- Firefox: ✅ Full support
- Safari: ✅ Full support
- Mobile browsers: ✅ Touch optimized

## Security Considerations
- JSON files stored server-side (not client-downloadable by default)
- .htaccess protection on /drawings/ directory
- Judge authentication required for save operations
- SQL injection protection via PDO prepared statements

## Performance Notes
- Canvas rendering: Client-side (no server load)
- File storage: JSON format (small file sizes ~5-50KB)
- Image generation: Deferred to Phase 2

## Next Steps
1. Create `api/judge_drawings_api.php` endpoint
2. Implement save/load functionality
3. Create background template library
4. Integrate with head judge approval workflow
5. Add thumbnail generation
6. Add drawing viewer component

---

**Status**: Ready for testing
**Version**: 1.0
**Last Updated**: <?= date('Y-m-d H:i:s') ?>
