# Diversity Rules Heat Configuration Feature

## Summary
Added the ability to enable/disable diversity rules per heat in the Heat Configuration page.

## Changes Made

### 1. Database Migration
- **File**: `add_diversity_rules_enabled_migration.php`
- **SQL**: `add_diversity_rules_enabled_column.sql`
- **Column Added**: `diversity_rules_enabled` (TINYINT, DEFAULT 1)
- **Table**: `event_heat_settings`
- **Status**: ✓ Migration completed successfully

### 2. Backend Updates (`admin/admin_heat_config.php`)

#### POST Handling (Line ~93)
Added processing for the diversity rules checkbox:
```php
// Handle diversity rules setting
$diversity_rules_enabled = isset($_POST['diversity_rules_enabled'][$heat_number]) ? 1 : 0;
```

#### Database INSERT/UPDATE (Line ~147)
Updated SQL statement to include `diversity_rules_enabled` column:
- Added to INSERT columns list
- Added to VALUES placeholder list
- Added to ON DUPLICATE KEY UPDATE list
- Added to execute() parameters array

### 3. Frontend UI (`admin/admin_heat_config.php`)

Added new card section after Flow Configuration (Line ~2045):
```html
<div class="col-md-4">
    <!-- Diversity Rules Configuration -->
    <div class="card">
        <div class="card-header">
            <h6><i class="fas fa-shapes me-2"></i>Diversity Rules</h6>
        </div>
        <div class="card-body">
            <div class="form-check form-switch">
                <input type="checkbox" name="diversity_rules_enabled[<?= $h ?>]" 
                       id="diversity_rules_<?= $h ?>" value="1" 
                       <?= ($setting['diversity_rules_enabled'] ?? 1) ? 'checked' : '' ?>>
                <label>Apply Diversity Rules</label>
            </div>
            <div class="alert alert-info">
                When enabled, diversity rules defined in the scoring format 
                will be applied to this heat. Disable to ignore diversity 
                requirements for this specific heat.
            </div>
        </div>
    </div>
</div>
```

## How to Use

### For Administrators:
1. Navigate to **Admin > Heat Configuration**
2. Select an event
3. Edit any heat in the **"Participants & Flow"** tab
4. Find the new **"Diversity Rules"** card (right column)
5. Toggle **"Apply Diversity Rules"** checkbox:
   - **Checked (Default)**: Diversity rules from scoring format will be enforced
   - **Unchecked**: Diversity rules will be ignored for this heat
6. Click **"Save Settings"** to apply changes

### For Developers:
To check if diversity rules should be applied in your code:

```php
// Load heat settings
$stmt = $pdo->prepare("SELECT diversity_rules_enabled FROM event_heat_settings WHERE event_id = ? AND heat_number = ?");
$stmt->execute([$event_id, $heat_number]);
$heat = $stmt->fetch(PDO::FETCH_ASSOC);

if ($heat['diversity_rules_enabled']) {
    // Apply diversity validation
    validateDiversityRules($figures, $diversity_rules);
} else {
    // Skip diversity validation
}
```

## Database Schema

```sql
CREATE TABLE `event_heat_settings` (
  ...
  `flow_position_range` VARCHAR(50) DEFAULT NULL,
  `diversity_rules_enabled` TINYINT(1) NOT NULL DEFAULT 1 
    COMMENT 'Whether to apply diversity rules from scoring format (1=apply, 0=ignore)',
  `is_active` TINYINT(1) DEFAULT 0,
  ...
);
```

## Default Behavior
- **New heats**: Diversity rules enabled by default (value = 1)
- **Existing heats**: Updated to have diversity rules enabled (value = 1)
- This ensures backward compatibility with existing heat configurations

## Integration Points

### Current Integration:
- ✓ Admin UI for heat configuration
- ✓ Database storage and retrieval
- ✓ Form submission and validation
- ✓ **Judge Scoring API** (`api/judge_score_api.php`) - Checks `diversity_rules_enabled` flag before loading diversity rules
- ✓ **Legacy Judge Scoring** (`judge/score.php`) - Checks `diversity_rules_enabled` flag before loading diversity rules

### How Diversity Rules Are Applied:
When `diversity_rules_enabled` = 0 (disabled):
- **NO diversity rules are loaded** from `scoring_format_diversity_rules` table
- Judge scoring panel will NOT display diversity validation
- Figure selection will NOT enforce diversity requirements
- This completely ignores all diversity-related settings from the scoring format

When `diversity_rules_enabled` = 1 (enabled - default):
- Diversity rules ARE loaded from scoring format
- All diversity validation and enforcement is active
- Judges see diversity validation feedback in real-time

### Implementation Details:

#### Judge Score API (`api/judge_score_api.php`)
```php
// Check if diversity rules are enabled for the active heat
$diversity_enabled_stmt = $pdo->prepare("
    SELECT diversity_rules_enabled 
    FROM event_heat_settings 
    WHERE event_id = ? AND status = 'active' 
    LIMIT 1
");
$diversity_enabled_stmt->execute([$event_id]);
$diversity_enabled = $diversity_enabled_stmt->fetchColumn();

// Only load diversity rules if enabled
if ($diversity_enabled === false || $diversity_enabled === 1 || $diversity_enabled === '1') {
    // Load diversity rules...
}
```

#### Legacy Judge Scoring (`judge/score.php`)
```php
// Check if diversity rules are enabled for the current heat
$diversity_enabled_stmt = $pdo->prepare("
    SELECT diversity_rules_enabled 
    FROM event_heat_settings 
    WHERE event_id = ? AND heat_number = ?
");
$diversity_enabled_stmt->execute([$selected_event_id, $heat_number]);
$diversity_enabled = $diversity_enabled_stmt->fetchColumn();

// Only load diversity rules if enabled
if ($diversity_enabled === false || $diversity_enabled === 1 || $diversity_enabled === '1') {
    // Load diversity rules...
} else {
    $diversity_rules = [];
}
```

### Future Integration (Optional Enhancements):
- Summary table calculations with diversity awareness
- Heat flow promotion with diversity validation
- Report generation showing diversity compliance per heat

## Files Modified
1. `admin/admin_heat_config.php` - UI and backend logic
2. `add_diversity_rules_enabled_migration.php` - Migration script (NEW)
3. `add_diversity_rules_enabled_column.sql` - SQL migration (NEW)

## Testing Checklist
- [x] Database column created successfully
- [x] UI field displays correctly in heat configuration
- [x] Default value (enabled) applied to existing heats
- [ ] Form submission saves diversity_rules_enabled value
- [ ] Value persists after page reload
- [ ] Integration with judge scoring validation
- [ ] Integration with figure selection validation

## Notes
- The diversity rules themselves are defined in the scoring format (table: `scoring_format_diversity_rules`)
- This feature provides per-heat control over whether those format-level rules should be enforced
- Useful for finals/semi-finals where diversity requirements might be relaxed
