# Data Attributes Reference - HTML Tables

## Overview

All HTML tables generated by the `summary_table_api.php` now include comprehensive `data-*` attributes on each `<tr>` element. These attributes enable client-side sorting, filtering, and manipulation without requiring server calls.

## Complete List of Data Attributes

### Basic Participant Information

| Attribute | Type | Description | Example |
|-----------|------|-------------|---------|
| `data-bib` | string | Participant's BIB number | `"42"` |
| `data-participant` | string | Full participant name | `"John Doe"` |
| `data-first-name` | string | First name only | `"John"` |
| `data-last-name` | string | Last name only | `"Doe"` |
| `data-category` | string | Event category | `"Male Senior"` |
| `data-club` | string | Club/team name | `"Mountain Riders"` |
| `data-gender` | string | Gender (M/F) | `"M"` |

### Scoring Data

| Attribute | Type | Description | Example |
|-----------|------|-------------|---------|
| `data-rank` | number | Current ranking (999 if no scores) | `"1"` |
| `data-has-scores` | boolean | Whether participant has scores | `"true"` |
| `data-overall-average` | number | Overall average score | `"85.50"` |
| `data-overall-best` | number | Overall best score | `"92.75"` |
| `data-heat-1-best` | number | Best score from heat 1 | `"88.00"` |
| `data-heat-1-average` | number | Average score from heat 1 | `"85.50"` |
| `data-heat-2-best` | number | Best score from heat 2 | `"90.25"` |
| `data-heat-N-best` | number | Best score from heat N | `"87.50"` |
| `data-heat-N-average` | number | Average score from heat N | `"84.75"` |

### Additional Information

| Attribute | Type | Description | Example |
|-----------|------|-------------|---------|
| `data-fis-code` | string | FIS code (if available) | `"12345"` |
| `data-licence-code` | string | Licence code (if available) | `"ABC123"` |
| `data-country` | string | Country code | `"USA"` |
| `data-birth-year` | string | Year of birth | `"1995"` |
| `data-age` | number | Current age | `"30"` |
| `data-approved-at` | string | Timestamp of last approved score | `"2025-11-16 14:30:00"` |
| `data-heat-number` | number | Heat assignment | `"2"` |

### Grouping & Status

| Attribute | Type | Description | Example |
|-----------|------|-------------|---------|
| `data-group` | string | Category group (for grouped tables) | `"Female Junior"` |
| `data-diversity-valid` | boolean | Diversity rules met | `"true"` |

## Usage Examples

### 1. Client-Side Sorting

```javascript
// Sort by overall average score (descending)
function sortByScore() {
    const tbody = document.querySelector('.data-table tbody');
    const rows = Array.from(tbody.querySelectorAll('tr'));
    
    rows.sort((a, b) => {
        const scoreA = parseFloat(a.dataset.overallAverage) || 0;
        const scoreB = parseFloat(b.dataset.overallAverage) || 0;
        return scoreB - scoreA; // Descending
    });
    
    rows.forEach(row => tbody.appendChild(row));
}

// Sort by name (ascending)
function sortByName() {
    const tbody = document.querySelector('.data-table tbody');
    const rows = Array.from(tbody.querySelectorAll('tr'));
    
    rows.sort((a, b) => {
        return a.dataset.participant.localeCompare(b.dataset.participant);
    });
    
    rows.forEach(row => tbody.appendChild(row));
}

// Generic sort function
function sortTable(attribute, direction = 'asc') {
    const tbody = document.querySelector('.data-table tbody');
    const rows = Array.from(tbody.querySelectorAll('tr'));
    
    rows.sort((a, b) => {
        let valA = a.dataset[attribute] || '';
        let valB = b.dataset[attribute] || '';
        
        // Try numeric comparison
        const numA = parseFloat(valA);
        const numB = parseFloat(valB);
        
        if (!isNaN(numA) && !isNaN(numB)) {
            return direction === 'asc' ? numA - numB : numB - numA;
        }
        
        // String comparison
        return direction === 'asc' 
            ? valA.localeCompare(valB) 
            : valB.localeCompare(valA);
    });
    
    rows.forEach(row => tbody.appendChild(row));
}
```

### 2. Client-Side Filtering

```javascript
// Filter by category
function filterByCategory(category) {
    const rows = document.querySelectorAll('.data-table tbody tr');
    
    rows.forEach(row => {
        if (category === '' || row.dataset.category === category) {
            row.style.display = '';
        } else {
            row.style.display = 'none';
        }
    });
}

// Filter by gender
function filterByGender(gender) {
    const rows = document.querySelectorAll('.data-table tbody tr');
    
    rows.forEach(row => {
        if (gender === '' || row.dataset.gender === gender) {
            row.style.display = '';
        } else {
            row.style.display = 'none';
        }
    });
}

// Filter by score presence
function filterByScores(hasScores) {
    const rows = document.querySelectorAll('.data-table tbody tr');
    
    rows.forEach(row => {
        if (hasScores === null || row.dataset.hasScores === hasScores.toString()) {
            row.style.display = '';
        } else {
            row.style.display = 'none';
        }
    });
}

// Search by participant name
function searchParticipants(query) {
    const rows = document.querySelectorAll('.data-table tbody tr');
    const searchTerm = query.toLowerCase();
    
    rows.forEach(row => {
        const name = row.dataset.participant.toLowerCase();
        if (name.includes(searchTerm)) {
            row.style.display = '';
        } else {
            row.style.display = 'none';
        }
    });
}
```

### 3. Data Extraction

```javascript
// Extract all participants as array
function extractParticipants() {
    const rows = document.querySelectorAll('.data-table tbody tr');
    const participants = [];
    
    rows.forEach(row => {
        participants.push({
            bib: row.dataset.bib,
            name: row.dataset.participant,
            firstName: row.dataset.firstName,
            lastName: row.dataset.lastName,
            category: row.dataset.category,
            club: row.dataset.club,
            gender: row.dataset.gender,
            rank: parseInt(row.dataset.rank),
            hasScores: row.dataset.hasScores === 'true',
            overallAverage: parseFloat(row.dataset.overallAverage) || null,
            overallBest: parseFloat(row.dataset.overallBest) || null
        });
    });
    
    return participants;
}

// Get top N participants
function getTopParticipants(n = 10) {
    const participants = extractParticipants();
    return participants
        .filter(p => p.hasScores)
        .sort((a, b) => b.overallAverage - a.overallAverage)
        .slice(0, n);
}

// Get statistics
function getStats() {
    const rows = document.querySelectorAll('.data-table tbody tr');
    const scores = [];
    
    rows.forEach(row => {
        if (row.dataset.hasScores === 'true') {
            scores.push(parseFloat(row.dataset.overallAverage));
        }
    });
    
    return {
        total: rows.length,
        withScores: scores.length,
        average: scores.reduce((a, b) => a + b, 0) / scores.length,
        max: Math.max(...scores),
        min: Math.min(...scores)
    };
}
```

### 4. Row Highlighting

```javascript
// Highlight top 3 participants
function highlightTop3() {
    const rows = Array.from(document.querySelectorAll('.data-table tbody tr'));
    
    // Sort by overall average
    const sorted = rows
        .filter(row => row.dataset.hasScores === 'true')
        .sort((a, b) => {
            const scoreA = parseFloat(a.dataset.overallAverage) || 0;
            const scoreB = parseFloat(b.dataset.overallAverage) || 0;
            return scoreB - scoreA;
        });
    
    // Add classes to top 3
    sorted.slice(0, 3).forEach((row, index) => {
        row.classList.add(`rank-${index + 1}`);
    });
}

// Highlight participants from specific club
function highlightClub(clubName) {
    const rows = document.querySelectorAll('.data-table tbody tr');
    
    rows.forEach(row => {
        if (row.dataset.club === clubName) {
            row.classList.add('highlight-club');
        } else {
            row.classList.remove('highlight-club');
        }
    });
}
```

### 5. Event Listeners

```javascript
// Make rows clickable to show details
document.querySelectorAll('.data-table tbody tr').forEach(row => {
    row.addEventListener('click', function() {
        const details = {
            BIB: this.dataset.bib,
            Name: this.dataset.participant,
            Category: this.dataset.category,
            Club: this.dataset.club,
            Score: this.dataset.overallAverage,
            Rank: this.dataset.rank
        };
        
        console.log('Participant Details:', details);
        // Or show in modal, etc.
    });
});

// Add hover effect with data display
document.querySelectorAll('.data-table tbody tr').forEach(row => {
    row.addEventListener('mouseenter', function() {
        this.title = `${this.dataset.participant} (${this.dataset.category}) - Score: ${this.dataset.overallAverage || 'N/A'}`;
    });
});
```

## Integration with Table Libraries

### DataTables.js Example

```javascript
// Use data attributes as sortable columns
$('.data-table').DataTable({
    columns: [
        { data: 'rank', render: (data, type, row) => row.dataset.rank },
        { data: 'bib', render: (data, type, row) => row.dataset.bib },
        { data: 'participant', render: (data, type, row) => row.dataset.participant },
        { data: 'category', render: (data, type, row) => row.dataset.category },
        { data: 'score', render: (data, type, row) => row.dataset.overallAverage }
    ]
});
```

### List.js Example

```javascript
// Initialize List.js with data attributes
const options = {
    valueNames: [
        { data: ['bib'] },
        { data: ['participant'] },
        { data: ['category'] },
        { data: ['overall-average'] }
    ]
};

const participantList = new List('table-container', options);
```

## Demo Page

A complete interactive demo is available at:
```
/v2/HTML_TABLE_SORTING_DEMO.html
```

This demo shows:
- ✅ Real-time search
- ✅ Multi-criteria filtering
- ✅ Dynamic sorting
- ✅ Statistics display
- ✅ Active filter badges

## Browser Support

Data attributes are supported in all modern browsers:
- ✅ Chrome/Edge 90+
- ✅ Firefox 88+
- ✅ Safari 14+
- ✅ Opera 76+

## Performance Notes

- Data attributes add ~500 bytes per row
- Sorting 1000 rows takes ~5-10ms on modern browsers
- Filtering is instant even with 1000+ rows
- No server calls needed for common operations

## Best Practices

1. **Camel Case Conversion**: `data-overall-average` → `dataset.overallAverage`
2. **Type Coercion**: Always parse numbers: `parseFloat(row.dataset.rank)`
3. **Null Checks**: Handle missing data: `row.dataset.score || 0`
4. **Event Delegation**: Use single listener on tbody instead of per-row
5. **CSS Classes**: Prefer adding/removing classes over inline styles

## Example: Complete Sortable Table

```html
<!DOCTYPE html>
<html>
<head>
    <style>
        th { cursor: pointer; user-select: none; }
        th::after { content: ' ⇅'; opacity: 0.3; }
        th.sort-asc::after { content: ' ▲'; opacity: 1; }
        th.sort-desc::after { content: ' ▼'; opacity: 1; }
    </style>
</head>
<body>
    <!-- Your table loaded from API -->
    <div id="table-container"></div>
    
    <script>
        // Load table
        fetch('/v2/api/summary_table_api.php?event_id=3&format=html&styling=partial')
            .then(r => r.text())
            .then(html => {
                document.getElementById('table-container').innerHTML = html;
                makeSortable();
            });
        
        // Make headers clickable for sorting
        function makeSortable() {
            const headers = document.querySelectorAll('.data-table thead th');
            headers.forEach((header, index) => {
                header.onclick = () => sortByColumn(index);
            });
        }
        
        let currentSort = { column: null, direction: 'asc' };
        
        function sortByColumn(columnIndex) {
            const headers = document.querySelectorAll('.data-table thead th');
            const header = headers[columnIndex];
            const headerText = header.textContent.trim().toLowerCase();
            
            // Map header text to data attribute
            const attrMap = {
                'rank': 'rank', 'bib': 'bib', 'name': 'participant',
                'category': 'category', 'score': 'overall-average'
            };
            
            const attr = attrMap[headerText];
            if (!attr) return;
            
            // Toggle direction
            if (currentSort.column === attr) {
                currentSort.direction = currentSort.direction === 'asc' ? 'desc' : 'asc';
            } else {
                currentSort.column = attr;
                currentSort.direction = 'asc';
            }
            
            // Sort rows
            const tbody = document.querySelector('.data-table tbody');
            const rows = Array.from(tbody.querySelectorAll('tr'));
            
            rows.sort((a, b) => {
                const valA = a.dataset[attr.replace(/-/g, '')] || '';
                const valB = b.dataset[attr.replace(/-/g, '')] || '';
                
                const numA = parseFloat(valA);
                const numB = parseFloat(valB);
                
                if (!isNaN(numA) && !isNaN(numB)) {
                    return currentSort.direction === 'asc' ? numA - numB : numB - numA;
                }
                
                return currentSort.direction === 'asc' 
                    ? valA.localeCompare(valB) 
                    : valB.localeCompare(valA);
            });
            
            rows.forEach(row => tbody.appendChild(row));
            
            // Update visual indicators
            headers.forEach(h => h.classList.remove('sort-asc', 'sort-desc'));
            header.classList.add(currentSort.direction === 'asc' ? 'sort-asc' : 'sort-desc');
        }
    </script>
</body>
</html>
```

## Summary

With these data attributes, you can build rich, interactive table experiences entirely client-side. The attributes provide:

- 🔍 **Instant search** without server calls
- 🗂️ **Multi-column sorting** in milliseconds
- 🎯 **Advanced filtering** by any criteria
- 📊 **Real-time statistics** calculation
- 🎨 **Dynamic styling** based on data
- 📱 **Mobile-friendly** progressive enhancement

No JavaScript libraries required - just vanilla JavaScript and the power of `data-*` attributes!
