# Custom Fields Feature for Participants

## Overview
The custom fields feature allows you to dynamically add custom columns to participant data without modifying the database schema. Custom fields are stored as JSON and are fully searchable, sortable, and importable.

## Database Schema

### participants.custom_fields (JSON Column)
Stores dynamic custom field data as JSON:
```json
{
  "team_name": "Blue Dragons",
  "coach_name": "John Smith",
  "sponsor": "Acme Corp",
  "skill_level": "Advanced",
  "notes": "National champion 2024"
}
```

### participant_custom_field_definitions Table
Stores field definitions and metadata:
- `id`: Primary key
- `field_key`: Unique identifier (e.g., "team_name")
- `field_label`: Display name (e.g., "Team Name")
- `field_type`: text, number, date, select, textarea
- `field_options`: JSON array of options for select fields
- `is_searchable`: Whether field can be searched
- `is_required`: Whether field is mandatory
- `display_order`: Display order in UI
- `is_active`: Whether field is currently active

## Installation

1. Run the migration script:
```bash
mysql -u root -p freestyle_db < add_custom_fields_column.sql
```

2. Verify the installation:
```sql
DESCRIBE participants;  -- Should show custom_fields column
SELECT * FROM participant_custom_field_definitions;  -- Should show example fields
```

## Usage

### 1. Managing Custom Field Definitions

Navigate to **Admin > Custom Fields Manager** (`/admin/custom_fields_manager.php`)

**Add a New Field:**
1. Click "Add Custom Field"
2. Enter field key (lowercase, underscores only, e.g., `team_name`)
3. Enter field label (display name, e.g., "Team Name")
4. Select field type
5. Set searchable/required options
6. Set display order

**Field Types:**
- **Text**: Single-line text input
- **Number**: Numeric input
- **Date**: Date picker
- **Select**: Dropdown with predefined options
- **Textarea**: Multi-line text input

### 2. Using Custom Fields in Participants Manager

Custom fields will automatically appear as columns in the participants spreadsheet.

**Editing Values:**
- Click any custom field cell to edit
- Values are saved when you click "Save Changes"

**Searching:**
- Custom fields marked as "searchable" can be filtered
- Use the search bar to search across all searchable fields

**Sorting:**
- Click column headers to sort by custom fields
- Hold Ctrl to sort by multiple columns

### 3. Importing Data with Custom Fields

**CSV Import:**
1. Include custom field columns in your CSV
2. Use the field key as column header (e.g., `team_name`)
3. Import as usual via "CSV Import"

**Excel Paste Import:**
1. Map your Excel columns to custom fields
2. Custom fields will appear in the column mapping dropdown
3. Complete the import process

**CSV Format Example:**
```csv
bib,first_name,last_name,team_name,coach_name,skill_level
101,John,Doe,Blue Dragons,Mike Johnson,Advanced
102,Jane,Smith,Red Hawks,Sarah Williams,Professional
```

### 4. Exporting Data with Custom Fields

Custom fields are automatically included in CSV exports with their field keys as column headers.

## API Endpoints

### Get Custom Field Definitions
```javascript
fetch('/api/custom_fields_api.php?action=get_custom_fields')
  .then(r => r.json())
  .then(data => console.log(data.fields));
```

### Update Participant Custom Fields
```javascript
const formData = new FormData();
formData.append('action', 'update_participant_custom_fields');
formData.append('participant_id', 123);
formData.append('custom_fields', JSON.stringify({
  team_name: 'Blue Dragons',
  coach_name: 'Mike Johnson'
}));

fetch('/api/custom_fields_api.php', {
  method: 'POST',
  body: formData
}).then(r => r.json());
```

### Search by Custom Field
```javascript
fetch('/api/custom_fields_api.php?action=search_by_custom_field&field_key=team_name&search_value=Dragons')
  .then(r => r.json())
  .then(data => console.log(data.results));
```

### Bulk Update Custom Field
```javascript
const formData = new FormData();
formData.append('action', 'bulk_update_custom_field');
formData.append('participant_ids[]', 101);
formData.append('participant_ids[]', 102);
formData.append('field_key', 'sponsor');
formData.append('field_value', 'Acme Corp');

fetch('/api/custom_fields_api.php', {
  method: 'POST',
  body: formData
}).then(r => r.json());
```

## Integration with Existing Code

### participants.php Modifications Needed

1. **Load Custom Field Definitions:**
```php
// Add after line ~160
$custom_fields_stmt = $pdo->query("
    SELECT * FROM participant_custom_field_definitions 
    WHERE is_active = 1 
    ORDER BY display_order ASC
");
$custom_field_defs = $custom_fields_stmt->fetchAll(PDO::FETCH_ASSOC);
```

2. **Include Custom Fields in Query:**
```php
// Modify the main SELECT query to include custom_fields
$query = "SELECT p.*, 
                 p.custom_fields as custom_fields_json,
                 GROUP_CONCAT(DISTINCT e1.name SEPARATOR ', ') as assigned_event_names,
                 ...
```

3. **Add Custom Field Columns to Spreadsheet:**
```javascript
// In the spreadsheet column definitions (around line 2500)
const customFieldColumns = <?= json_encode($custom_field_defs) ?>;

// Add custom field columns dynamically
customFieldColumns.forEach(field => {
    columns.push({
        type: field.field_type === 'select' ? 'dropdown' : 'text',
        title: field.field_label,
        width: '150',
        source: field.field_type === 'select' ? JSON.parse(field.field_options) : null
    });
});
```

4. **Handle Custom Fields in Save:**
```php
// In participants_api.php save operation
$custom_fields = json_decode($_POST['custom_fields'][$id] ?? '{}', true);

$stmt = $pdo->prepare("
    UPDATE participants 
    SET first_name = ?, 
        last_name = ?, 
        ..., 
        custom_fields = ?
    WHERE id = ?
");
```

## Advanced Features

### JSON Path Queries
Search specific nested values:
```sql
SELECT * FROM participants 
WHERE JSON_EXTRACT(custom_fields, '$.team_name') = 'Blue Dragons';
```

### Indexing for Performance
For better search performance on frequently searched fields:
```sql
-- Create virtual column + index
ALTER TABLE participants 
ADD COLUMN team_name_virtual VARCHAR(255) 
GENERATED ALWAYS AS (JSON_UNQUOTE(JSON_EXTRACT(custom_fields, '$.team_name'))) VIRTUAL;

CREATE INDEX idx_team_name ON participants(team_name_virtual);
```

### Validation Rules
Add custom validation in JavaScript:
```javascript
function validateCustomField(fieldKey, value, fieldDef) {
    if (fieldDef.is_required && !value) {
        return 'This field is required';
    }
    
    if (fieldDef.field_type === 'number' && isNaN(value)) {
        return 'Must be a valid number';
    }
    
    return null; // Valid
}
```

## Best Practices

1. **Field Keys:** Use lowercase with underscores (snake_case)
2. **Field Labels:** Use clear, descriptive names
3. **Required Fields:** Mark fields as required only if truly necessary
4. **Display Order:** Group related fields together
5. **Searchable:** Enable search only for fields that need it (performance)
6. **Field Types:** Choose appropriate types for data validation

## Troubleshooting

**Custom fields not appearing:**
- Check that fields are marked as "Active"
- Verify field definitions exist in database
- Clear browser cache and refresh

**Search not working:**
- Ensure field is marked as "Searchable"
- Check JSON syntax in database
- Verify MySQL version supports JSON functions (5.7+)

**Import errors:**
- Match CSV column names exactly to field keys
- Ensure select field values match defined options
- Check for JSON encoding issues

## Migration Path

If you have existing custom columns in the participants table:
```sql
-- Migrate existing column to custom_fields JSON
UPDATE participants 
SET custom_fields = JSON_SET(
    COALESCE(custom_fields, '{}'),
    '$.old_column_name',
    old_column_name
);

-- Then drop the old column
ALTER TABLE participants DROP COLUMN old_column_name;
```

## Future Enhancements

- [ ] Field groups/categories
- [ ] Conditional fields (show/hide based on other values)
- [ ] File upload field type
- [ ] Multi-select field type
- [ ] Field-level permissions
- [ ] Custom validation rules per field
- [ ] Calculated/formula fields
- [ ] Field history/audit log
