<?php
/**
 * Apply Common Translations Across All Views
 * This script automatically updates common text strings with translation calls
 */

require_once '../bootstrap/env.php';
require_once '../bootstrap/db.php';
require_once '../app/helpers/TranslationHelper.php';

// First, ensure common translations exist in database
echo "<h2>Applying Common Translations to Views</h2>";

$helper = new TranslationHelper();

// Add all common translations
$commonTranslations = [
    'common.actions' => 'Actions',
    'common.active' => 'Active', 
    'common.all' => 'All',
    'common.cancel' => 'Cancel',
    'common.confirm_password' => 'Confirm Password',
    'common.create' => 'Create',
    'common.created_at' => 'Created At',
    'common.delete' => 'Delete',
    'common.edit' => 'Edit',
    'common.email' => 'Email',
    'common.filter' => 'Filter',
    'common.inactive' => 'Inactive',
    'common.name' => 'Name',
    'common.no' => 'No',
    'common.password' => 'Password',
    'common.save' => 'Save',
    'common.search' => 'Search',
    'common.status' => 'Status',
    'common.submit' => 'Submit',
    'common.type' => 'Type',
    'common.update' => 'Update',
    'common.updated_at' => 'Updated At',
    'common.yes' => 'Yes'
];

echo "<p>Adding common translations to database...</p>";
$result = $helper->addEnglishTranslations($commonTranslations, 'common');

// Function to apply translations to a file
function applyTranslationsToFile($filePath, $replacements) {
    if (!file_exists($filePath)) {
        return false;
    }
    
    $content = file_get_contents($filePath);
    $originalContent = $content;
    
    foreach ($replacements as $search => $replace) {
        $content = str_replace($search, $replace, $content);
    }
    
    if ($content !== $originalContent) {
        file_put_contents($filePath, $content);
        return true;
    }
    
    return false;
}

// Define common text replacements
$commonReplacements = [
    // Form labels and buttons
    '>Name<' => '><?= __("common.name", "Name") ?><',
    '>Email<' => '><?= __("common.email", "Email") ?><',
    '>Status<' => '><?= __("common.status", "Status") ?><',
    '>Actions<' => '><?= __("common.actions", "Actions") ?><',
    '>Search<' => '><?= __("common.search", "Search") ?><',
    '>Filter<' => '><?= __("common.filter", "Filter") ?><',
    '>Create<' => '><?= __("common.create", "Create") ?><',
    '>Update<' => '><?= __("common.update", "Update") ?><',
    '>Save<' => '><?= __("common.save", "Save") ?><',
    '>Cancel<' => '><?= __("common.cancel", "Cancel") ?><',
    '>Delete<' => '><?= __("common.delete", "Delete") ?><',
    '>Edit<' => '><?= __("common.edit", "Edit") ?><',
    '>Type<' => '><?= __("common.type", "Type") ?><',
    '>Active<' => '><?= __("common.active", "Active") ?><',
    '>Inactive<' => '><?= __("common.inactive", "Inactive") ?><',
    '>Yes<' => '><?= __("common.yes", "Yes") ?><',
    '>No<' => '><?= __("common.no", "No") ?><',
    '>All<' => '><?= __("common.all", "All") ?><',
    
    // Button text
    'btn">Save</button>' => 'btn"><?= __("common.save", "Save") ?></button>',
    'btn">Update</button>' => 'btn"><?= __("common.update", "Update") ?></button>',
    'btn">Create</button>' => 'btn"><?= __("common.create", "Create") ?></button>',
    'btn">Cancel</a>' => 'btn"><?= __("common.cancel", "Cancel") ?></a>',
    'btn">Delete</button>' => 'btn"><?= __("common.delete", "Delete") ?></button>',
    'btn">Edit</button>' => 'btn"><?= __("common.edit", "Edit") ?></button>',
    
    // Placeholder text
    'placeholder="Search"' => 'placeholder="<?= __("common.search", "Search") ?>"',
    'placeholder="Name"' => 'placeholder="<?= __("common.name", "Name") ?>"',
    'placeholder="Email"' => 'placeholder="<?= __("common.email", "Email") ?>"',
    
    // Option values
    '"All"' => '"<?= __("common.all", "All") ?>"',
    '"Active"' => '"<?= __("common.active", "Active") ?>"',
    '"Inactive"' => '"<?= __("common.inactive", "Inactive") ?>"',
    "'Active'" => "'<?= __('common.active', 'Active') ?>'",
    "'Inactive'" => "'<?= __('common.inactive', 'Inactive') ?>'",
    "'All'" => "'<?= __('common.all', 'All') ?>'",
];

$viewsDir = __DIR__ . '/../app/Views';
$modifiedFiles = [];

// Function to recursively scan directories
function scanViews($dir, &$files) {
    $items = scandir($dir);
    foreach ($items as $item) {
        if ($item === '.' || $item === '..') continue;
        
        $path = $dir . '/' . $item;
        if (is_dir($path)) {
            scanViews($path, $files);
        } elseif (pathinfo($item, PATHINFO_EXTENSION) === 'php') {
            $files[] = $path;
        }
    }
}

$viewFiles = [];
scanViews($viewsDir, $viewFiles);

echo "<h3>Processing View Files:</h3>";

foreach ($viewFiles as $file) {
    $relativePath = str_replace($viewsDir . '/', '', $file);
    
    // Skip files that already have heavy translation usage to avoid conflicts
    $skipFiles = ['layout.php', 'languages/translations.php'];
    $skip = false;
    foreach ($skipFiles as $skipFile) {
        if (strpos($relativePath, $skipFile) !== false) {
            $skip = true;
            break;
        }
    }
    
    if ($skip) {
        echo "<p>⏭️ Skipped: $relativePath (already has translations)</p>";
        continue;
    }
    
    if (applyTranslationsToFile($file, $commonReplacements)) {
        $modifiedFiles[] = $relativePath;
        echo "<p>✅ Updated: $relativePath</p>";
    } else {
        echo "<p>⚪ No changes: $relativePath</p>";
    }
}

echo "<hr>";
echo "<h3>Summary:</h3>";
echo "<p><strong>✅ Modified " . count($modifiedFiles) . " files</strong></p>";

if (!empty($modifiedFiles)) {
    echo "<details><summary>Modified Files:</summary><ul>";
    foreach ($modifiedFiles as $file) {
        echo "<li>$file</li>";
    }
    echo "</ul></details>";
}

echo "<p><strong>All common translations have been applied!</strong></p>";
echo "<p><a href='/samanta_crm/languages'>Manage Translations</a> | ";
echo "<a href='/samanta_crm/public/test_auto_translations.php'>Test Auto-Creation</a></p>";
?>
