<?php
/**
 * Add New Translations Script
 * Demonstrates both manual and automatic translation creation methods
 */

require_once '../bootstrap/env.php';
require_once '../bootstrap/db.php';
require_once '../app/helpers/TranslationHelper.php';
require_once '../app/helpers/translation.php';

session_start();
if (!isset($_SESSION['language'])) {
    $_SESSION['language'] = 'en';
}

echo "<h2>Translation Methods Demonstration</h2>";
echo "<p>Auto-creation is: <strong>" . (AUTO_CREATE_TRANSLATIONS ? 'ENABLED' : 'DISABLED') . "</strong></p>";

echo "<hr>";

// Method 1: Manual bulk addition using TranslationHelper
echo "<h3>Method 1: Manual Bulk Addition</h3>";

$helper = new TranslationHelper();

$productTranslations = [
    'products.export.csv' => [
        'en' => 'Export CSV',
        'es' => 'Exportar CSV',
        'fr' => 'Exporter CSV'
    ],
    'products.import.success' => [
        'en' => 'Products imported successfully',
        'es' => 'Productos importados exitosamente',
        'fr' => 'Produits importés avec succès'
    ]
];

echo "<p>Adding translations manually for multiple languages...</p>";

foreach ($productTranslations as $key => $translations) {
    $result = $helper->addTranslationKey($key, $translations, 'products');
    
    foreach ($result['success'] as $success) {
        echo "<p style='color: green;'>✓ $success</p>";
    }
    
    foreach ($result['errors'] as $error) {
        echo "<p style='color: red;'>✗ $error</p>";
    }
}

echo "<hr>";

// Method 2: Automatic creation by just using the keys
echo "<h3>Method 2: Automatic Creation (Just Use The Keys!)</h3>";
echo "<p>These translations will be created automatically when first used:</p>";

// Just use new translation keys - they'll be created automatically!
$autoKeys = [
    'products.quick_edit' => 'Quick Edit',
    'products.bulk_export' => 'Bulk Export', 
    'orders.track_shipment' => 'Track Shipment',
    'orders.generate_invoice' => 'Generate Invoice',
    'customers.view_history' => 'View History',
    'dashboard.recent_orders' => 'Recent Orders'
];

foreach ($autoKeys as $key => $expectedValue) {
    $result = __($key, $expectedValue);
    echo "<p>✨ <code>__('$key', '$expectedValue')</code> → <strong>$result</strong></p>";
}

echo "<hr>";

// Method 3: English-only quick additions
echo "<h3>Method 3: English-Only Quick Addition</h3>";

$englishOnly = [
    'reports.weekly_sales' => 'Weekly Sales Report',
    'reports.export_data' => 'Export Data',
    'settings.backup_restore' => 'Backup & Restore'
];

$result = $helper->addEnglishTranslations($englishOnly, 'reports');
foreach ($result as $key => $keyResult) {
    foreach ($keyResult['success'] as $success) {
        echo "<p style='color: green;'>✓ $success</p>";
    }
}

echo "<hr>";

echo "<h3>Summary of Methods:</h3>";
echo "<div style='background: #f8f9fa; padding: 15px; border-radius: 5px;'>";
echo "<h4>🔧 Method 1: Manual Helper (Best for bulk/multiple languages)</h4>";
echo "<pre><code>\$helper = new TranslationHelper();
\$helper->addTranslationKey('key', ['en' => 'English', 'es' => 'Spanish']);</code></pre>";

echo "<h4>✨ Method 2: Auto-Creation (Best for development)</h4>";
echo "<pre><code>// Just use it - will be created automatically!
echo __('products.new_feature', 'New Feature');</code></pre>";

echo "<h4>⚡ Method 3: English-Only Helper (Best for initial setup)</h4>";
echo "<pre><code>\$helper->addEnglishTranslations(['key' => 'English text']);</code></pre>";
echo "</div>";

echo "<p><strong>All methods completed! Check your translation manager.</strong></p>";
echo "<p><a href='/samanta_crm/languages' style='background: #007bff; color: white; padding: 8px 16px; text-decoration: none; border-radius: 4px;'>Manage Translations</a></p>";
echo "<p><a href='/samanta_crm/public/test_auto_translations.php'>Test Auto-Creation Feature</a></p>";
?>
