<?php

class Language
{
    private $pdo;
    
    public function __construct()
    {
        $this->pdo = $GLOBALS['pdo'];
    }
    
    public function create($data)
    {
        $sql = "INSERT INTO languages (code, name, native_name, flag, is_active, is_default, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())";
        $stmt = $this->pdo->prepare($sql);
        
        return $stmt->execute([
            $data['code'],
            $data['name'],
            $data['native_name'],
            $data['flag'] ?? '',
            $data['is_active'] ?? 1,
            $data['is_default'] ?? 0
        ]);
    }
    
    public function findByCode($code)
    {
        $sql = "SELECT * FROM languages WHERE code = ?";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$code]);
        
        return $stmt->fetch();
    }
    
    public function findById($id)
    {
        $sql = "SELECT * FROM languages WHERE id = ?";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$id]);
        
        return $stmt->fetch();
    }
    
    public function getAll($activeOnly = false)
    {
        $sql = "SELECT * FROM languages";
        if ($activeOnly) {
            $sql .= " WHERE is_active = 1";
        }
        $sql .= " ORDER BY is_default DESC, name ASC";
        
        $stmt = $this->pdo->query($sql);
        return $stmt->fetchAll();
    }
    
    public function getActive()
    {
        return $this->getAll(true);
    }
    
    public function getDefault()
    {
        $sql = "SELECT * FROM languages WHERE is_default = 1 LIMIT 1";
        $stmt = $this->pdo->query($sql);
        $result = $stmt->fetch();
        
        // If no default is set, return English or first available
        if (!$result) {
            $result = $this->findByCode('en') ?: $this->getAll()[0] ?? null;
        }
        
        return $result;
    }
    
    public function setDefault($id)
    {
        // First, remove default from all languages
        $sql = "UPDATE languages SET is_default = 0";
        $this->pdo->exec($sql);
        
        // Then set the new default
        $sql = "UPDATE languages SET is_default = 1 WHERE id = ?";
        $stmt = $this->pdo->prepare($sql);
        
        return $stmt->execute([$id]);
    }
    
    public function update($id, $data)
    {
        $sql = "UPDATE languages SET name = ?, native_name = ?, flag = ?, is_active = ?, updated_at = NOW() WHERE id = ?";
        $stmt = $this->pdo->prepare($sql);
        
        return $stmt->execute([
            $data['name'],
            $data['native_name'],
            $data['flag'] ?? '',
            $data['is_active'] ?? 1,
            $id
        ]);
    }
    
    public function delete($id)
    {
        // Don't allow deleting the default language
        $language = $this->findById($id);
        if ($language && $language['is_default']) {
            return false;
        }
        
        // Check if any translations exist for this language
        $sql = "SELECT COUNT(*) FROM translations WHERE language_id = ?";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([$id]);
        
        if ($stmt->fetchColumn() > 0) {
            // Don't delete language with existing translations
            return false;
        }
        
        $sql = "DELETE FROM languages WHERE id = ?";
        $stmt = $this->pdo->prepare($sql);
        
        return $stmt->execute([$id]);
    }
}
