# Responsive Table Pattern for Smartphones and Tablets

## Goal
Make data tables functional and readable on smartphones and tablets while maintaining desktop functionality. Some elements can wrap to new rows for better mobile experience.

## Implementation Pattern

### 1. Wrap Table in Responsive Container
```html
<div class="card-body pt-0">
  <div class="table-responsive">
    <table class="table table-sm table-hover align-middle data-table mb-0" id="yourTable">
      <!-- table content -->
    </table>
  </div>
</div>
```

### 2. Progressive Column Hiding (Bootstrap Responsive Classes)

Use these classes on `<th>` and corresponding `<td>` elements:

- **Hide on phones only**: `d-none d-md-table-cell`
- **Hide on phones & tablets**: `d-none d-lg-table-cell`  
- **Hide on all but large desktops**: `d-none d-xl-table-cell`

**Priority for hiding:**
1. First hide: Order numbers, timestamps, secondary status indicators
2. Then hide: Less critical metadata (assigned user, item counts)
3. Keep always: ID, main name/title, primary status, action buttons

### 3. Mobile Info Wrapping Pattern

For important hidden info, display it under the main column on mobile:

```html
<td>
  <strong class="small"><?= htmlspecialchars($item['name']) ?></strong>
  
  <!-- Mobile-only info (phones) -->
  <div class="d-md-none small text-muted mt-1">
    <span class="badge bg-primary me-1"><?= $item['status'] ?></span>
    <span class="badge bg-success"><?= $item['secondary_info'] ?></span>
  </div>
  
  <!-- Tablet-only info (tablets but not desktop) -->
  <div class="d-lg-none small text-muted mt-1">
    <i class="fas fa-icon me-1"></i>
    Additional details: <?= $item['details'] ?>
  </div>
</td>
```

### 4. Responsive Action Buttons

Hide less critical actions on small screens:

```html
<td class="text-end">
  <div class="btn-group btn-group-sm">
    <!-- Always visible -->
    <a class="btn btn-outline-secondary" href="/view/<?= $id ?>">
      <i class="fas fa-eye"></i>
    </a>
    <!-- Hide on extra-small screens -->
    <a class="btn btn-outline-primary d-none d-sm-inline-block" href="/edit/<?= $id ?>">
      <i class="fas fa-edit"></i>
    </a>
    <form method="POST" class="d-inline d-none d-sm-inline-block">
      <button class="btn btn-outline-danger">
        <i class="fas fa-trash"></i>
      </button>
    </form>
  </div>
</td>
```

### 5. Compact Date/Time Format

For better mobile fit:

```html
<td class="text-nowrap small">
  <?= date('d/m/y', strtotime($date)) ?>
  <br>
  <span class="text-muted" style="font-size:0.7rem">
    <?= date('H:i', strtotime($date)) ?>
  </span>
</td>
```

### 6. Add Table Header Styling

```html
<thead class="table-light">
  <tr>
    <th>Column 1</th>
    <!-- ... -->
  </tr>
</thead>
```

## Complete Example Structure

```html
<div class="card">
  <div class="card-header py-2">
    <span class="small fw-bold">
      <i class="fas fa-table me-1"></i><?= __('Table Title') ?>
    </span>
  </div>
  <div class="card-body pt-0">
    <div class="table-responsive">
      <table class="table table-sm table-hover align-middle data-table mb-0">
        <thead class="table-light">
          <tr>
            <th>ID</th>
            <th>Name</th>
            <th class="d-none d-md-table-cell">Category</th>
            <th class="d-none d-lg-table-cell">Details</th>
            <th class="d-none d-xl-table-cell">Order</th>
            <th class="text-end">Actions</th>
          </tr>
        </thead>
        <tbody>
          <?php foreach($items as $item): ?>
          <tr>
            <td><code class="small"><?= $item['id'] ?></code></td>
            <td>
              <strong class="small"><?= htmlspecialchars($item['name']) ?></strong>
              <!-- Mobile wrapped info -->
              <div class="d-md-none small text-muted mt-1">
                <span class="badge bg-primary"><?= $item['category'] ?></span>
              </div>
            </td>
            <td class="d-none d-md-table-cell">
              <span class="badge bg-primary"><?= $item['category'] ?></span>
            </td>
            <td class="d-none d-lg-table-cell"><?= $item['details'] ?></td>
            <td class="d-none d-xl-table-cell"><?= $item['order'] ?></td>
            <td class="text-end">
              <div class="btn-group btn-group-sm">
                <a class="btn btn-outline-secondary" href="/view/<?= $item['id'] ?>">
                  <i class="fas fa-eye"></i>
                </a>
                <a class="btn btn-outline-primary d-none d-sm-inline-block" href="/edit/<?= $item['id'] ?>">
                  <i class="fas fa-edit"></i>
                </a>
              </div>
            </td>
          </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>
  <div class="card-footer py-2">
    <div class="small text-muted">
      <?= __('common.total', 'Total') ?>: <?= count($items) ?> <?= __('common.items', 'items') ?>
    </div>
  </div>
</div>
```

## Bootstrap Breakpoints Reference

- `xs` (< 576px): Extra small devices (phones)
- `sm` (≥ 576px): Small devices (landscape phones)
- `md` (≥ 768px): Medium devices (tablets)
- `lg` (≥ 992px): Large devices (desktops)
- `xl` (≥ 1200px): Extra large devices (large desktops)
- `xxl` (≥ 1400px): Extra extra large devices

## Key Principles

1. **Always keep essential info visible**: ID, name/title, primary status, at least one action button
2. **Progressive enhancement**: Add more columns as screen size increases
3. **Wrap don't hide critical info**: If data is important, show it wrapped on mobile rather than hiding it
4. **Use smaller fonts on mobile**: `class="small"` for better fit
5. **Horizontal scrolling as fallback**: `table-responsive` allows scrolling if content still doesn't fit
6. **Touch-friendly buttons**: Keep buttons at least 44x44px for easy tapping (btn-sm is fine)
7. **Test at multiple breakpoints**: Phone portrait, phone landscape, tablet portrait, tablet landscape, desktop

## Testing Checklist

- [ ] Table readable on phone (portrait)
- [ ] Table readable on phone (landscape)
- [ ] Table readable on tablet (portrait)
- [ ] Table readable on tablet (landscape)
- [ ] All critical data accessible on all screen sizes
- [ ] Action buttons easily tappable
- [ ] No horizontal overflow issues
- [ ] Text remains readable (not too small)
- [ ] Badges and icons render properly
