# Unit Conversion System

## Overview
The system allows automatic conversion between related units (e.g., g ↔ kg, ml ↔ l).

## Database Schema
```sql
units table:
- base_unit (INT) - References the base unit for conversion chain
- conversion_factor (DECIMAL) - Multiply by this to convert to base unit
```

## Conversion Examples
- **1g → kg**: 1 × 0.001 = 0.001 kg
- **1000g → kg**: 1000 × 0.001 = 1 kg
- **3520g → kg**: 3520 × 0.001 = 3.52 kg (auto-formatted)

## Usage in Code

### 1. Simple Formatting (Auto-conversion)
```php
<?= formatUnit(3520, 'g') ?>
// Output: "3.52 kg" (automatically converts g to kg for values >= 1000)

<?= formatUnit(750, 'g') ?>
// Output: "750 g" (stays in grams)

<?= formatUnit(2500, 'ml') ?>
// Output: "2.5 l" (auto-converts to liters)
```

### 2. Force Specific Unit
```php
<?= formatUnit(500, 'g', ['forceUnit' => 'kg']) ?>
// Output: "0.5 kg"

<?= formatUnit(3520, 'g', ['forceUnit' => 'g', 'decimals' => 0]) ?>
// Output: "3 520 g"
```

### 3. HTML Formatting
```php
<?= UnitHelper::formatHtml(3520, 'g') ?>
// Output: "3.52 <small>kg</small>"
```

### 4. Manual Conversion
```php
$kg = convertUnit(1000, 'g', 'kg');
// Returns: 1.0

$grams = convertUnit(2.5, 'kg', 'g');
// Returns: 2500.0
```

### 5. Get Array with Details
```php
$result = UnitHelper::formatArray(3520, 'g');
// Returns: [
//   'value' => 3.52,
//   'unit' => 'kg',
//   'display' => '3.52 kg'
// ]
```

## Setup Instructions

### 1. Run Migration
```bash
# Execute the SQL migration
mysql -u your_user -p samanta_crm < database/migrations/add_unit_conversions.sql
```

### 2. Configure Units
In the Units management page, set up conversions:

**Weight Units:**
- kg (base unit) - base_unit: NULL, conversion_factor: NULL
- g - base_unit: kg, conversion_factor: 0.001
- mg - base_unit: kg, conversion_factor: 0.000001

**Volume Units:**
- l (base unit) - base_unit: NULL, conversion_factor: NULL
- ml - base_unit: l, conversion_factor: 0.001

### 3. Use in Views
```php
<!-- In ingredient stock display -->
<h4>
  <?= formatUnit($ingredientStock['total_stock'], $ingredientStock['unit']) ?>
</h4>

<!-- In order items -->
<?php foreach($items as $item): ?>
  <td><?= formatUnit($item['quantity'], $item['unit']) ?></td>
<?php endforeach; ?>
```

## Auto-Conversion Rules
The system automatically converts to larger units when:
- **g → kg**: value >= 1000
- **ml → l**: value >= 1000
- **mg → g**: value >= 1000

## API Reference

### formatUnit($value, $unit, $options = [])
Global helper function for formatting with auto-conversion.

**Parameters:**
- `$value` (float) - Numeric value
- `$unit` (string) - Unit code (g, kg, ml, l, etc.)
- `$options` (array) - Optional:
  - `decimals` (int) - Decimal places (default: 2)
  - `forceUnit` (string) - Force specific unit instead of auto-convert

**Returns:** Formatted string

### convertUnit($value, $fromUnit, $toUnit)
Global helper for direct conversion.

**Parameters:**
- `$value` (float) - Value to convert
- `$fromUnit` (string) - Source unit code
- `$toUnit` (string) - Target unit code

**Returns:** Converted value or null if not possible

### UnitHelper Class Methods
- `format($value, $unit, $options)` - Format with auto-conversion
- `convert($value, $fromUnit, $toUnit)` - Convert between units
- `formatArray($value, $unit, $options)` - Get detailed array
- `formatHtml($value, $unit, $options)` - HTML formatted output
- `isConvertible($unit1, $unit2)` - Check if units are convertible
