<?php
/**
 * Translation Management Helper
 * Functions to add, update, and manage translations programmatically
 */

require_once __DIR__ . '/../Models/Language.php';
require_once __DIR__ . '/../Models/Translation.php';

class TranslationHelper 
{
    private $translationModel;
    private $languageModel;
    
    public function __construct() 
    {
        $this->translationModel = new Translation();
        $this->languageModel = new Language();
    }
    
    /**
     * Add a single translation key to all active languages
     * @param string $key Translation key (e.g., 'products.new_feature')
     * @param array $translations Array of language_code => translation_value
     * @param string $category Category for organization (default: 'general')
     */
    public function addTranslationKey($key, $translations, $category = 'general') 
    {
        $result = ['success' => [], 'errors' => []];
        
        foreach ($translations as $langCode => $translation) {
            $language = $this->languageModel->findByCode($langCode);
            
            if (!$language) {
                $result['errors'][] = "Language '$langCode' not found";
                continue;
            }
            
            $success = $this->translationModel->create([
                'language_id' => $language['id'],
                'key' => $key,
                'value' => $translation,
                'category' => $category
            ]);
            
            if ($success) {
                $result['success'][] = "Added '$key' for language '$langCode'";
            } else {
                $result['errors'][] = "Failed to add '$key' for language '$langCode'";
            }
        }
        
        return $result;
    }
    
    /**
     * Add multiple translation keys at once
     * @param array $keyTranslations Array of [key => [lang_code => translation]]
     * @param string $category Category for all keys
     */
    public function addMultipleKeys($keyTranslations, $category = 'general') 
    {
        $results = [];
        
        foreach ($keyTranslations as $key => $translations) {
            $results[$key] = $this->addTranslationKey($key, $translations, $category);
        }
        
        return $results;
    }
    
    /**
     * Quick method to add English translations with automatic fallbacks
     * @param array $englishTranslations Array of key => english_text
     * @param string $category Category
     */
    public function addEnglishTranslations($englishTranslations, $category = 'general') 
    {
        $results = [];
        
        foreach ($englishTranslations as $key => $englishText) {
            // Add to English
            $translations = ['en' => $englishText];
            
            // You can add logic here to auto-translate or leave empty for other languages
            $results[$key] = $this->addTranslationKey($key, $translations, $category);
        }
        
        return $results;
    }
    
    /**
     * Update existing translation
     */
    public function updateTranslation($key, $languageCode, $newValue) 
    {
        $language = $this->languageModel->findByCode($languageCode);
        if (!$language) {
            return false;
        }
        
        return $this->translationModel->updateByKeyAndLanguage($key, $language['id'], $newValue);
    }
    
    /**
     * Get all translation keys for a category
     */
    public function getKeysByCategory($category) 
    {
        return $this->translationModel->getAllKeys();
    }
}

// Helper function for quick access
function addTranslations($key, $translations, $category = 'general') 
{
    $helper = new TranslationHelper();
    return $helper->addTranslationKey($key, $translations, $category);
}

