<?php
/**
 * Unit Formatter Helper
 * Provides easy-to-use unit conversion and formatting functions
 */
class UnitHelper {
  private static $unitModel = null;
  
  private static function getUnitModel() {
    if (self::$unitModel === null) {
      require_once __DIR__ . '/../Models/Unit.php';
      self::$unitModel = new Unit();
    }
    return self::$unitModel;
  }
  
  /**
   * Format a value with automatic unit conversion
   * Example: formatUnit(3520, 'g') returns "3.52 kg"
   * 
   * @param float $value - The numeric value
   * @param string $unit - The unit code (g, kg, ml, l, etc.)
   * @param array $options - Optional: ['decimals' => 2, 'forceUnit' => 'kg']
   * @return string - Formatted string like "3.52 kg"
   */
  public static function format($value, $unit, $options = []) {
    $result = self::getUnitModel()->formatWithBestUnit($value, $unit, $options);
    return $result['display'];
  }
  
  /**
   * Convert value between units
   * Example: convert(1000, 'g', 'kg') returns 1.0
   * 
   * @param float $value - The value to convert
   * @param string $fromUnit - Source unit code
   * @param string $toUnit - Target unit code
   * @return float|null - Converted value or null if not possible
   */
  public static function convert($value, $fromUnit, $toUnit) {
    return self::getUnitModel()->convert($value, $fromUnit, $toUnit);
  }
  
  /**
   * Get formatted array with value and unit separated
   * Example: formatArray(3520, 'g') returns ['value' => 3.52, 'unit' => 'kg', 'display' => '3.52 kg']
   * 
   * @param float $value - The numeric value
   * @param string $unit - The unit code
   * @param array $options - Optional settings
   * @return array - ['value', 'unit', 'display']
   */
  public static function formatArray($value, $unit, $options = []) {
    return self::getUnitModel()->formatWithBestUnit($value, $unit, $options);
  }
  
  /**
   * Format for HTML display with unit in <small> tag
   * Example: formatHtml(3520, 'g') returns "3.52 <small>kg</small>"
   * 
   * @param float $value - The numeric value
   * @param string $unit - The unit code
   * @param array $options - Optional settings
   * @return string - HTML formatted string
   */
  public static function formatHtml($value, $unit, $options = []) {
    $result = self::getUnitModel()->formatWithBestUnit($value, $unit, $options);
    return number_format($result['value'], $options['decimals'] ?? 2, ',', ' ') . ' <small>' . htmlspecialchars($result['unit']) . '</small>';
  }
  
  /**
   * Check if two units are convertible
   * 
   * @param string $unit1 - First unit code
   * @param string $unit2 - Second unit code
   * @return bool - True if convertible
   */
  public static function isConvertible($unit1, $unit2) {
    $result = self::convert(1, $unit1, $unit2);
    return $result !== null;
  }
}

/**
 * Global helper function for quick unit formatting
 * Usage in views: <?= formatUnit(3520, 'g') ?> outputs "3.52 kg"
 */
function formatUnit($value, $unit, $options = []) {
  return UnitHelper::format($value, $unit, $options);
}

/**
 * Global helper function for unit conversion
 * Usage: convertUnit(1000, 'g', 'kg') returns 1.0
 */
function convertUnit($value, $fromUnit, $toUnit) {
  return UnitHelper::convert($value, $fromUnit, $toUnit);
}
