# Panel Layout Standardization Guide

This guide defines the standard card-based layout structure for all list/index views in the CRM system.

## Structure Overview

All list/index views should follow this three-part structure:
1. **Page Header** - Title and primary actions
2. **Filters Section** - Collapsible filter card
3. **Data Table Section** - Main content card with table

---

## 1. PAGE HEADER

Place before filters section. Contains page title and primary action buttons.

```php
<div class="d-flex justify-content-between align-items-center mb-4">
    <h1 class="h3 mb-0">
        <i class="fas fa-[ICON]"></i> <?= __('[namespace].[key]', 'Title') ?>
    </h1>
    <div class="btn-group">
        <a href="/samanta_crm/[module]/create" class="btn btn-primary">
            <i class="fas fa-plus"></i> <?= __('common.add', 'Add') ?>
        </a>
        <a href="/samanta_crm/[module]/import" class="btn btn-outline-secondary">
            <i class="fas fa-file-import"></i> <?= __('common.import', 'Import') ?>
        </a>
    </div>
</div>
```

**Key Points:**
- Use `h1` with `h3` class for title
- Use appropriate FontAwesome icon
- Use translation functions `__()` for all text
- Use regular button sizes (no `-sm`)
- Group action buttons in `btn-group`

---

## 2. FILTERS SECTION

Collapsible card with filter inputs. Must have class `data-filters`.

```php
<form class="row g-2 align-items-end data-filters">
  <div class="card mb-3">
    <div class="card-header py-2 d-flex justify-content-between align-items-center">
      <span class="small fw-bold">
        <i class="fas fa-filter me-1"></i><?= __('common.filters', 'Filters') ?>
      </span>
      <button class="btn btn-sm btn-outline-secondary" type="button" 
              data-bs-toggle="collapse" data-bs-target="#filtersWrap">
        <i class="fas fa-chevron-down"></i>
      </button>
    </div>
    <div class="collapse show" id="filtersWrap">
      <div class="card-body py-3">
        <div class="row g-2 align-items-end">
          
          <!-- Example: Search Input -->
          <div class="col-md-2 col-6">
            <label class="form-label small mb-0 d-none">
              <?= __('common.search', 'Search') ?>
            </label>
            <input type="text" name="q" value="<?= h($filter_q ?? '') ?>" 
                   class="form-control" 
                   placeholder="<?= __('common.search', 'Search') ?>">
          </div>
          
          <!-- Example: Category Dropdown -->
          <div class="col-md-2 col-6">
            <label class="form-label small mb-0 d-none">
              <?= __('common.category', 'Category') ?>
            </label>
            <select name="category" class="form-select">
              <option value=""><?= __('common.all', 'All') ?></option>
              <?php foreach($categories as $c): ?>
                <option value="<?= $c['id'] ?>" 
                        <?= $filter_category==$c['id']?'selected':'' ?>>
                  <?= h($c['name'] ?? '') ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          
          <!-- Example: Status Dropdown -->
          <div class="col-md-2 col-6">
            <label class="form-label small mb-0 d-none">
              <?= __('common.status', 'Status') ?>
            </label>
            <select name="status" class="form-select">
              <option value="all"><?= __('common.all', 'All') ?></option>
              <option value="active" <?= $filter_status=='active'?'selected':'' ?>>
                <?= __('common.active', 'Active') ?>
              </option>
              <option value="notactive" <?= $filter_status=='notactive'?'selected':'' ?>>
                <?= __('common.inactive', 'Inactive') ?>
              </option>
            </select>
          </div>
          
          <!-- Example: Page Size -->
          <div class="col-md-2 col-6">
            <label class="form-label small mb-0 d-none">
              <?= __('common.page_size', 'Page Size') ?>
            </label>
            <select name="ps" class="form-select">
              <?php foreach([10,25,50,100,200] as $ps): ?>
                <option value="<?= $ps ?>" <?= ($pageSize??25)==$ps?'selected':'' ?>>
                  <?= $ps ?>
                </option>
              <?php endforeach; ?>
            </select>
          </div>
          
          <!-- Submit Button -->
          <div class="col-auto">
            <button class="btn btn-outline-secondary">
              <?= __('common.filter', 'Filter') ?>
            </button>
          </div>
          
          <!-- Hidden fields for sorting -->
          <input type="hidden" name="sort" value="<?= h($sort) ?>">
          <input type="hidden" name="dir" value="<?= h($dir) ?>">
          
        </div>
      </div>
    </div>
  </div>
</form>
```

**Key Points:**
- Form wraps the entire card
- Labels hidden with `d-none` (use placeholders instead)
- Use `form-control` and `form-select` (no `-sm`)
- Collapse starts expanded with `show` class
- Use `py-3` for card-body padding
- Responsive columns: `col-md-2 col-6` for filters

---

## 3. DATA TABLE SECTION

Card containing the data table with header controls and footer pagination.

```php
<div class="card">
  <!-- CARD HEADER: Title and Controls -->
  <div class="card-header py-2 d-flex justify-content-between align-items-center">
    <span class="small fw-bold">
      <i class="fas fa-table me-1"></i><?= __('[namespace].[title]', 'Table Title') ?>
    </span>
    <div class="d-flex gap-2">
      <!-- Bulk Delete Form -->
      <form method="POST" action="/samanta_crm/[module]/bulk-delete" 
            id="bulkForm" onsubmit="return confirm('Delete selected items?');">
        <button type="submit" class="btn btn-outline-danger" disabled id="bulkDeleteBtn">
          <i class="fas fa-trash"></i> <?= __('common.remove_selected', 'Remove Selected') ?>
        </button>
      </form>
      
      <!-- Column Chooser -->
      <div class="dropdown">
        <button class="btn btn-outline-secondary dropdown-toggle" type="button" 
                data-bs-toggle="dropdown">
          <?= __('common.columns', 'Columns') ?>
        </button>
        <div class="dropdown-menu p-2 small" style="min-width:220px" id="colChooser">
          <div class="text-muted small mb-1">
            <?= __('common.show_hide', 'Show / Hide') ?>
          </div>
          <div id="colChooserList"></div>
          <hr class="my-1">
          <button type="button" class="btn btn-sm btn-outline-primary w-100" 
                  id="applyFiltersBtn">
            <?= __('common.apply_filters', 'Apply Filters') ?>
          </button>
        </div>
      </div>
    </div>
  </div>
  
  <!-- CARD BODY: Table Content -->
  <div class="card-body pt-0">
    <!-- Selection Counter -->
    <div class="small text-muted px-3 pt-2" id="selCount"></div>
    
    <!-- Table Form -->
    <form method="POST" action="/samanta_crm/[module]/bulk-delete" id="bulkFormTable">
      <table class="table table-sm table-hover align-middle data-table mb-0" id="[tableName]">
        <thead>
          <tr>
            <th data-col="select" style="width:28px">
              <input type="checkbox" id="checkAll">
            </th>
            <th data-col="code">
              <a href="<?= $queryBase('code') ?>" class="text-decoration-none">
                <?= __('common.code', 'Code') ?>
                <?= $sort==='code'?' '.($dir==='asc'?'▲':'▼'):'' ?>
              </a>
            </th>
            <th data-col="name">
              <a href="<?= $queryBase('name') ?>" class="text-decoration-none">
                <?= __('common.name', 'Name') ?>
                <?= $sort==='name'?' '.($dir==='asc'?'▲':'▼'):'' ?>
              </a>
            </th>
            <!-- More columns... -->
            <th data-col="actions"></th>
          </tr>
          <tr id="filterRow" class="align-middle"></tr>
        </thead>
        <tbody id="[tableName]Body" data-sortable="1">
          <?php foreach($items as $item): ?>
          <tr data-id="<?= $item['id'] ?>">
            <td data-col="select">
              <input type="checkbox" name="ids[]" value="<?= $item['id'] ?>" 
                     class="rowCheck">
            </td>
            <td data-col="code" class="editable" data-field="code">
              <span class="cell-value"><?= h($item['code'] ?? '') ?></span>
            </td>
            <td data-col="name" class="editable" data-field="name">
              <span class="cell-value"><?= h($item['name'] ?? '') ?></span>
            </td>
            <!-- More cells... -->
            <td data-col="actions" class="text-end">
              <a href="/samanta_crm/[module]/<?= $item['id'] ?>/edit" 
                 class="btn btn-sm btn-outline-primary">
                <?= __('common.edit', 'Edit') ?>
              </a>
              <form action="/samanta_crm/[module]/<?= $item['id'] ?>/delete" 
                    method="POST" class="d-inline" 
                    onsubmit="return confirm('Delete?')">
                <button class="btn btn-sm btn-outline-danger">
                  <?= __('common.delete', 'Del') ?>
                </button>
              </form>
            </td>
          </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </form>
  </div>
  
  <!-- CARD FOOTER: Pagination -->
  <div class="card-footer py-2 d-flex justify-content-between align-items-center">
    <span></span>
    <?php if(($pages??1) > 1): ?>
    <nav aria-label="Pagination">
      <ul class="pagination pagination-sm mb-0">
        <?php
          $baseParams = function($p) use($filter_q_str,$filter_category_str,
                                         $filter_status_str,$sort,$dir,$pageSize){
            return '?q='.urlencode($filter_q_str)
                  .'&category='.urlencode($filter_category_str)
                  .'&status='.urlencode($filter_status_str)
                  .'&sort='.urlencode($sort)
                  .'&dir='.urlencode($dir)
                  .'&ps='.urlencode($pageSize)
                  .'&page='.$p;
          };
          for($p=1;$p<=$pages;$p++):
        ?>
          <li class="page-item <?= $p==($page??1)?'active':'' ?>">
            <a class="page-link" href="<?= $baseParams($p) ?>"><?= $p ?></a>
          </li>
        <?php endfor; ?>
      </ul>
    </nav>
    <?php endif; ?>
    <div class="small text-muted">
      <?= __('common.total', 'Total') ?>: <?= (int)$total ?> 
      <?= __('common.items', 'items') ?>
    </div>
  </div>
</div>
```

**Key Points:**
- Table must have class `data-table` and `mb-0`
- Card body uses `pt-0` to reduce top padding
- Selection counter inside card-body, above table
- Pagination in card-footer
- Use `data-col` attributes for column visibility control
- Use `data-field` attributes for inline editing
- Action buttons use `btn-sm` sizing

---

## CSS Classes Summary

### Layout Classes
- **Page Header**: `d-flex justify-content-between align-items-center mb-4`
- **Card Header**: `py-2 d-flex justify-content-between align-items-center`
- **Card Body**: `pt-0` (for table cards), `py-3` (for filter cards)
- **Card Footer**: `py-2 d-flex justify-content-between align-items-center`

### Component Classes
- **Filters Form**: `data-filters` (custom identifier)
- **Data Table**: `data-table` (custom identifier) + `table table-sm table-hover align-middle mb-0`
- **Filter Labels**: `form-label small mb-0 d-none`
- **Filter Inputs**: `form-control` or `form-select` (no sizing modifiers)
- **Header Buttons**: `btn btn-[variant]` (no `-sm`)
- **Table Action Buttons**: `btn btn-sm btn-outline-[variant]`

### Responsive Columns
- **Filter Inputs**: `col-md-2 col-6` (2 columns on desktop, 6 on mobile)
- **Filter Button**: `col-auto`

---

## Icon Mapping

Common icons for different modules:

| Module | Icon Class |
|--------|------------|
| Products | `fas fa-box` |
| Customers | `fas fa-users` |
| Categories | `fas fa-layer-group` |
| Orders | `fas fa-receipt` |
| Stock | `fas fa-warehouse` |
| Reports | `fas fa-chart-bar` |
| Settings | `fas fa-cogs` |
| Languages | `fas fa-language` |
| Users | `fas fa-user` |

---

## Translation Keys

Standard translation keys to use:

```php
// Common
__('common.filters', 'Filters')
__('common.search', 'Search')
__('common.filter', 'Filter')
__('common.all', 'All')
__('common.add', 'Add')
__('common.import', 'Import')
__('common.export', 'Export')
__('common.edit', 'Edit')
__('common.delete', 'Del')
__('common.remove_selected', 'Remove Selected')
__('common.columns', 'Columns')
__('common.show_hide', 'Show / Hide')
__('common.apply_filters', 'Apply Filters')
__('common.total', 'Total')
__('common.items', 'items')
__('common.page_size', 'Page Size')
__('common.status', 'Status')
__('common.active', 'Active')
__('common.inactive', 'Inactive')
__('common.category', 'Category')
__('common.name', 'Name')
__('common.code', 'Code')
```

---

## Quick Checklist

When converting a view to this standard:

- [ ] Page header with h1.h3 title and action buttons
- [ ] Filters wrapped in collapsible card with `data-filters` class
- [ ] Filter labels hidden with `d-none`
- [ ] Filter inputs use placeholders instead of visible labels
- [ ] Table wrapped in card with header, body (pt-0), and footer
- [ ] Bulk actions and column chooser in card header
- [ ] Selection counter in card body above table
- [ ] Pagination moved to card footer
- [ ] Table has `mb-0` class
- [ ] All text uses translation functions `__()`
- [ ] Button sizes: regular in headers, `-sm` in tables
- [ ] Form controls: regular size (no `-sm`)

---

## Example: Converting a Simple View

**Before:**
```php
<h3>Products</h3>
<a href="/products/create">Add</a>

<form>
  <input name="q" placeholder="Search">
  <button>Filter</button>
</form>

<table class="table">
  <!-- table content -->
</table>
```

**After:**
```php
<div class="d-flex justify-content-between align-items-center mb-4">
    <h1 class="h3 mb-0">
        <i class="fas fa-box"></i> <?= __('products.products', 'Products') ?>
    </h1>
    <div class="btn-group">
        <a href="/samanta_crm/products/create" class="btn btn-primary">
            <i class="fas fa-plus"></i> <?= __('common.add', 'Add') ?>
        </a>
    </div>
</div>

<form class="row g-2 align-items-end data-filters">
  <div class="card mb-3">
    <div class="card-header py-2 d-flex justify-content-between align-items-center">
      <span class="small fw-bold"><i class="fas fa-filter me-1"></i><?= __('common.filters', 'Filters') ?></span>
      <button class="btn btn-sm btn-outline-secondary" type="button" data-bs-toggle="collapse" data-bs-target="#filtersWrap">
        <i class="fas fa-chevron-down"></i>
      </button>
    </div>
    <div class="collapse show" id="filtersWrap">
      <div class="card-body py-3">
        <div class="row g-2 align-items-end">
          <div class="col-md-2 col-6">
            <label class="form-label small mb-0 d-none"><?= __('common.search', 'Search') ?></label>
            <input type="text" name="q" class="form-control" placeholder="<?= __('common.search', 'Search') ?>">
          </div>
          <div class="col-auto">
            <button class="btn btn-outline-secondary"><?= __('common.filter', 'Filter') ?></button>
          </div>
        </div>
      </div>
    </div>
  </div>
</form>

<div class="card">
  <div class="card-header py-2 d-flex justify-content-between align-items-center">
    <span class="small fw-bold"><i class="fas fa-table me-1"></i><?= __('products.products', 'Products') ?></span>
    <div class="d-flex gap-2">
      <!-- Controls -->
    </div>
  </div>
  <div class="card-body pt-0">
    <table class="table table-sm table-hover align-middle data-table mb-0">
      <!-- table content -->
    </table>
  </div>
  <div class="card-footer py-2 d-flex justify-content-between align-items-center">
    <span></span>
    <!-- Pagination -->
    <div class="small text-muted">Total: X items</div>
  </div>
</div>
```

---

## Notes

- This structure ensures consistency across all list views
- Improves mobile responsiveness with collapsible sections
- Provides better visual hierarchy with cards
- Maintains accessibility with proper semantic HTML
- All components are Bootstrap 5 compatible
- Uses FontAwesome 6 icons
- Supports RTL languages through translation system
