<?php
/**
 * Simple Translation Cleanup
 * Fixes the most common double-translation issues
 */

require_once '../bootstrap/env.php';

echo "<h2>Simple Translation Cleanup</h2>";

function simpleCleanup($filePath) {
    if (!file_exists($filePath)) {
        return false;
    }
    
    $content = file_get_contents($filePath);
    $originalContent = $content;
    
    // Simple string replacements for the most common issues
    $fixes = [
        // Fix the specific pattern we saw
        "'<?= __('common.all', 'All') ?>'" => "'All'",
        "'<?= __('common.active', 'Active') ?>'" => "'Active'", 
        "'<?= __('common.inactive', 'Inactive') ?>'" => "'Inactive'",
        '"<?= __("common.all", "All") ?>"' => '"All"',
        '"<?= __("common.active", "Active") ?>"' => '"Active"',
        '"<?= __("common.inactive", "Inactive") ?>"' => '"Inactive"',
        
        // Fix other nested patterns
        "__('common.all', '<?= __('common.all', 'All') ?>')" => "__('common.all', 'All')",
        "__('common.active', '<?= __('common.active', 'Active') ?>')" => "__('common.active', 'Active')",
        "__('common.inactive', '<?= __('common.inactive', 'Inactive') ?>')" => "__('common.inactive', 'Inactive')",
        '__("common.all", "<?= __("common.all", "All") ?>")' => '__("common.all", "All")',
        '__("common.active", "<?= __("common.active", "Active") ?>")' => '__("common.active", "Active")',
        '__("common.inactive", "<?= __("common.inactive", "Inactive") ?>")' => '__("common.inactive", "Inactive")',
    ];
    
    foreach ($fixes as $search => $replace) {
        $content = str_replace($search, $replace, $content);
    }
    
    if ($content !== $originalContent) {
        file_put_contents($filePath, $content);
        return true;
    }
    
    return false;
}

$viewsDir = __DIR__ . '/../app/Views';
$fixedFiles = [];

function scanAndFix($dir, &$fixedFiles) {
    $items = scandir($dir);
    foreach ($items as $item) {
        if ($item === '.' || $item === '..') continue;
        
        $path = $dir . '/' . $item;
        if (is_dir($path)) {
            scanAndFix($path, $fixedFiles);
        } elseif (pathinfo($item, PATHINFO_EXTENSION) === 'php') {
            $relativePath = str_replace(__DIR__ . '/../app/Views/', '', $path);
            if (simpleCleanup($path)) {
                $fixedFiles[] = $relativePath;
                echo "<p>🔧 Fixed: $relativePath</p>";
            }
        }
    }
}

echo "<h3>Scanning and fixing files...</h3>";
scanAndFix($viewsDir, $fixedFiles);

echo "<hr>";
echo "<h3>Summary:</h3>";
echo "<p><strong>🔧 Fixed " . count($fixedFiles) . " files</strong></p>";

if (!empty($fixedFiles)) {
    echo "<details><summary>Fixed Files:</summary><ul>";
    foreach ($fixedFiles as $file) {
        echo "<li>$file</li>";
    }
    echo "</ul></details>";
}

echo "<h3>✅ Common Translation Guidelines:</h3>";
echo "<div style='background: #f8f9fa; padding: 15px; border-radius: 5px;'>";
echo "<h4>✅ Correct Usage:</h4>";
echo "<pre>";
echo "// In PHP arrays:\n";
echo "\$statuses = [\n";
echo "    'all' => __('common.all', 'All') . ' Statuses',\n";
echo "    'active' => __('common.active', 'Active'),\n";
echo "    'inactive' => __('common.inactive', 'Inactive')\n";
echo "];\n\n";

echo "// In HTML output:\n";
echo "&lt;th&gt;&lt;?= __('common.name', 'Name') ?&gt;&lt;/th&gt;\n";
echo "&lt;button&gt;&lt;?= __('common.save', 'Save') ?&gt;&lt;/button&gt;\n\n";

echo "// In conditional output:\n";
echo "\$status === 'active' ? __('common.active', 'Active') : __('common.inactive', 'Inactive')\n";
echo "</pre>";

echo "<h4>❌ Avoid:</h4>";
echo "<pre>";
echo "// Don't nest PHP tags in strings:\n";
echo "__('common.all', '&lt;?= __('common.all', 'All') ?&gt;')  // ❌ WRONG\n";
echo "__('common.all', 'All')                                // ✅ CORRECT\n";
echo "</pre>";
echo "</div>";

echo "<p><strong>Translation mess cleaned up! Your views should work properly now.</strong></p>";
echo "<p><a href='/samanta_crm/users'>Test Users Page</a> | ";
echo "<a href='/samanta_crm/products'>Test Products Page</a></p>";
?>
